/* SexyAL - Simple audio abstraction library.

Copyright (c) 2005 Mednafen Team

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <poll.h>
#include <sys/time.h>
#include <time.h>
#include <alsa/asoundlib.h>
#include "../sexyal.h"
#include "../smallc.h"

#include "alsa.h"

typedef struct
{
	snd_pcm_t *alsa_pcm;

	uint32_t period_size;
} ADStruct;

void SexyALI_ALSA_EnumerateDevices(int (*func)(uint8_t *name, char *id, void *udata), void *udata)
{

}

static int Pause(SexyAL_device *device, int state)
{
 if(0)
  snd_pcm_pause(((ADStruct *)device->private_data)->alsa_pcm, state);
 else
 {
  snd_pcm_drop(((ADStruct *)device->private_data)->alsa_pcm);
 }
 return(0);
}

static int Clear(SexyAL_device *device)
{
 snd_pcm_drop(((ADStruct *)device->private_data)->alsa_pcm);

 return(1);
}

static inline int CheckState(ADStruct *ads)
{
 int ret = 0;

 switch(snd_pcm_state(ads->alsa_pcm))
 {
  case SND_PCM_STATE_RUNNING: break;
  case SND_PCM_STATE_SUSPENDED:
  case SND_PCM_STATE_XRUN: ret = 1;
                          snd_pcm_prepare(ads->alsa_pcm);
                          break;

  case SND_PCM_STATE_PREPARED: snd_pcm_start(ads->alsa_pcm); break;

  default: ret = 1; break;
 }
 return(ret);
}

static uint32_t RawCanWrite(SexyAL_device *device)
{
 ADStruct *ads = (ADStruct *)device->private_data;
 uint32_t ret;

 if(CheckState(ads))
 {
  return(device->buffering.totalsize);
 }

 snd_pcm_sframes_t delay;
 snd_pcm_delay(ads->alsa_pcm, &delay);

 ret = device->buffering.totalsize - delay;
 if(ret < 0) ret = 0;
 return(ret);
}

static uint32_t RawWrite(SexyAL_device *device, void *data, uint32_t len)
{
 ADStruct *ads = (ADStruct *)device->private_data;
 uint32_t ret = len;

 while(len > 0)
 {
  CheckState(ads);

  int snore = snd_pcm_writei(ads->alsa_pcm, data, len / (device->format.sampformat>>4) / device->format.channels);

  if(snore <= 0)
  {
   continue;
   // UUUhhhhmmmmmm.... probably only happens on underruns...
   //printf("ACKAGE: %d, %d\n", snore, ret);
   //if(snore == -EBADFD)
   //{
   // if(!snd_pcm_prepare(ads->alsa_pcm)) // If it succeeds, loop back around and try to continue writing our sound data
   //  continue;
   //}
   //return(ret * (device->format.sampformat>>4) * device->format.channels);
  }
  data = (uint8_t*)data + snore * (device->format.sampformat>>4) * device->format.channels;

  len -= snore * (device->format.sampformat>>4) * device->format.channels;
 }

 return(ret * (device->format.sampformat>>4) * device->format.channels);
}

static int RawClose(SexyAL_device *device)
{
 if(device)
 {
  if(device->private_data)
  {
   ADStruct *ads = (ADStruct *)device->private_data;
   snd_pcm_close(ads->alsa_pcm);
   free(device->private_data);
  }
  free(device);
  return(1);
 }
 return(0);
}

#define ALSA_TRY(func) { 	\
	int error; 	\
	error = func; 	\
	if(error) 	\
	{ 	\
	 printf("ALSA Error: %s %s\n", #func, snd_strerror(error)); 	\
	 if(hw_params)  snd_pcm_hw_params_free(hw_params);	\
	 if(alsa_pcm) snd_pcm_close(alsa_pcm);	\
	 if(ads) free(ads);	\
	 if(device) free(device);	\
	 return(0); 	\
	} }

SexyAL_device *SexyALI_ALSA_Open(const char *id, SexyAL_format *format, SexyAL_buffering *buffering)
{
 ADStruct *ads = NULL;
 SexyAL_device *device = NULL;
 snd_pcm_t *alsa_pcm = NULL;
 snd_pcm_hw_params_t *hw_params = NULL;

 snd_pcm_format_t sampformat;

 ALSA_TRY(snd_pcm_open(&alsa_pcm, id ? id : "hw:0", SND_PCM_STREAM_PLAYBACK, 0));
 ALSA_TRY(snd_pcm_hw_params_malloc(&hw_params));

 switch(format->sampformat)
 {
  case SEXYAL_FMT_PCMU8: sampformat = SND_PCM_FORMAT_U8; break;
  case SEXYAL_FMT_PCMS8: sampformat = SND_PCM_FORMAT_S8; break;
  case SEXYAL_FMT_PCMU16: sampformat = SND_PCM_FORMAT_U16; break;
  case SEXYAL_FMT_PCMS16: sampformat = SND_PCM_FORMAT_S16; break;

  default: sampformat = 0; break; // FIXME
 }

 ALSA_TRY(snd_pcm_hw_params_any(alsa_pcm, hw_params));
 ALSA_TRY(snd_pcm_hw_params_set_periods_integer(alsa_pcm, hw_params));

 ALSA_TRY(snd_pcm_hw_params_set_access(alsa_pcm, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED));
 ALSA_TRY(snd_pcm_hw_params_set_format(alsa_pcm, hw_params, sampformat));

 #if SND_LIB_VERSION >= 0x10009
 ALSA_TRY(snd_pcm_hw_params_set_rate_resample(alsa_pcm, hw_params, 0));
 #endif

 unsigned int rrate = format->rate;
 ALSA_TRY(snd_pcm_hw_params_set_rate_near(alsa_pcm, hw_params, &rrate, 0));
 format->rate = rrate;

 if(snd_pcm_hw_params_set_channels(alsa_pcm, hw_params, format->channels) != 0)
 {
  if(format->channels == 2)
  {
   puts("Warning:  Couldn't set stereo sound, trying mono instead...");
   format->channels = 1;
  }
  else if(format->channels == 1)
   format->channels = 2;

  ALSA_TRY(snd_pcm_hw_params_set_channels(alsa_pcm, hw_params, format->channels));
 }

 if(snd_pcm_hw_params_set_period_size(alsa_pcm, hw_params, 128, 0) != 0)
 {
  //puts("Zoom");
  if(snd_pcm_hw_params_set_period_size(alsa_pcm, hw_params, 256, 0) != 0)
  {
   puts("ALSA Warning:  Could not set period size to a nice value. :(");
  }
 }
 snd_pcm_uframes_t tmp_uft;
 tmp_uft = buffering->ms * format->rate / 1000;
 ALSA_TRY(snd_pcm_hw_params_set_buffer_size_near(alsa_pcm, hw_params, &tmp_uft));

 ALSA_TRY(snd_pcm_hw_params(alsa_pcm, hw_params));
 snd_pcm_uframes_t buffer_size, period_size;
 unsigned int periods;

 ALSA_TRY(snd_pcm_hw_params_get_period_size(hw_params, &period_size, NULL));
 ALSA_TRY(snd_pcm_hw_params_get_periods(hw_params, &periods, NULL));
 snd_pcm_hw_params_free(hw_params);

 buffer_size = period_size * periods;

 buffering->totalsize = buffer_size;
 buffering->ms = buffering->totalsize * 1000 / format->rate;
 buffering->latency = buffering->totalsize;

 device = (SexyAL_device *)calloc(1, sizeof(SexyAL_device));
 ads = (ADStruct *)calloc(1, sizeof(ADStruct));

 ads->alsa_pcm = alsa_pcm;
 ads->period_size = period_size;

 device->private_data = ads;
 device->RawWrite = RawWrite;
 device->RawCanWrite = RawCanWrite;
 device->RawClose = RawClose;
 device->Pause = Pause;
 device->Clear = Clear;

 memcpy(&device->buffering,buffering,sizeof(SexyAL_buffering));
 memcpy(&device->format,format,sizeof(SexyAL_format));

 ALSA_TRY(snd_pcm_prepare(alsa_pcm));

 return(device);
}

