#include "pcfx.h"
#include "interrupt.h"
#include "timer.h"
#include "v810_cpu.h"
static uint16 control;
static uint16 period;
static int32 counter;

static int32 lastts;

static INLINE void SetEvent(void)
{
 v810_setevent(V810_EVENT_TIMER, (control & 0x2) ? counter : V810_EVENT_NONONO);
}

#define EFF_PERIOD ((period ? period : 0x10000) * 15)

void FXTIMER_Update(void)
{
 if(control & 0x2)
 {
  int32 cycles = v810_timestamp - lastts;
  counter -= cycles;
  while(counter <= 0)
  {
   counter += EFF_PERIOD;
   if(control & 0x1)
   {
    PCFXIRQ_Assert(9, TRUE);
    //puts("Timer IRQ");
   }
   if(control & 0x4) // One-shot?
   {
    control &= ~0x2;
    break;
   }
    
  }
 }

 lastts = v810_timestamp;

 SetEvent();
}

void FXTIMER_ResetTS(void)
{
 lastts = 0;
}

uint16 FXTIMER_Read16(uint32 A)
{
 //printf("Read16: %08x\n", A);
 FXTIMER_Update();
 switch(A & 0xF80)
 {
  case 0xF00: return(control);
  case 0xF80: return(period);
 }
 return(0);
}

uint8 FXTIMER_Read8(uint32 A)
{
 FXTIMER_Update();
 return(FXTIMER_Read16(A&~1) >> ((A & 1) * 8));
}

void FXTIMER_Write16(uint32 A, uint16 V)
{
 //printf("Write16: %08x %04x\n", A, V);
 FXTIMER_Update();
 PCFXIRQ_Assert(9, FALSE);

 switch(A & 0xF80)
 {
  case 0xF00: if(!(control & 0x2) && (V & 0x2))
		counter = EFF_PERIOD;
	      control = V; 
	      SetEvent();
	      break;
  case 0xF80: period = V; 
	      SetEvent();
	      break;
 }
}

int FXTIMER_StateAction(StateMem *sm, int load, int data_only)
{
 SFORMAT StateRegs[] =
 {
  SFVAR(counter),
  SFVAR(period),
  SFVAR(control),
  SFEND
 };

 int ret = MDFNSS_StateAction(sm, load, data_only, StateRegs, "TIMR");

 if(load)
 {
  SetEvent();
 }
 return(ret);
}



bool FXTIMER_GetRegister(const std::string &name, uint32 &value, std::string *special)
{
 if(name == "TIMCTRL")
 {
  value = control;
  if(special)
  {
   char buf[256];
   snprintf(buf, 256, "Counting Enabled: %d, IRQ Enabled: %d, One-shot Enabled: %d", (int)(bool)(control & 2), (int)(bool)(control & 1), (int)(bool)(control & 4));
   *special = std::string(buf);
  }
  return(TRUE);
 }
 else if(name == "TIMPRD")
 {
  value = period;
  if(special)
  {
   char buf[256];
   snprintf(buf, 256, "Effective Period: %d; 21477272 / %d = %fHz", EFF_PERIOD, EFF_PERIOD, (double)21477272 / (EFF_PERIOD));
   *special = std::string(buf);
  }
  return(TRUE);
 }
 else if(name == "TIMCNTR")
 {
  value = counter;
  if(special)
  {
   //char buf[256];
   //snprintf(buf, 256, "Pad: %d, ??: %d, Timer: %d, Reset: %d",
   //*special = std::string(buf);
  }
  return(TRUE);
 }
 else
  return(FALSE);
}

