/* Mednafen - Multi-system Emulator
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "pcfx.h"
#include "soundbox.h"
#include "v810_cpu.h"
#include "vdc.h"
#include "king.h"
#include "scsicd.h"
#include "timer.h"
#include "pad.h"

typedef Blip_Synth<blip_good_quality, 65536> ADSynth;
static ADSynth ADPCMSynth[2][2]; // [Ch0, Ch1][Left, right]
static Blip_Synth<blip_good_quality, 4096> WaveSynth;
static Blip_Synth<blip_med_quality, 4096> NoiseSynth;

Blip_Buffer FXsbuf[2];		// Used in the CDROM code

static t_soundbox psg;

void SoundBox_SetSoundMultiplier(double multiplier)
{
 for(int y = 0; y < 2; y++)
 {
  FXsbuf[y].clock_rate((long)(1789772.727272 * 4 * multiplier));
 }
}

static INLINE void RedoADPCMVolume(uint8 ch, uint8 lr) // bitmasks
{
 if((ch & 1) && (lr & 1))
  ADPCMSynth[0][0].volume((double)FSettings.SoundVolume / 100 / 6 * psg.ADPCMVolume[0][0] / 64);
 if((ch & 2) && (lr & 1))
  ADPCMSynth[1][0].volume((double)FSettings.SoundVolume / 100 / 6 * psg.ADPCMVolume[1][0] / 64);

 if((ch & 1) && (lr & 2))
  ADPCMSynth[0][1].volume((double)FSettings.SoundVolume / 100 / 6 * psg.ADPCMVolume[0][1] / 64);
 if((ch & 2) && (lr & 2))
  ADPCMSynth[1][1].volume((double)FSettings.SoundVolume / 100 / 6 * psg.ADPCMVolume[1][1] / 64);
}

static void RedoVolume(void)
{
 WaveSynth.volume((double)FSettings.SoundVolume / 100 / 6 * 0.33);
 NoiseSynth.volume((double)FSettings.SoundVolume / 100 / 6 * 0.33);
 RedoADPCMVolume(0x03, 0x03);
}

void SoundBox_SetSoundVolume(uint32 volume)
{
 RedoVolume();
}

void SoundBox_Sound(int rate)
{
 int x, y;

 for(y = 0; y < 2; y++)
 {
  FXsbuf[y].set_sample_rate(rate ? rate : 44100, 50);
  FXsbuf[y].clock_rate((long)(1789772.727272 * 4));
  FXsbuf[y].bass_freq(20);
 }

 if(psg.WaveIL)
  free(psg.WaveIL);

 psg.WaveFinalLen = rate / 60 * 2; // *2 for extra room
 psg.WaveIL = (int16 *)malloc(sizeof(int16) * psg.WaveFinalLen * 2); // *2 for stereo

 RedoVolume();
}

static int32 dbtable[32][32 * 3];
 static const int scale_tab[] = {
        0x00, 0x03, 0x05, 0x07, 0x09, 0x0B, 0x0D, 0x0F,
        0x10, 0x13, 0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F
        };

#define redo_ddacache(ch)	\
{	\
 int vll,vlr;	\
 vll = (ch)->lal + (ch)->al + psg.lmal;	\
 vlr = (ch)->ral + (ch)->al + psg.rmal;	\
 (ch)->dda_cache[0] = dbtable[(ch)->dda][vll];	\
 (ch)->dda_cache[1] = dbtable[(ch)->dda][vlr];	\
}

int SoundBox_Init(void)
{
    int x;

    memset(&psg, 0, sizeof(psg));
    if(FSettings.SndRate)
     SoundBox_Sound(FSettings.SndRate);

    for(x=0;x<32 * 3;x++)
    {
     int y;
     double flub;

     flub = 1;

     if(x)
      flub /= pow(2, (double)1/4*x); //4*x);		// ~1.5dB reduction per increment of x
     for(y=0;y<32;y++)
      dbtable[y][x] = (int32)(flub * (y - 0x10) * 128);
     //printf("%f\n",flub);
    }
    return (1);
}

void SoundBox_Kill(void)
{
	if(psg.WaveIL)
	 free(psg.WaveIL);
	psg.WaveIL = 0;
}

/*--------------------------------------------------------------------------*/
/* PSG emulation                                                            */
/*--------------------------------------------------------------------------*/

/* Macro to access currently selected PSG channel */
#define PSGCH   psg.channel[psg.select]
void SoundBox_Update(void);
void snortus(void);
void SoundBox_Write(uint32 A, uint16 V)
{
    int x;

    SoundBox_Update();

    switch(A & 0x3F)
    {
	//default: printf("HARUM: %04x %04x\n", A, V); break;
	case 0x20: psg.ADPCMControl = V; 
		   //printf("ADPCM Control: %04x\n", V);
		   break;
	case 0x22: psg.ADPCMVolume[0][0] = V & 0x3F; 
		   RedoADPCMVolume(0x01, 0x01);
		   break;
	case 0x24: psg.ADPCMVolume[0][1] = V & 0x3F;
		   RedoADPCMVolume(0x01, 0x02);
		   break;
	case 0x26: psg.ADPCMVolume[1][0] = V & 0x3F;
		   RedoADPCMVolume(0x02, 0x01);
		   break;
	case 0x28: psg.ADPCMVolume[1][1] = V & 0x3F; 
		   RedoADPCMVolume(0x02, 0x02);
		   break;
	case 0x2A: psg.CDDAVolume[0] = V & 0x3F; SCSICD_SetCDDAVolume(psg.CDDAVolume[0], psg.CDDAVolume[1]); break;
	case 0x2C: psg.CDDAVolume[1] = V & 0x3F; SCSICD_SetCDDAVolume(psg.CDDAVolume[0], psg.CDDAVolume[1]); break;

	case 0x00: psg.select = (V & 0x07); break;
        case 0x02: /* Global sound balance */
            psg.globalbalance = V;
	    psg.lmal = 0x1F - scale_tab[(psg.globalbalance >> 4) & 0xF];
	    psg.rmal = 0x1F - scale_tab[(psg.globalbalance >> 0) & 0xF];
	    for(x=0;x<6;x++)
	     redo_ddacache(&psg.channel[x]);
            break;

        case 0x04: /* Channel frequency (LSB) */
	    if(psg.select > 5) return; // no more than 6 channels, silly game.
            PSGCH.frequency = (PSGCH.frequency & 0x0F00) | V;
	    PSGCH.base_frequency = PSGCH.frequency;
            break;

        case 0x06: /* Channel frequency (MSB) */
	    if(psg.select > 5) return; // no more than 6 channels, silly game.
            PSGCH.frequency = (PSGCH.frequency & 0x00FF) | ((V & 0x0F) << 8);
            PSGCH.base_frequency = PSGCH.frequency;
            break;

        case 0x08: /* Channel enable, DDA, volume */
	    if(psg.select > 5) return; // no more than 6 channels, silly game.
            PSGCH.control = V;
	    PSGCH.al = 0x1F - (PSGCH.control & 0x1F);
            if((V & 0xC0) == 0x40) PSGCH.waveform_index = 0;
            redo_ddacache(&PSGCH);
            break;

        case 0x0A: /* Channel balance */
	    if(psg.select > 5) return; // no more than 6 channels, silly game.
            PSGCH.balance = V;
	    PSGCH.lal = 0x1F - scale_tab[(V >> 4) & 0xF];
	    PSGCH.ral = 0x1F - scale_tab[(V >> 0) & 0xF];
            redo_ddacache(&PSGCH);
            break;

        case 0x0C: /* Channel waveform data */
	    if(psg.select > 5) return; // no more than 6 channels, silly game.
            V &= 0x1F;

            PSGCH.waveform[PSGCH.waveform_index] = V;

	    if((PSGCH.control & 0xC0) == 0x00)
             PSGCH.waveform_index = ((PSGCH.waveform_index + 1) & 0x1F);

	    if(PSGCH.control & 0x80)
	     PSGCH.dda = V;

	    redo_ddacache(&PSGCH);
            break;

        case 0x0E: /* Noise enable and frequency */
	    if(psg.select > 5) return; // no more than 6 channels, silly game.
            if(psg.select >= 4) PSGCH.noisectrl = V;
            break;

        case 0x10: /* LFO frequency */
            psg.lfofreq = V;
            break;

        case 0x12: /* LFO trigger and control */
	    if((V & 0x80) && !(psg.lfoctrl & 0x80))
	     psg.channel[1].waveform_index = 0;
            psg.lfoctrl = V;
            break;
    }
}

/* MSM 6258 code transcribed/implemented/whatever from msm6258.c from NetBSD,
   Copyright (c) 2001 Tetsuya Isaki. All rights reserved.
*/

static const int EstimIndexTable[16] = 
{
          2,  6,  10,  14,  18,  22,  26,  30,
         -2, -6, -10, -14, -18, -22, -26, -30
};

static const int EstimTable[49] = 
{
         16,  17,  19,  21,  23,  25,  28,  31,  34,  37,
         41,  45,  50,  55,  60,  66,  73,  80,  88,  97,
        107, 118, 130, 143, 157, 173, 190, 209, 230, 253,
        279, 307, 337, 371, 408, 449, 494, 544, 598, 658,
        724, 796, 876, 963, 1060, 1166, 1282, 1411, 1552
};

static const int EstimStepTable[16] = 
{
        -1, -1, -1, -1, 2, 4, 6, 8,
        -1, -1, -1, -1, 2, 4, 6, 8
};

void ADPCM_ResetState(int ch)
{
 psg.ADPCMCurrent[ch] = 0;
 psg.ADPCMEstim[ch] = 0;
}

static uint32 KINGADPCMControl;
static uint32 LSampleFreq = 0xFFFF;

void SoundBox_SetKINGADPCMControl(uint32 value)
{
 KINGADPCMControl = value;

 uint32 SampleFreq = (KINGADPCMControl >> 2) & 0x3;
 if(LSampleFreq != SampleFreq)
 {
  int rolloff = (int)((double)0.80 * 21477272 * 2 / 1365 / (1 << SampleFreq) );

  for(int ch = 0; ch < 2; ch++)
   for(int lr = 0; lr < 2; lr++)
    ADPCMSynth[ch][lr].treble_eq(blip_eq_t::blip_eq_t(-1000, rolloff, FSettings.SndRate));
  LSampleFreq = SampleFreq;
 }
}

static int32 adpcm_lastts;

void DoADPCMUpdate(void)
{
 int32 run_time = v810_timestamp - adpcm_lastts;

 adpcm_lastts = v810_timestamp;

 psg.bigdiv -= run_time * 2;

 while(psg.bigdiv <= 0)
 {
  psg.smalldiv--;
  while(psg.smalldiv <= 0)
  {
   psg.smalldiv += 1 << ((KINGADPCMControl >> 2) & 0x3);
   for(int ch = 0; ch < 2; ch++)
   {
    if(psg.ADPCMHaveHalfWord[ch] || KINGADPCMControl & (1 << ch)) // Keep playing our last halfword fetched even if KING ADPCM is disabled
    {
     uint32 synthtime = ((v810_timestamp + (psg.bigdiv >> 19))) / 3;
     uint8 zenibble;
     int last_pcm = psg.ADPCMCurrent[ch];

     if(!psg.ADPCMWhichNibble[ch])
      psg.ADPCMHalfWord[ch] = KING_GetADPCMHalfWord(ch);

     zenibble = (psg.ADPCMHalfWord[ch] >> psg.ADPCMWhichNibble[ch]) & 0xF;

     psg.ADPCMWhichNibble[ch] = (psg.ADPCMWhichNibble[ch] + 4) & 0xF;

     if(!psg.ADPCMWhichNibble[ch])
      psg.ADPCMHaveHalfWord[ch] = FALSE;

     psg.ADPCMCurrent[ch] += EstimTable[psg.ADPCMEstim[ch]] * EstimIndexTable[zenibble];
     if(psg.ADPCMCurrent[ch] > 32767) psg.ADPCMCurrent[ch] = 32767;
     if(psg.ADPCMCurrent[ch] < -32768) psg.ADPCMCurrent[ch] = -32768;
     psg.ADPCMEstim[ch] += EstimStepTable[zenibble];

     if(psg.ADPCMEstim[ch] < 0)
	psg.ADPCMEstim[ch] = 0;
     if(psg.ADPCMEstim[ch] > 48)
	psg.ADPCMEstim[ch] = 48;

     if(FSettings.SndRate)
     {
      int diff = psg.ADPCMCurrent[ch] - psg.ADPCM_last[ch];
      ADPCMSynth[ch][0].offset(synthtime, diff, &FXsbuf[0]);
      ADPCMSynth[ch][1].offset(synthtime, diff, &FXsbuf[1]);
      psg.ADPCM_last[ch] = psg.ADPCMCurrent[ch];
     }
    }
   }
  }
  psg.bigdiv += 1365 * 2 / 2;
 }
 v810_setevent(V810_EVENT_ADPCM, (psg.bigdiv + 1) / 2);
}

static void DoLFOUpdate(int32 run_time, int32 timestamp)
{
  int chc;
  for(chc = 1; chc < 6; chc++)
  {
   psg_channel *ch = &psg.channel[chc];
   int disabled = ((ch->control & 0x80)^0x80) >> 7;
   //puts("ROO");
   if(disabled) 
    continue; 
   #include "psg-loop.h"
  }

  #define LFO_ON
  chc = 0;
  psg_channel *ch = &psg.channel[0];
  #include "psg-loop.h"
  #undef LFO_ON
}

static void DoNormalUpdate(int32 run_time, int32 timestamp)
{
 int chc;

 for(chc = 0; chc < 6; chc++)
 {
  psg_channel *ch = &psg.channel[chc];
  int disabled = ((ch->control & 0x80)^0x80) >> 7;
  if(disabled) 
   continue; 
  #include "psg-loop.h"
 }
}

void SoundBox_Update(void)
{
 int32 timestamp = v810_timestamp / 3;
 int32 run_time = timestamp - psg.lastts;

 if(!FSettings.SndRate) return;

 if(timestamp == psg.lastts) return;

 int lfo_on = psg.lfoctrl & 0x03;
 if(!(psg.channel[1].control & 0x80))
  lfo_on = 0;

 if(!(psg.channel[0].control & 0x80))
  lfo_on = 0;

 if(!(psg.lfoctrl & 0x80))
  lfo_on = 0;

 if(lfo_on)
  DoLFOUpdate(run_time, timestamp);
 else
  DoNormalUpdate(run_time, timestamp);
 
 psg.lastts = timestamp;
}

int16 *SoundBox_Flush(int32 *len, int needrew)
{
 int32 timestamp;

 if(FSettings.SndRate);
  SoundBox_Update();

 
 DoADPCMUpdate();
 SCSICD_Run();
 FXTIMER_Update();
 FXPAD_Update();

 timestamp = v810_timestamp / 3;
 if(FSettings.SndRate)
 {
  int love;
  for(int y = 0; y < 2; y++)
  {
   FXsbuf[y].end_frame(timestamp);
   love = FXsbuf[y].read_samples(&psg.WaveIL[y], psg.WaveFinalLen, 1);
  }
  *len = love;
 }

 v810_timestamp = 0;
 psg.lastts = 0;
 adpcm_lastts = 0;

 SCSICD_ResetTS();
 FXTIMER_ResetTS();
 FXPAD_ResetTS();
 
 if(!FSettings.SndRate)
 {
  *len = 0;
  return(NULL);
 }
 return(psg.WaveIL);
}

void SoundBox_Reset(void)
{
 memset(psg.channel, 0, sizeof(psg.channel));
 psg.select = 0;
 psg.globalbalance = 0;
 psg.lfofreq = 0;
 psg.lfoctrl = 0;
 psg.lfo_counter = 0;
}

void SoundBox_Power(void)
{
 memset(psg.channel, 0, sizeof(psg.channel));
 psg.select = 0;
 psg.globalbalance = 0;
 psg.lfofreq = 0;
 psg.lfoctrl = 0;
 psg.lfo_counter = 0;
}

int SoundBox_StateAction(StateMem *sm, int load, int data_only)
{
 int ret = 1;

 for(int ch = 0; ch < 6; ch++)
 {
  char tmpstr[5] = "SCHx";
  psg_channel *pt = &psg.channel[ch];

  SFORMAT CH_StateRegs[] = 
  {
   SFVARN(pt->counter, "counter"),
   SFVARN(pt->frequency, "frequency"),
   SFVARN(pt->control, "control"),
   SFVARN(pt->balance, "balance"),
   SFARRAYN(pt->waveform, 32, "waveform"),
   SFVARN(pt->waveform_index, "waveform_index"),
   SFVARN(pt->dda, "dda"),
   SFVARN(pt->noisectrl, "noisectrl"),
   SFVARN(pt->noisecount, "noisecount"),
   SFVARN(pt->lfsr, "lfsr"),
   SFEND
  };
  tmpstr[3] = '0' + ch;
  ret &= MDFNSS_StateAction(sm, load, data_only, CH_StateRegs, tmpstr);
 }

 SFORMAT SoundBox_StateRegs[] =
 {
  SFVARN(psg.select, "select"),
  SFVARN(psg.globalbalance, "globalbalance"),
  SFVARN(psg.lfofreq, "lfofreq"),
  SFVARN(psg.lfoctrl, "lfoctrl"),

  SFVARN(psg.ADPCMControl, "ADPCMControl"),
  SFARRAY32N(psg.ADPCMCurrent, 2, "ADPCMCurrent"),
  SFARRAY32N(psg.ADPCMEstim, 2, "ADPCMEstim"),
  SFARRAY32N(psg.ADPCMWhichNibble, 2, "ADPCMWNibble"),
  SFARRAY16N(psg.ADPCMHalfWord, 2, "ADPCMHalfWord"),
  SFARRAYN(psg.ADPCMHaveHalfWord, 2, "ADPCMHHW"),

  SFARRAYN(psg.ADPCMVolume, 2 * 2, "ADPCMVolume"),
  SFVARN(psg.bigdiv, "bigdiv"),
  SFVARN(psg.smalldiv, "smalldiv"),

  SFARRAYN(psg.CDDAVolume, 2, "CDDAVolume"),
  SFEND
 };
 
 ret &= MDFNSS_StateAction(sm, load, data_only, SoundBox_StateRegs, "SBOX");

 if(load)
 {
  psg.lmal = 0x1F - scale_tab[(psg.globalbalance >> 4) & 0xF];
  psg.rmal = 0x1F - scale_tab[(psg.globalbalance >> 0) & 0xF];

  for(int ch = 0; ch < 6; ch++)
  {
   psg.channel[ch].lal = 0x1F - scale_tab[(psg.channel[ch].balance >> 4) & 0xF];
   psg.channel[ch].ral = 0x1F - scale_tab[(psg.channel[ch].balance >> 0) & 0xF];
   psg.channel[ch].al = 0x1F - (psg.channel[ch].control & 0x1F);
   redo_ddacache(&psg.channel[ch]);
  }
  RedoADPCMVolume(0x03, 0x03);
  SCSICD_SetCDDAVolume(psg.CDDAVolume[0], psg.CDDAVolume[1]);

  v810_setevent(V810_EVENT_ADPCM, (psg.bigdiv + 1) / 2);
 }
 return(ret); 
}
