/* FCE Ultra - NES/Famicom Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Ben Parnell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "types.h"
#include "fce.h"
#include "debug.h"



void FCEUI_DumpMem(char *fname, uint32 start, uint32 end)
{
 FILE *fp=fopen(fname,"wb");
 fceuindbg=1;
 for(;start<=end;start++)
  fputc(ARead[start](start),fp);
 fclose(fp);
 fceuindbg=0;
}

void FCEUI_LoadMem(char *fname, uint32 start)
{
 int t;

 FILE *fp=fopen(fname,"rb");
 while((t=fgetc(fp))>=0)
 {
  if(start>0xFFFF) break;
  BWrite[start](start,t);
  start++;
 }
 fclose(fp);
}
static char *fstrings[12]=
{ 
        "#$%02X",       // immediate
        "$%04X",        // RELATIVE(jump)
        "$%02X",        // Z
        "$%02X,X",      // Z,x
        "$%02X,Y",      // Z,y
        "$%04X",        //ABS
        "$%04X,X",      // ABS,x
        "$%04X,Y",      // ABS,y
        "($%04X)",      // IND
        "($%02X,X)",    // INX
        "($%02X),Y",    // INY
	""
};
static int flengths[12]={1,1,1,1,1,2,2,2,2,1,1,0};

#define IMD(x)	((0<<16)|x)
#define	REL(x)	((1<<16)|x)
#define	ZP(x)	((2<<16)|x)
#define	ZPX(x)	((3<<16)|x)
#define	ZPY(x)	((4<<16)|x)
#define	ABS(x)	((5<<16)|x)
#define	ABX(x)	((6<<16)|x)
#define ABY(x)	((7<<16)|x)
#define	IND(x)	((8<<16)|x)
#define	INX(x)	((9<<16)|x)
#define	INY(x)	((10<<16)|x)
#define IMP(x)	((11<<16)|x)

typedef struct {
	char *name;
	int32 modes[10];
} OPS;
#define NUMOPS 56
static OPS optable[NUMOPS]=
{
 {"BRK",{IMP(0x00),-1}},
 {"RTI",{IMP(0x40),-1}},
 {"RTS",{IMP(0x60),-1}},
 {"PHA",{IMP(0x48),-1}},
 {"PHP",{IMP(0x08),-1}},
 {"PLA",{IMP(0x68),-1}},
 {"PLP",{IMP(0x28),-1}},
 {"JMP",{ABS(0x4C),IND(0x6C),-1}},
 {"JSR",{ABS(0x20),-1}},
 {"TAX",{IMP(0xAA),-1}},
 {"TXA",{IMP(0x8A),-1}},
 {"TAY",{IMP(0xA8),-1}},
 {"TYA",{IMP(0x98),-1}},
 {"TSX",{IMP(0xBA),-1}},
 {"TXS",{IMP(0x9A),-1}},
 {"DEX",{IMP(0xCA),-1}},
 {"DEY",{IMP(0x88),-1}},
 {"INX",{IMP(0xE8),-1}},
 {"INY",{IMP(0xC8),-1}},
 {"CLC",{IMP(0x18),-1}},
 {"CLD",{IMP(0xD8),-1}},
 {"CLI",{IMP(0x58),-1}},
 {"CLV",{IMP(0xB8),-1}},
 {"SEC",{IMP(0x38),-1}},
 {"SED",{IMP(0xF8),-1}},
 {"SEI",{IMP(0x78),-1}},
 {"NOP",{IMP(0xEA),-1}},
 {"ASL",{IMP(0x0a),ZP(0x06),ZPX(0x16),ABS(0x0E),ABX(0x1E),-1}},
 {"DEC",{ZP(0xc6),ZPX(0xd6),ABS(0xcE),ABX(0xdE),-1}},
 {"INC",{ZP(0xe6),ZPX(0xf6),ABS(0xeE),ABX(0xfE),-1}},
 {"LSR",{IMP(0x4a),ZP(0x46),ZPX(0x56),ABS(0x4E),ABX(0x5E),-1}},
 {"ROL",{IMP(0x2a),ZP(0x26),ZPX(0x36),ABS(0x2E),ABX(0x3E),-1}},
 {"ROR",{IMP(0x6a),ZP(0x66),ZPX(0x76),ABS(0x6E),ABX(0x7E),-1}},
 {"ADC",{IMD(0x69),ZP(0x65),ZPX(0x75),ABS(0x6D),ABX(0x7d),ABY(0x79),
	 INX(0x61),INY(0x71),-1}},
 {"AND",{IMD(0x29),ZP(0x25),ZPX(0x35),ABS(0x2D),ABX(0x3d),ABY(0x39),
         INX(0x21),INY(0x31),-1}},
 {"BIT",{ZP(0x24),ABS(0x2c),-1}},
 {"CMP",{IMD(0xc9),ZP(0xc5),ZPX(0xd5),ABS(0xcD),ABX(0xdd),ABY(0xd9),
         INX(0xc1),INY(0xd1),-1}},
 {"CPX",{IMD(0xe0),ZP(0xe4),ABS(0xec),-1}},
 {"CPY",{IMD(0xc0),ZP(0xc4),ABS(0xcc),-1}},
 {"EOR",{IMD(0x49),ZP(0x45),ZPX(0x55),ABS(0x4D),ABX(0x5d),ABY(0x59),
         INX(0x41),INY(0x51),-1}},
 {"LDA",{IMD(0xa9),ZP(0xa5),ZPX(0xb5),ABS(0xaD),ABX(0xbd),ABY(0xb9),
         INX(0xa1),INY(0xb1),-1}},
 {"LDX",{IMD(0xa2),ZP(0xa6),ZPY(0xB6),ABS(0xae),ABY(0xbe),-1}},
 {"LDY",{IMD(0xa0),ZP(0xa4),ZPX(0xB4),ABS(0xac),ABY(0xbc),-1}},
 {"ORA",{IMD(0x09),ZP(0x05),ZPX(0x15),ABS(0x0D),ABX(0x1d),ABY(0x19),
         INX(0x01),INY(0x11),-1}},
 {"SBC",{IMD(0xEB),IMD(0xe9),ZP(0xe5),ZPX(0xf5),ABS(0xeD),ABX(0xfd),ABY(0xf9),
         INX(0xe1),INY(0xf1),-1}},
 {"STA",{ZP(0x85),ZPX(0x95),ABS(0x8D),ABX(0x9d),ABY(0x99),
         INX(0x81),INY(0x91),-1}},
 {"STX",{ZP(0x86),ZPY(0x96),ABS(0x8E),-1}},
 {"STY",{ZP(0x84),ZPX(0x94),ABS(0x8C),-1}},
 {"BCC",{REL(0x90),-1}},
 {"BCS",{REL(0xb0),-1}},
 {"BEQ",{REL(0xf0),-1}},
 {"BNE",{REL(0xd0),-1}},
 {"BMI",{REL(0x30),-1}},
 {"BPL",{REL(0x10),-1}},
 {"BVC",{REL(0x50),-1}},
 {"BVS",{REL(0x70),-1}},
};

void FCEUI_Disassemble(uint16 a, int (*callb)(uint16 a, char *s))
{
 char stringo[80];
 uint8 buf;
 unsigned int arg;
 int x;
 int y;
 uint16 savea;

 fceuindbg=1;

 for(;;)
 {
  int32 info=-1;

  buf=ARead[a](a);
  savea=a;
  a++;

  for(x=0;x<NUMOPS;x++)
  {
   y=0;
   while(optable[x].modes[y]>=0)
   {
    if((optable[x].modes[y]&0xFF)==buf)
    {
     info=optable[x].modes[y];
     goto endy;
    }
    y++;
   }
  }

  endy:
  sprintf(stringo,"$%04X: %02X ",savea,buf);
  if(info>=0)
  {
   int z=flengths[(info>>16)];
  
   if(z)
   {
    arg=ARead[a](a);
    sprintf(stringo+strlen(stringo),"%02X ",arg);
    a++;
    if(z==2) {arg|=ARead[a](a)<<8;sprintf(stringo+strlen(stringo),"%02X ",arg>>8);a++;}
    else
     strcat(stringo,"   ");
    if((info>>16)==1)
     arg=a+(char)arg;
    sprintf(stringo+strlen(stringo),"%s ",optable[x].name);
    sprintf(stringo+strlen(stringo),fstrings[info>>16],arg);
   }
   else
   {
    strcat(stringo,"      ");
    strcat(stringo,optable[x].name);
   }
  }
  else
   sprintf(stringo+strlen(stringo),"      .db $%02X",buf);
  if(!callb(savea,stringo)) break;
 }
 fceuindbg=0;
}

void FCEUI_MemDump(uint16 a, int32 len, void (*callb)(uint16 a, uint8 v))
{
 fceuindbg=1;
 while(len)
 {
  callb(a,ARead[a](a));
  a++;
  len--;
 }
 fceuindbg=0;
}

void FCEUI_MemPoke(uint16 a, uint8 v, int hl)
{
 extern uint8 *Page[32];
 if(hl)
 {
  if(Page[a/2048])
   Page[a/2048][a]=v;
 }
 else
  BWrite[a](a,v);
}
