/*
 *  Copyright (C) 2002-2004  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: sdlmain.cpp,v 1.79 2004/09/29 19:14:10 harekiet Exp $ */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>

//#include "SDL.h"

#include "dosbox.h"
#include "video.h"
#include "mouse.h"
#include "pic.h"
#include "timer.h"
#include "setup.h"
#include "support.h"
#include "debug.h"
#include "mapper.h"
#include "joystick.h"

#define Sint16 signed short
#define Uint16 unsigned short

//#define DISABLE_JOYSTICK
#ifdef _XBOX

#include <xtl.h>
#include "..\..\..\iso9660.h"

//extern HANDLE m_cdrom ;
//extern CIoSupport m_ioM ;
extern CISO9660 m_iso9660 ;

void RENDER_SetFrameskip( Bitu fskip ) ;
float mouseMotionX, mouseMotionY ;

float joyAxisFlipX, joyAxisFlipY, mouseAxisFlipX, mouseAxisFlipY ;

char g_confFile[500] ;

Uint16 modifiers ;

extern Bits CPU_CycleMax;

int flipAnalog ;

unsigned int lastHAT ;

int can_move_screen ;


#define DEFAULT_CONFIG_FILE "Z:\\dosbox.conf"

#ifdef __cplusplus
extern "C" {
#endif

unsigned char * xbox_get_screen_buffer();
unsigned int xbox_get_pitch();
void xbox_put_image( unsigned char* msg) ;
void sprintfx( const char *fmt, ... );
int xbox_check_events( );
void xbox_lock( unsigned char **ptr );
void xbox_process_sound(  short *sampleBuffer, unsigned short sampleCount ) ;

#ifdef __cplusplus
}
#endif

#endif



#if C_SET_PRIORITY
#include <sys/resource.h>
#define PRIO_TOTAL (PRIO_MAX-PRIO_MIN)
#endif

enum SCREEN_TYPES	{ 
	SCREEN_SURFACE,
	SCREEN_SURFACE_DDRAW,
	SCREEN_OVERLAY,
	SCREEN_OPENGL
};

enum PRIORITY_LEVELS {
	PRIORITY_LEVEL_LOWER,
	PRIORITY_LEVEL_NORMAL,
	PRIORITY_LEVEL_HIGHER,
	PRIORITY_LEVEL_HIGHEST
};

typedef struct SDLRECT {
	int x ;
	int y ;
	int w ; 
	int h ;
} SDL_Rect ;

struct SDL_Block {
	bool active;							//If this isn't set don't draw
	bool updating;
	struct {
		Bit32u width;
		Bit32u height;
		Bitu flags;
		GFX_Modes mode;
		double scalex,scaley;
		GFX_ResetCallBack reset;
	} draw;
	bool wait_on_error;
	struct {
		Bit32u width,height,bpp;
		bool fixed;
		bool fullscreen;
		bool doublebuf;
		SCREEN_TYPES type;
		SCREEN_TYPES want_type;
		double hwscale;
	} desktop;
	struct {
		PRIORITY_LEVELS focus;
		PRIORITY_LEVELS nofocus;
	} priority;
	SDL_Rect clip;
	//SDL_Surface * surface;
	//SDL_Overlay * overlay;
	//SDL_cond *cond;
	struct {
		bool autolock;
		bool autoenable;
		bool requestlock;
		bool locked;
		Bitu sensitivity;
	} mouse;
};

static SDL_Block sdl;
static void CaptureMouse(void);

extern char * RunningProgram;
void GFX_SetTitle(Bits cycles,Bits frameskip,bool paused){
}


/* Reset the screen with current values in the sdl structure */
Bitu GFX_GetBestMode(Bitu flags) {

	if (flags & CAN_16) flags&=~(CAN_8|CAN_32);

	return flags;
}


void GFX_ResetScreen(void) {
	GFX_Stop();
	if (sdl.draw.reset) (sdl.draw.reset)();
	GFX_Start();
}

static int int_log2 (int val) {
    int log = 0;
    while ((val >>= 1) != 0)
	log++;
    return log;
}


static void GFX_SetupSurfaceScaled(Bit32u sdl_flags,Bit32u bpp) {
	if (sdl.desktop.fullscreen) {
		if (sdl.desktop.fixed) {
			double ratio_w=(double)sdl.desktop.width/(sdl.draw.width*sdl.draw.scalex);
			double ratio_h=(double)sdl.desktop.height/(sdl.draw.height*sdl.draw.scaley);
			if ( ratio_w < ratio_h) {
				sdl.clip.w=(Bit16u)sdl.desktop.width;
				sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley*ratio_w);
			} else {
				sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex*ratio_h);
				sdl.clip.h=(Bit16u)sdl.desktop.height;
			}
			sdl.clip.x=(Sint16)((sdl.desktop.width-sdl.clip.w)/2);
			sdl.clip.y=(Sint16)((sdl.desktop.height-sdl.clip.h)/2);
			//return sdl.surface=SDL_SetVideoMode(sdl.desktop.width,sdl.desktop.height,bpp,sdl_flags|SDL_FULLSCREEN|SDL_HWSURFACE);
		} else {
			sdl.clip.x=0;sdl.clip.y=0;
			sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex);
			sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley);
			//return sdl.surface=SDL_SetVideoMode(sdl.clip.w,sdl.clip.h,bpp,sdl_flags|SDL_FULLSCREEN|SDL_HWSURFACE);
		}
	} else {
		sdl.clip.x=0;sdl.clip.y=0;
		sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex*sdl.desktop.hwscale);
		sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley*sdl.desktop.hwscale);
		//return sdl.surface=SDL_SetVideoMode(sdl.clip.w,sdl.clip.h,bpp,sdl_flags|SDL_HWSURFACE);
	}
}

GFX_Modes GFX_SetSize(Bitu width,Bitu height,Bitu flags,double scalex,double scaley,GFX_ResetCallBack reset) {
	if (sdl.updating) GFX_EndUpdate();
	sdl.draw.width=width;
	sdl.draw.height=height;
	sdl.draw.flags=flags;
	sdl.draw.mode=GFX_NONE;
	sdl.draw.reset=reset;
	sdl.draw.scalex=scalex;
	sdl.draw.scaley=scaley;

	Bitu bpp;
	switch (sdl.desktop.want_type) {
	case SCREEN_SURFACE:
dosurface:
		if (flags & CAN_8) bpp=8;
		if (flags & CAN_16) bpp=16;
		if (flags & CAN_32) bpp=32;
		sdl.desktop.type=SCREEN_SURFACE;
		sdl.clip.w=width;
		sdl.clip.h=height;
		if (sdl.desktop.fullscreen) {
			if (sdl.desktop.fixed) {
				sdl.clip.x=(Sint16)((sdl.desktop.width-width)/2);
				sdl.clip.y=(Sint16)((sdl.desktop.height-height)/2);
				//sdl.surface=SDL_SetVideoMode(sdl.desktop.width,sdl.desktop.height,bpp,
					//SDL_FULLSCREEN|SDL_HWSURFACE|(sdl.desktop.doublebuf ? SDL_DOUBLEBUF|SDL_ASYNCBLIT  : 0)|SDL_HWPALETTE);
			} else {
				sdl.clip.x=0;sdl.clip.y=0;
				//sdl.surface=SDL_SetVideoMode(width,height,bpp,
					//SDL_FULLSCREEN|SDL_HWSURFACE|(sdl.desktop.doublebuf ? SDL_DOUBLEBUF|SDL_ASYNCBLIT  : 0)|SDL_HWPALETTE);
			}
		} else {
			sdl.clip.x=0;sdl.clip.y=0;
			//sdl.surface=SDL_SetVideoMode(width,height,bpp,SDL_HWSURFACE);
		}

		sdl.draw.mode=GFX_16;

		break;
	}//CASE
	if (sdl.draw.mode!=GFX_NONE) GFX_Start();
	return sdl.draw.mode;
}

bool mouselocked; //Global variable for mapper
static void CaptureMouse(void) {
}

void GFX_CaptureMouse(void) {
	CaptureMouse();
}

static void SwitchFullScreen(void) {
}

void GFX_SwitchFullScreen(void) {
    SwitchFullScreen();
}

bool GFX_StartUpdate(Bit8u * & pixels,Bitu & pitch) {
	unsigned char *ptr ;

	if (!sdl.active || sdl.updating) return false;
	sdl.updating=true;
	switch (sdl.desktop.type) {
	case SCREEN_SURFACE:

		//XTODO lock?
		xbox_lock( &ptr ) ;
		pixels=(Bit8u *)xbox_get_screen_buffer();

		pixels+=sdl.clip.y*xbox_get_pitch();
		pixels+=sdl.clip.x*2;
		pitch=xbox_get_pitch() ;
		return true;
	}
	return false;
}

void GFX_EndUpdate(void) {
	int ret;
	if (!sdl.updating) return;
	sdl.updating=false;
	switch (sdl.desktop.type) {
	case SCREEN_SURFACE:

		//XTODO lock?

		xbox_put_image( NULL ) ;
		//SDL_Flip(sdl.surface);
		break;

	}
}


void GFX_SetPalette(Bitu start,Bitu count,GFX_PalEntry * entries) {
	int i ;

	i = 5 ;
}

Bitu GFX_GetRGB(Bit8u red,Bit8u green,Bit8u blue) {
	switch (sdl.desktop.type) {
	case SCREEN_SURFACE:
	case SCREEN_SURFACE_DDRAW:
		return ( (red >> 3) << 11 ) | ( (green >> 2) << 5 ) | ( (blue >> 3) << 0 ) ;
	}
	return 0;
}

void GFX_Stop() {
	if (sdl.updating) GFX_EndUpdate();
	sdl.active=false;
}

void GFX_Start() {
	sdl.active=true;
}

static void GUI_ShutDown(Section * sec) {
	GFX_Stop();
	if (sdl.mouse.locked) CaptureMouse();
	if (sdl.desktop.fullscreen) SwitchFullScreen();
	//SDL_Quit();   //Becareful this should be removed if on the fly renderchanges are allowed
}

static void KillSwitch(void){
	throw 1;
}


static void GUI_StartUp(Section * sec) {
	sec->AddDestroyFunction(&GUI_ShutDown);
	Section_prop * section=static_cast<Section_prop *>(sec);
	sdl.active=false;
	sdl.updating=false;
	sdl.desktop.fullscreen=true ; 
	sdl.wait_on_error=false ;
	const char * priority=section->Get_string("priority");


	sdl.mouse.locked=false;
	mouselocked=false; //Global for mapper
	sdl.mouse.requestlock=false;
	sdl.desktop.fixed=false ;
	const char* fullresolution=section->Get_string("fullresolution");

	sdl.desktop.width  = 640;
	sdl.desktop.height = 480;

	sdl.desktop.doublebuf=section->Get_bool("fulldouble");
	sdl.desktop.hwscale=section->Get_float("hwscale");
	if (sdl.desktop.hwscale<0.1f) {
		LOG_MSG("SDL:Can't hwscale lower than 0.1");
		sdl.desktop.hwscale=0.1f;
	}


	sdl.mouse.autoenable=section->Get_bool("autolock");
	sdl.mouse.autolock=false;
	sdl.mouse.sensitivity=section->Get_int("sensitivity");
	const char * output=section->Get_string("output");

	sdl.desktop.want_type=SCREEN_SURFACE;


	/* Initialize screen for first time */
	//sdl.surface=SDL_SetVideoMode(640,400,0,0);
	sdl.desktop.bpp=16;
	if (sdl.desktop.bpp==24) {
		LOG_MSG("SDL:You are running in 24 bpp mode, this will slow down things!");
	}
	GFX_Stop();
/* Get some Event handlers */
	//MAPPER_AddHandler(KillSwitch,MK_f9,MMOD1,"shutdown","ShutDown");
	//MAPPER_AddHandler(CaptureMouse,MK_f10,MMOD1,"capmouse","Cap Mouse");
	//MAPPER_AddHandler(SwitchFullScreen,MK_return,MMOD2,"fullscr","Fullscreen");

}

void Mouse_AutoLock(bool enable) {
	sdl.mouse.autolock=enable;
	if (enable && sdl.mouse.autoenable) sdl.mouse.requestlock=true;
	else sdl.mouse.requestlock=false;
}

#if 0
static void HandleMouseMotion(SDL_MouseMotionEvent * motion) {
	if (sdl.mouse.locked) 
		Mouse_CursorMoved((float)motion->xrel*sdl.mouse.sensitivity/100,(float)motion->yrel*sdl.mouse.sensitivity/100);
}

static void HandleMouseButton(SDL_MouseButtonEvent * button) {
	switch (button->state) {
	case SDL_PRESSED:
		if (sdl.mouse.requestlock && !sdl.mouse.locked) {
			CaptureMouse();
			// Dont pass klick to mouse handler
			break;
		}
		switch (button->button) {
		case SDL_BUTTON_LEFT:
			Mouse_ButtonPressed(0);
			break;
		case SDL_BUTTON_RIGHT:
			Mouse_ButtonPressed(1);
			break;
		case SDL_BUTTON_MIDDLE:
			Mouse_ButtonPressed(2);
			break;
		}
		break;
	case SDL_RELEASED:
		switch (button->button) {
		case SDL_BUTTON_LEFT:
			Mouse_ButtonReleased(0);
			break;
		case SDL_BUTTON_RIGHT:
			Mouse_ButtonReleased(1);
			break;
		case SDL_BUTTON_MIDDLE:
			Mouse_ButtonReleased(2);
			break;
		}
		break;
	}
}

static Bit8u laltstate = SDL_KEYUP;
static Bit8u raltstate = SDL_KEYUP;
#endif


void GFX_Events() {

	if ( xbox_check_events() )
	{
		throw(0);
	}
	//xbox_process_sound( NULL, 0 ) ;

#if 0
	SDL_Event event;
	while (SDL_PollEvent(&event)) {
	    switch (event.type) {
		case SDL_ACTIVEEVENT:
			if (event.active.state & SDL_APPINPUTFOCUS) {
				if (event.active.gain) {
					if (sdl.desktop.fullscreen && !sdl.mouse.locked)
						CaptureMouse();	
					SetPriority(sdl.priority.focus);
				} else {
					if (sdl.mouse.locked) 
						CaptureMouse();	
					SetPriority(sdl.priority.nofocus);
				}
			}
			break;
		case SDL_MOUSEMOTION:
			HandleMouseMotion(&event.motion);
			break;
		case SDL_MOUSEBUTTONDOWN:
		case SDL_MOUSEBUTTONUP:
			HandleMouseButton(&event.button);
			break;
		case SDL_VIDEORESIZE:
//			HandleVideoResize(&event.resize);
			break;
		case SDL_QUIT:
			throw(0);
			break;
#ifdef WIN32
		case SDL_KEYDOWN:
		case SDL_KEYUP:
			// ignore event alt+tab
			if (event.key.keysym.sym==SDLK_LALT) laltstate = event.key.type;
			if (event.key.keysym.sym==SDLK_RALT) raltstate = event.key.type;
			if (((event.key.keysym.sym==SDLK_TAB)) &&
				((laltstate==SDL_KEYDOWN) || (raltstate==SDL_KEYDOWN))) break;
#endif

		default:
			void MAPPER_CheckEvent(SDL_Event * event);
			MAPPER_CheckEvent(&event);
		}
    }
#endif
}

void GFX_ShowMsg(char * format,...) {
	char buf[512];
	va_list msg;
	va_start(msg,format);
	vsprintf(buf,format,msg);
        strcat(buf,"\n");
	va_end(msg);
	sprintfx(buf);       
};
void MAPPER_AddHandler(MAPPER_Handler * handler,MapKeys key,Bitu mods,char * eventname,char * buttonname) {
	return ;
}
int getSDLMouseSensitivity()
{
	return sdl.mouse.sensitivity ;
}


void SDLUpdateConfig() {

	Section *sec ;

	sec = control->GetSection("sdl") ;
	Section_prop * section=static_cast<Section_prop *>(sec);

	if ( section!=NULL )
	{
		sdl.mouse.sensitivity=section->Get_int("sensitivity");


		joyAxisFlipX = section->Get_int("xbox_joy_flip_x");
		joyAxisFlipY = section->Get_int("xbox_joy_flip_y");
		mouseAxisFlipX = section->Get_int("xbox_mouse_flip_x");
		mouseAxisFlipY = section->Get_int("xbox_mouse_flip_y");
		flipAnalog = section->Get_int("xbox_analog_flip");

	}

	sec = control->GetSection("cpu") ;
	section=static_cast<Section_prop *>(sec);

	if ( section!=NULL )
	{
		CPU_CycleMax=section->Get_int("cycles");
	}

	sec = control->GetSection("render") ;
	section=static_cast<Section_prop *>(sec);

	if ( section!=NULL )
	{
		Bitu fskip =section->Get_int("frameskip");
		RENDER_SetFrameskip( fskip ) ;
	}


}

void SHELL_initvars();



int dosbox_main(int argc, char* argv[]) {
	try {
		CommandLine com_line(argc,argv);
		Config myconf(&com_line);
		control=&myconf;

		SHELL_initvars();



		Section_prop * sdl_sec=control->AddSection_prop("sdl",&GUI_StartUp);
		//sdl_sec->AddInitFunction(&MAPPER_StartUp);
		sdl_sec->Add_bool("fullscreen",false);
		sdl_sec->Add_bool("fulldouble",false);
		sdl_sec->Add_bool("fullfixed",false);
		sdl_sec->Add_string("fullresolution","1024x768");
		sdl_sec->Add_string("output","surface");
		sdl_sec->Add_float("hwscale",1.0);
		sdl_sec->Add_bool("autolock",true);
		sdl_sec->Add_int("sensitivity",100);
		sdl_sec->Add_bool("waitonerror",true);
		sdl_sec->Add_string("priority","higher,normal");
		sdl_sec->Add_string("mapperfile","mapper.txt");

		MSG_Add("SDL_CONFIGFILE_HELP",
			"fullscreen -- Start dosbox directly in fullscreen.\n"
			"fulldouble -- Use double buffering in fullscreen.\n"
			"fullfixed -- Don't resize the screen when in fullscreen.\n"
			"fullresolution -- What resolution to use for fullscreen, use together with fullfixed.\n"
			"output -- What to use for output: surface,overlay"
#if C_OPENGL
			",opengl,openglnb"
#endif
#if defined(HAVE_DDRAW_H) && defined(WIN32)
			",ddraw"
#endif
			".\n"
			"hwscale -- Extra scaling of window if the output device supports hardware scaling.\n"
			"autolock -- Mouse will automatically lock, if you click on the screen.\n"
			"sensitiviy -- Mouse sensitivity.\n"
			"waitonerror -- Wait before closing the console if dosbox has an error.\n"
			"priority -- Priority levels for dosbox: lower,normal,higher,highest.\n"
			"            Second entry behind the comma is for when dosbox is not focused/minimized.\n"
			"mapperfile -- File used to load/save the key/event mappings from.\n"
			);
		/* Init all the dosbox subsystems */
		DOSBOX_Init();
		std::string config_file;

		config_file="Z:\\dosbox.conf";

		/* Parse the config file
		 * try open config file in $HOME if can't open dosbox.conf or specified file
		 */
		if (control->ParseConfigFile(config_file.c_str()) == false)  {
		}


		/* Init all the sections */
		control->Init();

		JOYSTICK_Enable(0,true);
		JOYSTICK_Enable(1,true);


		/* Start up main machine */
		control->StartUp();
		/* Shutdown everything */
	} catch (char * error) {
		LOG_MSG("Exit to error: %s",error);

	}
	catch (int){ 
		;//nothing pressed killswitch
	}
	catch(...){   
		throw;//dunno what happened. rethrow for sdl to catch
	}

	//YM3812Shutdown() ;
	//YM3526Shutdown();
	//Y8950Shutdown();
	//YMF262Shutdown();

	return 0;
};
