/*****************************************************************************
** File:        moonsound.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the YM2413 sound chip. 
**              Wraps the c++ implementation taken from openMSX
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "Moonsound.h"
#include <string.h>
#include "OpenMsxYMF262.h"
#include "OpenMsxYMF278.h"
extern "C" {
#include "IoPort.h"
#include "romMapper.h"
#include "SlotManager.h"
}

#define FREQUENCY        GLOBAL_FREQ //3579545
#define SAMPLERATE       44100
#define BUFFER_SIZE      2 * 1024
 
extern "C" Int32* moonsoundSync(void* ref, UInt32 count);

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))

struct Moonsound {
    Moonsound() : opl3latch(0), opl4latch(0),
        timer1(0), timer2(0), timerRef1(-1), timerRef2(-1) {
        memset(defaultBuffer, 0, sizeof(defaultBuffer));
    }

    Mixer* mixer;
    int slotHandle;
    YMF278* ymf278;
    YMF262* ymf262;
    Int32  buffer[BUFFER_SIZE];
    Int32  defaultBuffer[BUFFER_SIZE];
    UInt32 clock;
    UInt32 timer1;
    UInt32 counter1;
    UInt8  timerRef1;
    UInt32 timer2;
    UInt32 counter2;
    UInt8  timerRef2;
    int opl3latch;
    UInt8 opl4latch;
    int sramSize;
 };

Moonsound* theMoonsound = NULL;

extern "C" Int32* moonsoundSync(void* ref, UInt32 count) 
{
    Moonsound* moonsound = (Moonsound*)ref;
    int* genBuf1 = NULL;
    int* genBuf2 = NULL;
    UInt32 i;

    genBuf1 = moonsound->ymf262->updateBuffer(count);
    if (genBuf1 == NULL) {
        genBuf1 = (int*)moonsound->defaultBuffer;
    }

    genBuf2 = moonsound->ymf278->updateBuffer(count);
    if (genBuf2 == NULL) {
        genBuf2 = (int*)moonsound->defaultBuffer;
    }

    for (i = 0; i < 2 * count; i++) {
        moonsound->buffer[i] = genBuf1[i] + genBuf2[i];
    }

    return moonsound->buffer;
}

void moonsoundTimerSet(int timer, int count)
{
    if (timer == 1) {
        if (theMoonsound->counter1 != -1) {
            theMoonsound->counter1 = count;
        }
        theMoonsound->timer1 = count;
    }
    else {
        if (theMoonsound->counter2 != -1) {
            theMoonsound->counter2 = count;
        }
        theMoonsound->timer2 = count;
    }
}

void moonsoundTimerStart(int timer, int start, UInt8 ref)
{
    if (timer == 1) {
        theMoonsound->timerRef1 = ref;
        theMoonsound->counter1  = start ? theMoonsound->timer1 : -1;
    }
    else {
        theMoonsound->timerRef2 = ref;
        theMoonsound->counter2  = start ? theMoonsound->timer2 : -1;
    }
}

extern "C" void moonsoundTick(UInt32 cpuClock) 
{
    if (theMoonsound != NULL) {
        while (cpuClock - theMoonsound->clock >= 286) {
            theMoonsound->clock += 286;
     
            if (theMoonsound != NULL) {
                if (theMoonsound->counter1 != -1) {
                    if (theMoonsound->counter1-- == 0) {
                        theMoonsound->counter1 = theMoonsound->timer1;
                        theMoonsound->ymf262->callback(theMoonsound->timerRef1);
                    }
                }

                if (theMoonsound->counter2 != -1) {
                    if (theMoonsound->counter2-- == 0) {
                        theMoonsound->counter2 = theMoonsound->timer2;
                        theMoonsound->ymf262->callback(theMoonsound->timerRef2);
                    }
                }
            }
        }
    }
}

extern "C" void moonsoundDestroy(void* rm) {
    Moonsound* moonsound = (Moonsound*)rm;
    slotUnregisterUnslotted(moonsound->slotHandle);

    ioPortUnregister(0x7e);
    ioPortUnregister(0x7f);
    ioPortUnregister(0xc4);
    ioPortUnregister(0xc5);
    ioPortUnregister(0xc6);
    ioPortUnregister(0xc7);

    mixerUnregisterChannel(moonsound->mixer, MIXER_CHANNEL_MSX_MOONSOUND);

    delete moonsound->ymf262;
    delete moonsound->ymf278;

	if ( theMoonsound )
	{
		delete theMoonsound ;
		theMoonsound = NULL;
	}
}



extern "C" int moonsoundGetState(void* rm, UInt8* buffer, UInt32 systemTime)
{
    Moonsound* moonsound = (Moonsound*)rm;
    
    int offset1 = OFFSETOF(YMF278, firstMember);
    int size1   = sizeof(YMF278) - offset1;
    int offset2 = OFFSETOF(YMF262, firstMember);
    int size2   = sizeof(YMF262) - offset2;
    int offset3 = OFFSETOF(Moonsound, clock);
    int size3   = sizeof(Moonsound) - offset3;

    memcpy(buffer, (char*)moonsound->ymf278 + offset1, size1);
    memcpy(buffer + size1, (char*)moonsound->ymf262 + offset2, size2);
    memcpy(buffer + size1 + size2, (char*)moonsound + offset3, size3);
    memcpy(buffer + size1 + size2 + size3, moonsound->ymf278->getRam(), moonsound->sramSize);

    return size1 + size2 + size3 + moonsound->sramSize;
}

extern "C" int moonsoundSetState(void* rm, UInt8* buffer, UInt32 systemTime)
{
    Moonsound* moonsound = (Moonsound*)rm;
    
    int offset1 = OFFSETOF(YMF278, firstMember);
    int size1   = sizeof(YMF278) - offset1;
    int offset2 = OFFSETOF(YMF262, firstMember);
    int size2   = sizeof(YMF262) - offset2;
    int offset3 = OFFSETOF(Moonsound, clock);
    int size3   = sizeof(Moonsound) - offset3;

    memcpy((char*)moonsound->ymf278 + offset1, buffer, size1);
    memcpy((char*)moonsound->ymf262 + offset2, buffer + size1, size2);
    memcpy((char*)moonsound + offset3, buffer + size1 + size2, size3);
    memcpy(moonsound->ymf278->getRam(), buffer + size1 + size2 + size3, moonsound->sramSize);

    return size1 + size2 + size3 + moonsound->sramSize;
}

extern "C" UInt8 moonsoundRead(Moonsound* moonsound, UInt16 ioPort, UInt32 cpuClock)
{
	UInt8 result = 0xff;

	if (ioPort < 0xC0) {
		switch (ioPort & 0x01) {
		case 1: // read wave register
			result = moonsound->ymf278->readRegOPL4(moonsound->opl4latch, cpuClock);
			break;
		}
	} else {
		switch (ioPort & 0x03) {
		case 0: // read status
		case 2:
			result = moonsound->ymf262->readStatus() | 
                     moonsound->ymf278->readStatus(cpuClock);
			break;
		case 1:
		case 3: // read fm register
			result = moonsound->ymf262->readReg(moonsound->opl3latch);
			break;
		}
	}

    return result;
}
extern "C" void moonsoundWrite(Moonsound* moonsound, UInt16 ioPort, UInt8 value, UInt32 cpuClock)
{
	if (ioPort < 0xC0) {
		switch (ioPort & 0x01) {
		case 0: // select register
			moonsound->opl4latch = value;
			break;
		case 1:
  			moonsound->ymf278->writeRegOPL4(moonsound->opl4latch, value, cpuClock);
			break;
		}
	} else {
		switch (ioPort & 0x03) {
		case 0:
			moonsound->opl3latch = value;
			break;
		case 2: // select register bank 1
			moonsound->opl3latch = value | 0x100;
			break;
		case 1:
		case 3: // write fm register
			moonsound->ymf262->writeReg(moonsound->opl3latch, value, cpuClock);
			break;
		}
	}
}

extern "C" int moonsoundCreate(Mixer* mixer, void* romData, int romSize, int sramSize, UInt32 cpuClock)
{
    SlotCallbacks callbacks = { moonsoundDestroy, NULL, NULL, moonsoundGetState, moonsoundSetState };
    Moonsound* moonsound = new Moonsound;

    theMoonsound = moonsound;

    moonsound->mixer = mixer;
    moonsound->clock = cpuClock;
    moonsound->counter1 = -1;
    moonsound->counter2 = -1;
    moonsound->sramSize = 1024 * sramSize;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_MSX_MOONSOUND, 1, moonsoundSync, moonsound);

    moonsound->slotHandle = slotRegisterUnslotted(AUDIO_MOONSOUND, &callbacks, moonsound);

    moonsound->ymf262 = new YMF262(0, cpuClock);
    moonsound->ymf262->setSampleRate(SAMPLERATE);
	moonsound->ymf262->setVolume(32767);

    moonsound->ymf278 = new YMF278(0, sramSize, romData, romSize, cpuClock);
    moonsound->ymf278->setSampleRate(SAMPLERATE);
    moonsound->ymf278->setVolume(32767);

    ioPortRegister(0x7e, (IoPortRead)moonsoundRead, (IoPortWrite)moonsoundWrite, moonsound);
    ioPortRegister(0x7f, (IoPortRead)moonsoundRead, (IoPortWrite)moonsoundWrite, moonsound);
    ioPortRegister(0xc4, (IoPortRead)moonsoundRead, (IoPortWrite)moonsoundWrite, moonsound);
    ioPortRegister(0xc5, (IoPortRead)moonsoundRead, (IoPortWrite)moonsoundWrite, moonsound);
    ioPortRegister(0xc6, (IoPortRead)moonsoundRead, (IoPortWrite)moonsoundWrite, moonsound);
    ioPortRegister(0xc7, (IoPortRead)moonsoundRead, (IoPortWrite)moonsoundWrite, moonsound);

    return 1;
}


