/*****************************************************************************
** File:        romMapperKorean90.c
**
** Author:      Daniel Vik
**
** Description: Rom mapper for Korean90 cartridges
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "romMapperKorean90.h"
#include "romMapper.h"
#include "SlotManager.h"
#include "IoPort.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

typedef struct {
    UInt8* romData;
    int slot;
    int sslot;
    int startPage;
    int size;
    int romMapper[4];
} RomMapperKorean90;

static void destroy(RomMapperKorean90* rm)
{
    ioPortUnregister(0x77);
    slotUnregister(rm->slot, rm->sslot, rm->startPage);

    free(rm->romData);
    free(rm);
}

static int getState(RomMapperKorean90* rm, UInt8* buffer, UInt32 systemTime)
{
    memcpy(buffer, rm->romMapper, sizeof(rm->romMapper));
    return sizeof(rm->romMapper);
}

static int setState(RomMapperKorean90* rm, UInt8* buffer, UInt32 systemTime)
{
    int i;

    memcpy(rm->romMapper, buffer, sizeof(rm->romMapper));

    for (i = 0; i < 4; i++) {   
        slotMapPage(rm->slot, rm->sslot, rm->startPage + i, rm->romData + rm->romMapper[i] * 0x2000, 1, 0);
    }

    return sizeof(rm->romMapper);
}

static void write(RomMapperKorean90* rm, UInt16 address, UInt8 value, UInt32 systemTime) 
{
    int page = (value & 0x3f) << 1;
    int mode = value >> 6;
    int i;

    switch (mode) {
    case 0:
    case 1:
        rm->romMapper[0] = page + 0;
        rm->romMapper[1] = page + 1;
        rm->romMapper[2] = page + 0;
        rm->romMapper[3] = page + 1;
        break;
    case 2:
        rm->romMapper[0] = (page & 0xfc) + 0;
        rm->romMapper[1] = (page & 0xfc) + 1;
        rm->romMapper[2] = (page & 0xfc) + 2;
        rm->romMapper[3] = (page & 0xfc) + 3;
        break;
    case 3:
        rm->romMapper[0] = page + 0;
        rm->romMapper[1] = page + 1;
        rm->romMapper[2] = page + 1;
        rm->romMapper[3] = page + 0;
        break;
    }

    for (i = 0; i < 4; i++) {   
        slotMapPage(rm->slot, rm->sslot, rm->startPage + i, rm->romData + rm->romMapper[i] * 0x2000, 1, 0);
    }
}

int romMapperKorean90Create(char* filename, UInt8* romData, 
                            int size, int slot, int sslot, int startPage) 
{
    SlotCallbacks callbacks = { destroy, NULL, NULL, getState, setState };
    RomMapperKorean90* rm;
    int i;

    if (size < 0x8000) {
        return 0;
    }

    rm = malloc(sizeof(RomMapperKorean90));

    slotRegister(ROM_KOREAN90, slot, sslot, startPage, 4, &callbacks, rm);

    rm->romData = malloc(size);
    memcpy(rm->romData, romData, size);
    rm->size = size;
    rm->slot  = slot;
    rm->sslot = sslot;
    rm->startPage  = startPage;

    rm->romMapper[0] = 0;
    rm->romMapper[1] = 1;
    rm->romMapper[2] = 2;
    rm->romMapper[3] = 3;

    for (i = 0; i < 4; i++) {   
        slotMapPage(rm->slot, rm->sslot, rm->startPage + i, rm->romData + rm->romMapper[i] * 0x2000, 1, 0);
    }
    
    ioPortRegister(0x77, NULL, write, rm);

    return 1;
}

