/*****************************************************************************
** File:
**      MSX.c
**
** Author:
**      Daniel Vik
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>

#include "MSX.h"
#include "RTC.h"
#include "JoystickIO.h"
#include "AY8910.h"
#include "YM2413.h"
#include "Y8950.h"
#include "SCC.h"
#include "KeyClick.h"
#include "Moonsound.h"
#include "audioMixer.h"
#include "romMapper.h"
#include "VDP.h"
#include "Z80.h"
#include "V9938.h"
#include "romMapper.h"
#include "Casette.h"
#include "Disk.h"
#include "MegaromCartridge.h"
#include "IoPort.h"
#include "SlotManager.h"
#include "RomLoader.h"
#include "ramMapper.h"
#include "ramNormal.h"
#include "romMapperNormal.h"
#include "romMapperKanji.h"
#include "romMapperKanji12.h"
#include "romMapperBasic.h"
#include "romMapperDisk.h"
#include "romMapperStandard.h"
#include "romMapperMsxDos2.h"
#include "romMapperKonami5.h"
#include "romMapperKonami4.h"
#include "romMapperKoei.h"
#include "romMapperHolyQuran.h"
#include "romMapperMegaRam.h"
#include "romMapperASCII8.h"
#include "romMapperASCII16.h"
#include "romMapperGameMaster2.h"
#include "romMapperASCII8sram.h"
#include "romMapperASCII16sram.h"
#include "romMapperASCII16nf.h"
#include "romMapperKonami4nf.h"
#include "romMapperPlain.h"
#include "romMapperHarryFox.h"
#include "romMapperHalnote.h"
#include "romMapperMsxAudio.h"
#include "romMapperRType.h"
#include "romMapperCrossBlaim.h"
#include "romMapperKorean80.h"
#include "romMapperKorean90.h"
#include "romMapperKorean126.h"
#include "romMapperPAC.h"
#include "romMapperFMPAC.h"
#include "romMapperLodeRunner.h"
#include "romMapperSCCplus.h"
#include "romMapperPanasonic.h"
#include "romMapperNational.h"
#include "romFMPAC.h"
#include "sramMapperMatsuchita.h"
#include "sramMapperS1985.h"
#include "romMapperF4device.h"
#include "romMapperBunsetu.h"

extern int  WaitForSync(void);
extern void Keyboard(UInt8* keybardMap);


UInt8* msxLoadStateBuffer(const char* stateFile, int* stateSize);
int msxStateValid(UInt8* stateBuffer, int stateSize);
int msxSetState(UInt8* stateBuffer, int stateSize);


static char saveStateVersion[32] = "blueMSX - state  v 2";

// Hardware
static Machine*        msxMachine;
static DeviceInfo*     msxDevInfo;
static AY8910*         ay8910;
static SCC*            scc;
static AudioKeyClick*  keyClick;
static Z80             z80;
static RTC*            rtc;
static JoystickIO*     joyIO;
static UInt32          nextSyncTime;
static UInt32          loopTime;
static int             syncCount;
static int             SyncPeriod;
UInt8           KeyMap[16];
UInt32                 systemTime;
int                    MsxFrequency;
static UInt8*          msxRam;
static UInt32          msxRamSize;


static UInt8 readMisc(void* ref, UInt16 ioPort, UInt32 systemTime)
{
    switch (ioPort & 0xff) {
    case 0x04: 
        return 2;

    case 0x05: 
        return 0;

    case 0x90:
        return 0xfd;
    }

    return 0xff;
}

void xbox_set_memory( unsigned char *ptr, unsigned int ramsize );

static int initMachine(Machine* machine, 
                       Mixer* mixer, 
                       int clearRAM,
                       int enableYM2413,
                       int enableY8950,
                       int enableMoonsound,
                       int moonsoundSRAM)
{
    char cmosName[128];
    void* jisyoRom = NULL;
    int jisyoRomSize = 0;
    int success = 1;
    UInt8* buf;
    int size;
    int i;

    sprintf(cmosName, "SRAM\\%s.cmos", machine->name);
    rtc = rtcCreate(cmosName);

    msxRam = NULL;

    cartridgeInit(scc);

    vdpInit(&z80, machine->video.vramSize / 0x4000, systemTime, machine->video.vdpVersion);

    for (i = 0; i < 4; i++) {
        slotSetSubslotted(i, machine->slot[i].subslotted);
    }

    /* Map megarom cartridges */
    for (i = 0; i < 2; i++) {
        cartridgeSetSlotInfo(i, machine->cart[i].slot, machine->cart[i].subslot);
    }

    for (i = 0; i < machine->slotInfoCount; i++) {
        int slot;
        int subslot;
        int startPage;
        char* romName;
        
        // Don't map slots with error
        if (machine->slotInfo[i].error) {
            continue;
        }

        romName   = strlen(machine->slotInfo[i].inZipName) ? machine->slotInfo[i].inZipName : machine->slotInfo[i].name;
        slot      = machine->slotInfo[i].slot;
        subslot   = machine->slotInfo[i].subslot;
        startPage = machine->slotInfo[i].startPage;
        size      = 0x2000 * machine->slotInfo[i].pageCount;

        if (machine->slotInfo[i].romType == RAM_NORMAL) {
            success &= ramNormalCreate(size, slot, subslot, startPage, clearRAM, &msxRam, &msxRamSize);
            continue;
        }

        if (machine->slotInfo[i].romType == RAM_MAPPER) {
            success &= ramMapperCreate(size, slot, subslot, startPage, clearRAM, &msxRam, &msxRamSize);
            continue;
        }

        if (machine->slotInfo[i].romType == ROM_SCCPLUS) {
            success &= romMapperSCCplusCreate(NULL, NULL, 0x10000, slot, subslot, startPage, scc);
            continue;
        }

        if (machine->slotInfo[i].romType == ROM_PAC) {
            success &= romMapperPACCreate("Pac.rom", NULL, 0, slot, subslot, startPage);
            continue;
        }

        if (machine->slotInfo[i].romType == ROM_MEGARAM) {
            success &= romMapperMegaRAMCreate(size, slot, subslot, startPage);
            continue;
        }

        if (machine->slotInfo[i].romType == SRAM_MATSUCHITA) {
            success &= sramMapperMatsushitaCreate();
            continue;
        }

        if (machine->slotInfo[i].romType == SRAM_S1985) {
            success &= sramMapperS1985Create();
            continue;
        }

        if (machine->slotInfo[i].romType == ROM_F4DEVICE) {
            success &= romMapperF4deviceCreate(0);
            continue;
        }

        if (machine->slotInfo[i].romType == ROM_F4INVERTED) {
            success &= romMapperF4deviceCreate(1);
            continue;
        }

        buf = romLoad(machine->slotInfo[i].name, machine->slotInfo[i].inZipName, &size);

        if (buf == NULL) {
            success = 0;
            continue;
        }

        switch (machine->slotInfo[i].romType) {
        case ROM_0x4000:
            romMapperNormalCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_BASIC:
            romMapperBasicCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_PLAIN:
            romMapperPlainCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_STANDARD:
            success &= romMapperStandardCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_MSXDOS2:
            success &= romMapperMsxDos2Create(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_KONAMI5:
            success &= romMapperKonami5Create(romName, buf, size, slot, subslot, startPage, scc);
            break;
            
        case ROM_KONAMI4:
            success &= romMapperKonami4Create(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_HOLYQURAN:
            success &= romMapperHolyQuranCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_ASCII8:
            success &= romMapperASCII8Create(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_ASCII16:
            success &= romMapperASCII16Create(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_PANASONIC:
            success &= romMapperPanasonicCreate(romName, buf, size, slot, subslot, startPage, 0x4000);
            break;
            
        case ROM_ASCII8SRAM:
            success &= romMapperASCII8sramCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_ASCII16SRAM:
            success &= romMapperASCII16sramCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_MSXAUDIO:
            success &= romMapperMsxAudioCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_KOEI:
            success &= romMapperKoeiCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_NATIONAL:
            success &= romMapperNationalCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_KONAMI4NF:
            success &= romMapperKonami4nfCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_ASCII16NF:
            success &= romMapperASCII16nfCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_GAMEMASTER2:
            success &= romMapperGameMaster2Create(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_HARRYFOX:
            success &= romMapperHarryFoxCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_HALNOTE:
            success &= romMapperHalnoteCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_RTYPE:
            success &= romMapperRTypeCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_CROSSBLAIM:
            success &= romMapperCrossBlaimCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_LODERUNNER:
            success &= romMapperLodeRunnerCreate(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_KOREAN80:
            success &= romMapperKorean80Create(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_KOREAN90:
            success &= romMapperKorean90Create(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_KOREAN126:
            success &= romMapperKorean126Create(romName, buf, size, slot, subslot, startPage);
            break;
            
        case ROM_FMPAC:
            success &= romMapperFMPACCreate(romName, buf, size, slot, subslot, startPage);
            break;

        case ROM_NORMAL:
            success &= romMapperNormalCreate("", buf, size, slot, subslot, startPage);
            break;

        case ROM_DISK:
            success &= romMapperDiskCreate("", buf, size, slot, subslot, startPage);
           break;

        case ROM_KANJI:
            success &= romMapperKanjiCreate(buf, size);
            break;

        case ROM_KANJI12:
            success &= romMapperKanji12Create(buf, size);
            break;

        case ROM_JISYO:
            if (jisyoRom == NULL) {
                jisyoRom = malloc(size);
                memcpy(jisyoRom, buf, size);
                jisyoRomSize = size;
            }
            break;

        case ROM_BUNSETU:
            success &= romMapperBunsetuCreate(romName, buf, size, slot, subslot, startPage, jisyoRom, jisyoRomSize);
            break;
        }
        free(buf);
    }

    if (jisyoRom != NULL) {
        free(jisyoRom);
    }

    // Enable sound chips
    if (enableY8950) {
        success &= y8950Create(mixer, systemTime);
    }

    if (enableYM2413) {
        success &= ym2413Create(mixer, systemTime);
    }

    if (enableMoonsound) {
        buf = romLoad("internal/MOONSOUND.rom", NULL, &size);
        success &= moonsoundCreate(mixer, buf, size, moonsoundSRAM, 0);
        free(buf);
    }

    for (i = 0; i < 8; i++) {
        slotMapRamPage(0, 0, i);
    }

    /* Register misc io ports */
    ioPortRegister(0x04, readMisc, NULL, NULL);
    ioPortRegister(0x05, readMisc, NULL, NULL);
    ioPortRegister(0x90, readMisc, NULL, NULL);

	xbox_set_memory( msxRam, msxRamSize );

    return success;
}

int msxRun(Machine* machine, 
           DeviceInfo* devInfo,
           Mixer* mixer,
           char* stateFile,
           int syncPeriod,
           int frequency,
           int clearRam)
{
    UInt8* stateBuffer;
    int stateBufferSize = 0;
    int success;
    int i;

    // If we're running from a state file, use its machine
    // and user configuration
    stateBuffer = msxLoadStateBuffer(stateFile, &stateBufferSize);
    if (stateBuffer != NULL) {
        if (!msxStateValid(stateBuffer, stateBufferSize)) {
            free(stateBuffer);
            return 0;
        }
        memcpy(machine, stateBuffer + sizeof(saveStateVersion), sizeof(Machine));
        memcpy(devInfo, stateBuffer + sizeof(saveStateVersion) + sizeof(Machine), sizeof(DeviceInfo));
    }

    SyncPeriod   = syncPeriod;
    MsxFrequency = frequency;

    systemTime    = 0;
    nextSyncTime  = 0;
    loopTime      = 0;
    syncCount     = 0;

    msxMachine      = machine;
    msxDevInfo      = devInfo;

    ioPortReset();

    mixerReset(mixer, systemTime);

    ay8910    = ay8910Create(mixer, systemTime);
    scc       = sccCreate(mixer, systemTime);
    keyClick  = audioKeyClickCreate(mixer, systemTime);

    slotManagerCreate(KeyMap, keyClick);

    success = initMachine(machine, mixer, clearRam, 
                          devInfo->audio.enableYM2413, 
                          devInfo->audio.enableY8950, 
                          devInfo->audio.enableMoonsound,
                          devInfo->audio.moonsoundSRAM);

    joyIO = joystickIoCreate(ay8910);

    for (i = 0; i < 2; i++) {
        if (devInfo->cartridge[i].inserted) {
            msxChangeCartridge(i, devInfo->cartridge[i].type, 
                                  devInfo->cartridge[i].name,
                                  devInfo->cartridge[i].inZipName);
        }
    }

    for (i = 0; i < 2; i++) {
        if (devInfo->diskette[i].inserted) {
            msxChangeDiskette(i, devInfo->diskette[i].name,
                                 devInfo->diskette[i].inZipName);
        }
    }

    if (devInfo->diskette[i].inserted) {
        msxChangeCassette(devInfo->cassette.name,
                          devInfo->cassette.inZipName);
    }

    memset(KeyMap, 0xff, 16);
    
    z80Reset(&z80, 0);

    if (stateBuffer != NULL) {
        success &= msxSetState(stateBuffer, stateBufferSize);
        free(stateBuffer);
    }

    if (success) {
        /* Start execution of the code */
        for (;;) {
            if (syncCount >= SyncPeriod) {
                int rv = WaitForSync();
                if (rv) {
                    break;
                }
                syncCount -= SyncPeriod;
                mixerSync(mixer, systemTime);    
                Keyboard(KeyMap);
            }

            syncCount += loopTime;
            nextSyncTime += loopTime;

            moonsoundTick(nextSyncTime);
            y8950Tick(nextSyncTime); 

            z80execute(&z80, nextSyncTime);
            vdpSync(nextSyncTime);

            loopTime = vdpRefreshLine(nextSyncTime);
        }
    }

    rtcDestroy(rtc);

    joystickIoDestroy(joyIO);

    ay8910Destroy(ay8910);
    sccDestroy(scc);
    audioKeyClickDestroy(keyClick);

    slotManagerDestroy();

    msxChangeDiskette(0, NULL, NULL);
    msxChangeDiskette(1, NULL, NULL);

    msxChangeCassette(0, 0);

    vdpDestroy();

    msxMachine = NULL;
    msxDevInfo = NULL;

    return success;
}

void msxSetFrequency(int syncPeriod, int frequency)
{
    SyncPeriod   = syncPeriod;
    MsxFrequency = frequency;
}

static int frontSwitchState = 0;
static int audioSwitchState = 0;

int frontSwitchEnabled()
{
    return frontSwitchState;
}

void msxSetFrontSwitch(int state)
{
    frontSwitchState = state;
}

void msxSetAudioSwitch(int state)
{
    audioSwitchState = state;
}

int audioSwitchEnabled()
{
    return audioSwitchState;
}

UInt8* msxGetRamPage(int page) {
    static UInt8 emptyRam[0x2000];

    if (msxRam == NULL) {
        return emptyRam;
    }

	return msxRam + ((page * 0x2000) & (msxRamSize - 1));
}

void msxChangeCassette(char *name, const char *fileInZipFile)
{
    if (name && strlen(name) == 0) {
        name = NULL;
    }

    if (fileInZipFile && strlen(fileInZipFile) == 0) {
        fileInZipFile = NULL;
    }

    if (msxDevInfo != NULL) {
        msxDevInfo->cassette.inserted = name != NULL;

        strcpy(msxDevInfo->cassette.name, name ? name : "");
        strcpy(msxDevInfo->cassette.inZipName, fileInZipFile ? fileInZipFile : "");
    }

    tapeInsert(name, fileInZipFile);
}

void msxChangeCartridge(int cartNo, RomType romType, char* cart, char* cartZip)
{
    if (cart && strlen(cart) == 0) {
        cart = NULL;
    }

    if (cartZip && strlen(cartZip) == 0) {
        cartZip = NULL;
    }

    if (msxDevInfo != NULL) {
        msxDevInfo->cartridge[cartNo].inserted = cart != NULL;
        msxDevInfo->cartridge[cartNo].type = romType;

        strcpy(msxDevInfo->cartridge[cartNo].name, cart ? cart : "");
        strcpy(msxDevInfo->cartridge[cartNo].inZipName, cartZip ? cartZip : "");
    }

    cartridgeInsert(cartNo, romType, cart, cartZip);
}

void msxChangeDiskette(int driveId, char* fileName, const char* fileInZipFile)
{
    if (fileName && strlen(fileName) == 0) {
        fileName = NULL;
    }

    if (fileInZipFile && strlen(fileInZipFile) == 0) {
        fileInZipFile = NULL;
    }

    if (msxDevInfo != NULL) {
        msxDevInfo->diskette[driveId].inserted = fileName != NULL;

        strcpy(msxDevInfo->diskette[driveId].name, fileName ? fileName : "");
        strcpy(msxDevInfo->diskette[driveId].inZipName, fileInZipFile ? fileInZipFile : "");
    }

    ChangeDisk(driveId ,fileName, fileInZipFile);
}

UInt8 RdZ80(UInt16 address)
{
    return slotRead(address, systemTime);
}

void WrZ80(UInt16 address, UInt8 value)
{
    slotWrite(address, value, systemTime);
}

UInt8 InZ80(UInt16 ioPort)
{
    return ioPortRead(ioPort, systemTime);
}

void OutZ80(UInt16 ioPort, UInt8 value)
{
    ioPortWrite(ioPort, value, systemTime);
}

void* msxGetState(int* stateSize)
{
    UInt8* stateBuffer = malloc(0x800000);
    int size = 0;

    memcpy(stateBuffer + size, saveStateVersion, sizeof(saveStateVersion));
    size += sizeof(saveStateVersion);

    memcpy(stateBuffer + size, msxMachine, sizeof(Machine));
    size += sizeof(Machine);

    memcpy(stateBuffer + size, msxDevInfo, sizeof(DeviceInfo));
    size += sizeof(DeviceInfo);

    memcpy(stateBuffer + size, &z80, sizeof(Z80));
    size += sizeof(Z80);

    memcpy(stateBuffer + size, &systemTime, sizeof(systemTime));
    size += sizeof(systemTime);

    memcpy(stateBuffer + size, &nextSyncTime, sizeof(nextSyncTime));
    size += sizeof(nextSyncTime);

    memcpy(stateBuffer + size, &loopTime, sizeof(loopTime));
    size += sizeof(loopTime);

    size += rtcGetState(rtc, stateBuffer + size, systemTime);
    size += ay8910GetState(ay8910, stateBuffer + size, systemTime);
    size += sccGetState(scc, stateBuffer + size, systemTime);
    size += joystickIoGetState(joyIO, stateBuffer + size, systemTime);
    size += vdpGetState(NULL, stateBuffer + size, systemTime);
    size += slotGetState(NULL, stateBuffer + size, systemTime);

    *stateSize = size;

    return stateBuffer;
}

void* msxGetStateMem(int* stateSize)
{
    UInt8* stateBuffer = malloc(0x300000);
    int size = 0;

    memcpy(stateBuffer + size, saveStateVersion, sizeof(saveStateVersion));
    size += sizeof(saveStateVersion);

    memcpy(stateBuffer + size, msxMachine, sizeof(Machine));
    size += sizeof(Machine);

    memcpy(stateBuffer + size, msxDevInfo, sizeof(DeviceInfo));
    size += sizeof(DeviceInfo);

    memcpy(stateBuffer + size, &z80, sizeof(Z80));
    size += sizeof(Z80);

    memcpy(stateBuffer + size, &systemTime, sizeof(systemTime));
    size += sizeof(systemTime);

    memcpy(stateBuffer + size, &nextSyncTime, sizeof(nextSyncTime));
    size += sizeof(nextSyncTime);

    memcpy(stateBuffer + size, &loopTime, sizeof(loopTime));
    size += sizeof(loopTime);

    size += rtcGetState(rtc, stateBuffer + size, systemTime);
    size += ay8910GetState(ay8910, stateBuffer + size, systemTime);
    size += sccGetState(scc, stateBuffer + size, systemTime);
    size += joystickIoGetState(joyIO, stateBuffer + size, systemTime);
    size += vdpGetState(NULL, stateBuffer + size, systemTime);

	size += slotGetState(NULL, stateBuffer + size, systemTime);

    *stateSize = size;

    return stateBuffer;
}

int msxStateValid(UInt8* stateBuffer, int stateSize) 
{
    if (stateSize < sizeof(saveStateVersion)) {
        return 0;
    }

    if (memcmp(saveStateVersion, stateBuffer, strlen(saveStateVersion))) {
        return 0;
    }

    return 1;
}

int msxSetState(UInt8* stateBuffer, int stateSize)
{
    int size = 0;

    size += sizeof(saveStateVersion);

    memcpy(msxMachine, stateBuffer + size, sizeof(Machine));
    size += sizeof(Machine);

    memcpy(msxDevInfo, stateBuffer + size, sizeof(DeviceInfo));
    size += sizeof(DeviceInfo);

	ReopenDisks( msxDevInfo ) ;

    memcpy(&z80, stateBuffer + size, sizeof(Z80));
    size += sizeof(Z80);

    memcpy(&systemTime, stateBuffer + size, sizeof(systemTime));
    size += sizeof(systemTime);

    memcpy(&nextSyncTime, stateBuffer + size, sizeof(nextSyncTime));
    size += sizeof(nextSyncTime);

    memcpy(&loopTime, stateBuffer + size, sizeof(loopTime));
    size += sizeof(loopTime);

    size += rtcSetState(rtc, stateBuffer + size, systemTime);
    size += ay8910SetState(ay8910, stateBuffer + size, systemTime);
    size += sccSetState(scc, stateBuffer + size, systemTime);
    size += joystickIoSetState(joyIO, stateBuffer + size, systemTime);
    size += vdpSetState(NULL, stateBuffer + size, systemTime);
    size += slotSetState(NULL, stateBuffer + size, systemTime);

    return size == stateSize ? 1 : 0;
}

int msxSetStateMem(UInt8* stateBuffer, int stateSize)
{
    int size = 0;

    size += sizeof(saveStateVersion);

    memcpy(msxMachine, stateBuffer + size, sizeof(Machine));
    size += sizeof(Machine);

    memcpy(msxDevInfo, stateBuffer + size, sizeof(DeviceInfo));
    size += sizeof(DeviceInfo);

	ReopenDisks( msxDevInfo ) ;

    memcpy(&z80, stateBuffer + size, sizeof(Z80));
    size += sizeof(Z80);

    memcpy(&systemTime, stateBuffer + size, sizeof(systemTime));
    size += sizeof(systemTime);

    memcpy(&nextSyncTime, stateBuffer + size, sizeof(nextSyncTime));
    size += sizeof(nextSyncTime);

    memcpy(&loopTime, stateBuffer + size, sizeof(loopTime));
    size += sizeof(loopTime);

    size += rtcSetState(rtc, stateBuffer + size, systemTime);
    size += ay8910SetState(ay8910, stateBuffer + size, systemTime);
    size += sccSetState(scc, stateBuffer + size, systemTime);
    size += joystickIoSetState(joyIO, stateBuffer + size, systemTime);
    size += vdpSetState(NULL, stateBuffer + size, systemTime);
    size += slotSetState(NULL, stateBuffer + size, systemTime);

    return size == stateSize ? 1 : 0;
}

//size_t xbox_mfread( const void *buffer, size_t len1, size_t len2, void *st ) ;
//size_t xbox_mfwrite( const void *buffer, size_t len1, size_t len2, void *st ) ;
//unsigned char *xbox_memgetchunk( unsigned int *memsize, void *st ) ;

void xbox_savestatemem( void *sm)
{
    int size;
    //UInt8* stateBuffer = msxGetStateMem(&size);
	//xbox_mfwrite( stateBuffer, 1, size, sm ) ;
    //free(stateBuffer);
}

void xbox_loadstatemem( void *sm)
{
	unsigned char *stateBuffer ;
	unsigned int  statesize ;

	//stateBuffer = xbox_memgetchunk( &statesize, sm ) ;
	//msxSetStateMem(stateBuffer, statesize);
}

int msxSaveState(const char* stateFile)
{   
    int size;
    UInt8* stateBuffer = msxGetState(&size);
    FILE* file;

    file = fopen(stateFile, "wb");
    fwrite(stateBuffer, 1, size, file);
    fclose(file);
    free(stateBuffer);
    return 1;
}


UInt8* msxLoadStateBuffer(const char* stateFile, int* stateSize)
{
    UInt8* stateBuffer = NULL ;
    FILE* file;
    int size;

    if (stateFile == NULL) {
		//free( stateBuffer ) ;
        return NULL;
    }

    file = fopen(stateFile, "rb");

	if ( file == NULL )
	{
		//free( stateBuffer ) ;
        return NULL;
	}

    fseek(file, 0, SEEK_END);
    size = ftell(file);
    fseek(file, 0, SEEK_SET);

	stateBuffer = malloc(size) ;
    fread(stateBuffer, 1, size, file);   
    fclose(file);

    *stateSize = size;

    return stateBuffer;
}

int msxLoadState(const char* stateFile)
{   
    UInt8* stateBuffer;
    int stateBufferSize = 0;
    int success;
    int i;

    // If we're running from a state file, use its machine
    // and user configuration
    stateBuffer = msxLoadStateBuffer(stateFile, &stateBufferSize);
    if (stateBuffer != NULL) {
        if (!msxStateValid(stateBuffer, stateBufferSize)) {
            free(stateBuffer);
            return 0;
        }
    }
    if (stateBuffer != NULL) {
        msxSetState(stateBuffer, stateBufferSize);
        free(stateBuffer);
    }

	return 1 ;
}
