/*****************************************************************************
**
** This command engine may be used in any MSX emulator. It does not matter 
** under which conditions that the emulator gets distributed. Be it open-
** source or closed-source. Be it commercially or free-of-charge.
** 
** This command engine may not be used in any other kind of softwares.
** Using this command engine is at own risk. The author is not responsible 
** nor liable for any damages that may occur intentionally or unintentionally 
** from it's usage.
**
******************************************************************************
*/
#include "V9938.h"
#include <string.h>
#include "VDP.h"

/*************************************************************
** Different compilers inline C functions differently.
**************************************************************
*/
#ifdef __GNUC__
#define INLINE inline
#else
#define INLINE static
#endif

#define VDPSTATUS_TR 0x80
#define VDPSTATUS_BO 0x10
#define VDPSTATUS_CE 0x01

/*************************************************************
** Other useful defines
**************************************************************
*/
#define VDP_VRMP5(X, Y) (VRAM + ((Y & 1023) << 7) + ((X & 255) >> 1))
#define VDP_VRMP6(X, Y) (VRAM + ((Y & 1023) << 7) + ((X & 511) >> 2))
#define VDP_VRMP7(X, Y) (VRAM + ((Y &  511) << 7) + ((X & 511) >> 2) + ((X & 2) << 15))
#define VDP_VRMP8(X, Y) (VRAM + ((Y &  511) << 7) + ((X & 255) >> 1) + ((X & 1) << 16))

#define GET_COMMAND_TIME(timeList) timeList[((VDP[1] >> 6) & 1) | (VDP[8] & 2)]

#define CM_ABRT  0x0
#define CM_POINT 0x4
#define CM_PSET  0x5
#define CM_SRCH  0x6
#define CM_LINE  0x7
#define CM_LMMV  0x8
#define CM_LMMM  0x9
#define CM_LMCM  0xA
#define CM_LMMC  0xB
#define CM_HMMV  0xC
#define CM_HMMM  0xD
#define CM_YMMM  0xE
#define CM_HMMC  0xF

/*************************************************************
** Many VDP commands are executed in some kind of loop but
** essentially, there are only a few basic loop structures
** that are re-used. We define the loop structures that are
** re-used here so that they have to be entered only once
**************************************************************
*/
#define pre_loop \
    while ((cnt-=MMC.delta) > 0) {

/* Loop over DX, DY */
#define post__x_y(MX) \
    if (!--ANX || ((ADX+=TX)&MX)) { \
    if (!(--NY&1023) || (DY+=TY)==-1) \
    break; \
      else { \
      ADX=DX; \
      ANX=NX; \
      } \
    } \
    }

/* Loop over DX, SY, DY */
#define post__xyy(MX) \
    if ((ADX+=TX)&MX) { \
    if (!(--NY&1023) || (SY+=TY)==-1 || (DY+=TY)==-1) \
    break; \
      else \
      ADX=DX; \
    } \
  }

/* Loop over SX, DX, SY, DY */
#define post_xxyy(MX) \
    if (!--ANX || ((ASX+=TX)&MX) || ((ADX+=TX)&MX)) { \
    if (!(--NY&1023) || (SY+=TY)==-1 || (DY+=TY)==-1) \
    break; \
      else { \
      ASX=SX; \
      ADX=DX; \
      ANX=NX; \
      } \
    } \
  }


/*************************************************************
** VdpCmd State instance
**************************************************************
*/
typedef struct {
    int   SX;
    int   SY;
    int   DX;
    int   DY;
    int   TX;
    int   TY;
    int   NX;
    int   NY;
    int   MX;
    int   ASX;
    int   ADX;
    int   ANX;
    UInt8 CL;
    UInt8 LO;
    UInt8 CM;
    int   VdpOpsCnt;
    int   VdpEngine;
    int   delta;
} VdpCmdState;

VdpCmdState MMC;


/*************************************************************
** Forward declarations
**************************************************************
*/
static UInt8 *getVramPointer(UInt8 M, int X, int Y);

static UInt8 getPixel(UInt8 SM, int SX, int SY);
static UInt8 getPixel5(int SX, int SY);
static UInt8 getPixel6(int SX, int SY);
static UInt8 getPixel7(int SX, int SY);
static UInt8 getPixel8(int SX, int SY);

static void setPixel(UInt8 SM, int DX, int DY, UInt8 CL, UInt8 OP);
static void setPixel5(int DX, int DY, UInt8 CL, UInt8 OP);
static void setPixel6(int DX, int DY, UInt8 CL, UInt8 OP);
static void setPixel7(int DX, int DY, UInt8 CL, UInt8 OP);
static void setPixel8(int DX, int DY, UInt8 CL, UInt8 OP);

static void setPixelLow(UInt8 *P, UInt8 CL, UInt8 M, UInt8 OP);

static void SrchEngine(void);
static void LineEngine(void);
static void LmmvEngine(void);
static void LmmmEngine(void);
static void LmcmEngine(void);
static void LmmcEngine(void);
static void HmmvEngine(void);
static void HmmmEngine(void);
static void YmmmEngine(void);
static void HmmcEngine(void);


/*************************************************************
** Constants
**************************************************************
*/
static UInt8 Mask[4] = { 0x0f, 0x03, 0x0f, 0xff };
static int   PPB[4]  = { 2, 4, 2, 1 };
static int   PPL[4]  = { 256, 512, 512, 256 };

static int srch_timing[8] = { 92,  125, 92,  92  };
static int line_timing[8] = { 120, 147, 120, 132 };
static int hmmv_timing[8] = { 49,  65,  49,  62  };
static int lmmv_timing[8] = { 98,  137, 98,  124 };
static int ymmm_timing[8] = { 65,  125, 65,  68  };
static int hmmm_timing[8] = { 92,  136, 92,  97  };
static int lmmm_timing[8] = { 129, 157, 129, 132 };


/*************************************************************
** getVramPointer
**
** Description:
**      Calculate addr of a pixel in vram
**************************************************************
*/
INLINE UInt8 *getVramPointer(UInt8 M,int X,int Y)
{
    switch(M) {
    case 0: 
        return VDP_VRMP5(X,Y);
    case 1: 
        return VDP_VRMP6(X,Y);
    case 2: 
        return VDP_VRMP7(X,Y);
    case 3: 
        return VDP_VRMP8(X,Y);
    }

    return(VRAM);
}


/*************************************************************
** getPixel5
**
** Description:
**      Get a pixel on screen 5
**************************************************************
*/
INLINE UInt8 getPixel5(int SX, int SY)
{
    return (*VDP_VRMP5(SX, SY) >> (((~SX)&1)<<2))&15;
}

/*************************************************************
** getPixel6
**
** Description:
**      Get a pixel on screen 6
**************************************************************
*/
INLINE UInt8 getPixel6(int SX, int SY)
{
    return (*VDP_VRMP6(SX, SY) >>(((~SX)&3)<<1))&3;
}

/*************************************************************
** getPixel7
**
**    Get a pixel on screen 7
**************************************************************
*/
INLINE UInt8 getPixel7(int SX, int SY)
{
    return (*VDP_VRMP7(SX, SY) >>(((~SX)&1)<<2))&15;
}

/*************************************************************
** getPixel8
**
** Description:
**      Get a pixel on screen 8
**************************************************************
*/
INLINE UInt8 getPixel8(int SX, int SY)
{
    return *VDP_VRMP8(SX, SY);
}

/*************************************************************
** getPixel
**
**    Get a pixel on a screen
**************************************************************
*/
INLINE UInt8 getPixel(UInt8 SM, int SX, int SY)
{
    switch(SM) {
    case 0: 
        return getPixel5(SX,SY);
    case 1: 
        return getPixel6(SX,SY);
    case 2: 
        return getPixel7(SX,SY);
    case 3: 
        return getPixel8(SX,SY);
    }

    return(0);
}

/*************************************************************
** setPixelLow
**
** Description:
**      Low level function to set a pixel on a screen
**************************************************************
*/
INLINE void setPixelLow(UInt8 *P, UInt8 CL, UInt8 M, UInt8 OP)
{
    switch (OP) {
    case 0: 
        *P = (*P & M) | CL; 
        break;
    case 1: 
        *P = *P & (CL | M); 
        break;
    case 2: 
        *P |= CL; 
        break;
    case 3: 
        *P ^= CL; 
        break;
    case 4: 
        *P = (*P & M) | ~(CL | M); 
        break;
    case 8: 
        if (CL) *P = (*P & M) | CL; 
        break;
    case 9: 
        if (CL) *P = *P & (CL | M); 
        break;
    case 10: 
        if (CL) {
            *P |= CL; 
        }
        break;
    case 11: 
        if (CL) {
            *P ^= CL; 
        }
        break;
    case 12: 
        if (CL) {
            *P = (*P & M) | ~(CL|M); 
        }
        break;
    }
}

/*************************************************************
** setPixel5
**
** Description:
**      Set a pixel on screen 5
**************************************************************
*/
INLINE void setPixel5(int DX, int DY, UInt8 CL, UInt8 OP)
{
    UInt8 SH = ((~DX)&1)<<2;

    setPixelLow(VDP_VRMP5(DX, DY), CL << SH, ~(15<<SH), OP);
}
 
/*************************************************************
** setPixel6
**
**    Set a pixel on screen 6
**************************************************************
*/
INLINE void setPixel6(int DX, int DY, UInt8 CL, UInt8 OP)
{
    UInt8 SH = ((~DX)&3)<<1;

    setPixelLow(VDP_VRMP6(DX, DY), CL << SH, ~(3<<SH), OP);
}

/*************************************************************
** setPixel7
**
** Description:
**      Set a pixel on screen 7
**************************************************************
*/
INLINE void setPixel7(int DX, int DY, UInt8 CL, UInt8 OP)
{
    UInt8 SH = ((~DX)&1)<<2;

    setPixelLow(VDP_VRMP7(DX, DY), CL << SH, ~(15<<SH), OP);
}

/*************************************************************
** setPixel8
**
**    Set a pixel on screen 8
**************************************************************
*/
INLINE void setPixel8(int DX, int DY, UInt8 CL, UInt8 OP)
{
    setPixelLow(VDP_VRMP8(DX, DY), CL, 0, OP);
}

/*************************************************************
** setPixel
**
** Description:
**      Set a pixel on a screen
**************************************************************
*/
INLINE void setPixel(UInt8 SM, int DX, int DY, UInt8 CL, UInt8 OP)
{
    switch (SM) {
    case 0: setPixel5(DX, DY, CL, OP); break;
    case 1: setPixel6(DX, DY, CL, OP); break;
    case 2: setPixel7(DX, DY, CL, OP); break;
    case 3: setPixel8(DX, DY, CL, OP); break;
    }
}

/*************************************************************
** SrchEgine
**
** Description:
**      Search a dot
**************************************************************
*/
static void SrchEngine(void)
{
    int SX=MMC.SX;
    int SY=MMC.SY;
    int TX=MMC.TX;
    int ANX=MMC.ANX;
    UInt8 CL=MMC.CL;
    int cnt;

    cnt = MMC.VdpOpsCnt;

#define pre_srch \
    pre_loop \
    if ((
#define post_srch(MX) \
    ==CL) ^ANX) { \
    VDPStatus[2]|=VDPSTATUS_BO; /* Border detected */ \
    break; \
    } \
    if ((SX+=TX) & MX) { \
    VDPStatus[2]&=~VDPSTATUS_BO; /* Border not detected */ \
    break; \
    } \
} 

    switch (ScrMode) {
    case 5: 
        pre_srch getPixel5(SX, SY) post_srch(256)
        break;
    case 6: 
        pre_srch getPixel6(SX, SY) post_srch(512)
        break;
    case 7: 
        pre_srch getPixel7(SX, SY) post_srch(512)
        break;
    case 8: 
        pre_srch getPixel8(SX, SY) post_srch(256)
        break;
    }

    if ((MMC.VdpOpsCnt=cnt)>0) {
        /* Command execution done */
        VDPStatus[2] &= ~VDPSTATUS_CE;
        MMC.VdpEngine=0;
        /* Update SX in VDP registers */
        VDPStatus[8] = SX & 0xFF;
        VDPStatus[9] = (SX >> 8) | 0xFE;
    }
    else {
        MMC.SX=SX;
    }
}

/*************************************************************
** LineEgine
**
** Description:
**      Draw a line
**************************************************************
*/
static void LineEngine(void)
{
    int DX=MMC.DX;
    int DY=MMC.DY;
    int TX=MMC.TX;
    int TY=MMC.TY;
    int NX=MMC.NX;
    int NY=MMC.NY;
    int ASX=MMC.ASX;
    int ADX=MMC.ADX;
    UInt8 CL=MMC.CL;
    UInt8 LO=MMC.LO;
    int cnt;

    cnt = MMC.VdpOpsCnt;

#define post_linexmaj(MX) \
    DX+=TX; \
    if ((ASX-=NY)<0) { \
    ASX+=NX; \
    DY+=TY; \
    } \
    ASX&=1023; /* Mask to 10 bits range */ \
    if (ADX++==NX || (DX&MX)) \
    break; \
}
#define post_lineymaj(MX) \
    DY+=TY; \
    if ((ASX-=NY)<0) { \
    ASX+=NX; \
    DX+=TX; \
    } \
    ASX&=1023; /* Mask to 10 bits range */ \
    if (ADX++==NX || (DX&MX)) \
    break; \
    }

    if ((VDP[45]&0x01)==0) {
        /* X-Axis is major direction */
        switch (ScrMode) {
        case 5: 
            pre_loop setPixel5(DX, DY, CL, LO); post_linexmaj(256)
            break;
        case 6: 
            pre_loop setPixel6(DX, DY, CL, LO); post_linexmaj(512)
            break;
        case 7: 
            pre_loop setPixel7(DX, DY, CL, LO); post_linexmaj(512)
            break;
        case 8: 
            pre_loop setPixel8(DX, DY, CL, LO); post_linexmaj(256)
            break;
        }
    }
    else {
        /* Y-Axis is major direction */
        switch (ScrMode) {
        case 5: 
            pre_loop setPixel5(DX, DY, CL, LO); post_lineymaj(256)
            break;
        case 6: 
            pre_loop setPixel6(DX, DY, CL, LO); post_lineymaj(512)
            break;
        case 7: 
            pre_loop setPixel7(DX, DY, CL, LO); post_lineymaj(512)
            break;
        case 8: 
            pre_loop setPixel8(DX, DY, CL, LO); post_lineymaj(256)
            break;
        }
    }

    if ((MMC.VdpOpsCnt=cnt)>0) {
        /* Command execution done */
        VDPStatus[2] &= ~VDPSTATUS_CE;
        MMC.VdpEngine=0;
        VDP[38]=DY & 0xFF;
        VDP[39]=(DY>>8) & 0x03;
    }
    else {
        MMC.DX=DX;
        MMC.DY=DY;
        MMC.ASX=ASX;
        MMC.ADX=ADX;
    }
}

/*************************************************************
** LmmvEngine
**
** Description:
**      VDP -> Vram
**************************************************************
*/
static void LmmvEngine(void)
{
    int DX=MMC.DX;
    int DY=MMC.DY;
    int TX=MMC.TX;
    int TY=MMC.TY;
    int NX=MMC.NX;
    int NY=MMC.NY;
    int ADX=MMC.ADX;
    int ANX=MMC.ANX;
    UInt8 CL=MMC.CL;
    UInt8 LO=MMC.LO;
    int cnt;

    cnt = MMC.VdpOpsCnt;

    switch (ScrMode) {
    case 5: 
        pre_loop setPixel5(ADX, DY, CL, LO); post__x_y(256)
        break;
    case 6: 
        pre_loop setPixel6(ADX, DY, CL, LO); post__x_y(512)
        break;
    case 7: 
        pre_loop setPixel7(ADX, DY, CL, LO); post__x_y(512)
        break;
    case 8: 
        pre_loop setPixel8(ADX, DY, CL, LO); post__x_y(256)
        break;
    }

    if ((MMC.VdpOpsCnt=cnt)>0) {
        /* Command execution done */
        VDPStatus[2] &= ~VDPSTATUS_CE;
        MMC.VdpEngine=0;
        if (!NY)
            DY+=TY;
        VDP[38]=DY & 0xFF;
        VDP[39]=(DY>>8) & 0x03;
        VDP[42]=NY & 0xFF;
        VDP[43]=(NY>>8) & 0x03;
    }
    else {
        MMC.DY=DY;
        MMC.NY=NY;
        MMC.ANX=ANX;
        MMC.ADX=ADX;
    }
}

/*************************************************************
** LmmmEngine
**
** Description:
**      Vram -> Vram
**************************************************************
*/
static void LmmmEngine(void)
{
    int SX=MMC.SX;
    int SY=MMC.SY;
    int DX=MMC.DX;
    int DY=MMC.DY;
    int TX=MMC.TX;
    int TY=MMC.TY;
    int NX=MMC.NX;
    int NY=MMC.NY;
    int ASX=MMC.ASX;
    int ADX=MMC.ADX;
    int ANX=MMC.ANX;
    UInt8 LO=MMC.LO;
    int cnt;

    cnt = MMC.VdpOpsCnt;

    switch (ScrMode) {
    case 5: 
        pre_loop setPixel5(ADX, DY, getPixel5(ASX, SY), LO); post_xxyy(256)
        break;
    case 6: 
        pre_loop setPixel6(ADX, DY, getPixel6(ASX, SY), LO); post_xxyy(512)
        break;
    case 7: 
        pre_loop setPixel7(ADX, DY, getPixel7(ASX, SY), LO); post_xxyy(512)
        break;
    case 8: 
        pre_loop setPixel8(ADX, DY, getPixel8(ASX, SY), LO); post_xxyy(256)
        break;
    }

    if ((MMC.VdpOpsCnt=cnt)>0) {
        /* Command execution done */
        VDPStatus[2] &= ~VDPSTATUS_CE;
        MMC.VdpEngine=0;
        if (!NY) {
            SY+=TY;
            DY+=TY;
        }
        else {
            if (SY==-1) {
                DY+=TY;
            }
        }
        VDP[42]=NY & 0xFF;
        VDP[43]=(NY>>8) & 0x03;
        VDP[34]=SY & 0xFF;
        VDP[35]=(SY>>8) & 0x03;
        VDP[38]=DY & 0xFF;
        VDP[39]=(DY>>8) & 0x03;
    }
    else {
        MMC.SY=SY;
        MMC.DY=DY;
        MMC.NY=NY;
        MMC.ANX=ANX;
        MMC.ASX=ASX;
        MMC.ADX=ADX;
    }
}

/*************************************************************
** LmcmEngine
**
** Description:
**      Vram -> CPU
**************************************************************
*/
static void LmcmEngine()
{
    if (!(VDPStatus[2] & VDPSTATUS_TR)) {
        VDPStatus[7] = VDP[44] = getPixel(ScrMode-5, MMC.ASX, MMC.SY);
        MMC.VdpOpsCnt -= MMC.delta;
        VDPStatus [2] |= VDPSTATUS_TR;

        if (!--MMC.ANX || ((MMC.ASX+=MMC.TX)&MMC.MX)) {
            if (!(--MMC.NY & 1023) || (MMC.SY+=MMC.TY)==-1) {
                VDPStatus[2] &= ~VDPSTATUS_CE;
                MMC.VdpEngine = 0;
                if (!MMC.NY)
                    MMC.DY+=MMC.TY;
                VDP[42]=MMC.NY & 0xFF;
                VDP[43]=(MMC.NY>>8) & 0x03;
                VDP[34]=MMC.SY & 0xFF;
                VDP[35]=(MMC.SY>>8) & 0x03;
            }
            else {
                MMC.ASX=MMC.SX;
                MMC.ANX=MMC.NX;
            }
        }
    }
}

/*************************************************************
** LmmcEngine
**
** Description:
**      CPU -> Vram
**************************************************************
*/
static void LmmcEngine(void)
{
    if (!(VDPStatus[2] & VDPSTATUS_TR)) {
        UInt8 SM=ScrMode-5;

        VDPStatus[7]=VDP[44]&=Mask[SM];
        setPixel(SM, MMC.ADX, MMC.DY, VDP[44], MMC.LO);
        MMC.VdpOpsCnt-=MMC.delta;
        VDPStatus[2] |= VDPSTATUS_TR;

        if (!--MMC.ANX || ((MMC.ADX+=MMC.TX)&MMC.MX)) {
            if (!(--MMC.NY&1023) || (MMC.DY+=MMC.TY)==-1) {
                VDPStatus[2] &= ~VDPSTATUS_CE;
                MMC.VdpEngine=0;
                if (!MMC.NY)
                    MMC.DY+=MMC.TY;
                VDP[42]=MMC.NY & 0xFF;
                VDP[43]=(MMC.NY>>8) & 0x03;
                VDP[38]=MMC.DY & 0xFF;
                VDP[39]=(MMC.DY>>8) & 0x03;
            }
            else {
                MMC.ADX=MMC.DX;
                MMC.ANX=MMC.NX;
            }
        }
    }
}

/*************************************************************
** HmmvEngine
**
** Description:
**      VDP --> Vram
**************************************************************
*/
static void HmmvEngine(void)
{
    int DX=MMC.DX;
    int DY=MMC.DY;
    int TX=MMC.TX;
    int TY=MMC.TY;
    int NX=MMC.NX;
    int NY=MMC.NY;
    int ADX=MMC.ADX;
    int ANX=MMC.ANX;
    UInt8 CL=MMC.CL;
    int cnt;

    cnt = MMC.VdpOpsCnt;

    switch (ScrMode) {
    case 5: 
        pre_loop *VDP_VRMP5(ADX, DY) = CL; post__x_y(256)
        break;
    case 6: 
        pre_loop *VDP_VRMP6(ADX, DY) = CL; post__x_y(512)
        break;
    case 7: 
        pre_loop *VDP_VRMP7(ADX, DY) = CL; post__x_y(512)
        break;
    case 8: 
        pre_loop *VDP_VRMP8(ADX, DY) = CL; post__x_y(256)
        break;
    }

    if ((MMC.VdpOpsCnt=cnt)>0) {
        /* Command execution done */
        VDPStatus[2] &= ~VDPSTATUS_CE;
        MMC.VdpEngine=0;
        if (!NY) {
            DY+=TY;
        }
        VDP[42]=NY & 0xFF;
        VDP[43]=(NY>>8) & 0x03;
        VDP[38]=DY & 0xFF;
        VDP[39]=(DY>>8) & 0x03;
    }
    else {
        MMC.DY=DY;
        MMC.NY=NY;
        MMC.ANX=ANX;
        MMC.ADX=ADX;
    }
}

/*************************************************************
** HmmmEngine
**
** Description:
**      Vram -> Vram
**************************************************************
*/
static void HmmmEngine(void)
{
    int SX=MMC.SX;
    int SY=MMC.SY;
    int DX=MMC.DX;
    int DY=MMC.DY;
    int TX=MMC.TX;
    int TY=MMC.TY;
    int NX=MMC.NX;
    int NY=MMC.NY;
    int ASX=MMC.ASX;
    int ADX=MMC.ADX;
    int ANX=MMC.ANX;
    int cnt;

    cnt = MMC.VdpOpsCnt;

    switch (ScrMode) {
    case 5: 
        pre_loop *VDP_VRMP5(ADX, DY) = *VDP_VRMP5(ASX, SY); post_xxyy(256)
        break;
    case 6: 
        pre_loop *VDP_VRMP6(ADX, DY) = *VDP_VRMP6(ASX, SY); post_xxyy(512)
        break;
    case 7: 
        pre_loop *VDP_VRMP7(ADX, DY) = *VDP_VRMP7(ASX, SY); post_xxyy(512)
        break;
    case 8: 
        pre_loop *VDP_VRMP8(ADX, DY) = *VDP_VRMP8(ASX, SY); post_xxyy(256)
        break;
    }

    if ((MMC.VdpOpsCnt=cnt)>0) {
        /* Command execution done */
        VDPStatus[2] &= ~VDPSTATUS_CE;
        MMC.VdpEngine=0;
        if (!NY) {
            SY+=TY;
            DY+=TY;
        }
        else {
            if (SY==-1) {
                DY+=TY;
            }
        }
        VDP[42]=NY & 0xFF;
        VDP[43]=(NY>>8) & 0x03;
        VDP[34]=SY & 0xFF;
        VDP[35]=(SY>>8) & 0x03;
        VDP[38]=DY & 0xFF;
        VDP[39]=(DY>>8) & 0x03;
    }
    else {
        MMC.SY=SY;
        MMC.DY=DY;
        MMC.NY=NY;
        MMC.ANX=ANX;
        MMC.ASX=ASX;
        MMC.ADX=ADX;
    }
}

/*************************************************************
** YmmmEngine
**
** Description:
**      Vram -> Vram 
**************************************************************
*/
static void YmmmEngine(void)
{
    int SY=MMC.SY;
    int DX=MMC.DX;
    int DY=MMC.DY;
    int TX=MMC.TX;
    int TY=MMC.TY;
    int NY=MMC.NY;
    int ADX=MMC.ADX;
    int cnt;

    cnt = MMC.VdpOpsCnt;

    switch (ScrMode) {
    case 5: 
        pre_loop *VDP_VRMP5(ADX, DY) = *VDP_VRMP5(ADX, SY); post__xyy(256)
        break;
    case 6: 
        pre_loop *VDP_VRMP6(ADX, DY) = *VDP_VRMP6(ADX, SY); post__xyy(512)
        break;
    case 7: 
        pre_loop *VDP_VRMP7(ADX, DY) = *VDP_VRMP7(ADX, SY); post__xyy(512)
        break;
    case 8: 
        pre_loop *VDP_VRMP8(ADX, DY) = *VDP_VRMP8(ADX, SY); post__xyy(256)
        break;
    }

    if ((MMC.VdpOpsCnt=cnt)>0) {
        /* Command execution done */
        VDPStatus[2] &=~VDPSTATUS_CE;
        MMC.VdpEngine=0;
        if (!NY) {
            SY+=TY;
            DY+=TY;
        }
        else {
            if (SY==-1) {
                DY+=TY;
            }
        }
        VDP[42]=NY & 0xFF;
        VDP[43]=(NY>>8) & 0x03;
        VDP[34]=SY & 0xFF;
        VDP[35]=(SY>>8) & 0x03;
        VDP[38]=DY & 0xFF;
        VDP[39]=(DY>>8) & 0x03;
    }
    else {
        MMC.SY=SY;
        MMC.DY=DY;
        MMC.NY=NY;
        MMC.ADX=ADX;
    }
}

/*************************************************************
** HmmcEngine
**
** Description:
**      CPU -> Vram 
**************************************************************
*/
static void HmmcEngine(void)
{
    if (!(VDPStatus[2] & VDPSTATUS_TR)) {
        *getVramPointer(ScrMode - 5, MMC.ADX, MMC.DY)=VDP[44];
        MMC.VdpOpsCnt-=MMC.delta;
        VDPStatus[2] |= VDPSTATUS_TR;

        if (!--MMC.ANX || ((MMC.ADX+=MMC.TX)&MMC.MX)) {
            if (!(--MMC.NY&1023) || (MMC.DY+=MMC.TY)==-1) {
                VDPStatus[2] &= ~VDPSTATUS_CE;
                MMC.VdpEngine=0;
                if (!MMC.NY) {
                    MMC.DY+=MMC.TY;
                }
                VDP[42]=MMC.NY & 0xFF;
                VDP[43]=(MMC.NY>>8) & 0x03;
                VDP[38]=MMC.DY & 0xFF;
                VDP[39]=(MMC.DY>>8) & 0x03;
            }
            else {
                MMC.ADX=MMC.DX;
                MMC.ANX=MMC.NX;
            }
        }
    }
}


/*************************************************************
** vdpCmdInit
**
** Description:
**      Initializes the command engine.
**************************************************************
*/
void vdpCmdInit()
{
    memset(&MMC, 0, sizeof(MMC));
}


/*************************************************************
** vdpCmdGetState
**
** Description:
**      Gets the state of the command engine. The method
**      returns the size of the sate.
**************************************************************
*/
int vdpCmdGetState(void* buffer)
{
    memcpy(buffer, &MMC, sizeof(MMC));

    return sizeof(MMC);
}


/*************************************************************
** vdpCmdSetState
**
** Description:
**      Sets the state of the command engine. The method
**      returns the size of the sate.
**************************************************************
*/
int vdpCmdSetState(void* buffer)
{
    memcpy(&MMC, buffer, sizeof(MMC));

    return sizeof(MMC);
}

/*************************************************************
** vdpCmdWrite
**
** Description:
**      Writes a new command to the VDP
**************************************************************
*/
void vdpCmdWrite(UInt8 V)
{
    VDPStatus[2] &= ~VDPSTATUS_TR;
    VDPStatus[7]  = V;
    VDP[44]       = V;
}


/*************************************************************
** vdpCmdRead
**
** Description:
**      Reads current command from the VDP
**************************************************************
*/
UInt8 vdpCmdRead(void)
{
    VDPStatus[2] &= ~VDPSTATUS_TR;

    return VDP[44];
}


/*************************************************************
** vdpCmdSetCommand
**
** Description:
**      Set VDP command to ececute
**************************************************************
*/
UInt8 vdpCmdSetCommand(UInt8 command)
{
    int SM = ScrMode - 5;         /* Screen mode index 0..3  */

    MMC.delta = 0;
    MMC.VdpOpsCnt = 0;

    /* V9938 ops only work in SCREENs 5-8 */
    if (SM < 0) {
        return 0;
    }

    MMC.CM = command >> 4;
    if ((MMC.CM & 0x0C) != 0x0C && MMC.CM != 0)
        /* Dot operation: use only relevant bits of color */
        VDPStatus[7] = (VDP[44] &= Mask[SM]);

    MMC.VdpEngine = command >> 4;
    switch (MMC.VdpEngine) {
    case CM_ABRT:
        VDPStatus[2] &= ~VDPSTATUS_CE;
        return 1;

    case CM_POINT:
        VDPStatus[2] &= ~VDPSTATUS_CE;
        VDPStatus[7]=VDP[44]=
            getPixel(SM, VDP[32]+((int)VDP[33]<<8),
            VDP[34]+((int)VDP[35]<<8));
        return 1;

    case CM_PSET:
        VDPStatus[2] &= ~VDPSTATUS_CE;
        setPixel(SM, 
            VDP[36]+((int)VDP[37]<<8),
            VDP[38]+((int)VDP[39]<<8),
            VDP[44],
            command&0x0F);
        return 1;

    case CM_SRCH:
        MMC.delta = GET_COMMAND_TIME(srch_timing);
        break;
    case CM_LINE:
        MMC.delta = GET_COMMAND_TIME(line_timing);
        break;
    case CM_LMMV:
        MMC.delta = GET_COMMAND_TIME(lmmv_timing);
        break;
    case CM_LMMM:
        MMC.delta = GET_COMMAND_TIME(lmmm_timing);
        break;
    case CM_LMCM:
        MMC.delta = GET_COMMAND_TIME(lmmv_timing);
        break;
    case CM_LMMC:
        MMC.delta = GET_COMMAND_TIME(lmmv_timing);
        break;
    case CM_HMMV:
        MMC.delta = GET_COMMAND_TIME(hmmv_timing);
        break;
    case CM_HMMM:
        MMC.delta = GET_COMMAND_TIME(hmmm_timing);
        break;
    case CM_YMMM:
        MMC.delta = GET_COMMAND_TIME(ymmm_timing);
        break;
    case CM_HMMC:
        MMC.delta = GET_COMMAND_TIME(hmmv_timing);
        break;

    default:
        return 0;
    }

    /* Fetch unconditional arguments */
    MMC.SX = (VDP[32]+((int)VDP[33]<<8)) & 511;
    MMC.SY = (VDP[34]+((int)VDP[35]<<8)) & 1023;
    MMC.DX = (VDP[36]+((int)VDP[37]<<8)) & 511;
    MMC.DY = (VDP[38]+((int)VDP[39]<<8)) & 1023;
    MMC.NY = (VDP[42]+((int)VDP[43]<<8)) & 1023;
    MMC.TY = VDP[45]&0x08? -1:1;
    MMC.MX = PPL[SM]; 
    MMC.CL = VDP[44];
    MMC.LO = command & 0x0F;

    /* Argument depends on UInt8 or dot operation */
    if ((MMC.CM & 0x0C) == 0x0C) {
        MMC.TX = VDP[45]&0x04? -PPB[SM]:PPB[SM];
        MMC.NX = ((VDP[40]+((int)VDP[41]<<8)) & 1023)/PPB[SM];
    }
    else {
        MMC.TX = VDP[45]&0x04? -1:1;
        MMC.NX = (VDP[40]+((int)VDP[41]<<8)) & 1023;
    }

    /* X loop variables are treated specially for LINE command */
    if (MMC.CM == CM_LINE) {
        MMC.ASX=((MMC.NX-1)>>1);
        MMC.ADX=0;
    }
    else {
        MMC.ASX = MMC.SX;
        MMC.ADX = MMC.DX;
    }    

    /* NX loop variable is treated specially for SRCH command */
    if (MMC.CM == CM_SRCH)
        MMC.ANX=(VDP[45]&0x02)!=0; /* Do we look for "==" or "!="? */
    else
        MMC.ANX = MMC.NX;

    /* Command execution started */
    VDPStatus[2] |= VDPSTATUS_CE;

    /* Operation successfull initiated */
    return 1;
}


/*************************************************************
** vdpCmdFlush
**
** Description:
**      Flushes current VDP command
**************************************************************
*/
void vdpCmdFlush() 
{
    while (MMC.VdpEngine != 0 && !(VDPStatus[2] & VDPSTATUS_TR)) {
        int opsCnt = MMC.VdpOpsCnt + 6 * 1000000;
        vdpCmdExecute(1000000);
        if (MMC.VdpOpsCnt == 0 || MMC.VdpOpsCnt == opsCnt) {
            break;
        }
    }
}


/*************************************************************
** vdpCmdExecute
**
** Description:
**      Executes a number of cycles of the VDP Command engine
**      (in 3579545 Hz)
**************************************************************
*/
void vdpCmdExecute(int cycles)
{
    MMC.VdpOpsCnt += 6 * cycles;
    if (MMC.VdpOpsCnt < MMC.delta) {
        return;
    }

    switch (MMC.VdpEngine) {
    case CM_SRCH:
        SrchEngine();
        break;
    case CM_LINE:
        LineEngine();
        break;
    case CM_LMMV:
        LmmvEngine();
        break;
    case CM_LMMM:
        LmmmEngine();
        break;
    case CM_LMCM:
        LmcmEngine();
        break;
    case CM_LMMC:
        LmmcEngine();
        break;
    case CM_HMMV:
        HmmvEngine();
        break;
    case CM_HMMM:
        HmmmEngine();
        break;
    case CM_YMMM:
        YmmmEngine();
        break;
    case CM_HMMC:
        HmmcEngine();  
        break;
    default:
        MMC.VdpOpsCnt = 0;
    }
}
