/*****************************************************************************
** File:        romMapperPanasonic.c
**
** Author:      Daniel Vik
**
** Description: Rom mapper for Panasonic cartridges with SRAM
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "romMapperPanasonic.h"
#include "romMapper.h"
#include "SlotManager.h"
#include "sramLoader.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

extern UInt8* msxGetRamPage(int page);


typedef struct {
    UInt8* romData;
    UInt8* sram;
    UInt8* readBlock;
    int    sramSize;
    char   sramFilename[512];
    int    maxSRAMBank;
    UInt8  control;
    int    romMapper[8];
    int    slot;
    int    sslot;
    int    startPage;
} RomMapperPanasonic;

static int SRAM_BASE = 0x80;
static int RAM_BASE  = 0x180;

static void changeBank(RomMapperPanasonic* rm, int region, int bank)
{
	if (bank == rm->romMapper[region]) {
		return;
	}
	rm->romMapper[region] = bank;
	if (rm->sramSize > 0 && bank >= SRAM_BASE && bank < rm->maxSRAMBank) {
		int offset = (bank - SRAM_BASE) * 0x2000;
		if (offset >= rm->sramSize) {
			offset &= (rm->sramSize - 1);
		}
        if (region == 3) rm->readBlock = rm->sram + offset;
        slotMapPage(rm->slot, rm->sslot, region, rm->sram + offset, region != 3, 0);
	} else if (bank >= RAM_BASE) {
        if (region == 3) rm->readBlock = msxGetRamPage(bank - RAM_BASE);
        slotMapPage(rm->slot, rm->sslot, region, msxGetRamPage(bank - RAM_BASE), region != 3, 0);
	} else {
        if (region == 3) rm->readBlock = rm->romData + bank * 0x2000;
        slotMapPage(rm->slot, rm->sslot, region, rm->romData + bank * 0x2000, region != 3, 0);
	}
}

static void destroy(RomMapperPanasonic* rm)
{
    sramSave(rm->sramFilename, rm->sram, 0x2000, NULL, 0);

    slotUnregister(rm->slot, rm->sslot, rm->startPage);

    free(rm->sram);
    free(rm->romData);
    free(rm);
}

static int getState(RomMapperPanasonic* rm, UInt8* buffer, UInt32 systemTime)
{
    memcpy(buffer, rm->romMapper, sizeof(rm->romMapper));
    memcpy(buffer + sizeof(rm->romMapper), &rm->control, sizeof(rm->control));
    return sizeof(rm->romMapper) + sizeof(rm->control);
}

static int setState(RomMapperPanasonic* rm, UInt8* buffer, UInt32 systemTime)
{
    int romMapper[8];
    int i;

    memcpy(romMapper, buffer, sizeof(romMapper));
    memcpy(&rm->control, buffer + sizeof(rm->romMapper), sizeof(rm->control));

    for (i = 0; i < 8; i++) {
        changeBank(rm, i, romMapper[i]);
    }

    return sizeof(rm->romMapper) + sizeof(rm->control);
}

static UInt8 read(RomMapperPanasonic* rm, UInt16 address, UInt32 systemTime) 
{
	UInt8 result;

	if ((rm->control & 0x04) && address >= 0x7ff0 && address < 0x7ff8) {
		result = rm->romMapper[address & 7] & 0xff;
	} else if ((rm->control & 0x10) && address == 0x7ff8) {
        int i;
		result = 0;
		for (i = 7; i >= 0; i--) {
			result <<= 1;
			if (rm->romMapper[i] & 0x100) {
				result++;
			}
		}
	} else if ((rm->control & 0x08) && address == 0x7ff9) {
		result = rm->control;
	} else {
        result = rm->readBlock[address & 0x1fff];
	}

    return result;
}

static void write(RomMapperPanasonic* rm, UInt16 address, UInt8 value, UInt32 systemTime) 
{
    int region;
    int selectedBank;
    int newBank;

    if (address >= 0x6000 && address < 0x7ff0) {
		region = (address & 0x1c00) >> 10;
        if (region == 5 || region == 6) {
            region ^= 3;
        }

		selectedBank = rm->romMapper[region];
		newBank = (selectedBank & ~0xff) | value;
		changeBank(rm, region, newBank);

	} 
    else if (address == 0x7ff8) {
		for (region = 0; region < 8; region++) {
			if (value & 1) {
				changeBank(rm, region, rm->romMapper[region] |  0x100);
			} else {
				changeBank(rm, region, rm->romMapper[region] & ~0x100);
			}
			value >>= 1;
		}
	} 
    else if (address == 0x7ff9) {
		rm->control = value;
	} 
    else if (address >= 0x8000 && address < 0xC000) {
		region = address >> 13;
		selectedBank = rm->romMapper[region];
		
        if (rm->sramSize > 0 && selectedBank >= SRAM_BASE && selectedBank < rm->maxSRAMBank) {
            rm->sram[0x2000 * (selectedBank - SRAM_BASE) + (address & 0x1fff)] = value;
		} 
        else if (selectedBank >= RAM_BASE) {
        volatile int x = 0;
        x++;
//			bank[region][address & 0x1fff] = value;
		}
	} 
}

int romMapperPanasonicCreate(char* filename, UInt8* romData, 
                             int size, int slot, int sslot, int startPage,
                             int sramSize) 
{
    SlotCallbacks callbacks = { destroy, read, write, getState, setState };
    RomMapperPanasonic* rm;
    int i;

    if (size < 0x8000 || startPage != 0) {
        return 0;
    }

    rm = malloc(sizeof(RomMapperPanasonic));

    slotRegister(ROM_PANASONIC, slot, sslot, 0, 8, &callbacks, rm);

    rm->romData = malloc(size);
    memcpy(rm->romData, romData, size);
    rm->sramSize = sramSize;
    rm->sram = malloc(sramSize);
    memset(rm->sram, 0xff, sramSize);
    rm->maxSRAMBank = SRAM_BASE + sramSize / 0x2000;
    memset(rm->romMapper, 0, sizeof(rm->romMapper));
    rm->control = 0;
    rm->readBlock = rm->romData;
    rm->slot  = slot;
    rm->sslot = sslot;
    rm->startPage  = startPage;
    strcpy(rm->sramFilename, sramCreateFilename(filename));

    sramLoad(rm->sramFilename, rm->sram, 0x2000, NULL, 0);

    for (i = 0; i < 8; i++) {
        rm->romMapper[i] = 0;
        slotMapPage(rm->slot, rm->sslot, i, rm->romData, i != 3, 0);
    }

    return 1;
}

