/*****************************************************************************
** File:        ramMapper.c
**
** Author:      Daniel Vik
**
** Description: Ram mapper
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "ramMapper.h"
#include "romMapper.h"
#include "SlotManager.h"
#include "IoPort.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


static UInt8 ramData[256 * 0x4000];


typedef struct {
    UInt8* ramData;
    int slot;
    int sslot;
    UInt8 mask;
    int ramMapper[4];
} RamMapper;

static int getState(RamMapper* rm, UInt8* buffer, UInt32 systemTime)
{
    int ramSize = 0x4000 * (rm->mask + 1);

    memcpy(buffer, rm, sizeof(RamMapper));
    memcpy(buffer + sizeof(RamMapper), ramData, ramSize);
    return sizeof(RamMapper) + ramSize;
}

static int setState(RamMapper* rm, UInt8* buffer, UInt32 systemTime)
{
    int ramSize = 0x4000 * (rm->mask + 1);
    int i;

    memcpy(rm, buffer, sizeof(RamMapper));
    memcpy(ramData, buffer + sizeof(RamMapper), ramSize);

    rm->ramData = ramData;

    for (i = 0; i < 4; i++) {
        slotMapPage(rm->slot, rm->sslot, 2 * i, rm->ramData + 0x4000 * rm->ramMapper[i], 1, 1);
        slotMapPage(rm->slot, rm->sslot, 2 * i + 1, rm->ramData + 0x4000 * rm->ramMapper[i] + 0x2000, 1, 1);
    }

    return sizeof(RamMapper) + ramSize;
}

static UInt8 read(RamMapper* rm, UInt16 address, UInt32 systemTime)
{
    return rm->ramMapper[address & 3] | ~rm->mask;
}

static void write(RamMapper* rm, UInt16 address, UInt8 value, UInt32 systemTime)
{
    value &= rm->mask;
    address &= 3;

    if (rm->ramMapper[address] != value) {
        rm->ramMapper[address] = value;

        slotMapPage(rm->slot, rm->sslot, 2 * address,     rm->ramData + 0x4000 * value, 1, 1);
        slotMapPage(rm->slot, rm->sslot, 2 * address + 1, rm->ramData + 0x4000 * value + 0x2000, 1, 1);
    }
}

static void destroy(void* ref)
{
    RamMapper* rm = (RamMapper*)ref;

    ioPortUnregister(0xfc);
    ioPortUnregister(0xfd);
    ioPortUnregister(0xfe);
    ioPortUnregister(0xff);

    slotUnregister(rm->slot, rm->sslot, 0);

    free(rm);
}

int ramMapperCreate(int size, int slot, int sslot, int startPage, int clearRAM, UInt8** ramPtr, UInt32* ramSize) 
{
    SlotCallbacks callbacks = { destroy, NULL, NULL, getState, setState };
    RamMapper* rm;
    int pages = size / 0x4000;
    int i;

    // Check that memory is a power of 2 and at least 64kB
    for (i = 4; i < pages; i <<= 1);

    if (i != pages) {
        return 0;
    }

    // Start page must be zero (only full slot allowed)
    if (startPage != 0) {
        return 0;
    }

    if (clearRAM) {
        memset(ramData, 0xff, sizeof(ramData));
    }

    rm = malloc(sizeof(RamMapper));

    rm->ramData = ramData;
    rm->slot    = slot;
    rm->sslot   = sslot;
    rm->mask    = pages - 1;

    rm->ramMapper[0] = 3;
    rm->ramMapper[1] = 2;
    rm->ramMapper[2] = 1;
    rm->ramMapper[3] = 0;

    for (i = 0; i < 4; i++) {
        slotMapPage(slot, sslot, 2 * i, rm->ramData + 0x4000 * rm->ramMapper[i], 1, 1);
        slotMapPage(slot, sslot, 2 * i + 1, rm->ramData + 0x4000 * rm->ramMapper[i] + 0x2000, 1, 1);
    }

    ioPortRegister(0xfc, read, write, rm);
    ioPortRegister(0xfd, read, write, rm);
    ioPortRegister(0xfe, read, write, rm);
    ioPortRegister(0xff, read, write, rm);
    
    slotRegister(RAM_MAPPER, slot, sslot, 0, 8, &callbacks, rm);

    *ramPtr = rm->ramData;
    *ramSize = pages * 0x4000;

    return 1;
}

