/*****************************************************************************
** File:        Win32keyboard.c
**
** Author:      Daniel Vik
**
** Description: Keyboard input methods
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#define DIRECTINPUT_VERSION     0x0500
#include "Win32keyboard.h"
#include <windows.h>
#include <stdio.h>
#include <winioctl.h>
#include <dinput.h>
 

#ifndef DIK_WEBFORWARD
#define DIK_WEBFORWARD      0xE9
#define DIK_WEBBACK         0xEA
#endif

typedef struct { 
    DWORD Code; 
    WORD  VirtKey;
    DWORD DCode;
    byte Pos; 
    byte Mask; 
} KeyMapEntry;

/* Key mapping table */
static KeyMapEntry keyMapEnglish[] = {
    { 0x0B, 0,           DIK_0,          0,  0x01 }, /* 0          */
    { 0x02, 0,           DIK_1,          0,  0x02 }, /* 1          */
    { 0x03, 0,           DIK_2,          0,  0x04 }, /* 2          */
    { 0x04, 0,           DIK_3,          0,  0x08 }, /* 3          */
    { 0x05, 0,           DIK_4,          0,  0x10 }, /* 4          */
    { 0x06, 0,           DIK_5,          0,  0x20 }, /* 5          */
    { 0x07, 0,           DIK_6,          0,  0x40 }, /* 6          */
    { 0x08, 0,           DIK_7,          0,  0x80 }, /* 7          */
    { 0x09, 0,           DIK_8,          1,  0x01 }, /* 8          */
    { 0x0A, 0,           DIK_9,          1,  0x02 }, /* 9          */
    { 0x0C, 0,           DIK_MINUS,      1,  0x04 }, /* -          */
    { 0x0D, 0,           DIK_EQUALS,     1,  0x08 }, /* =          */
    { 0x2B, 0,           DIK_BACKSLASH,  1,  0x10 }, /* \          */
    { 0x1A, 0,           DIK_LBRACKET,   1,  0x20 }, /* [          */
    { 0x1B, 0,           DIK_RBRACKET,   1,  0x40 }, /* ]          */
    { 0x27, 0,           DIK_SEMICOLON,  1,  0x80 }, /* ;          */
    { 0x28, 0,           DIK_APOSTROPHE, 2,  0x01 }, /* '          */
    { 0x29, 0,           DIK_GRAVE,      2,  0x02 }, /* `          */
    { 0x33, 0,           DIK_COMMA,      2,  0x04 }, /* ,          */
    { 0x34, 0,           DIK_PERIOD,     2,  0x08 }, /* .          */
    { 0x35, 0,           DIK_SLASH,      2,  0x10 }, /* /          */
    { 0x1E, 0,           DIK_A,          2,  0x40 }, /* A          */
    { 0x30, 0,           DIK_B,          2,  0x80 }, /* B          */
    { 0x2E, 0,           DIK_C,          3,  0x01 }, /* C          */
    { 0x20, 0,           DIK_D,          3,  0x02 }, /* D          */
    { 0x12, 0,           DIK_E,          3,  0x04 }, /* E          */
    { 0x21, 0,           DIK_F,          3,  0x08 }, /* F          */
    { 0x22, 0,           DIK_G,          3,  0x10 }, /* G          */
    { 0x23, 0,           DIK_H,          3,  0x20 }, /* H          */
    { 0x17, 0,           DIK_I,          3,  0x40 }, /* I          */
    { 0x24, 0,           DIK_J,          3,  0x80 }, /* J          */
    { 0x25, 0,           DIK_K,          4,  0x01 }, /* K          */
    { 0x26, 0,           DIK_L,          4,  0x02 }, /* L          */
    { 0x32, 0,           DIK_M,          4,  0x04 }, /* M          */
    { 0x31, 0,           DIK_N,          4,  0x08 }, /* N          */
    { 0x18, 0,           DIK_O,          4,  0x10 }, /* O          */
    { 0x19, 0,           DIK_P,          4,  0x20 }, /* P          */
    { 0x10, 0,           DIK_Q,          4,  0x40 }, /* Q          */
    { 0x13, 0,           DIK_R,          4,  0x80 }, /* R          */
    { 0x1F, 0,           DIK_S,          5,  0x01 }, /* S          */
    { 0x14, 0,           DIK_T,          5,  0x02 }, /* T          */
    { 0x16, 0,           DIK_U,          5,  0x04 }, /* U          */
    { 0x2f, 0,           DIK_V,          5,  0x08 }, /* V          */
    { 0x11, 0,           DIK_W,          5,  0x10 }, /* W          */
    { 0x2D, 0,           DIK_X,          5,  0x20 }, /* X          */
    { 0x15, 0,           DIK_Y,          5,  0x40 }, /* Y          */
    { 0x2C, 0,           DIK_Z,          5,  0x80 }, /* Z          */
    { 0x3B, 0,           DIK_F1,         6,  0x20 }, /* F1         */
    { 0x3C, 0,           DIK_F2,         6,  0x40 }, /* F2         */
    { 0x3D, 0,           DIK_F3,         6,  0x80 }, /* F3         */
    { 0x3E, 0,           DIK_F4,         7,  0x01 }, /* F4         */
    { 0x3F, 0,           DIK_F5,         7,  0x02 }, /* F5         */
    { 0x01, 0,           DIK_ESCAPE,     7,  0x04 }, /* ESCAPE     */
    { 0x0F, 0,           DIK_TAB,        7,  0x08 }, /* TAB        */
    { 0x49, 0,           DIK_PRIOR,      7,  0x10 }, /* STOP/BREAK */ //VK 69 - PAGE UP key
    { 0x0E, 0,           DIK_BACK,       7,  0x20 }, /* BACKSPACE  */
    { 0x4F, 0,           DIK_END,        7,  0x40 }, /* SELECT     */ //VK 23 - END key
    { 0x1C, 0,           DIK_RETURN,     7,  0x80 }, /* RETURN     */
    { 0x9C, 0,           DIK_NUMPADENTER,7,  0x80 }, /* Enter on numeric keypad */
    { 0x39, 0,           DIK_SPACE,      8,  0x01 }, /* SPACE      */
    { 0x47, 0,           DIK_HOME,       8,  0x02 }, /* HOME/CLS   */ //VK 67 - HOME key
    { 0x52, 0,           DIK_INSERT,     8,  0x04 }, /* INSERT     */ //VK 2d - INS key
    { 0x53, 0,           DIK_DELETE,     8,  0x08 }, /* DELETE     */ //VK 6e - DEL key
    { 0x4B, 0,           DIK_LEFT,       8,  0x10 }, /* LEFT ARROW */
    { 0x48, 0,           DIK_UP,         8,  0x20 }, /* UP ARROW   */
    { 0x4D, 0,           DIK_RIGHT,      8,  0x80 }, /* RIGHT ARROW */
    { 0x50, 0,           DIK_DOWN,       8,  0x40 }, /* DOWN ARROW */
    { 0x37, VK_MULTIPLY, DIK_MULTIPLY,   9,  0x01 }, /* NUMPAD *   */
    { 0x4E, VK_ADD,      DIK_ADD,        9,  0x02 }, /* NUMPAD +   */
    { 0x35, VK_DIVIDE,   DIK_DIVIDE,     9,  0x04 }, /* NUMPAD /   */
    { 0x00, VK_NUMPAD0,  DIK_NUMPAD0,    9,  0x08 }, /* NUMPAD 0   */
    { 0x00, VK_NUMPAD1,  DIK_NUMPAD1,    9,  0x10 }, /* NUMPAD 1   */
    { 0x00, VK_NUMPAD2,  DIK_NUMPAD2,    9,  0x20 }, /* NUMPAD 2   */
    { 0x00, VK_NUMPAD3,  DIK_NUMPAD3,    9,  0x40 }, /* NUMPAD 3   */
    { 0x00, VK_NUMPAD4,  DIK_NUMPAD4,    9,  0x80 }, /* NUMPAD 4   */
    { 0x00, VK_NUMPAD5,  DIK_NUMPAD5,    10, 0x01 }, /* NUMPAD 5   */
    { 0x00, VK_NUMPAD6,  DIK_NUMPAD6,    10, 0x02 }, /* NUMPAD 6   */
    { 0x00, VK_NUMPAD7,  DIK_NUMPAD7,    10, 0x04 }, /* NUMPAD 7   */
    { 0x00, VK_NUMPAD8,  DIK_NUMPAD8,    10, 0x08 }, /* NUMPAD 8   */
    { 0x00, VK_NUMPAD9,  DIK_NUMPAD9,    10, 0x10 }, /* NUMPAD 9   */
    { 0x4A, VK_SUBTRACT, DIK_SUBTRACT,   10, 0x20 }, /* NUMPAD -   */
    { 0x00, VK_NEXT,     DIK_NEXT,       10, 0x40 }, /* NUMPAD ,   */ //VK 22 - PAGE DOWN key
    { 0x45, VK_DECIMAL,  DIK_DECIMAL,    10, 0x80 }, /* NUMPAD .   */
    { 0x00, 0,           0,              11, 0x01 },
    { 0x00, 0,           DIK_RWIN,       11, 0x02 },
    { 0x00, 0,           DIK_WEBBACK,    11, 0x02 },
    { 0x00, 0,           0,              11, 0x04 },
    { 0x00, 0,           DIK_LWIN,       11, 0x08 },
    { 0x00, 0,           DIK_WEBFORWARD, 11, 0x08 },
    { 0x00, 0,           0,              11, 0x10 },
    { 0x00, 0,           0,              11, 0x20 },
    { 0x00, 0,           0,              11, 0x40 },
    { 0x00, 0,           0,              11, 0x80 },
    { 0x00, VK_LSHIFT,   DIK_LSHIFT,     6,  0x01 }, /* SHIFT      */
    { 0x00, VK_RSHIFT,   DIK_RSHIFT,     6,  0x01 }, /* SHIFT      */
    { 0x00, VK_LCONTROL, DIK_LCONTROL,   6,  0x02 }, /* CONTROL    */
    { 0x00, VK_RCONTROL, DIK_RCONTROL,   6,  0x02 }, /* CONTROL    */
    { 0x00, VK_LMENU,    DIK_LMENU,      6,  0x04 }, /* GRAPH      */
    { 0x00, VK_RMENU,    DIK_RMENU,      6,  0x10 }, /* CODE       */
    { 0x00, VK_CAPITAL,  DIK_CAPITAL,    6,  0x08 }, /* CAPSLOCK   */
    { 0x00, 0, 0, 0,  0x00 }  /** The End. **/
};

/* Key mapping table */
KeyMapEntry keyMapJapanese[] = {
    { 0x0B, 0,           DIK_0,          0,  0x01 }, /* 0          */
    { 0x02, 0,           DIK_1,          0,  0x02 }, /* 1          */
    { 0x03, 0,           DIK_2,          0,  0x04 }, /* 2          */
    { 0x04, 0,           DIK_3,          0,  0x08 }, /* 3          */
    { 0x05, 0,           DIK_4,          0,  0x10 }, /* 4          */
    { 0x06, 0,           DIK_5,          0,  0x20 }, /* 5          */
    { 0x07, 0,           DIK_6,          0,  0x40 }, /* 6          */
    { 0x08, 0,           DIK_7,          0,  0x80 }, /* 7          */
    { 0x09, 0,           DIK_8,          1,  0x01 }, /* 8          */
    { 0x0A, 0,           DIK_9,          1,  0x02 }, /* 9          */
    { 0x0C, 0,           DIK_MINUS,      1,  0x04 }, /* -          */
    { 0x0D, 0,           DIK_CIRCUMFLEX, 1,  0x08 }, /* ^          */
    { 0x7D, 0,           DIK_YEN,		 1,  0x10 }, /* \          */
    { 0x1A, 0,           DIK_AT,	     1,  0x20 }, /* @          */
    { 0x1B, 0,           DIK_LBRACKET,   1,  0x40 }, /* [          */
    { 0x27, 0,           DIK_SEMICOLON,  1,  0x80 }, /* ;          */
    { 0x28, 0,           DIK_COLON, 	 2,  0x01 }, /* :          */
    { 0x2B, 0,           DIK_RBRACKET,   2,  0x02 }, /* ]          */
    { 0x33, 0,           DIK_COMMA,      2,  0x04 }, /* ,          */
    { 0x34, 0,           DIK_PERIOD,     2,  0x08 }, /* .          */
    { 0x35, 0,           DIK_SLASH,      2,  0x10 }, /* /          */
    { 0x73, 0,           DIK_BACKSLASH,  2,  0x20 }, /* \          */
    { 0x1E, 0,           DIK_A,          2,  0x40 }, /* A          */
    { 0x30, 0,           DIK_B,          2,  0x80 }, /* B          */
    { 0x2E, 0,           DIK_C,          3,  0x01 }, /* C          */
    { 0x20, 0,           DIK_D,          3,  0x02 }, /* D          */
    { 0x12, 0,           DIK_E,          3,  0x04 }, /* E          */
    { 0x21, 0,           DIK_F,          3,  0x08 }, /* F          */
    { 0x22, 0,           DIK_G,          3,  0x10 }, /* G          */
    { 0x23, 0,           DIK_H,          3,  0x20 }, /* H          */
    { 0x17, 0,           DIK_I,          3,  0x40 }, /* I          */
    { 0x24, 0,           DIK_J,          3,  0x80 }, /* J          */
    { 0x25, 0,           DIK_K,          4,  0x01 }, /* K          */
    { 0x26, 0,           DIK_L,          4,  0x02 }, /* L          */
    { 0x32, 0,           DIK_M,          4,  0x04 }, /* M          */
    { 0x31, 0,           DIK_N,          4,  0x08 }, /* N          */
    { 0x18, 0,           DIK_O,          4,  0x10 }, /* O          */
    { 0x19, 0,           DIK_P,          4,  0x20 }, /* P          */
    { 0x10, 0,           DIK_Q,          4,  0x40 }, /* Q          */
    { 0x13, 0,           DIK_R,          4,  0x80 }, /* R          */
    { 0x1F, 0,           DIK_S,          5,  0x01 }, /* S          */
    { 0x14, 0,           DIK_T,          5,  0x02 }, /* T          */
    { 0x16, 0,           DIK_U,          5,  0x04 }, /* U          */
    { 0x2f, 0,           DIK_V,          5,  0x08 }, /* V          */
    { 0x11, 0,           DIK_W,          5,  0x10 }, /* W          */
    { 0x2D, 0,           DIK_X,          5,  0x20 }, /* X          */
    { 0x15, 0,           DIK_Y,          5,  0x40 }, /* Y          */
    { 0x2C, 0,           DIK_Z,          5,  0x80 }, /* Z          */
    { 0x3B, 0,           DIK_F1,         6,  0x20 }, /* F1         */
    { 0x3C, 0,           DIK_F2,         6,  0x40 }, /* F2         */
    { 0x3D, 0,           DIK_F3,         6,  0x80 }, /* F3         */
    { 0x3E, 0,           DIK_F4,         7,  0x01 }, /* F4         */
    { 0x3F, 0,           DIK_F5,         7,  0x02 }, /* F5         */
    { 0x01, 0,           DIK_ESCAPE,     7,  0x04 }, /* ESCAPE     */
    { 0x0F, 0,           DIK_TAB,        7,  0x08 }, /* TAB        */
    { 0x49, 0,           DIK_PRIOR,      7,  0x10 }, /* STOP/BREAK */ //VK 69 - PAGE UP key
    { 0x0E, 0,           DIK_BACK,       7,  0x20 }, /* BACKSPACE  */
    { 0x4F, 0,           DIK_END,        7,  0x40 }, /* SELECT     */ //VK 23 - END key
    { 0x1C, 0,           DIK_RETURN,     7,  0x80 }, /* RETURN     */
    { 0x9C, 0,           DIK_NUMPADENTER,7,  0x80 }, /* Enter on numeric keypad */
    { 0x39, 0,           DIK_SPACE,      8,  0x01 }, /* SPACE      */
    { 0x47, 0,           DIK_HOME,       8,  0x02 }, /* HOME/CLS   */ //VK 67 - HOME key
    { 0x52, 0,           DIK_INSERT,     8,  0x04 }, /* INSERT     */ //VK 2d - INS key
    { 0x53, 0,           DIK_DELETE,     8,  0x08 }, /* DELETE     */ //VK 6e - DEL key
    { 0x4B, 0,           DIK_LEFT,       8,  0x10 }, /* LEFT ARROW */
    { 0x48, 0,           DIK_UP,         8,  0x20 }, /* UP ARROW   */
    { 0x4D, 0,           DIK_RIGHT,      8,  0x80 }, /* RIGHT ARROW */
    { 0x50, 0,           DIK_DOWN,       8,  0x40 }, /* DOWN ARROW */
    { 0x37, VK_MULTIPLY, DIK_MULTIPLY,   9,  0x01 }, /* NUMPAD *   */
    { 0x4E, VK_ADD,      DIK_ADD,        9,  0x02 }, /* NUMPAD +   */
    { 0x35, VK_DIVIDE,   DIK_DIVIDE,     9,  0x04 }, /* NUMPAD /   */
    { 0x00, VK_NUMPAD0,  DIK_NUMPAD0,    9,  0x08 }, /* NUMPAD 0   */
    { 0x00, VK_NUMPAD1,  DIK_NUMPAD1,    9,  0x10 }, /* NUMPAD 1   */
    { 0x00, VK_NUMPAD2,  DIK_NUMPAD2,    9,  0x20 }, /* NUMPAD 2   */
    { 0x00, VK_NUMPAD3,  DIK_NUMPAD3,    9,  0x40 }, /* NUMPAD 3   */
    { 0x00, VK_NUMPAD4,  DIK_NUMPAD4,    9,  0x80 }, /* NUMPAD 4   */
    { 0x00, VK_NUMPAD5,  DIK_NUMPAD5,    10, 0x01 }, /* NUMPAD 5   */
    { 0x00, VK_NUMPAD6,  DIK_NUMPAD6,    10, 0x02 }, /* NUMPAD 6   */
    { 0x00, VK_NUMPAD7,  DIK_NUMPAD7,    10, 0x04 }, /* NUMPAD 7   */
    { 0x00, VK_NUMPAD8,  DIK_NUMPAD8,    10, 0x08 }, /* NUMPAD 8   */
    { 0x00, VK_NUMPAD9,  DIK_NUMPAD9,    10, 0x10 }, /* NUMPAD 9   */
    { 0x4A, VK_SUBTRACT, DIK_SUBTRACT,   10, 0x20 }, /* NUMPAD -   */
    { 0x00, VK_NEXT,     DIK_NEXT,       10, 0x40 }, /* NUMPAD ,   */ //VK 22 - PAGE DOWN key
    { 0x45, VK_DECIMAL,  DIK_DECIMAL,    10, 0x80 }, /* NUMPAD .   */
    { 0x00, 0,           0,              11, 0x01 },
    { 0x00, 0,           DIK_RWIN,       11, 0x02 },
    { 0x00, 0,           DIK_LWIN,       11, 0x08 },
    { 0x00, 0,           0,              11, 0x04 },
    { 0x00, 0,           DIK_WEBBACK,    11, 0x02 },
    { 0x00, 0,           DIK_WEBFORWARD, 11, 0x08 },
    { 0x00, 0,           DIK_CONVERT,    11, 0x02 }, /* Jikkou     */ 
    { 0x00, 0,           DIK_NOCONVERT,  11, 0x08 }, /* Torikeshi  */
    { 0x00, 0,           0,              11, 0x10 },
    { 0x00, 0,           0,              11, 0x20 },
    { 0x00, 0,           0,              11, 0x40 },
    { 0x00, 0,           0,              11, 0x80 },
    { 0x00, VK_LSHIFT,   DIK_LSHIFT,     6,  0x01 }, /* SHIFT      */
    { 0x00, VK_RSHIFT,   DIK_RSHIFT,     6,  0x01 }, /* SHIFT      */
    { 0x00, VK_LCONTROL, DIK_LCONTROL,   6,  0x02 }, /* CONTROL    */
    { 0x00, VK_RCONTROL, DIK_RCONTROL,   6,  0x02 }, /* CONTROL    */
    { 0x00, VK_LMENU,    DIK_LMENU,      6,  0x04 }, /* GRAPH      */
    { 0x00, VK_RMENU,    DIK_RMENU,      6,  0x10 }, /* CODE       */
    { 0x00, VK_CAPITAL,  DIK_CAPITAL,    6,  0x08 }, /* CAPSLOCK   */
    { 0x00, 0, 0, 0,  0x00 }  /** The End. **/
};

static LPDIRECTINPUT        dinput;
static int                  dinputVersion;
static LPDIRECTINPUTDEVICE  kbdDevice = NULL;
static LPDIRECTINPUTDEVICE2 kbdDevice2 = NULL;
static HWND                 dinputWindow;
static KeyMapEntry*         keyMap;

#define STRUCTSIZE(x) ((dinputVersion == 0x0300) ? sizeof(x##_DX3) : sizeof(x))


static void modifyKeyMapVirtKey(WORD oldKey, WORD newKey) {
    int i;
    for (i = 0; keyMap[i].Mask != 0; i++) {
        if (keyMap[i].VirtKey == oldKey) {
            keyMap[i].VirtKey = newKey;
        }
    }
}

static void keyboardInitVK() 
{    
    int i;
    BOOL success;
    char klId[KL_NAMELENGTH];
    OSVERSIONINFO ovi;

    keyMap = keyMapEnglish;

    memset(klId, 0, sizeof(klId));

    /* Modify scan code map if nessecary */
    success = GetKeyboardLayoutName(klId) ;
    if (success) {
        if (0 == strcmp(klId + 4, "0411")) {
            keyMap = keyMapJapanese;
        }
    }

    ovi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
    GetVersionEx(&ovi);
    if (ovi.dwMajorVersion <= 4) {
        modifyKeyMapVirtKey(VK_NEXT,     0);
        modifyKeyMapVirtKey(VK_LSHIFT,   VK_SHIFT);
        modifyKeyMapVirtKey(VK_RSHIFT,   VK_SHIFT);
        modifyKeyMapVirtKey(VK_LCONTROL, VK_CONTROL);
        modifyKeyMapVirtKey(VK_RCONTROL, VK_CONTROL);
        modifyKeyMapVirtKey(VK_LMENU,    VK_MENU);
        modifyKeyMapVirtKey(VK_RMENU,    VK_NEXT);
    } 

    for (i = 0; keyMap[i].Mask != 0; i++) {
        if (keyMap[i].Code != 0) {
            keyMap[i].VirtKey = MapVirtualKey(keyMap[i].Code, 1);
        }
    }
}

static BOOL CALLBACK enumKeyboards(LPCDIDEVICEINSTANCE devInst, LPVOID ref)
{
    DIDEVCAPS kbdCaps;
    HRESULT rv;

    if (kbdDevice != NULL) {
        return DIENUM_CONTINUE;
    }

    rv = IDirectInput_CreateDevice(dinput, &devInst->guidInstance, &kbdDevice, NULL);
    if (rv != DI_OK) {
        return DIENUM_CONTINUE;
    }

    IDirectInputDevice_QueryInterface(kbdDevice, &IID_IDirectInputDevice2, (void **)&kbdDevice2);

    kbdCaps.dwSize = STRUCTSIZE(DIDEVCAPS);
    rv = IDirectInputDevice_GetCapabilities(kbdDevice, &kbdCaps);
    if (rv == DI_OK) {
        rv = IDirectInputDevice_SetDataFormat(kbdDevice, &c_dfDIKeyboard);
        if (rv == DI_OK) {
            rv = IDirectInputDevice_SetCooperativeLevel(kbdDevice, dinputWindow,
                DISCL_FOREGROUND | DISCL_NONEXCLUSIVE);
            if (rv != DI_OK) {
                if (kbdDevice2 != NULL) {
                    IDirectInputDevice_Release(kbdDevice2);
                }
                IDirectInputDevice_Release(kbdDevice);
                kbdDevice = NULL;
                kbdDevice2 = NULL;
            }
        }
    }

    return DIENUM_CONTINUE;
}

int keyboardInit(HWND hwnd)
{
    HRESULT rv = 234;

    keyboardInitVK();

    dinputWindow = hwnd;
    dinputVersion = DIRECTINPUT_VERSION;
	rv = DirectInputCreate(GetModuleHandle(NULL), dinputVersion, &dinput, NULL);
    if (rv != DI_OK) {
        dinputVersion = 0x0300;
    	rv = DirectInputCreate(GetModuleHandle(NULL), dinputVersion, &dinput, NULL);
        if (rv != DI_OK) {
            printf("Failed to initialize DirectInput\n");
            return 0;
        }
    }

	rv = IDirectInput_EnumDevices(dinput, DIDEVTYPE_KEYBOARD, enumKeyboards, 0, DIEDFL_ATTACHEDONLY);
    if (rv != DI_OK) {
        IDirectInput_Release(dinput);
        printf("Failed to find DirectInput device\n");
        return 0;
    }

    if (kbdDevice == NULL) {
        printf("Failed to create DirectInput device\n");
        return 0;
    }

    return 1;
}

void keyboardDestroy(void)
{
    if (kbdDevice) {
        IDirectInputDevice_Release(kbdDevice);
    }
    
    if (kbdDevice2) {
        IDirectInputDevice_Release(kbdDevice2);
    }

    if (dinput) {
        IDirectInput_Release(dinput);
    }

    dinput = NULL;
}

void keyboardEnable(int enable)
{
    if (kbdDevice != NULL) {
        if (enable) {
            IDirectInputDevice_Acquire(kbdDevice);
        }
        else {
            IDirectInputDevice_Unacquire(kbdDevice);
        }
    }
}

void keyboardUpdate(BYTE* MSXKeyMap) 
{ 
    BYTE msxKeyMap[16];
    int i;
    
    memset(msxKeyMap, 0xff, 16);

    if (!GetFocus()) {
        return;
    }

    if (kbdDevice != NULL) {
        char buffer[256]; 
        HRESULT  rv; 
        
        if (kbdDevice2) {
			IDirectInputDevice2_Poll(kbdDevice2);
        }

        rv = IDirectInputDevice_GetDeviceState(kbdDevice, sizeof(buffer), (LPVOID)&buffer); 
        if (rv == DIERR_INPUTLOST || rv == DIERR_NOTACQUIRED) {
            rv = IDirectInputDevice_Acquire(kbdDevice);
            if (rv == DI_OK) {
                rv = IDirectInputDevice_GetDeviceState(kbdDevice, sizeof(buffer), (LPVOID)&buffer); 
            }
        }

        if (rv >= 0) { 
            int shiftCtrlPressed = (buffer[DIK_LSHIFT] | buffer[DIK_RSHIFT] | buffer[DIK_LCONTROL] | buffer[DIK_RCONTROL]) >> 7;
            if (!(shiftCtrlPressed &&
                  ((buffer[DIK_F6]  | buffer[DIK_F7]  | buffer[DIK_F8]  | buffer[DIK_F9]  | 
                    buffer[DIK_F10] | buffer[DIK_F11] | buffer[DIK_F12]) >> 7)))
            {
                for (i = 0; keyMap[i].Mask != 0; i++) {
                    if (buffer[keyMap[i].DCode] >> 7) {
                        msxKeyMap[keyMap[i].Pos] &= ~keyMap[i].Mask;
                    }
                }

#if 0
                {
                    static char oldbuffer[256] = {0};
                    int j;

                    for (i = 0; i < 256; i++) {
                        if ((buffer[i] ^ oldbuffer[i]) >> 7) {
                            printf("%s: %d \t", buffer[i] >> 7 ? "Down" : "Up  ", i);
                            
                            for (j = 0; keyMap[j].Mask != 0; j++) {
                                if (keyMap[j].DCode == i) {
                                    printf("MSX key %.2x:%.2x", keyMap[j].Pos, keyMap[j].Mask);
                                }
                            }
                            printf("\n");
                        }
                    }
                    memcpy(oldbuffer, buffer, 256);
                }
#endif
            }

            memcpy(MSXKeyMap, msxKeyMap, 16);

            return;
        }
    }

    // If we got here, DirectInput failed. Use old input mechanism
    for (i = 0; keyMap[i].Mask != 0; i++) {
        if (keyMap[i].VirtKey != 0 && GetAsyncKeyState(keyMap[i].VirtKey) > 1UL) {
            msxKeyMap[keyMap[i].Pos] &= ~keyMap[i].Mask;
        }
    }

    memcpy(MSXKeyMap, msxKeyMap, 16);
} 
