/** fMSX: portable MSX emulator ******************************/
/**                                                         **/
/**                         Common.h                        **/
/**                                                         **/
/** This file contains standard screen refresh drivers      **/
/** common for X11, VGA, and other "chunky" bitmapped video **/
/** implementations. It also includes dummy sound drivers   **/
/** for fMSX.                                               **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-2003                 **/
/**               John Stiles     1996                      **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

#include "SpriteLine.h"

#define WIDTH  320
#define HEIGHT 240

extern UInt32* emuFrameBuffer;
extern int*    emuLineWidth;
extern UInt32  emuFixedPalette[256];
extern UInt32  emuPalette0;
extern UInt32  emuPalette[300];

static int FirstLine = 18;     /* First scanline in the emuFrameBuffer */

static void  ColorSprites(UInt8 Y,UInt8 *ZBuf);
static UInt32 *RefreshBorder(UInt8 Y,UInt32 C);
static void  ClearLine(UInt32 *P,UInt32 C);
static UInt32 YJKColor(int Y,int J,int K);

/** ClearLine() **********************************************/
/** Clear 256 pixels from P with color C.                   **/
/*************************************************************/
static void ClearLine(register UInt32 *P,register UInt32 C)
{
    register int J;

    for(J=0;J<256;J++) P[J]=C;
}

static void ClearLine512(register UInt32 *P,register UInt32 C)
{
    register int J;

    for(J=0;J<512;J++) P[J]=C;
}

/** YJKColor() ***********************************************/
/** Given a color in YJK format, return the corresponding   **/
/** palette entry.                                          **/
/*************************************************************/
extern UInt32 YJKtoYCbCrTable[32][32][32];

static UInt32 YJKColor(register int Y,register int J,register int K)
{
    return YJKtoYCbCrTable[Y][J>>1][K>>1];
}

/** RefreshBorder() ******************************************/
/** This function is called from RefreshLine#() to refresh  **/
/** the screen border. It returns a pointer to the start of **/
/** scanline Y in emuFrameBuffer or 0 if scanline is beyond emuFrameBuffer.     **/
/*************************************************************/
UInt32 *RefreshBorder(register UInt8 Y,register UInt32 C)
{
    register UInt32 *P;
    int btScanLine;
    register int H;

    /* First line number in the buffer */
    if(!Y) {
        FirstLine=(ScanLines212? 8:18)+VAdjust;
        evenOddPage = 1 - evenOddPage;
    }

    /* Return 0 if we've run out of the screen buffer due to overscan */
    if(Y+FirstLine>=HEIGHT) return(0);

    /* Set up the transparent color */
    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];

    /* Start of the buffer */
    P = emuFrameBuffer + evenOddPage * 2 * WIDTH * HEIGHT;

    /* Paint top of the screen */
    if(!Y) {
        for(H=2*WIDTH*FirstLine-1;H>=0;H--) P[H]=C;
        for(H=0; H < FirstLine;H++) emuLineWidth[H] = 0;
    }

    /* Start of the line */
    P+=2*WIDTH*(FirstLine+Y);
    emuLineWidth[FirstLine+Y] = 0;

    /* Paint left/right borders */
    for(H=(WIDTH-256)/2+HAdjust;H>0;H--) {
        P[H-1 - 2 * WIDTH]=C;
    }

    for(H=(WIDTH-256)/2-HAdjust;H>0;H--) {
        P[WIDTH-H - 2 * WIDTH]=C;
    }

    /* Paint bottom of the screen */
    btScanLine=ScanLines212? 212:192;
    if(Y==btScanLine-1) {
        for(H=btScanLine + FirstLine; H < 256;H++) emuLineWidth[H] = 0;
        for(H=2*WIDTH*(HEIGHT-btScanLine-FirstLine+1)-1;H>=0;H--) P[H]=C;
    }

    /* Return pointer to the scanline in emuFrameBuffer */
    return(P+(WIDTH-256)/2+HAdjust);
}

UInt32 *RefreshBorder512(register UInt8 Y,register UInt32 C)
{
    UInt32* P;
    int btScanLine;
    int H;

    /* First line number in the buffer */
    if(!Y) {
        FirstLine=(ScanLines212? 8:18)+VAdjust;
        evenOddPage = 1 - evenOddPage;
    }

    /* Return 0 if we've run out of the screen buffer due to overscan */
    if(Y+FirstLine>=HEIGHT) return(0);

    /* Set up the transparent color */
    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];

    /* Start of the buffer */
    P = emuFrameBuffer + evenOddPage * 2 * WIDTH * HEIGHT;

    /* Paint top of the screen */
    if(!Y) {
        for(H=2*WIDTH*FirstLine-1;H>=0;H--) P[H]=C;
        for(H=0; H < FirstLine;H++) emuLineWidth[H] = 1;
    }

    /* Start of the line */
    P+=2*WIDTH*(FirstLine+Y);
    emuLineWidth[FirstLine+Y] = 1;

    /* Paint left/right borders */
    for(H=(2*WIDTH-512)/2+2 * HAdjust;H>0;H--) {
        P[H-1 - 2 * WIDTH]=C;
    }

    for(H=(2*WIDTH-512)/2-2 * HAdjust;H>0;H--) {
        P[2*WIDTH-H - 2 * WIDTH]=C;
    }

    /* Paint bottom of the screen */
    btScanLine=ScanLines212? 212:192;
    if(Y==btScanLine-1) {
        for(H=btScanLine + FirstLine; H < 256;H++) emuLineWidth[H] = 1;
        for(H=2*WIDTH*(HEIGHT-btScanLine-FirstLine+1)-1;H>=0;H--) P[H]=C;
    }

    /* Return pointer to the scanline in emuFrameBuffer */
    return(P+(2*WIDTH-512)/2+2 * HAdjust);
}


/** RefreshLineF() *******************************************/
/** Dummy refresh function called for non-existing screens. **/
/*************************************************************/
void RefreshLineF(register UInt8 Y)
{
    register UInt32 *P;

    P=RefreshBorder(Y,emuPalette[BGColor]);
    if(P) ClearLine(P,emuPalette[BGColor]);
}


/** RefreshLine0() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN0.                 **/
/*************************************************************/
void RefreshLine0(register UInt8 Y)
{
    register UInt32 *P,FC,BC;
    register UInt8 X,*T,*G;

    BC=emuPalette[BGColor];
    P=RefreshBorder(Y,BC);
    if(!P) return;

    if(!ScreenON) ClearLine(P,BC);
    else
    {
        P -= 2*WIDTH;
        P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=P[7]=P[8]=BC;
        P += 2*WIDTH;

        G = &VRAM[ChrGenO + ((Y + VScroll) & 0x07)];
        T = &VRAM[ChrTabO + 40 * (Y >> 3)];
        FC=emuPalette[FGColor];
        P+=9;

        for(X=0;X<40;X++,T++,P+=6)
        {
            Y=G[(int)*T<<3];
            P[0]=Y&0x80? FC:BC;
            P[1]=Y&0x40? FC:BC;
            P[2]=Y&0x20? FC:BC;
            P[3]=Y&0x10? FC:BC;
            P[4]=Y&0x08? FC:BC;
            P[5]=Y&0x04? FC:BC;
        }

        P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=BC;
    }
}

/** RefreshLine1() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN1, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine1(register UInt8 Y)
{
    register UInt32 *P,FC,BC;
    register UInt8 K,X,*T,*G;
    UInt8* sprLine;
    UInt8 col;

    P = RefreshBorder(Y, emuPalette[BGColor]);
    if (!P) {
        return;
    }

    if (!ScreenON) {
        ClearLine(P, emuPalette[BGColor]);
        return;
    }

    sprLine = spritesLine(Y);

    Y += VScroll;
    G = &VRAM[ChrGenO + (Y & 0x07)];
    T = &VRAM[ChrTabO + ((int)(Y & 0xF8) << 2)];

    for (X = 0; X < 32; X++, T++, P += 8, sprLine += 8) {
        K = VRAM[ColTabO + (*T >> 3)];
        FC = emuPalette[K >> 4];
        BC = emuPalette[K & 0x0F];
        K = G[(int)*T << 3];

        col = sprLine[0]; P[0] = col ? emuPalette[col] : K & 0x80 ? FC : BC; 
        col = sprLine[1]; P[1] = col ? emuPalette[col] : K & 0x40 ? FC : BC;
        col = sprLine[2]; P[2] = col ? emuPalette[col] : K & 0x20 ? FC : BC; 
        col = sprLine[3]; P[3] = col ? emuPalette[col] : K & 0x10 ? FC : BC;
        col = sprLine[4]; P[4] = col ? emuPalette[col] : K & 0x08 ? FC : BC; 
        col = sprLine[5]; P[5] = col ? emuPalette[col] : K & 0x04 ? FC : BC;
        col = sprLine[6]; P[6] = col ? emuPalette[col] : K & 0x02 ? FC : BC; 
        col = sprLine[7]; P[7] = col ? emuPalette[col] : K & 0x01 ? FC : BC;
    }
}

/** RefreshLine2() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN2, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine2(register UInt8 Y)
{
    UInt32 *P,FC,BC;
    UInt8 K,X,*T;
    int I,J;
    UInt8* sprLine;
    UInt8 col;

    P = RefreshBorder(Y, emuPalette[BGColor]);
    if (!P) {
        return;
    }

    if(!ScreenON) {
        ClearLine(P, emuPalette[BGColor]);
        return;
    }

    sprLine = spritesLine(Y);

    Y += VScroll;
    T = &VRAM[ChrTabO + ((int)(Y & 0xF8) << 2)];
    I =((int)(Y&0xC0)<<5)+(Y&0x07);

    for( X =0;X < 32; X++, T++, P += 8, sprLine += 8) {
        J=(int)*T<<3;
        K =  VRAM[ColTabO + ((I + J) & ColTabM)];
        FC = emuPalette[K>>4];
        BC = emuPalette[K&0x0F];
        K =  VRAM[ChrGenO + ((I + J) & ChrGenM)];

        col = sprLine[0]; P[0] = col ? emuPalette[col] : K & 0x80 ? FC : BC; 
        col = sprLine[1]; P[1] = col ? emuPalette[col] : K & 0x40 ? FC : BC;
        col = sprLine[2]; P[2] = col ? emuPalette[col] : K & 0x20 ? FC : BC; 
        col = sprLine[3]; P[3] = col ? emuPalette[col] : K & 0x10 ? FC : BC;
        col = sprLine[4]; P[4] = col ? emuPalette[col] : K & 0x08 ? FC : BC; 
        col = sprLine[5]; P[5] = col ? emuPalette[col] : K & 0x04 ? FC : BC;
        col = sprLine[6]; P[6] = col ? emuPalette[col] : K & 0x02 ? FC : BC; 
        col = sprLine[7]; P[7] = col ? emuPalette[col] : K & 0x01 ? FC : BC;
    }
}

/** RefreshLine3() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN3, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine3(register UInt8 Y)
{
    register UInt32 *P,FC,BC;
    register UInt8 X,K,*T,*G;
    UInt8* sprLine;
    UInt8 col;

    P = RefreshBorder(Y, emuPalette[BGColor]);
    if (!P) {
        return;
    }

    if (!ScreenON) {
        ClearLine(P,emuPalette[BGColor]);
        return;
    }

    sprLine = spritesLine(Y);

    Y += VScroll;
    T = &VRAM[ChrTabO + ((int)(Y & 0xF8) << 2)];
    G = &VRAM[ChrGenO + ((Y & 0x1C) >> 2)];

    for (X = 0; X < 32; X++, T++, P += 8, sprLine += 8) {
        K = G[(int)*T << 3];
        FC = emuPalette[K >> 4];
        BC = emuPalette[K & 0x0F];

        col = sprLine[0]; P[0] = col ? emuPalette[col] : FC; 
        col = sprLine[1]; P[1] = col ? emuPalette[col] : FC;
        col = sprLine[2]; P[2] = col ? emuPalette[col] : FC; 
        col = sprLine[3]; P[3] = col ? emuPalette[col] : FC;
        col = sprLine[4]; P[4] = col ? emuPalette[col] : BC; 
        col = sprLine[5]; P[5] = col ? emuPalette[col] : BC;
        col = sprLine[6]; P[6] = col ? emuPalette[col] : BC; 
        col = sprLine[7]; P[7] = col ? emuPalette[col] : BC;
    }
}

/** RefreshLine4() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN4, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine4(register UInt8 Y)
{
    register UInt32 *P,FC,BC;
    register UInt8 K,X,C,*T,*R;
    register int I,J;

    P=RefreshBorder(Y,emuPalette[BGColor]);
    if(!P) {
        return;
    }

    if(!ScreenON) { 
        ClearLine(P,emuPalette[BGColor]);
        return;
    }

    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];

    R=colorSpritesLine(Y, SolidColor0);
    Y+=VScroll;
    T = &VRAM[ChrTabO + ((int)(Y & 0xF8) << 2)];
    I=((int)(Y&0xC0)<<5)+(Y&0x07);

    for(X=0;X<32;X++,R+=8,P+=8,T++)
    {
        J=(int)*T<<3;
        K = VRAM[ColTabO + ((I + J) & ColTabM)];
        FC=emuPalette[K>>4];
        BC=emuPalette[K&0x0F];
        K = VRAM[ChrGenO + ((I + J) & ChrGenM)];

        C=R[0];P[0]=C ? emuPalette[C >> 1]:(K&0x80)? FC:BC;
        C=R[1];P[1]=C ? emuPalette[C >> 1]:(K&0x40)? FC:BC;
        C=R[2];P[2]=C ? emuPalette[C >> 1]:(K&0x20)? FC:BC;
        C=R[3];P[3]=C ? emuPalette[C >> 1]:(K&0x10)? FC:BC;
        C=R[4];P[4]=C ? emuPalette[C >> 1]:(K&0x08)? FC:BC;
        C=R[5];P[5]=C ? emuPalette[C >> 1]:(K&0x04)? FC:BC;
        C=R[6];P[6]=C ? emuPalette[C >> 1]:(K&0x02)? FC:BC;
        C=R[7];P[7]=C ? emuPalette[C >> 1]:(K&0x01)? FC:BC;
    }
}

/** RefreshLine5() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN5, including       **/
/** sprites in this line.                                   **/
/*************************************************************/

void RefreshLine5(register UInt8 Y)
{
    static int jumpTable[] = { -128, -128, -0x8080, 0x7f80 };
    register UInt32 *P;
    register UInt8 I,X=0,*T,*R;
    int hScroll;
    int hScroll512;
    int* jump;
    int page;
    int scroll;

    P=RefreshBorder(Y,emuPalette[BGColor]);

    if(!P) {
        return;
    }

    if(!ScreenON) {
        ClearLine(P,emuPalette[BGColor]);
        return;
    }

    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];

    hScroll    = HScroll;
    hScroll512 = HScroll512;
    jump       = jumpTable + hScroll512 * 2;
    page       = (ChrTabO / 0x8000) & 1;
    scroll     = hScroll >> 1;

    R = colorSpritesLine(Y, SolidColor0);
    T = &VRAM[ChrTabO + (((int)(Y + VScroll) << 7) & ChrTabM & 0x7FFF) + scroll];

    if (VDP[25] & 2) {
        I=R[0];P[0]=emuPalette[I? I >> 1:BGColor];
        I=R[1];P[1]=emuPalette[I? I >> 1:BGColor];
        I=R[2];P[2]=emuPalette[I? I >> 1:BGColor];
        I=R[3];P[3]=emuPalette[I? I >> 1:BGColor];
        I=R[4];P[4]=emuPalette[I? I >> 1:BGColor];
        I=R[5];P[5]=emuPalette[I? I >> 1:BGColor];
        I=R[6];P[6]=emuPalette[I? I >> 1:BGColor];
        I=R[7];P[7]=emuPalette[I? I >> 1:BGColor];
        X++;R+=8;P+=8;T+=4;scroll+=4;
    }

    if (hScroll512) {
        if (scroll & 0x80) T += jump[page ^= 1];
        if (VDP[2] & 0x60) T += jump[page ^= 1] + 128;
    }
    else {
        if (scroll < 0) T += 128;
    }

#define UPDATE_T() if ((++scroll & 0x7f) == 0) T += jump[page ^= 1];

    if (VDP[9] & 0x04) {
        if (hScroll & 1) {
            for(;X<32;X++,R+=8,P+=8,T+=4) {
                I=R[0];P[0]=emuPalette[I? I >> 1:T[1]&0x0F];UPDATE_T();
                I=R[1];P[1]=emuPalette[I? I >> 1:T[0]>>4];
                I=R[2];P[2]=emuPalette[I? I >> 1:T[0]&0x0F];UPDATE_T();
                I=R[3];P[3]=emuPalette[I? I >> 1:T[3]>>4];
                I=R[4];P[4]=emuPalette[I? I >> 1:T[3]&0x0F];UPDATE_T();
                I=R[5];P[5]=emuPalette[I? I >> 1:T[2]>>4];
                I=R[6];P[6]=emuPalette[I? I >> 1:T[2]&0x0F];UPDATE_T();
                I=R[7];P[7]=emuPalette[I? I >> 1:T[5]>>4];
            }
        }
        else {
            for(;X<32;X++,R+=8,P+=8,T+=4) {
                I=R[0];P[0]=emuPalette[I? I >> 1:T[1]>>4];
                I=R[1];P[1]=emuPalette[I? I >> 1:T[1]&0x0F];UPDATE_T();
                I=R[2];P[2]=emuPalette[I? I >> 1:T[0]>>4];
                I=R[3];P[3]=emuPalette[I? I >> 1:T[0]&0x0F];UPDATE_T();
                I=R[4];P[4]=emuPalette[I? I >> 1:T[3]>>4];
                I=R[5];P[5]=emuPalette[I? I >> 1:T[3]&0x0F];UPDATE_T();
                I=R[6];P[6]=emuPalette[I? I >> 1:T[2]>>4];
                I=R[7];P[7]=emuPalette[I? I >> 1:T[2]&0x0F];UPDATE_T();
            }
        }
    }
    else {
        if (hScroll & 1) {
            for(;X<32;X++,R+=8,P+=8,T+=4) {
                I=R[0];P[0]=emuPalette[I? I >> 1:T[0]&0x0F];UPDATE_T();
                I=R[1];P[1]=emuPalette[I? I >> 1:T[1]>>4];
                I=R[2];P[2]=emuPalette[I? I >> 1:T[1]&0x0F];UPDATE_T();
                I=R[3];P[3]=emuPalette[I? I >> 1:T[2]>>4];
                I=R[4];P[4]=emuPalette[I? I >> 1:T[2]&0x0F];UPDATE_T();
                I=R[5];P[5]=emuPalette[I? I >> 1:T[3]>>4];
                I=R[6];P[6]=emuPalette[I? I >> 1:T[3]&0x0F];UPDATE_T();
                I=R[7];P[7]=emuPalette[I? I >> 1:T[4]>>4];
            }
        }
        else {     
            for(;X<32;X++,R+=8,P+=8,T+=4) {
                I=R[0];P[0]=emuPalette[I? I >> 1:T[0]>>4];
                I=R[1];P[1]=emuPalette[I? I >> 1:T[0]&0x0F];UPDATE_T();
                I=R[2];P[2]=emuPalette[I? I >> 1:T[1]>>4];
                I=R[3];P[3]=emuPalette[I? I >> 1:T[1]&0x0F];UPDATE_T();
                I=R[4];P[4]=emuPalette[I? I >> 1:T[2]>>4];
                I=R[5];P[5]=emuPalette[I? I >> 1:T[2]&0x0F];UPDATE_T();
                I=R[6];P[6]=emuPalette[I? I >> 1:T[3]>>4];
                I=R[7];P[7]=emuPalette[I? I >> 1:T[3]&0x0F];UPDATE_T();
            }
        }
    }
#undef UPDATE_T
}

/** RefreshLine6() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN6, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine6(register UInt8 Y)
{
    register UInt32 *P;
    register UInt8 X,*T,*R,C;

    P = RefreshBorder512(Y, emuPalette[BGColor & 0x03]);
    if (!P) {
        return;
    }

    if (!ScreenON) {
        ClearLine512(P, emuPalette[BGColor&0x03]);
        return;
    }

    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];
    
    R = colorSpritesLine(Y, SolidColor0);
    T = &VRAM[ChrTabO + (((int)(Y + VScroll) << 7) & ChrTabM & 0x7FFF)];

    for(X=0;X<32;X++,R+=8,P+=16,T+=4) {
        C=R[0];P[0]=emuPalette[C? C >> 1:(T[0]>>6)&0x03];
        C=R[0];P[1]=emuPalette[C? C >> 1:(T[0]>>4)&0x03];
        C=R[1];P[2]=emuPalette[C? C >> 1:(T[0]>>2)&0x03];
        C=R[1];P[3]=emuPalette[C? C >> 1:(T[0]>>0)&0x03];
        C=R[2];P[4]=emuPalette[C? C >> 1:(T[1]>>6)&0x03];
        C=R[2];P[5]=emuPalette[C? C >> 1:(T[1]>>4)&0x03];
        C=R[3];P[6]=emuPalette[C? C >> 1:(T[1]>>2)&0x03];
        C=R[3];P[7]=emuPalette[C? C >> 1:(T[1]>>0)&0x03];
        C=R[4];P[8]=emuPalette[C? C >> 1:(T[2]>>6)&0x03];
        C=R[4];P[9]=emuPalette[C? C >> 1:(T[2]>>4)&0x03];
        C=R[5];P[10]=emuPalette[C? C >> 1:(T[2]>>2)&0x03];
        C=R[5];P[11]=emuPalette[C? C >> 1:(T[2]>>0)&0x03];
        C=R[6];P[12]=emuPalette[C? C >> 1:(T[3]>>6)&0x03];
        C=R[6];P[13]=emuPalette[C? C >> 1:(T[3]>>4)&0x03];
        C=R[7];P[14]=emuPalette[C? C >> 1:(T[3]>>2)&0x03];
        C=R[7];P[15]=emuPalette[C? C >> 1:(T[3]>>0)&0x03];
    }
}

/** RefreshLine7() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN7, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine7(register UInt8 Y)
{
    static int jumpTable[] = { -128, -128, -0x8080, 0x7f80 };
    register UInt32 *P;
    register UInt8 I,X = 0,*T,*R;
    int hScroll512;
    int scrPage;
    int* jump;
    int scroll;

    P = RefreshBorder512(Y, emuPalette[BGColor]);
    if(!P) {
        return;
    }

    if (!ScreenON) {
        ClearLine512(P, emuPalette[BGColor]);
        return;
    }

    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];
    
    hScroll512 = HScroll512;
    jump       = jumpTable + hScroll512 * 2;
    scrPage    = (ChrTabO / 0x8000) & 1;
    scroll     = HScroll;

    R = colorSpritesLine(Y, SolidColor0);
    T = &VRAM[ChrTabO + (((((int)(Y + VScroll) << 8) & ChrTabM & 0xFFFF) + scroll) >> 1)];

    if (VDP[25] & 2) {
        I = R[0]; P[0]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[0]; P[1]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[1]; P[2]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[1]; P[3]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[2]; P[4]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[2]; P[5]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[3]; P[6]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[3]; P[7]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[4]; P[8]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[4]; P[9]  = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[5]; P[10] = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[5]; P[11] = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[6]; P[12] = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[6]; P[13] = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[7]; P[14] = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        I = R[7]; P[15] = emuPalette[I ? I >> 1 : emuPalette[BGColor]];
        X++, R += 8, P += 16, T += 4; scroll += 8;
    }

    if (InterlaceON && !evenOddPage && FlipEvenOdd) T += jump[2 | (scrPage ^= 1)] + 128;

    if (hScroll512) {
        if (scroll & 0x100) T += jump[scrPage ^= 1];
    }
    else {
        if (scroll < 0) T += 128;
    }

#define UPDATE_T() if ((++scroll & 0xff) == 0) T += jump[scrPage ^= 1];

    if (scroll & 1) {
        for (; X < 32; X++, R += 8, P += 16, T += 4) {
            I = R[0]; P[0]  = emuPalette[I ? I >> 1 : T[0x10000] >> 4];
            I = R[0]; P[1]  = emuPalette[I ? I >> 1 : T[0x10000] & 0x0F]; UPDATE_T();
            I = R[1]; P[2]  = emuPalette[I ? I >> 1 : T[1] >> 4];
            I = R[1]; P[3]  = emuPalette[I ? I >> 1 : T[1] & 0x0F]; UPDATE_T();
            I = R[2]; P[4]  = emuPalette[I ? I >> 1 : T[0x10001] >> 4];
            I = R[2]; P[5]  = emuPalette[I ? I >> 1 : T[0x10001] & 0x0F]; UPDATE_T();
            I = R[3]; P[6]  = emuPalette[I ? I >> 1 : T[2] >> 4];
            I = R[3]; P[7]  = emuPalette[I ? I >> 1 : T[2] & 0x0F]; UPDATE_T();
            I = R[4]; P[8]  = emuPalette[I ? I >> 1 : T[0x10002] >> 4];
            I = R[4]; P[9]  = emuPalette[I ? I >> 1 : T[0x10002] & 0x0F]; UPDATE_T();
            I = R[5]; P[10] = emuPalette[I ? I >> 1 : T[3] >> 4];
            I = R[5]; P[11] = emuPalette[I ? I >> 1 : T[3] & 0x0F]; UPDATE_T();
            I = R[6]; P[12] = emuPalette[I ? I >> 1 : T[0x10003] >> 4];
            I = R[6]; P[13] = emuPalette[I ? I >> 1 : T[0x10003] & 0x0F]; UPDATE_T();
            I = R[7]; P[14] = emuPalette[I ? I >> 1 : T[4] >> 4];
            I = R[7]; P[15] = emuPalette[I ? I >> 1 : T[4] & 0x0F]; UPDATE_T();
        }
    }
    else {
        for (; X < 32; X++, R += 8, P += 16, T += 4) {
            I = R[0]; P[0]  = emuPalette[I ? I >> 1 : T[0] >> 4];
            I = R[0]; P[1]  = emuPalette[I ? I >> 1 : T[0] & 0x0F]; UPDATE_T();
            I = R[1]; P[2]  = emuPalette[I ? I >> 1 : T[0x10000] >> 4];
            I = R[1]; P[3]  = emuPalette[I ? I >> 1 : T[0x10000] & 0x0F]; UPDATE_T();
            I = R[2]; P[4]  = emuPalette[I ? I >> 1 : T[1] >> 4];
            I = R[2]; P[5]  = emuPalette[I ? I >> 1 : T[1] & 0x0F]; UPDATE_T();
            I = R[3]; P[6]  = emuPalette[I ? I >> 1 : T[0x10001] >> 4];
            I = R[3]; P[7]  = emuPalette[I ? I >> 1 : T[0x10001] & 0x0F]; UPDATE_T();
            I = R[4]; P[8]  = emuPalette[I ? I >> 1 : T[2] >> 4];
            I = R[4]; P[9]  = emuPalette[I ? I >> 1 : T[2] & 0x0F]; UPDATE_T();
            I = R[5]; P[10] = emuPalette[I ? I >> 1 : T[0x10002] >> 4];
            I = R[5]; P[11] = emuPalette[I ? I >> 1 : T[0x10002] & 0x0F]; UPDATE_T();
            I = R[6]; P[12] = emuPalette[I ? I >> 1 : T[3] >> 4];
            I = R[6]; P[13] = emuPalette[I ? I >> 1 : T[3] & 0x0F]; UPDATE_T();
            I = R[7]; P[14] = emuPalette[I ? I >> 1 : T[0x10003] >> 4];
            I = R[7]; P[15] = emuPalette[I ? I >> 1 : T[0x10003] & 0x0F]; UPDATE_T();
        }
    }

#undef UPDATE_T
}

/** RefreshLine8() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN8, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine8(register UInt8 Y)
{
    static int jumpTable[] = { -128, -128, -0x8080, 0x7f80 };
    static UInt8 SprToScr[16] = { 0x00,0x02,0x10,0x12,0x80,0x82,0x90,0x92, 0x49,0x4B,0x59,0x5B,0xC9,0xCB,0xD9,0xDB };
    register UInt32 *P;
    register UInt8 C,X=0,*T,*R;
    int hScroll;
    int hScroll512;
    int* jump;
    int scrPage;
    int scroll;

    if (Y == 50) {
        volatile int x = 0;
        x++;
    }

    P=RefreshBorder(Y,emuFixedPalette[VDP[7]]);
    if(!P) return;

    if(!ScreenON) {
        ClearLine(P,emuFixedPalette[VDP[7]]);
        return;
    }

    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];
    
    hScroll    = HScroll;
    hScroll512 = HScroll512;
    jump       = jumpTable + hScroll512 * 2;
    scrPage    = (ChrTabO / 0x8000) & 1;
    scroll     = hScroll;

    R = colorSpritesLine(Y, SolidColor0);
    T = &VRAM[ChrTabO + (((((int)(Y + VScroll) << 8) & ChrTabM & 0xFFFF) + scroll) >> 1)];

    if (VDP[25] & 2) {
        C=R[0];P[0]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        C=R[1];P[1]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        C=R[2];P[2]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        C=R[3];P[3]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        C=R[4];P[4]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        C=R[5];P[5]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        C=R[6];P[6]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        C=R[7];P[7]=emuFixedPalette[C? SprToScr[C >> 1]:BGColor];
        X++, T += 4, R += 8, P += 8; scroll += 8;
    }

    if (hScroll512) {
        if (scroll & 0x100) T += jump[scrPage ^= 1];
    }
    else {
        if (scroll < 0) T += 128;
    }
    if (InterlaceON && !evenOddPage && FlipEvenOdd) T += jump[2 | (scrPage ^= 1)] + 128;

#define UPDATE_T() if ((++scroll & 0xff) == 0) T += jump[scrPage ^= 1];

    if (scroll & 1) {
        for(; X < 32; X++, T += 4, R += 8, P += 8) {
            C = R[0]; P[0] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10000]]; UPDATE_T();
            C = R[1]; P[1] = emuFixedPalette[C ? SprToScr[C >> 1] : T[1]]; UPDATE_T();
            C = R[2]; P[2] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10001]]; UPDATE_T();
            C = R[3]; P[3] = emuFixedPalette[C ? SprToScr[C >> 1] : T[2]]; UPDATE_T();
            C = R[4]; P[4] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10002]]; UPDATE_T();
            C = R[5]; P[5] = emuFixedPalette[C ? SprToScr[C >> 1] : T[3]]; UPDATE_T();
            C = R[6]; P[6] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10003]]; UPDATE_T();
            C = R[7]; P[7] = emuFixedPalette[C ? SprToScr[C >> 1] : T[4]]; UPDATE_T();
        }
    }
    else {
        for(; X < 32; X++, T += 4, R += 8, P += 8) {
            C = R[0]; P[0] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0]]; UPDATE_T();
            C = R[1]; P[1] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10000]]; UPDATE_T();
            C = R[2]; P[2] = emuFixedPalette[C ? SprToScr[C >> 1] : T[1]]; UPDATE_T();
            C = R[3]; P[3] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10001]]; UPDATE_T();
            C = R[4]; P[4] = emuFixedPalette[C ? SprToScr[C >> 1] : T[2]]; UPDATE_T();
            C = R[5]; P[5] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10002]]; UPDATE_T();
            C = R[6]; P[6] = emuFixedPalette[C ? SprToScr[C >> 1] : T[3]]; UPDATE_T();
            C = R[7]; P[7] = emuFixedPalette[C ? SprToScr[C >> 1] : T[0x10003]]; UPDATE_T();
        }
    }
#undef UPDATE_T
}

/** RefreshLine10() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN10/11, including   **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine10(register UInt8 Y)
{
    static int jumpTable[] = { -128, -128, -0x8080, 0x7f80 };
    register UInt32 *P;
    register UInt8 C,X=1,*T,*R;
    register int J,K;
    int hScroll;
    int hScroll512;
    int* jump;
    int scrPage;
    int scroll;

    P = RefreshBorder(Y, emuFixedPalette[VDP[7]]);
    if (!P) {
        return;
    }

    if (!ScreenON) {
        ClearLine(P, emuFixedPalette[VDP[7]]);
        return;
    }

    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];
    
    hScroll    = HScroll;
    hScroll512 = HScroll512;
    jump       = jumpTable + hScroll512 * 2;
    scrPage    = (ChrTabO / 0x8000) & 1;
    scroll     = hScroll;

    R = colorSpritesLine(Y, SolidColor0);
    T = &VRAM[ChrTabO + (((((int)(Y + VScroll) << 8) & ChrTabM & 0xFFFF) + scroll) >> 1)];

    if (VDP[25] & 2) {
        C=R[0];P[0]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[1];P[1]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[2];P[2]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[3];P[3]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[0];P[0]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[1];P[1]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[2];P[2]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[3];P[3]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        X++, T += 4, R += 8, P += 8; scroll += 8;
    }

    if (hScroll512) {
        if (scroll & 0x100) T += jump[scrPage ^= 1];
    }
    else {
        if (scroll < 0) T += 128;
    }
    if (InterlaceON && !evenOddPage && FlipEvenOdd) T += jump[2 | (scrPage ^= 1)] + 128;

    /* Draw first 4 pixels */
    C = R[0]; P[0] = C ? emuPalette[C >> 1] : emuFixedPalette[VDP[7]];
    C = R[1]; P[1] = C ? emuPalette[C >> 1] : emuFixedPalette[VDP[7]];
    C = R[2]; P[2] = C ? emuPalette[C >> 1] : emuFixedPalette[VDP[7]];
    C = R[3]; P[3] = C ? emuPalette[C >> 1] : emuFixedPalette[VDP[7]];
    R += 4; P += 4;

#define UPDATE_T() if ((++scroll & 0xff) == 0) T += jump[scrPage ^= 1];

    if (scroll & 1) {
        for(X; X < 64; X++, T += 2, R += 4, P += 4) {
            UInt8 t0, t1, t2, t3;

            t0 = T[0x10000];  UPDATE_T();
            t1 = T[1];        UPDATE_T();
            t2 = T[0x10001];  UPDATE_T();
            t3 = T[2];        UPDATE_T();

            K = (t0 & 0x07) | ((t1 & 0x07) << 3);
            J = (t2 & 0x07) | ((t3 & 0x07) << 3);

            C = R[0]; Y = t0 >> 3; P[0] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
            C = R[1]; Y = t1 >> 3; P[1] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
            C = R[2]; Y = t2 >> 3; P[2] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
            C = R[3]; Y = t3 >> 3; P[3] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
        }
    }
    else {
        for(X = 0; X < 63; X++, T += 2, R += 4, P += 4) {
            UInt8 t0, t1, t2, t3;

            t0 = T[0];        UPDATE_T();
            t1 = T[0x10000];  UPDATE_T();
            t2 = T[1];        UPDATE_T();
            t3 = T[0x10001];  UPDATE_T();

            K = (t0 & 0x07) | ((t1 & 0x07) << 3);
            J = (t2 & 0x07) | ((t3 & 0x07) << 3);

            C = R[0]; Y = t0 >> 3; P[0] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
            C = R[1]; Y = t1 >> 3; P[1] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
            C = R[2]; Y = t2 >> 3; P[2] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
            C = R[3]; Y = t3 >> 3; P[3] = C ? emuPalette[C >> 1] : Y & 1 ? emuPalette[Y >> 1] : YJKColor(Y, J, K);
        }
    }
}

/** RefreshLine12() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN12, including      **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine12(register byte Y)
{
    static int jumpTable[] = { -128, -128, -0x8080, 0x7f80 };
    register UInt32 *P;
    register UInt8 C,X=1,*T,*R;
    register int J,K;
    int hScroll;
    int hScroll512;
    int* jump;
    int scrPage;
    int scroll;

    P=RefreshBorder(Y, emuFixedPalette[VDP[7]]);
    if(!P) {
        return;
    }

    if(!ScreenON) {
        ClearLine(P,emuFixedPalette[VDP[7]]);
        return;
    }

    emuPalette[0] = (!BGColor || SolidColor0) ? emuPalette0 : emuPalette[BGColor];
    
    hScroll    = HScroll;
    hScroll512 = HScroll512;
    jump       = jumpTable + hScroll512 * 2;
    scrPage    = (ChrTabO / 0x8000) & 1;
    scroll     = hScroll;

    R = colorSpritesLine(Y, SolidColor0);
    T = &VRAM[ChrTabO + (((((int)(Y + VScroll) << 8) & ChrTabM & 0xFFFF) + scroll) >> 1)];

    if (VDP[25] & 2) {
        C=R[0];P[0]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[1];P[1]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[2];P[2]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[3];P[3]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[0];P[0]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[1];P[1]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[2];P[2]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        C=R[3];P[3]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
        X++, T += 4, R += 8, P += 8; scroll += 8;
    }

    if (hScroll512) {
        if (scroll & 0x100) T += jump[scrPage ^= 1];
    }
    else {
        if (scroll < 0) T += 128;
    }
    if (InterlaceON && !evenOddPage && FlipEvenOdd) T += jump[2 | (scrPage ^= 1)] + 128;

    /* Draw first 4 pixels */
    C=R[0];P[0]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
    C=R[1];P[1]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
    C=R[2];P[2]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
    C=R[3];P[3]=C? emuPalette[C >> 1]:emuFixedPalette[VDP[7]];
    R+=4;P+=4;

#define UPDATE_T() if ((++scroll & 0xff) == 0) T += jump[scrPage ^= 1];

    if (scroll & 1) {
        for(;X<64;X++,T+=2,R+=4,P+=4) {
            UInt8 t0, t1, t2, t3;

            t0 = T[0x10000];  UPDATE_T();
            t1 = T[1];        UPDATE_T();
            t2 = T[0x10001];  UPDATE_T();
            t3 = T[2];        UPDATE_T();

            K=(t0 & 0x07) | ((t1 & 0x07) << 3);
            J=(t2 & 0x07) | ((t3 & 0x07) << 3);

            C = R[0]; P[0] = C ? emuPalette[C >> 1] : YJKColor(t0 >> 3, J, K);
            C = R[1]; P[1] = C ? emuPalette[C >> 1] : YJKColor(t1 >> 3, J, K);
            C = R[2]; P[2] = C ? emuPalette[C >> 1] : YJKColor(t2 >> 3, J, K);
            C = R[3]; P[3] = C ? emuPalette[C >> 1] : YJKColor(t3 >> 3, J, K);
        }
    }
    else {
        for(;X<64;X++,T+=2,R+=4,P+=4) {
            UInt8 t0, t1, t2, t3;

            t0 = T[0];        UPDATE_T();
            t1 = T[0x10000];  UPDATE_T();
            t2 = T[1];        UPDATE_T();
            t3 = T[0x10001];  UPDATE_T();

            K=(t0 & 0x07) | ((t1 & 0x07) << 3);
            J=(t2 & 0x07) | ((t3 & 0x07) << 3);

            C = R[0]; P[0] = C ? emuPalette[C >> 1] : YJKColor(t0 >> 3, J, K);
            C = R[1]; P[1] = C ? emuPalette[C >> 1] : YJKColor(t1 >> 3, J, K);
            C = R[2]; P[2] = C ? emuPalette[C >> 1] : YJKColor(t2 >> 3, J, K);
            C = R[3]; P[3] = C ? emuPalette[C >> 1] : YJKColor(t3 >> 3, J, K);
        }
    }
#undef UPDATE_T
}

/** RefreshLineTx80() ****************************************/
/** Refresh line Y (0..191/211) of TEXT80.                  **/
/*************************************************************/
void RefreshLineTx80(register UInt8 Y)
{
    register UInt32 *P,FC,BC;
    register UInt8 X,M,*T,*C,*G;

    BC = emuPalette[BGColor];
    P = RefreshBorder512(Y,BC);

    if (!P) {
        return;
    }

    if (!ScreenON) {
        ClearLine512(P,BC);
        return;
    }

    P -= 2*WIDTH;
    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=P[7]=P[8]=BC;
    P[9]=P[10]=P[11]=P[12]=P[13]=P[14]=P[15]=P[16]=P[17]=BC;
    P +=2*WIDTH;

    G = &VRAM[ChrGenO + ((Y + VScroll) & 0x07)];
    T = &VRAM[ChrTabO + ((80 * (Y >> 3)) & ChrTabM)];
    C = &VRAM[ColTabO + ((10 * (Y >> 3)) & ColTabM)];
    P+=18;

    for(X=0,M=0x00;X<80;X++,T++,P+=6) {
        if(!(X&0x07)) M=*C++;
        if(M&0x80) { FC=emuPalette[XFGColor];BC=emuPalette[XBGColor]; }
        else       { FC=emuPalette[FGColor];BC=emuPalette[BGColor]; }
        M<<=1;
        Y=G[(int)*T<<3];
        P[0]=Y&0x80? FC:BC;
        P[1]=Y&0x40? FC:BC;
        P[2]=Y&0x20? FC:BC;
        P[3]=Y&0x10? FC:BC;
        P[4]=Y&0x08? FC:BC;
        P[5]=Y&0x04? FC:BC;
    }
    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=
        P[7]=P[8]=P[9]=P[10]=P[11]=P[12]=P[13]=emuPalette[BGColor];
}
