#include "MSX.h"
#include "Casette.h"
#include "Disk.h"
#include <string.h>

 typedef struct { 
    int sectors;
    UInt8 heads;
    UInt8 dirEntries;
    UInt8 sectorsPerTrack;
    UInt8 sectorsPerFAT;
    UInt8 sectorsPerCluster; 
} FormatInfo;
 
static const UInt8 bootSector[] = {
    0xEB, 0xFE, 0x90, 0x56, 0x46, 0x42, 0x2D, 0x31, 
    0x39, 0x38, 0x39, 0x00, 0x02, 0x02, 0x01, 0x00,
    0x02, 0x70, 0x00, 0xA0, 0x05, 0xF9, 0x03, 0x00, 
    0x09, 0x00, 0x02, 0x00, 0x00, 0x00, 0xD0, 0xED,
    0x53, 0x58, 0xC0, 0x32, 0xC2, 0xC0, 0x36, 0x55, 
    0x23, 0x36, 0xC0, 0x31, 0x1F, 0xF5, 0x11, 0x9D,
    0xC0, 0x0E, 0x0F, 0xCD, 0x7D, 0xF3, 0x3C, 0x28, 
    0x28, 0x11, 0x00, 0x01, 0x0E, 0x1A, 0xCD, 0x7D,
    0xF3, 0x21, 0x01, 0x00, 0x22, 0xAB, 0xC0, 0x21, 
    0x00, 0x3F, 0x11, 0x9D, 0xC0, 0x0E, 0x27, 0xCD,
    0x7D, 0xF3, 0xC3, 0x00, 0x01, 0x57, 0xC0, 0xCD, 
    0x00, 0x00, 0x79, 0xE6, 0xFE, 0xFE, 0x02, 0x20,
    0x07, 0x3A, 0xC2, 0xC0, 0xA7, 0xCA, 0x22, 0x40, 
    0x11, 0x77, 0xC0, 0x0E, 0x09, 0xCD, 0x7D, 0xF3,
    0x0E, 0x07, 0xCD, 0x7D, 0xF3, 0x18, 0xB4, 0x42, 
    0x6F, 0x6F, 0x74, 0x20, 0x65, 0x72, 0x72, 0x6F,
    0x72, 0x0D, 0x0A, 0x50, 0x72, 0x65, 0x73, 0x73, 
    0x20, 0x61, 0x6E, 0x79, 0x20, 0x6B, 0x65, 0x79,
    0x20, 0x66, 0x6F, 0x72, 0x20, 0x72, 0x65, 0x74, 
    0x72, 0x79, 0x0D, 0x0A, 0x24, 0x00, 0x4D, 0x53,
    0x58, 0x44, 0x4F, 0x53, 0x20, 0x20, 0x53, 0x59, 
    0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF3, 0x2A,
    0x51, 0xF3, 0x11, 0x00, 0x01, 0x19, 0x01, 0x00, 
    0x01, 0x11, 0x00, 0xC1, 0xED, 0xB0, 0x3A, 0xEE,
    0xC0, 0x47, 0x11, 0xEF, 0xC0, 0x21, 0x00, 0x00, 
    0xCD, 0x51, 0x52, 0xF3, 0x76, 0xC9, 0x18, 0x64,
    0x3A, 0xAF, 0x80, 0xF9, 0xCA, 0x6D, 0x48, 0xD3, 
    0xA5, 0x0C, 0x8C, 0x2F, 0x9C, 0xCB, 0xE9, 0x89,
    0xD2, 0x00, 0x32, 0x26, 0x40, 0x94, 0x61, 0x19, 
    0x20, 0xE6, 0x80, 0x6D, 0x8A, 0x00, 0x00, 0x00
};

static const FormatInfo formatInfo[8] = {
    { 720,  1, 112, 9, 2, 2 },
    { 1440, 2, 112, 9, 3, 2 },
    { 640,  1, 112, 8, 1, 2 },
    { 1280, 2, 112, 8, 2, 2 },
    { 360,  1, 64,  9, 2, 1 },
    { 720,  2, 112, 9, 2, 2 },
    { 320,  1, 64,  8, 1, 1 },
    { 640,  2, 112, 8, 1, 2 }
};

static void phydio(Z80* z80);
static void dskchg(Z80* z80);
static void getdpb(Z80* z80);
static void dskfmt(Z80* z80);
static void drvoff(Z80* z80);
static void tapion(Z80* z80);
static void tapin(Z80* z80);
static void tapiof(Z80* z80);
static void tapoon(Z80* z80);
static void stmotr(Z80* z80);
static void tapoof(Z80* z80);
static void tapout(Z80* z80);


void PatchZ80(Z80* z80)
{
    switch (z80->PC.W - 2) {
    case 0x4010: phydio(z80); break;
    case 0x4013: dskchg(z80); break;
    case 0x4016: getdpb(z80); break;
    case 0x401c: dskfmt(z80); break;
    case 0x401f: drvoff(z80); break;
    case 0x00e1: tapion(z80); break;
    case 0x00E4: tapin(z80);  break;
    case 0x00e7: tapiof(z80); break;
    case 0x00ea: tapoon(z80); break;
    case 0x00ed: tapout(z80); break;
    case 0x00f0: tapoof(z80); break;
    case 0x00f3: stmotr(z80); break;
    }
}


extern byte *RAM[],PSL[],SSLReg;

static byte RdZ80(word address) {
    if(address != 0xFFFF) {
        return(RAM[address >> 13][address & 0x1FFF]);
    }
    else {
        return((PSL[3] == 3) ? ~SSLReg : RAM[7][0x1FFF]);
    }
}


static void phydio(Z80* z80) {
    UInt8 buffer[512];
    UInt8 drive;
	UInt16 sector;
	UInt16 address;
	int write;
	UInt8 origSlotPri;
	UInt8 origSlotSec;
	UInt8 slotPri;
	UInt8 slotSec;
    int i;
    
    z80->IFF |= 1;

    drive   = z80->AF.B.h;
    sector  = z80->DE.W;
	address = z80->HL.W;
	write   = z80->AF.B.l & C_FLAG;
	
    if (!DiskPresent(drive)) { 
        z80->AF.W=0x0201;
        return; 
    }

    if (address + z80->BC.B.h * 512 > 0x10000) {
		z80->BC.B.h = (0x10000 - address) / 512;
	}

	origSlotPri = InZ80(0xa8);
	origSlotSec = RdZ80(0xffff) ^ 0xff;
	slotPri     = 0x55 * ((origSlotPri >> 6) & 0x03);
	slotSec     = 0x55 * ((origSlotSec >> 6) & 0x03);

    OutZ80(0xa8, slotPri);
    WrZ80(0xffff, slotSec);

	while (z80->BC.B.h) {
		if (write) {
            for (i = 0; i < 512; i++) {
                buffer[i]=RdZ80(address++);
            }

            if (!DiskWrite(drive, buffer, sector)) {
                z80->AF.W=0x0a01;
                WrZ80(0xffff,origSlotSec);
                OutZ80(0xa8,origSlotPri);
                return;
            }
		} 
        else {
            if (!DiskRead(drive, buffer, sector)) {
                z80->AF.W = 0x0401;
                WrZ80(0xffff, origSlotSec);
                OutZ80(0xa8, origSlotPri);
                return;
            }
            for (i = 0; i < 512; i++) {
                WrZ80(address++, buffer[i]);
            }
		}
		z80->BC.B.h--;
		sector++;
	}

	// restore memory settings
    WrZ80(0xffff, origSlotSec);
    OutZ80(0xa8, origSlotPri);

    z80->AF.B.l&=~C_FLAG;
}

static void dskchg(Z80* z80) {
    UInt8 buffer[512];
    UInt8 drive = z80->AF.B.h;

    z80->IFF |= 1;

    if (!DiskPresent(z80->AF.B.h)) { 
        z80->AF.W = 0x0201;
        return; 
    }

    if (!DiskRead(drive, buffer, 1)) { 
        z80->AF.W = 0x0a01; 
        return; 
    }

	z80->BC.B.h = buffer[0];

    getdpb(z80);

	if (z80->AF.B.l & C_FLAG) {
		z80->AF.W = 0x0a01;
	}
    
	z80->BC.B.h = 0;
}

static void getdpb(Z80* z80) {
    UInt16 dirSectorNo;
    UInt16 dataSectorNo;
    UInt8  fatSectorNo;
    UInt16 maxClusters;
    UInt16 address;
    UInt8  mediaDescriptor;
    
    if (!DiskPresent(z80->AF.B.h)) { 
        z80->AF.W = 0x0201;
        return; 
    }

	mediaDescriptor = z80->BC.B.h;
	address = z80->HL.W;

    switch (mediaDescriptor) {
    case 0xf8:
        fatSectorNo = 2;
        maxClusters = 355;
        break;
    case 0xf9:
        fatSectorNo = 3;
        maxClusters = 714;
        break;
    case 0xfa:
        fatSectorNo = 1;
        maxClusters = 316;
        break;
    case 0xfb:
        fatSectorNo = 2;
        maxClusters = 635;
        break;
    case 0xfc:
        fatSectorNo = 2;
        maxClusters = 316;
        break;
    default:
        z80->AF.W=0x0C01;
        return;
    }

    dirSectorNo = 1 + fatSectorNo * 2;
    dataSectorNo = dirSectorNo + 7;

    WrZ80(address +  1, mediaDescriptor);  
    WrZ80(address +  2, 0);  
    WrZ80(address +  3, 2);  
    WrZ80(address +  4, 15);  
    WrZ80(address +  5, 4);  
    WrZ80(address +  6, 1);  
    WrZ80(address +  7, 2);  
    WrZ80(address +  8, 1);  
    WrZ80(address +  9, 0);  
    WrZ80(address + 10, 2);  
    WrZ80(address + 11, 112);  
    WrZ80(address + 12, dataSectorNo & 0xff);  
    WrZ80(address + 13, dataSectorNo >> 8);  
    WrZ80(address + 14, maxClusters & 0xff);  
    WrZ80(address + 15, maxClusters >> 8);  
    WrZ80(address + 16, fatSectorNo);  
    WrZ80(address + 17, dirSectorNo & 0xff);  
    WrZ80(address + 18, dirSectorNo >> 8);  

    /* Return success      */
    z80->AF.B.l&=~C_FLAG;
}

static void dskfmt(Z80* z80) {
    UInt8 buffer[512];
    UInt8 index;
    int j;
    int i;
    int dirSize;
    int dataSize;
    int sector;
    
    z80->AF.B.l |= C_FLAG;

    z80->IFF |= 1;

    /* If invalid choice, return "Bad parameter": */
    if (z80->AF.B.h == 0x87) {
        z80->AF.B.h = 2;
    }
    
    index = z80->AF.B.h - 1;
    if (index > 1) {
        z80->AF.W = 0x0c01;
        return; 
    }

    /* If no disk, return "Not ready": */
    if(!DiskPresent(z80->DE.B.h)) { 
        z80->AF.W=0x0201;
        return; 
    }

	memset(buffer, 0, 512);
    memcpy(buffer, bootSector, sizeof(bootSector));

    memcpy(buffer + 3, "blueMSX", 8);

    /* sectors per cluster */
    buffer[13] = formatInfo[index].sectorsPerCluster;  

    /* Number of names     */
    buffer[17] = formatInfo[index].dirEntries;       
    buffer[18] = 0;

    /* Number of sectors   */
    buffer[19] = formatInfo[index].sectors & 0xff;
    buffer[20] = (formatInfo[index].sectors >> 8) & 0xff;

    /* Format ID [F8h-FFh] */
    buffer[21] = index + 0xf8;

    /* sectors per FAT     */
    buffer[22] = formatInfo[index].sectorsPerFAT;
    buffer[23] = 0;

    /* sectors per track   */
    buffer[24] = formatInfo[index].sectorsPerTrack;
    buffer[25] = 0;

    /* Number of heads     */
    buffer[26] = formatInfo[index].heads;
    buffer[27] = 0;

    /* If can't write bootblock, return "Write protected": */
    if (!DiskWrite(z80->DE.B.h, buffer, 0)) { 
        z80->AF.W = 0x0001;
        return; 
    }

    /* Writing FATs: */
    sector = 1;
    for (j = 0; j < 2; j++) {
        buffer[0] = index + 0xf8;
        buffer[1] = 0xff;
        buffer[2] = 0xff;
        memset(buffer + 3, 0x00, 509);

        if (!DiskWrite(z80->DE.B.h, buffer, sector++)) { 
            z80->AF.W = 0x0a01;
            return; 
        }

        memset(buffer, 0x00, 512);

        for(i = formatInfo[index].sectorsPerFAT; i > 1; i--) {
            if (!DiskWrite(z80->DE.B.h, buffer, sector++)) { 
                z80->AF.W = 0x0A01;
                return; 
            }
        }
    }

    dirSize = formatInfo[index].dirEntries / 16;                     
    dataSize = formatInfo[index].sectors - 2*formatInfo[index].sectorsPerFAT - dirSize - 1; 

    memset(buffer, 0x00, 512);
    while (dirSize--) {
        if (!DiskWrite(z80->DE.B.h, buffer, sector++)) { 
            z80->AF.W = 0x0A01;
            return; 
        }
    }

    memset(buffer, 0xFF, 512);
    while (dataSize--) {
        if (!DiskWrite(z80->DE.B.h, buffer, sector++)) { 
            z80->AF.W = 0x0a01;
            return; 
        }
    }

    /* Return success      */
    z80->AF.B.l &= ~C_FLAG;
}


static void drvoff(Z80* z80) {
}

static void tapion(Z80* z80) {
    z80->AF.B.l|=C_FLAG;

    if (TapeReadHeader()) {
        z80->AF.B.l&=~C_FLAG;
    }
}

static void tapin(Z80* z80) {
    UInt8 value;

    z80->AF.B.l |= C_FLAG;

    if (TapeRead(&value)) {
        z80->AF.B.h = value;
        z80->AF.B.l &= ~C_FLAG;
    }
}

static void tapiof(Z80* z80) {
    z80->AF.B.l &= ~C_FLAG;
}

static void tapoon(Z80* z80) {
    z80->AF.B.l |= C_FLAG;

    if (TapeWriteHeader()) {
        z80->AF.B.l &= ~C_FLAG;
    }
}

static void tapout(Z80* z80) {
    z80->AF.B.l |= C_FLAG;
}

static void tapoof(Z80* z80) {
    z80->AF.B.l |= C_FLAG;

    if (TapeWrite(z80->AF.B.h)) {
        z80->AF.B.l &= ~C_FLAG;
    }
}

static void stmotr(Z80* z80) {
    z80->AF.B.l &= ~C_FLAG;
}
