/*****************************************************************************
** File:        MegaromCartridge.c
**
** Author:      Daniel Vik
**
** Description: Megarom cartdidge methods
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "MegaromCartridge.h"
#include "romMapper.h"
#include "RomLoader.h"
#include "SlotManager.h"

#include "romMapperStandard.h"
#include "romMapperMsxDos2.h"
#include "romMapperKonami5.h"
#include "romMapperNormal.h"
#include "romMapperKonami4.h"
#include "romMapperASCII8.h"
#include "romMapperASCII16.h"
#include "romMapperGameMaster2.h"
#include "romMapperASCII8sram.h"
#include "romMapperASCII16sram.h"
#include "romMapperKoei.h"
#include "romMapperASCII16nf.h"
#include "romMapperKonami4nf.h"
#include "romMapperKanji.h"
#include "romMapperKanji12.h"
#include "romMapperHolyQuran.h"
#include "romMapperMegaRam.h"
#include "romMapperMsxAudio.h"
#include "romMapperPlain.h"
#include "romMapperBasic.h"
#include "romMapperHarryFox.h"
#include "romMapperHalnote.h"
#include "romMapperRType.h"
#include "romMapperCrossBlaim.h"
#include "romMapperLodeRunner.h"
#include "romMapperKorean80.h"
#include "romMapperKorean90.h"
#include "romMapperKorean126.h"
#include "romMapperPAC.h"
#include "romMapperFMPAC.h"
#include "romMapperSCCplus.h"
#include "romFMPAC.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>


typedef struct {
    SCC*     scc;
    struct {
        int slot;
        int sslot;
    } cart[2];
} CartridgeInfo;


CartridgeInfo cartridgeInfo;


void cartridgeInit(SCC* scc)
{
    cartridgeInfo.scc    = scc;
}

void cartridgeSetSlotInfo(int cartNo, int slot, int sslot) 
{
    cartridgeInfo.cart[cartNo].slot  = slot;
    cartridgeInfo.cart[cartNo].sslot = sslot;
}


void cartridgeInsert(int cartNo, RomType romType, char* cart, char* cartZip) 
{
    char* romName = cartZip != NULL ? cartZip : cart;
    UInt8* buf;
    int size;
    int page;
    int slot  = cartridgeInfo.cart[cartNo].slot;
    int sslot = cartridgeInfo.cart[cartNo].sslot;

    // Delete old cartridge
    for (page = 0; page < 8; page++) {
        slotRemove(slot, sslot, page);
    }

    sccReset(cartridgeInfo.scc);

    if (cart == NULL) {
        return;
    }

    switch (romType) {
    case ROM_SCCPLUS:
        romMapperSCCplusCreate(NULL, NULL, 0x10000, slot, sslot, 2, cartridgeInfo.scc);
        break;

    case ROM_MEGARAM128:
        romMapperMegaRAMCreate(0x20000, slot, sslot, 0);
        break;

    case ROM_MEGARAM256:
        romMapperMegaRAMCreate(0x40000, slot, sslot, 0);
        break;

    case ROM_MEGARAM512:
        romMapperMegaRAMCreate(0x80000, slot, sslot, 0);
        break;

    case ROM_MEGARAM768:
        romMapperMegaRAMCreate(0xc0000, slot, sslot, 0);
        break;

    case ROM_MEGARAM2M:
        romMapperMegaRAMCreate(0x200000, slot, sslot, 0);
        break;

    case ROM_PAC:
        romMapperPACCreate("PacA.rom", NULL, 0, slot, sslot, 2);
        break;

    case ROM_FMPAC:
        romMapperFMPACCreate("FmPacA.rom", romFMPAC, 0x10000, slot, sslot, 2);
        break;

    default:
        buf = romLoad(cart, cartZip, &size);
        if (buf == NULL) {
            break;
        }
        
        romType = romMapperGuessRom(buf, size, 1);

        if (romType != ROM_MSXAUDIO && romType != ROM_KONAMI5 && romType != ROM_ASCII16 && 
            romType != ROM_0x4000 && 
            romType != ROM_HARRYFOX && romType != ROM_BASIC && romType != ROM_FMPAC &&
            (size < 0x10000 || (size == 0x10000 && buf[0x4000] == 'A' && buf[0x4001] == 'B'))) {
            romType = ROM_PLAIN;
        }

        switch (romType) {
        case ROM_0x4000:
            romMapperNormalCreate(romName, buf, size, slot, sslot, 2);
            break;

        case ROM_BASIC:
            romMapperBasicCreate(romName, buf, size, slot, sslot, 4);
            break;

        case ROM_PLAIN:
            romMapperPlainCreate(romName, buf, size, slot, sslot, 0);
            break;

        case ROM_STANDARD:
            romMapperStandardCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_MSXDOS2:
            romMapperMsxDos2Create(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_KONAMI5:
            romMapperKonami5Create(romName, buf, size, slot, sslot, 2, cartridgeInfo.scc);
            break;
            
        case ROM_KONAMI4:
            romMapperKonami4Create(romName, buf, size, slot, sslot, 2);
            break;

        case ROM_HOLYQURAN:
            romMapperHolyQuranCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_ASCII8:
            romMapperASCII8Create(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_ASCII16:
            romMapperASCII16Create(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_ASCII8SRAM:
            romMapperASCII8sramCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_ASCII16SRAM:
            romMapperASCII16sramCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_KOEI:
            romMapperKoeiCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_KONAMI4NF:
            romMapperKonami4nfCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_ASCII16NF:
            romMapperASCII16nfCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_GAMEMASTER2:
            romMapperGameMaster2Create(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_HARRYFOX:
            romMapperHarryFoxCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_HALNOTE:
            romMapperHalnoteCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_RTYPE:
            romMapperRTypeCreate(romName, buf, size, slot, sslot, 2);
            break;

        case ROM_LODERUNNER:
            romMapperLodeRunnerCreate(romName, buf, size, slot, sslot, 4);
            break;
            
        case ROM_CROSSBLAIM:
            romMapperCrossBlaimCreate(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_KOREAN80:
            romMapperKorean80Create(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_KOREAN90:
            romMapperKorean90Create(romName, buf, size, slot, sslot, 2);
            break;
            
        case ROM_KOREAN126:
            romMapperKorean126Create(romName, buf, size, slot, sslot, 2);
            break;

        case ROM_MSXAUDIO:
            romMapperMsxAudioCreate(romName, buf, size, slot, sslot, 0);
            break;

        case ROM_KANJI:
            romMapperKanjiCreate(buf, size);
            break;

        case ROM_KANJI12:
            romMapperKanji12Create(buf, size);
            break;
            
        case ROM_FMPAC:
            romMapperFMPACCreate(romName, buf, size, slot, sslot, 2);
            break;
        }

        free(buf);
    }
}

