/*****************************************************************************
** File:        SlotManager.c
**
** Author:      Daniel Vik
**
** Description: Manages PPI slots and ram mapper
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "SlotManager.h"
#include "IoPort.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


extern void SetCapslockLed(int enable);


#define OFFSETOF(s, a) ((int)(&((s*)0)->a))


typedef struct {
    UInt8          reg[4];
    UInt8          outReg[3]; 
    UInt8          inReg[3];
    UInt8          oldOutReg[3];
    UInt8*         keymap;
    AudioKeyClick* keyClick;
} I8255;

typedef struct {
    UInt8* pageData;
    int    readEnable;
    int    writeEnable;
} RamSlotState;

typedef struct {
    int   subslotted;
    UInt8 state;
    UInt8 substate;
    UInt8 sslReg;
} PrimarySlotState;


typedef struct {
    int           type;
    UInt16        startpage;
    UInt16        pages;
    UInt8*        pageData;
    int           writeEnable;
    int           readEnable;
    SlotCallbacks callbacks;
    void*         ref;
} Slot;

static I8255            i8255;
static RamSlotState     ramslot[8];
static PrimarySlotState pslot[4];
static Slot             slotTable[4][4][8];
static Slot             slotUnslottedTable[16];
static Slot             slotAddr0;
static UInt8            emptyRAM[0x2000];

int slotGetState(void* ref, UInt8* buffer, UInt32 systemTime)
{
    int size = 0;
    int slot;
    int sslot;
    int page;

    memcpy(buffer + size, &i8255, OFFSETOF(I8255, keymap));
    size += OFFSETOF(I8255, keymap);

    memcpy(buffer + size, pslot, sizeof(pslot));
    size += sizeof(pslot);

    for (slot = 0; slot < 4; slot++) {
        for (sslot = 0; sslot < 4; sslot++) {
            for (page = 0; page < 8; page++) {
                Slot* slotInfo = &slotTable[slot][sslot][page];
                if (slotInfo->type != 0 && slotInfo->callbacks.getState != NULL) {
                    size += slotInfo->callbacks.getState(slotInfo->ref, buffer + size, systemTime);
                }
            }
        }
    }

    for (slot = 0; slot < 16; slot++) {
        Slot* slotInfo = &slotUnslottedTable[slot];
        if (slotInfo->type != 0 && slotInfo->callbacks.getState != NULL) {
            size += slotInfo->callbacks.getState(slotInfo->ref, buffer + size, systemTime);
        }
    }

    return size;
}

int slotSetState(void* ref, UInt8* buffer, UInt32 systemTime)
{
    int size = 0;
    int slot;
    int sslot;
    int page;

    memcpy(&i8255, buffer + size, OFFSETOF(I8255, keymap));
    size += OFFSETOF(I8255, keymap);

    memcpy(pslot, buffer + size, sizeof(pslot));
    size += sizeof(pslot);

    for (slot = 0; slot < 4; slot++) {
        for (sslot = 0; sslot < 4; sslot++) {
            for (page = 0; page < 8; page++) {
                Slot* slotInfo = &slotTable[slot][sslot][page];
                if (slotInfo->type != 0 && slotInfo->callbacks.setState != NULL) {
                    size += slotInfo->callbacks.setState(slotInfo->ref, buffer + size, systemTime);
                }
            }
        }
    }

    for (slot = 0; slot < 16; slot++) {
        Slot* slotInfo = &slotUnslottedTable[slot];
        if (slotInfo->type != 0 && slotInfo->callbacks.setState != NULL) {
            size += slotInfo->callbacks.setState(slotInfo->ref, buffer + size, systemTime);
        }
    }

    for (page = 0; page < 4; page++) {
        int psl = pslot[page].state;
        int ssl = pslot[psl].subslotted ? pslot[page].substate : 0;

        slotMapRamPage(psl, ssl, 2 * page);
        slotMapRamPage(psl, ssl, 2 * page + 1);
    }

    return size;
}


static UInt8 i8255read(void* ref, UInt16 ioPort, UInt32 systemTime)
{
    ioPort &= 3;

    // Read keyboard input
    i8255.inReg[1] = i8255.keymap[i8255.outReg[2] & 0x0f];

    switch(ioPort) {
    case 0: 
        return i8255.reg[3] & 0x10 ? i8255.inReg[0] : i8255.reg[0];
    case 1: 
        return i8255.reg[3] & 0x02? i8255.inReg[1] : i8255.reg[1];
    case 2: 
        return ((i8255.reg[3] & 0x01 ? i8255.inReg[2] : i8255.reg[2]) & 0x0f)|
               ((i8255.reg[3] & 0x08 ? i8255.inReg[2] : i8255.reg[2]) & 0xf0);
    case 3: 
        return i8255.reg[3];
    }

    return 0xff;
}

static void i8255write(void* ref, UInt16 ioPort, UInt8 value, UInt32 systemTime)
{
    int i;

    ioPort &= 3;

    // Update registers and ports
    switch(ioPort) {
    case 0:
    case 1:
    case 2:
        /* Data registers */
        i8255.reg[ioPort] = value;
        break;
    case 3:
        /* Control register */
        if (value & 0x80) {
            i8255.reg[ioPort] = value;
        }
        else {
            ioPort = 1 << ((value & 0x0e) >> 1);
            if (value&0x01) {
                i8255.reg[2] |= ioPort; 
            }
            else {
                i8255.reg[2] &= ~ioPort;
            }
        }
        break;
    }

    /* Set output ports */
    value = i8255.reg[3];
    i8255.outReg[0] = value & 0x10 ? 0x00 : i8255.reg[0];
    i8255.outReg[1] = value & 0x02 ? 0x00 : i8255.reg[1];
    i8255.outReg[2] = ((value & 0x01 ? 0x00 : i8255.reg[2]) & 0x0f) | 
                    ((value & 0x08 ? 0x00 : i8255.reg[2]) & 0xf0);

    // Do post processing
    if (i8255.outReg[2] != i8255.oldOutReg[2]) { 
        if ((i8255.outReg[2] & 0x80) && !(i8255.oldOutReg[2] & 0x80)) {
            audioKeyClick(i8255.keyClick, systemTime);
        }

        SetCapslockLed(0 == (i8255.outReg[2] & 0x40));

        i8255.oldOutReg[2] = i8255.outReg[2]; 
    }

    if (i8255.outReg[0] != i8255.oldOutReg[0]) {
        i8255.oldOutReg[0] = i8255.outReg[0];
        value  = i8255.outReg[0];

        for (i = 0; i < 4; i++) {
            int psl = value & 3;
            int ssl;
            
            pslot[i].state = psl;
            pslot[i].substate = (pslot[psl].sslReg >> (i * 2)) & 3;

            ssl = pslot[psl].subslotted ? pslot[i].substate : 0;

            slotMapRamPage(psl, ssl, 2 * i);
            slotMapRamPage(psl, ssl, 2 * i + 1);

            value >>= 2;
        }
    }
}

void slotMapRamPage(int slot, int sslot, int page)
{
    ramslot[page].readEnable  = slotTable[slot][sslot][page].readEnable;
    ramslot[page].writeEnable = slotTable[slot][sslot][page].writeEnable;
    ramslot[page].pageData    = slotTable[slot][sslot][page].pageData;
}

void slotMapPage(int slot, int sslot, int page, UInt8* pageData, 
                 int readEnable, int writeEnable) 
{
    slotTable[slot][sslot][page].readEnable  = readEnable;
    slotTable[slot][sslot][page].writeEnable = writeEnable;

    if (pageData != NULL) {
        slotTable[slot][sslot][page].pageData = pageData;
    }

    if (pslot[page >> 1].state == slot && 
        (!pslot[slot].subslotted || pslot[page >> 1].substate == sslot)) 
    {
        slotMapRamPage(slot, sslot, page);
    }
}

void slotUpdatePage(int slot, int sslot, int page, UInt8* pageData, 
                    int readEnable, int writeEnable) 
{
    slotTable[slot][sslot][page].readEnable  = readEnable;
    slotTable[slot][sslot][page].writeEnable = writeEnable;

    if (pageData != NULL) {
        slotTable[slot][sslot][page].pageData = pageData;
    }
}

void slotUnmapPage(int slot, int sslot, int page)
{
    slotTable[slot][sslot][page].readEnable  = 0;
    slotTable[slot][sslot][page].writeEnable = 1;
    slotTable[slot][sslot][page].pageData = emptyRAM;

    if (pslot[page >> 1].state == slot && 
        (!pslot[slot].subslotted || pslot[page >> 1].substate == sslot))  
    {
        slotMapRamPage(slot, sslot, page);
    }
}

void slotRegisterWrite0(SlotCallbacks* callbacks, void* ref) {
    memcpy(&slotAddr0.callbacks, callbacks, sizeof(SlotCallbacks));
    slotAddr0.ref = ref;
}

void slotUnregisterWrite0() {
    memset(&slotAddr0, 0, sizeof(Slot));
}

void slotRegister(int type, int slot, int sslot, int startpage, int pages,
                  SlotCallbacks* callbacks, void* ref)
{
    Slot* slotInfo = &slotTable[slot][sslot][startpage];

    slotInfo->type  = type;
    slotInfo->pages = pages;

    while (pages--) {
        slotInfo->startpage = startpage;
        memcpy(&slotInfo->callbacks, callbacks, sizeof(SlotCallbacks));
        slotInfo->ref = ref;
        slotInfo++;
    }
}


void slotUnregister(int slot, int sslot, int startpage)
{
    Slot* slotInfo = &slotTable[slot][sslot][startpage];
    int pages = slotInfo->pages;

    if (slotInfo->type != 0) {
        while (pages--) {
            memset(slotInfo, 0, sizeof(Slot));
            slotUnmapPage(slot, sslot, startpage + pages);
            slotInfo++;
        }
    }
}

void slotRemove(int slot, int sslot, int startpage)
{
    Slot* slotInfo = &slotTable[slot][sslot][startpage];
    if (slotInfo->type != 0 && slotInfo->callbacks.destroy != NULL) {
        slotInfo->callbacks.destroy(slotInfo->ref);
    }
}

int slotRegisterUnslotted(int type, SlotCallbacks* callbacks, void* ref)
{
    int i;

    for (i = 0; i < 15 && slotUnslottedTable[i].type != 0; i++);

    slotUnslottedTable[i].type = type;
    slotUnslottedTable[i].ref  = ref;
    memcpy(&slotUnslottedTable[i].callbacks, callbacks, sizeof(SlotCallbacks));

    return i;
}

void slotUnregisterUnslotted(int slotHandle)
{
    memset(&slotUnslottedTable[slotHandle], 0, sizeof(Slot));
}

void slotSetSubslotted(int slot, int subslotted)
{
    pslot[slot].subslotted = subslotted;
}

void slotManagerCreate(UInt8* keymap, AudioKeyClick* keyClick)
{
    int slot;
    int sslot;
    int page;

    memset(emptyRAM, 0xff, 0x2000);
    memset(ramslot, 0, sizeof(ramslot));
    memset(pslot, 0, sizeof(pslot));
    memset(slotTable, 0, sizeof(slotTable));
    memset(&slotAddr0, 0, sizeof(slotAddr0));
    memset(slotUnslottedTable, 0, sizeof(slotUnslottedTable));
    memset(&i8255, 0, sizeof(i8255));

    i8255.keymap   = keymap;
    i8255.keyClick = keyClick;
    i8255.reg[3]   = 0x9b;

    for (slot = 0; slot < 4; slot++) {
        for (sslot = 0; sslot < 4; sslot++) {
            for (page = 0; page < 8; page++) {
                slotUnmapPage(slot, sslot, page);
            }
        }
    }

    ioPortRegister(0xa8, i8255read, i8255write, NULL);
    ioPortRegister(0xa9, i8255read, i8255write, NULL);
    ioPortRegister(0xaa, i8255read, i8255write, NULL);
    ioPortRegister(0xab, i8255read, i8255write, NULL);

    SetCapslockLed(0);
}

void slotManagerDestroy() 
{
    int slot;
    int sslot;
    int page;

    ioPortUnregister(0xa8);
    ioPortUnregister(0xa9);
    ioPortUnregister(0xaa);
    ioPortUnregister(0xab);

    for (slot = 0; slot < 4; slot++) {
        for (sslot = 0; sslot < 4; sslot++) {
            for (page = 0; page < 8; page++) {
                Slot* slotInfo = &slotTable[slot][sslot][page];
                if (slotInfo->type != 0 && slotInfo->callbacks.destroy != NULL) {
                    slotInfo->callbacks.destroy(slotInfo->ref);
                }
            }
        }
    }

    for (slot = 0; slot < 16; slot++) {
        Slot* slotInfo = &slotUnslottedTable[slot];
        if (slotInfo->type != 0 && slotInfo->callbacks.destroy != NULL) {
            slotInfo->callbacks.destroy(slotInfo->ref);
        }
    }
}

UInt8 slotRead(UInt16 address, UInt32 systemTime)
{
    Slot* slotInfo;
    int psl;
    int ssl;

    if (address == 0xffff) {
        UInt8 sslReg = pslot[3].state;
        if (pslot[sslReg].subslotted) {
            return ~pslot[sslReg].sslReg;
        }
    }

    if (ramslot[address >> 13].readEnable) {
        return ramslot[address >> 13].pageData[address & 0x1fff];
    }

    psl = pslot[address >> 14].state;
    ssl = pslot[psl].subslotted ? pslot[address >> 14].substate : 0;

    slotInfo = &slotTable[psl][ssl][address >> 13];

    if (slotInfo->callbacks.read != NULL) {
        address -= slotInfo->startpage << 13;
        return slotInfo->callbacks.read(slotInfo->ref, address, systemTime);
    }

    return 0xff;
}

void slotWrite(UInt16 address, UInt8 value, UInt32 systemTime)
{
    Slot* slotInfo;
    int psl;
    int ssl;
    int page;

    if (address == 0xffff) {
        UInt8 pslReg = pslot[3].state;

        if (pslot[pslReg].subslotted) {
            pslot[pslReg].sslReg = value;

            for (page = 0; page < 4; page++) {
                if(pslot[page].state == pslReg) {
                    pslot[page].substate = value & 3;
                    slotMapRamPage(pslReg, value & 3, 2 * page);
                    slotMapRamPage(pslReg, value & 3, 2 * page + 1);
                }

                value >>= 2;
            }

            return;
        }
    }

    if (address == 0) {
        if (slotAddr0.callbacks.write != NULL) {
            slotAddr0.callbacks.write(slotAddr0.ref, address, value, systemTime);
            return;
        }
    }

    if (ramslot[address >> 13].writeEnable) {
        ramslot[address >> 13].pageData[address & 0x1FFF] = value;
        return;
    }

    psl = pslot[address >> 14].state;
    ssl = pslot[psl].subslotted ? pslot[address >> 14].substate : 0;

    slotInfo = &slotTable[psl][ssl][address >> 13];

    if (slotInfo->callbacks.write != NULL) {
        address -= slotInfo->startpage << 13;
        slotInfo->callbacks.write(slotInfo->ref, address, value, systemTime);
    }
}

