/*****************************************************************************
** File:        romMapper.c
**
** Author:      Daniel Vik
**
** Description: Rom Mapper
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "romMapper.h"
#include "msxTypes.h"
#include <stdio.h>
#include <string.h>

extern char* getCmdArgument(char* szLine, int argNo);


static char romdbInit[] = 
"## The romdb.dat file contains information about rom mapper types\r\n"
"## Used by blueMSX. The format is one rom per line. The format of\r\n"
"## each line is:\r\n"
"## \r\n"
"## [crc]  [romtype]  [description (optional)]\r\n"
"##\r\n"
"## All letters in the crc and rom type must be lower case.\r\n"
"##\r\n"
"## Rom types supported by the romdb are:\r\n"
"##\r\n"
"##        standard        Standard rom\r\n"
"##        msxdos2         MSX DOS 2 rom\r\n"
"##        konami5         Konami5 rom \r\n"
"##        konami4         Konami4 rom\r\n"
"##        ascii8          ASCII8 rom\r\n"
"##        ascii16         ASCII16rom\r\n"
"##        ascii8sram      ASCII8 rom with SRAM\r\n"
"##        ascii16sram     ASCII16 rom with SRAM\r\n"
"##        koei            Koei rom with SRAM\r\n"
"##        gamemaster2     Game Master 2 rom\r\n"
"##        konami4nf       Konami4 variant (for cracked roms)\r\n"
"##        ascii16nf       ASCII16 variant (for cracked roms)\r\n"
"##        halnote         Halnote rom\r\n"
"##        basic           Basic rom (start at 0x8000)\r\n"
"##        0x4000          Standard rom (start at 0x4000)\r\n"
"##        fmpac           FM-PAC rom\r\n"
"##        rtype           R-TYPE rom\r\n"
"##        loderunner      Lode Runner rom\r\n"
"##        crossblaim      Crossblaim rom\r\n"
"##        harryfox        Harry Fox rom\r\n"
"##        korean80        Korean80 rom\r\n"
"##        korean126       Korean126 rom\r\n"
"##        holyquran       Holy Quran rom\r\n"
"##        scc             SCC rom games (for cracked roms)\r\n"
"##        sccplus         SCC+ rom games (for cracked roms)\r\n"
"##\r\n"
"                                           \r\n"
"CRC         ROM type          Description  \r\n"
"===         ========          ========     \r\n"
"cb935a50    basic             3d Golf      \r\n"
"27405eed    konami5           Space Manbow \r\n"
"68fe9580    ascii16           Pinball Blaster";



static char romdbFilename[512] = "romdb.dat";


static struct {
    UInt32  crc;
    RomType romType;
} romTable[] = {
    /* MSX roms */
    { 0x4e20d256, ROM_STANDARD },
    { 0xa27787af, ROM_ASCII8 },
    { 0xfa8f9bbc, ROM_ASCII8 },
    { 0x2608c959, ROM_KONAMI5 },
    { 0xe4dbcdbd, ROM_STANDARD },
    { 0x96e02df3, ROM_STANDARD },
    { 0x2e1b3dd4, ROM_STANDARD },
    { 0x3ee454b0, ROM_STANDARD },
    { 0x2f72b1e3, ROM_STANDARD },
    { 0xc6fc7bd7, ROM_ASCII16 },
    { 0x93a25be1, ROM_STANDARD },
    { 0x4ef7c4e7, ROM_STANDARD },
    { 0x055254e0, ROM_STANDARD },
    { 0xee8bc1b7, ROM_STANDARD },
    { 0x1dd9b4d9, ROM_STANDARD },
    { 0x7dc880eb, ROM_STANDARD },
    { 0x20143ec7, ROM_STANDARD },
    { 0x15a0f98a, ROM_ASCII8 },
    { 0xaae7028b, ROM_STANDARD },
    { 0xb1ace0a0, ROM_STANDARD },
    { 0xf115257a, ROM_STANDARD },
    { 0xf3297895, ROM_STANDARD },
    { 0xc9a22dfc, ROM_STANDARD },
    { 0xe27f41df, ROM_STANDARD },
    { 0x11bac0e6, ROM_STANDARD },
    { 0xd14361b2, ROM_STANDARD },
    { 0x7b1acdea, ROM_STANDARD },
    { 0x7fdf81e9, ROM_STANDARD },
    { 0xb632b6bb, ROM_STANDARD },
    { 0x70c79a87, ROM_STANDARD },
    { 0xdbdb64ac, ROM_STANDARD },
    { 0x336fdcd9, ROM_STANDARD },
    { 0xcdabd75b, ROM_STANDARD },
    { 0x6aa7cbe0, ROM_STANDARD },
    { 0xd5e18df0, ROM_STANDARD },
    { 0x637f0494, ROM_ASCII8 },
    { 0x42262bd4, ROM_STANDARD },
    { 0xe403ebea, ROM_STANDARD },
    { 0x25e675ea, ROM_STANDARD },
    { 0x85f4767b, ROM_STANDARD },
    { 0x9c9420d0, ROM_STANDARD },
    { 0x1553e408, ROM_STANDARD },
    { 0xa580b72a, ROM_STANDARD },
    { 0x0ad3707d, ROM_STANDARD },
    { 0xf3c1119a, ROM_STANDARD },
    { 0x50475a7c, ROM_STANDARD },
    { 0x8334b431, ROM_STANDARD },
    { 0xec089246, ROM_STANDARD },
    { 0xdd6d9aad, ROM_STANDARD },
    { 0x8cf0e6c0, ROM_ASCII8 },
    { 0x9a5aef05, ROM_STANDARD },
    { 0x042ad44d, ROM_STANDARD },
    { 0x90ccea11, ROM_STANDARD },
    { 0x0a140b27, ROM_STANDARD },
    { 0xe040e8a1, ROM_ASCII16 },
    { 0x02f1997a, ROM_STANDARD },
    { 0x8801b31e, ROM_STANDARD },
    { 0xa20b196d, ROM_STANDARD },
    { 0xe4a7c230, ROM_STANDARD },
    { 0x9cf39bd6, ROM_STANDARD },
    { 0x4b2aa972, ROM_STANDARD },
    { 0xf302aef4, ROM_STANDARD },
    { 0x057d2790, ROM_STANDARD },
    { 0xb6ab6786, ROM_STANDARD },
    { 0xa667ad8a, ROM_STANDARD },
    { 0x8e985857, ROM_STANDARD },
    { 0x5780acfd, ROM_STANDARD },
    { 0x303187f5, ROM_STANDARD },
    { 0x94fc9169, ROM_STANDARD },
    { 0x3f0db564, ROM_STANDARD },
    { 0x5312db08, ROM_STANDARD },
    { 0x338491f5, ROM_STANDARD },
    { 0x90f38029, ROM_STANDARD },
    { 0x04f7e0b1, ROM_STANDARD },
    { 0x0fff4d26, ROM_STANDARD },
    { 0xa90daa22, ROM_STANDARD },
    { 0xa3a56524, ROM_STANDARD },
    { 0xc4b7a5b9, ROM_STANDARD },
    { 0xbf4b018f, ROM_STANDARD },
    { 0x0103c767, ROM_STANDARD },
    { 0x6686c84e, ROM_STANDARD },
    { 0x67ab6f8f, ROM_STANDARD },
    { 0x2f75e79c, ROM_STANDARD },
    { 0xd2b8c058, ROM_STANDARD },
    { 0x1f5eafc8, ROM_STANDARD },
    { 0x5506bf9b, ROM_STANDARD },
    { 0x83b8d8f3, ROM_STANDARD },
    { 0x6031a583, ROM_STANDARD },
    { 0x399fd445, ROM_STANDARD },
    { 0x12552a1f, ROM_STANDARD },
    { 0x40e2f32a, ROM_STANDARD },
    { 0x436c3f29, ROM_STANDARD },
    { 0x98b1cc99, ROM_STANDARD },
    { 0xa4097e41, ROM_STANDARD },
    { 0x9098682d, ROM_STANDARD },
    { 0x11eed1c7, ROM_STANDARD },
    { 0x6936c965, ROM_STANDARD },
    { 0xd9006ebe, ROM_STANDARD },
    { 0x3f534824, ROM_STANDARD },
    { 0xb6d707c7, ROM_STANDARD },
    { 0xd83966d0, ROM_ASCII16 },
    { 0x125a9884, ROM_STANDARD },
    { 0xd8fe84b2, ROM_STANDARD },
    { 0xe167fede, ROM_STANDARD },
    { 0x0b69dd50, ROM_STANDARD },
    { 0x5d284ea6, ROM_STANDARD },
    { 0xfc17c9bc, ROM_STANDARD },
    { 0x99198ed9, ROM_ASCII8 },
    { 0xe04bffe5, ROM_STANDARD },
    { 0x1fc65e80, ROM_STANDARD },
    { 0xcd016b93, ROM_STANDARD },
    { 0x27fd8f9a, ROM_ASCII8SRAM },
    { 0x101db19c, ROM_ASCII8SRAM },
    { 0xce08f27d, ROM_STANDARD },
    { 0x3c749758, ROM_STANDARD },
    { 0x652d0e39, ROM_STANDARD },
    { 0xf8ad9717, ROM_STANDARD },
    { 0x5aa63a76, ROM_STANDARD },
    { 0xa0ff8771, ROM_STANDARD },
    { 0x49f5478b, ROM_ASCII8 },
    { 0x8076fec6, ROM_ASCII8 },
    { 0x6a515349, ROM_STANDARD },
    { 0x87dcd309, ROM_ASCII8 },
    { 0xdb803e8a, ROM_STANDARD },
    { 0x2526e568, ROM_ASCII16 },
    { 0x33700744, ROM_STANDARD },
    { 0x6c366b32, ROM_STANDARD },
    { 0x99b7eab1, ROM_STANDARD },
    { 0x47ec57da, ROM_ASCII16 },
    { 0x5c808e73, ROM_STANDARD },
    { 0x232b1050, ROM_STANDARD },
    { 0x4ff88059, ROM_STANDARD },
    { 0x39886593, ROM_STANDARD },
    { 0x85ac5626, ROM_STANDARD },
    { 0x7abefd3d, ROM_STANDARD },
    { 0x0b6c146f, ROM_STANDARD },
    { 0x6e19c254, ROM_STANDARD },
    { 0xad529df0, ROM_STANDARD },
    { 0x70a0cb2c, ROM_STANDARD },
    { 0x12bfd3a9, ROM_KONAMI5 },
    { 0x64d2df7c, ROM_KONAMI5 },
    { 0x314728c3, ROM_STANDARD },
    { 0x3e96d005, ROM_ASCII8 },
    { 0x41a86301, ROM_STANDARD },
    { 0x851ba4bb, ROM_STANDARD },
    { 0xb120b314, ROM_STANDARD },
    { 0x14e2efcc, ROM_ASCII8 },
    { 0x8005a9ba, ROM_STANDARD },
    { 0xb6285a0b, ROM_STANDARD },
    { 0x4a4f3084, ROM_STANDARD },
    { 0x5c187cf7, ROM_STANDARD },
    { 0x88250e5d, ROM_STANDARD },
    { 0xa00526d0, ROM_STANDARD },
    { 0xa6165bd4, ROM_ASCII8 },
    { 0xfd7de91e, ROM_STANDARD },
    { 0x37b55d09, ROM_STANDARD },
    { 0x97e2fcb4, ROM_STANDARD },
    { 0xb9d03f7b, ROM_STANDARD },
    { 0xeba95a38, ROM_STANDARD },
    { 0xb5c0dace, ROM_STANDARD },
    { 0x89f7fb19, ROM_STANDARD },
    { 0x8856961d, ROM_STANDARD },
    { 0xe223ffd1, ROM_STANDARD },
    { 0x4980ffac, ROM_STANDARD },
    { 0x99ef860e, ROM_ASCII16 },
    { 0x91955bcd, ROM_ASCII8 },
    { 0x004976d3, ROM_STANDARD },
    { 0x4ead5098, ROM_KOEI },       // Genghis Khan
    { 0xc9bcbe5a, ROM_STANDARD },
    { 0xae35e4ad, ROM_STANDARD },
    { 0x4b61ae91, ROM_KONAMI5 },
    { 0x0db7132f, ROM_KONAMI5 },
    { 0x69ecb2ed, ROM_STANDARD },
    { 0xd83a6a3f, ROM_STANDARD },
    { 0x40d3889f, ROM_STANDARD },
    { 0x5eac55df, ROM_ASCII16 },
    { 0x269f079f, ROM_STANDARD },
    { 0xbe071826, ROM_STANDARD },
    { 0xed4a211d, ROM_STANDARD },
    { 0x6498865f, ROM_STANDARD },
    { 0x9dbdd4bc, ROM_STANDARD },
    { 0x4dfcc009, ROM_KONAMI4 },
    { 0x4d44255f, ROM_KONAMI4 },
    { 0xdb847b2b, ROM_KONAMI5 },
    { 0x32aba4e3, ROM_KONAMI5 },
    { 0xc8fac21a, ROM_KONAMI5 },
    { 0x5f74ae0e, ROM_STANDARD },
    { 0x6aebb9d3, ROM_STANDARD },
    { 0xa02029d0, ROM_STANDARD },
    { 0xf877b3d6, ROM_STANDARD },
    { 0x174f409d, ROM_STANDARD },
    { 0x0e89433b, ROM_STANDARD },
    { 0x4d105f57, ROM_STANDARD },
    { 0x6a603b8c, ROM_ASCII8 },
    { 0x5d6655c5, ROM_STANDARD },
    { 0x17b160cb, ROM_STANDARD },
    { 0x48e7212c, ROM_STANDARD },
    { 0x145bb27b, ROM_STANDARD },
    { 0x96b7faca, ROM_ASCII16SRAM },
    { 0xc7fe3ee1, ROM_HARRYFOX }, // ROM_STANDARD
    { 0xf1637c31, ROM_STANDARD },
    { 0x11e46700, ROM_STANDARD },
    { 0xcd63cd50, ROM_STANDARD },
    { 0x8fdad3af, ROM_STANDARD },
    { 0x24851440, ROM_STANDARD },
    { 0x136bba04, ROM_STANDARD },
    { 0x11502a96, ROM_STANDARD },
    { 0x86731751, ROM_STANDARD },
    { 0x5bd60572, ROM_STANDARD },
    { 0x98e7b01b, ROM_STANDARD },
    { 0xacfcbba6, ROM_STANDARD },
    { 0x99518a12, ROM_STANDARD },
    { 0x0f9d6f56, ROM_STANDARD },
    { 0xd8055f5f, ROM_ASCII16SRAM },
    { 0x00c5d5b5, ROM_ASCII8 },
    { 0xfef1d8fa, ROM_STANDARD },
    { 0x38cb690b, ROM_STANDARD },
    { 0xf94d452e, ROM_STANDARD },
    { 0x18db4ff2, ROM_STANDARD },
    { 0x968fa8d6, ROM_STANDARD },
    { 0x80a831e1, ROM_STANDARD },
    { 0x16e7b4be, ROM_STANDARD },
    { 0x51727e48, ROM_STANDARD },
    { 0x4aa97644, ROM_STANDARD },
    { 0x3a550788, ROM_STANDARD },
    { 0xa7c43855, ROM_STANDARD },
    { 0xd6465702, ROM_ASCII16 },
    { 0x61ea6742, ROM_STANDARD },
    { 0x9191c890, ROM_STANDARD },
    { 0xf278c6f3, ROM_STANDARD },
    { 0xe8df4aa8, ROM_STANDARD },
    { 0x57e1221c, ROM_STANDARD },
    { 0x0599617b, ROM_STANDARD },
    { 0xc15a25da, ROM_STANDARD },
    { 0xd902be7c, ROM_STANDARD },
    { 0x6567f14d, ROM_STANDARD },
    { 0x7382f3ac, ROM_STANDARD },
    { 0xc8ff3829, ROM_STANDARD },
    { 0x69367e10, ROM_STANDARD },
    { 0x6a30c707, ROM_ASCII8 },
    { 0x7ff117f9, ROM_STANDARD },
    { 0xe4f63cb7, ROM_STANDARD },
    { 0x2aba2253, ROM_STANDARD },
    { 0x1eeba987, ROM_STANDARD },
    { 0xa0481321, ROM_ASCII16 },
    { 0x5a141c44, ROM_STANDARD },
    { 0xdfac2125, ROM_STANDARD },
    { 0xe82a8e8e, ROM_KONAMI5 },
    { 0x166781b7, ROM_STANDARD },
    { 0xb575c44a, ROM_STANDARD },
    { 0xa3b2fe71, ROM_ASCII16 },
    { 0xb660494b, ROM_STANDARD },
    { 0x3a442408, ROM_STANDARD },
    { 0x08c7e406, ROM_STANDARD },
    { 0x0e3380fe, ROM_STANDARD },
    { 0x114198e5, ROM_STANDARD },
    { 0x3f200491, ROM_STANDARD },
    { 0xbbcdd731, ROM_STANDARD },
    { 0x58ea53b9, ROM_STANDARD },
    { 0x74bc8295, ROM_STANDARD },
    { 0x61726be4, ROM_STANDARD },
    { 0xcfce0a28, ROM_STANDARD },
    { 0x5a0b8739, ROM_STANDARD },
    { 0x08f23b3e, ROM_STANDARD },
    { 0xc1aaf8cb, ROM_STANDARD },
    { 0x091fe438, ROM_STANDARD },
    { 0xadc58732, ROM_STANDARD },
    { 0x32ab8b20, ROM_STANDARD },
    { 0xc1679aeb, ROM_STANDARD },
    { 0xd8a116d8, ROM_STANDARD },
    { 0x6e742024, ROM_STANDARD },
    { 0x8c2f2c59, ROM_STANDARD },
    { 0x827038a2, ROM_STANDARD },
    { 0xd446ba1e, ROM_ASCII16 },
    { 0x0db84205, ROM_STANDARD },
    { 0xb7002f08, ROM_STANDARD },
    { 0xa97a19ec, ROM_STANDARD },
    { 0xa0358870, ROM_STANDARD },
    { 0x3b702e9a, ROM_STANDARD },
    { 0x44aa5422, ROM_ASCII8 },
    { 0xd34d74f7, ROM_STANDARD },
    { 0x25fc11fa, ROM_ASCII8 },
    { 0xcf744d2e, ROM_STANDARD },
    { 0x2c2020a0, ROM_ASCII16 },
    { 0xce2882f6, ROM_STANDARD },
    { 0x7b5282e4, ROM_STANDARD },
    { 0x855ca028, ROM_STANDARD },
    { 0x72eb9d0e, ROM_ASCII8 },
    { 0x4faccae0, ROM_ASCII8 },
    { 0x0427df83, ROM_STANDARD },
    { 0xeba19259, ROM_STANDARD },
    { 0x1d27d31f, ROM_STANDARD },
    { 0x309f886b, ROM_STANDARD },
    { 0xe73c12e6, ROM_STANDARD },
    { 0x98cd6912, ROM_STANDARD },
    { 0x3ddcac18, ROM_STANDARD },
    { 0x1ba6e461, ROM_STANDARD },
    { 0xfebb1155, ROM_STANDARD },
    { 0x9d73e4c5, ROM_STANDARD },
    { 0x097f4e4a, ROM_STANDARD },
    { 0xa94d7d08, ROM_STANDARD },
    { 0xd40f481d, ROM_STANDARD },
    { 0x06a64527, ROM_STANDARD },
    { 0x414d29bc, ROM_STANDARD },
    { 0x4098d71d, ROM_STANDARD },
    { 0x09090135, ROM_STANDARD },
    { 0x6786a7ee, ROM_STANDARD },
    { 0xf928f075, ROM_STANDARD },
    { 0xf79d3088, ROM_STANDARD },
    { 0x0cb78f0e, ROM_STANDARD },
    { 0x61b33748, ROM_STANDARD },
    { 0x18ad3380, ROM_STANDARD },
    { 0x6824e45d, ROM_STANDARD },
    { 0xf31d13b9, ROM_STANDARD },
    { 0x519be53e, ROM_ASCII8 },
    { 0x53c25b6b, ROM_ASCII8 },
    { 0x790d5335, ROM_ASCII8 },
    { 0x2afbf7d1, ROM_STANDARD },
    { 0x72ddb449, ROM_STANDARD },
    { 0xf06a58da, ROM_STANDARD },
    { 0x2a28ff97, ROM_STANDARD },
    { 0xb202f481, ROM_STANDARD },
    { 0xd388cfd1, ROM_STANDARD },
    { 0xef339b82, ROM_STANDARD },
    { 0x203ef741, ROM_STANDARD },
    { 0x3aa33a30, ROM_ASCII8SRAM },
    { 0x02ae3eda, ROM_STANDARD },
    { 0x6d81d863, ROM_STANDARD },
    { 0x3c7f3767, ROM_STANDARD },
    { 0xb05ffed2, ROM_STANDARD },
    { 0xfd3921cd, ROM_STANDARD },
    { 0x01a24ca7, ROM_STANDARD },
    { 0xce7f6c91, ROM_STANDARD },
    { 0x4088efaa, ROM_STANDARD },
    { 0x505acae7, ROM_STANDARD },
    { 0xd74dffa2, ROM_STANDARD },
    { 0x6fd70773, ROM_STANDARD },
    { 0x33b9968c, ROM_STANDARD },
    { 0x9bb308f5, ROM_KONAMI5 },
    { 0xd6d8d1d7, ROM_STANDARD },
    { 0xd15b0b1a, ROM_STANDARD },
    { 0x941069ae, ROM_STANDARD },
    { 0xc23fa0d5, ROM_STANDARD },
    { 0x6d2b3e0c, ROM_STANDARD },
    { 0x9e51ae0a, ROM_STANDARD },
    { 0x7411c6f2, ROM_STANDARD },
    { 0x2eff15e7, ROM_STANDARD },
    { 0x6f172cd8, ROM_STANDARD },
    { 0xbe6a5e19, ROM_STANDARD },
    { 0x5a009c55, ROM_STANDARD },
    { 0xd307a7b8, ROM_STANDARD },
    { 0xd178833b, ROM_STANDARD },
    { 0x9d59f2d1, ROM_STANDARD },
    { 0x558a09f6, ROM_STANDARD },
    { 0xbd387377, ROM_STANDARD },
    { 0x8ca4cd58, ROM_STANDARD },
    { 0xf1ec5e18, ROM_STANDARD },
    { 0xb8fc19a4, ROM_ASCII8SRAM },
    { 0x1313c0c7, ROM_STANDARD },
    { 0x0aec8ddb, ROM_STANDARD },
    { 0x0e988f0e, ROM_STANDARD },
    { 0x27b78cb3, ROM_STANDARD },
    { 0x40eef5cd, ROM_STANDARD },
    { 0xa884911c, ROM_RTYPE },
    { 0x71e94fce, ROM_RTYPE }, 
    { 0x827919e4, ROM_RTYPE }, 
    { 0xa87a666d, ROM_STANDARD },
    { 0x9fd2f1dc, ROM_STANDARD },
    { 0x2b5cb04e, ROM_STANDARD },
    { 0x63413493, ROM_STANDARD },
    { 0x6a2e3726, ROM_STANDARD },
    { 0xdba4377b, ROM_STANDARD },
    { 0x25fe441c, ROM_STANDARD },
    { 0xe9b5b6ff, ROM_STANDARD },
    { 0xb612d79a, ROM_ASCII8 },
    { 0x6a6d37cf, ROM_STANDARD },
    { 0x0f22a553, ROM_STANDARD },
    { 0x01ddb68f, ROM_STANDARD },
    { 0x788637d5, ROM_ASCII16 },
    { 0x430e5789, ROM_STANDARD },
    { 0x56200fef, ROM_STANDARD },
    { 0x387c1de7, ROM_ASCII16 },
    { 0xc36f559d, ROM_KONAMI5 },
    { 0x803cc690, ROM_ASCII8 },
    { 0x10621440, ROM_ASCII8 },
    { 0x7faf00c0, ROM_STANDARD },
    { 0xc2ed4c08, ROM_STANDARD },
    { 0xba3a8ea1, ROM_STANDARD },
    { 0x02dc77e8, ROM_STANDARD },
    { 0x999dd794, ROM_STANDARD },
    { 0x126bc4cd, ROM_STANDARD },
    { 0x427b3f14, ROM_ASCII8 },
    { 0x925c0aee, ROM_STANDARD },
    { 0xcf60fa7d, ROM_KONAMI4 },
    { 0xa44495d5, ROM_STANDARD },
    { 0xece16709, ROM_STANDARD },
    { 0xb90dee1d, ROM_STANDARD },
    { 0xead8e29c, ROM_STANDARD },
    { 0x729d2540, ROM_STANDARD },
    { 0x8273fd0e, ROM_STANDARD },
    { 0xd8750242, ROM_STANDARD },
    { 0x53b87deb, ROM_STANDARD },
    { 0x54f84047, ROM_STANDARD },
    { 0xe4f725fd, ROM_STANDARD },
    { 0x4a44bf23, ROM_STANDARD },
    { 0x14811951, ROM_ASCII8 },
    { 0x030ddf11, ROM_STANDARD },
    { 0xeb197b9d, ROM_STANDARD },
    { 0xde02932d, ROM_STANDARD },
    { 0x4a45cbc0, ROM_STANDARD },
    { 0x26119f0a, ROM_STANDARD },
    { 0x3e14b585, ROM_STANDARD },
    { 0xdc948a3a, ROM_STANDARD },
    { 0xdd5cf5c8, ROM_STANDARD },
    { 0xf3b22c91, ROM_STANDARD },
    { 0xfd902e5d, ROM_STANDARD },
    { 0xc14e53a1, ROM_STANDARD },
    { 0x0b3d975d, ROM_STANDARD },
    { 0xdcde05dd, ROM_STANDARD },
    { 0x236d2ab1, ROM_STANDARD },
    { 0x60a8927b, ROM_STANDARD },
    { 0xcabcddf1, ROM_STANDARD },
    { 0x0fb7df8f, ROM_STANDARD },
    { 0x6554c0c5, ROM_STANDARD },
    { 0x97425d70, ROM_STANDARD },
    { 0x3254017e, ROM_STANDARD },
    { 0x66aadfa8, ROM_STANDARD },
    { 0x5dc45624, ROM_ASCII8 },
    { 0x8aebddd2, ROM_STANDARD },
    { 0x491c6af0, ROM_STANDARD },
    { 0x6408ed24, ROM_STANDARD },
    { 0xd560d9c0, ROM_STANDARD },
    { 0x58670a22, ROM_STANDARD },
    { 0xc93fadf4, ROM_STANDARD },
    { 0x0e691c47, ROM_ASCII8SRAM },
    { 0x892266ca, ROM_STANDARD },
    { 0xc57272c3, ROM_STANDARD },
    { 0xf48a0c3b, ROM_STANDARD },
    { 0x7914f7a6, ROM_STANDARD },
    { 0x589ce03e, ROM_STANDARD },
    { 0x867ba302, ROM_STANDARD },
    { 0x377fefef, ROM_STANDARD },
    { 0xb7322600, ROM_STANDARD },
    { 0xc58fbb99, ROM_STANDARD },
    { 0xb6c02ae7, ROM_STANDARD },
    { 0x929d6786, ROM_STANDARD },
    { 0xe258c032, ROM_STANDARD },
    { 0x67bf8337, ROM_ASCII16 },
    { 0x2149c32d, ROM_STANDARD },
    { 0x6fde5bca, ROM_STANDARD },
    { 0x11de4dfd, ROM_STANDARD },
    { 0x26e34f05, ROM_ASCII8 },
    { 0x309d996c, ROM_ASCII8 },
    { 0xdb327847, ROM_STANDARD },
    { 0x04e454c5, ROM_STANDARD },
    { 0xfe23d253, ROM_KONAMI4 },
    { 0x7867d044, ROM_STANDARD },
    { 0x5747e69d, ROM_STANDARD },
    { 0x8c7a7435, ROM_STANDARD },
    { 0x47bef309, ROM_STANDARD },
    { 0x80495007, ROM_STANDARD },
    { 0x599aa9ac, ROM_STANDARD },
    { 0x86f902a9, ROM_STANDARD },
    { 0x6b63047c, ROM_STANDARD },
    { 0xd8739501, ROM_STANDARD },
    { 0xe4abe008, ROM_STANDARD },
    { 0x23152fc3, ROM_STANDARD },
    { 0xca28cbb2, ROM_STANDARD },
    { 0x190f4ce5, ROM_STANDARD },
    { 0xb80ebeb4, ROM_STANDARD },
    { 0xc0a81a48, ROM_STANDARD },
    { 0xdfada43c, ROM_STANDARD },
    { 0xb9b69a43, ROM_ASCII16 },
    { 0xc7abc409, ROM_STANDARD },
    { 0x17683f42, ROM_STANDARD },
    { 0xc730d4a4, ROM_STANDARD },
    { 0xd792c827, ROM_STANDARD },
    { 0x973d816d, ROM_STANDARD },
    { 0xfea70207, ROM_ASCII16 },
    { 0xee0b8452, ROM_STANDARD },
    { 0x64853262, ROM_STANDARD },
    { 0x874e59fb, ROM_STANDARD },
    { 0x90f5f414, ROM_STANDARD },
    { 0xa0a19fd5, ROM_STANDARD },
    { 0x5c9d8f62, ROM_ASCII8 },
    { 0xd640deaf, ROM_ASCII8SRAM },
    { 0xc7303910, ROM_STANDARD },
    { 0x24b94274, ROM_STANDARD },
    { 0x71e6605f, ROM_STANDARD },
    { 0x8a12ec4f, ROM_STANDARD },
    { 0xd66034a9, ROM_STANDARD },
    { 0x8099f5c7, ROM_STANDARD },
    { 0x8d636963, ROM_STANDARD },
    { 0x2faf6e26, ROM_STANDARD },
    { 0xbdf332f2, ROM_ASCII8 },
    { 0x0f6418d3, ROM_KONAMI4 },
    { 0xbeccf133, ROM_STANDARD },
    { 0x425e0d34, ROM_STANDARD },
    { 0x0b3ba595, ROM_STANDARD },
    { 0x1a4aebb2, ROM_STANDARD },
    { 0xefefa02a, ROM_STANDARD },
    { 0x64283863, ROM_STANDARD },
    { 0xee0a750e, ROM_STANDARD },
    { 0x13014324, ROM_STANDARD },
    { 0x325bb241, ROM_STANDARD },
    { 0x097e4a7e, ROM_STANDARD },
    { 0x50bb1930, ROM_STANDARD },
    { 0xf7848700, ROM_STANDARD },
    { 0x3755ca3d, ROM_STANDARD },
    { 0x2de03477, ROM_STANDARD },
    { 0x4a4a6148, ROM_STANDARD },
    { 0xfd8a5c1d, ROM_STANDARD },
    { 0x6f2e48ab, ROM_STANDARD },
    { 0x8c13ccea, ROM_STANDARD },
    { 0x2cbfe03e, ROM_STANDARD },
    { 0xfad0860b, ROM_STANDARD },
    { 0x0e84505d, ROM_KOREAN126 },
//    { 0x0e84505d, ROM_FMPAC },
    { 0x70b36f17, ROM_STANDARD },
    { 0x906c24ae, ROM_STANDARD },
    { 0xef02e4f3, ROM_ASCII16 },
    { 0x46e90b77, ROM_STANDARD },
    { 0x6f9deb1b, ROM_STANDARD },
    { 0x3b8fdf44, ROM_STANDARD },
    { 0x8f1a102e, ROM_STANDARD },
    { 0xbd1c5a59, ROM_STANDARD },
    { 0x084c5803, ROM_GAMEMASTER2 },
    { 0xb9b0999a, ROM_STANDARD },
    { 0x91daf424, ROM_STANDARD },
    { 0xa7c4ac70, ROM_STANDARD },
    { 0x90e893ba, ROM_STANDARD },
    { 0xdc405f8c, ROM_STANDARD },
    { 0x0ea62a4d, ROM_STANDARD },
    { 0x381b3431, ROM_STANDARD },
    { 0x1b10cf79, ROM_ASCII8 },
    { 0x74cf041b, ROM_STANDARD },
    { 0x4583548c, ROM_STANDARD },
    { 0xba215cfe, ROM_STANDARD },
    { 0x85d47f39, ROM_ASCII8 },
    { 0x87b72f40, ROM_STANDARD },
    { 0x351b6deb, ROM_STANDARD },
    { 0xc903a4ec, ROM_STANDARD },
    { 0x309ba37a, ROM_STANDARD },
    { 0x77c3424c, ROM_STANDARD },
    { 0x2d46778a, ROM_STANDARD },
    { 0xc7b2e638, ROM_STANDARD },
    { 0x0dd5ea7d, ROM_STANDARD },
    { 0x168832a2, ROM_STANDARD },
    { 0xdf958ed5, ROM_STANDARD },
    { 0x24783377, ROM_STANDARD },
    { 0x7092ca0c, ROM_STANDARD },
    { 0x74d6a1b1, ROM_STANDARD },
    { 0xf3606c66, ROM_STANDARD },
    { 0x353dd532, ROM_STANDARD },
    { 0xb2d806d9, ROM_STANDARD },
    { 0xcfb61e99, ROM_STANDARD },
    { 0x62b1f1df, ROM_KONAMI4 },
    { 0x9728023b, ROM_ASCII16 },

    /* MSX-II roms */
    { 0xc99e5601, ROM_ASCII8 },
    { 0xf5174422, ROM_ASCII16 },
    { 0xc956e1df, ROM_ASCII16SRAM },
    { 0xd2ee4051, ROM_ASCII16 },
    { 0x0aedea40, ROM_KONAMI4 },
    { 0x5953d084, ROM_KONAMI4 },
    { 0x038c250d, ROM_KONAMI4 },
    { 0x217935d0, ROM_KONAMI4 },
    { 0xc29edd84, ROM_ASCII16 },
    { 0xbcf03d8b, ROM_ASCII8 },
    { 0xc6f74489, ROM_ASCII16 },
    { 0xae3167ee, ROM_ASCII16 },
    { 0x00d36af6, ROM_ASCII8 },
    { 0xbfe150c4, ROM_ASCII8 },
    { 0xc4096e6b, ROM_ASCII8 },
    { 0x225705cc, ROM_ASCII8 },
    { 0x3cb34efb, ROM_ASCII8 },
    { 0x47a82d74, ROM_ASCII8 },
    { 0x353382ce, ROM_ASCII8 },
    { 0x6524d9fc, ROM_ASCII8 },
    { 0x8528a795, ROM_ASCII8 },
    { 0xec7a8012, ROM_KOEI },       // Daikoukai Jidai
    { 0xa987a0d8, ROM_ASCII16SRAM },
    { 0x7098faf0, ROM_ASCII8 },
    { 0xf221b845, ROM_ASCII8 },
    { 0x5acbd591, ROM_ASCII8SRAM },
    { 0xe1148738, ROM_ASCII8 },
    { 0x49252882, ROM_ASCII8 },
    { 0x7729e7a9, ROM_ASCII8 },
    { 0x0bf0c3f5, ROM_ASCII8 },
    { 0x98095852, ROM_ASCII8 },
    { 0x94982598, ROM_ASCII8 },
    { 0x9935e9bd, ROM_ASCII16 },
    { 0x14913432, ROM_ASCII8 },
    { 0xa8e9ac9b, ROM_ASCII8SRAM },
    { 0x413d4700, ROM_KOEI },       // Europe War
    { 0x53cce88d, ROM_ASCII8 },
    { 0x0e03847b, ROM_ASCII8 },
    { 0x7c867d98, ROM_ASCII8 },
    { 0x7030f2ab, ROM_ASCII8 },
    { 0x356b7d2a, ROM_ASCII16 },
    { 0x1bdfaeeb, ROM_ASCII16 },
    { 0xcdcdc6c0, ROM_ASCII8 },
    { 0x9f152e9b, ROM_ASCII8 },
    { 0xfbdafe4e, ROM_ASCII16 },
    { 0x91955bcd, ROM_ASCII8 },
    { 0xeb67d4d6, ROM_ASCII8 },
    { 0x40e53ea3, ROM_KONAMI4 },
    { 0x2fb0ed30, ROM_ASCII8 },
    { 0xb0a7800f, ROM_KONAMI5 },
    { 0xc08faf0e, ROM_KONAMI5 },
    { 0xc44c6330, ROM_KONAMI5 },
    { 0x075ab646, ROM_KOEI },       // Genchohisi
    { 0x3b5b3fd8, ROM_KOEI },       // Genghis Khan
    { 0x28094960, ROM_ASCII16 },
    { 0x4b61ae91, ROM_KONAMI5 },
    { 0x0db7132f, ROM_KONAMI5 },
    { 0x8df56d42, ROM_ASCII16 },
    { 0xe0e72736, ROM_KOREAN80 },
    { 0x1d7040af, ROM_ASCII8 },
    { 0x6ac18445, ROM_ASCII8SRAM },
    { 0x1123545c, ROM_ASCII8 },
    { 0x3ab5ae57, ROM_KONAMI4 },
    { 0xd95116b0, ROM_KONAMI4 },
    { 0xf35df283, ROM_ASCII16 },
    { 0x4708bdd1, ROM_ASCII16 },
    { 0x00db86c8, ROM_ASCII8 },
    { 0x935f1bde, ROM_ASCII8 },
    { 0xa9a66075, ROM_ASCII16 },
    { 0x881b3849, ROM_KOEI },       // Inindo Tado Nobunaga
    { 0x8bc34dca, ROM_KOEI },       // Isin No Arashi
    { 0x2c2924ed, ROM_ASCII16 },
    { 0x00b8e82e, ROM_ASCII16 },
    { 0xc8ff3829, ROM_STANDARD },
    { 0x053aa66e, ROM_ASCII16 },
    { 0x5c859406, ROM_ASCII8 },
    { 0x23b78a2d, ROM_KONAMI4 },
    { 0xb1cb1052, ROM_KONAMI5 },
    { 0x774c70fc, ROM_KONAMI5 },
    { 0x167a53da, ROM_ASCII8SRAM },
    { 0x0767af40, ROM_ASCII8 },
    { 0x4e82660d, ROM_KONAMI5 },
    { 0xfb348de6, ROM_ASCII16 },
    { 0xdcb1a61e, ROM_ASCII8 },
    { 0x9926f43c, ROM_ASCII8 },
    { 0xc7521809, ROM_KOEI },       // Lempereur
    { 0xf87bd172, ROM_ASCII8 },
    { 0xf4d127c0, ROM_ASCII8 },
    { 0x94191aae, ROM_ASCII8 },
    { 0x17baf3ed, ROM_ASCII16 },
    { 0xf7a9591d, ROM_ASCII16 },
    { 0xfafe1303, ROM_KONAMI4 },
    { 0x18b1f34b, ROM_KONAMI5 },
    { 0xdcee5080, ROM_KONAMI5 },
    { 0x70aa93cf, ROM_ASCII8 },
    { 0x3f69a464, ROM_ASCII8 },
    { 0x68745c64, ROM_ASCII8 },
    { 0xebccb796, ROM_ASCII8 },
    { 0x53c25b6b, ROM_ASCII8 },
    { 0x790d5335, ROM_ASCII8 },
    { 0x758c6087, ROM_ASCII8 },
    { 0xf56a00e6, ROM_ASCII8 },
    { 0x0ff8e2c7, ROM_KOEI },       // Nobunaga No Yabou - Bushouhuunroku
    { 0x69c11d81, ROM_KOEI },       // Nobunaga No Yabou - Senkokugunyuden
    { 0x653f1c73, ROM_KOEI },       // Nobunaga No Yabou - Zenkokuhan
    { 0xf2544e2e, ROM_ASCII16 },
    { 0x0a9f8d6c, ROM_ASCII8 },
    { 0x7df16fc8, ROM_ASCII8 },
    { 0x252798f6, ROM_ASCII8 },
    { 0xb1b24da1, ROM_ASCII16 },
    { 0xe85a4a6b, ROM_ASCII8 },
    { 0xc065bc29, ROM_ASCII8 },
    { 0x2abb1ec4, ROM_ASCII16SRAM },
    { 0x1b285fe3, ROM_ASCII8 },
    { 0xa0014e7f, ROM_KONAMI5 },
    { 0xc4ccb75f, ROM_KONAMI5 },
    { 0x8d94af7d, ROM_ASCII8 },
    { 0x84461e73, ROM_ASCII8 },
    { 0xbabc68f2, ROM_ASCII8 },
    { 0x6eee12c4, ROM_ASCII8 },
    { 0x5ccc09ab, ROM_ASCII8 },
    { 0xe79aa962, ROM_ASCII8 },
    { 0x17f9ee7f, ROM_ASCII16 },
    { 0xfb9e42e1, ROM_KOEI },       // Royal Blood
    { 0xb83aa6f5, ROM_KOEI },       // Sangokusi 1
    { 0xc89b93b5, ROM_KOEI },       // Sangokusi 2
    { 0xfe797d88, ROM_ASCII8 },
    { 0x226795e0, ROM_ASCII8 },
    { 0xe37b681e, ROM_ASCII8 },
    { 0x3b815dc6, ROM_ASCII16 },
    { 0xf640670c, ROM_ASCII8SRAM },
    { 0x39e1c37a, ROM_ASCII8SRAM },
    { 0x27405eed, ROM_KONAMI5 },
    { 0xcbdfcc35, ROM_ASCII8 },
    { 0x4a3abb65, ROM_ASCII16 },
    { 0x187d3a21, ROM_KOEI },       // Suikoden
    { 0x8df9945a, ROM_ASCII16SRAM },
    { 0xb2392ef3, ROM_ASCII16 },
    { 0x5a5be701, ROM_ASCII8 },
    { 0xcbb00ef1, ROM_ASCII8 },
    { 0xae82a874, ROM_ASCII8 },
    { 0xc31f459c, ROM_KOEI },       // Teitoku No Ketsudan
    { 0x7e637801, ROM_ASCII8 },
    { 0xaa598dfd, ROM_ASCII8 },
    { 0x5f354e65, ROM_ASCII16 },
    { 0x8ee8212c, ROM_ASCII16 },
    { 0x96b33497, ROM_ASCII8 },
    { 0xff7395a3, ROM_ASCII8 },
    { 0x24238b21, ROM_ASCII8 },
    { 0x37b9e10f, ROM_ASCII8SRAM },
    { 0xe564a25f, ROM_ASCII8 },
    { 0x1f1f1f26, ROM_KONAMI4 },
    { 0x30396650, ROM_KONAMI4 },
    { 0x94ebc319, ROM_ASCII16 },
    { 0x08a5ca60, ROM_ASCII8SRAM },
    { 0xaea8d064, ROM_ASCII16 },
    { 0x97591725, ROM_ASCII16 },
    { 0xff974ee3, ROM_ASCII8 },
    { 0x712ae3c6, ROM_ASCII8 },
    { 0xd153b2f2, ROM_ASCII16 },
    { 0x2605031e, ROM_ASCII16 },
    { 0xa29e6ab4, ROM_ASCII8 },
    { 0xd0ececc6, ROM_ASCII8 },
    { 0x4968e84e, ROM_ASCII8 },
    { 0x2ffb49c3, ROM_STANDARD },
    { 0x33c37313, ROM_ASCII16 },
    { 0x40313fec, ROM_HALNOTE }, // Halnote
    { 0xef02e4f3, ROM_ASCII16 },
    { 0x459478cb, ROM_ASCII8SRAM },
    { 0x3b8fdf44, ROM_STANDARD },
    { 0x17e58926, ROM_STANDARD },
    { 0x084c5803, ROM_GAMEMASTER2 },
    { 0x90e893ba, ROM_STANDARD },
    { 0xa13615d2, ROM_STANDARD },
    { 0x7122e8e3, ROM_ASCII8 },
    { 0xb35b30f4, ROM_STANDARD },
    { 0xce3b09c0, ROM_STANDARD },
    { 0x2ec98b57, ROM_STANDARD },
    { 0x5899d0ae, ROM_STANDARD }, 
    { 0x1c430991, ROM_MSXDOS2 },
    { 0x845f5ab6, ROM_MSXDOS2 },
    { 0xf4045aa8, ROM_RTYPE },
    { 0x47273220, ROM_CROSSBLAIM },
    { 0x3459ce01, ROM_ASCII16 },
    { 0x8b73f133, ROM_ASCII8 },
    { 0xe012ee7a, ROM_ASCII8 },
    { 0x21489e1d, ROM_ASCII16 },
    { 0xfea70207, ROM_ASCII16 },
    { 0x2f8065eb, ROM_ASCII16 },

    /* Misc roms */
    { 0x089c8164, ROM_ASCII16 }, // zanac-ex
    { 0xe8c06841, ROM_ASCII8 },  // American soccer
    { 0xeb74e4ad, ROM_KONAMI4NF }, // Animal Wars 2
    { 0x25a3c30a, ROM_KONAMI4NF }, // Ashguine 3
    { 0xd982df38, ROM_KONAMI4NF }, // Ashguine T&E Version
    { 0xef43664a, ROM_KONAMI4NF }, // MWorld
    { 0xf00b4bbc, ROM_ASCII16NF }, // Super Pierrot
    { 0x872b7752, ROM_ASCII16 },   // Eggland (modified)
    { 0xeaf2bfcd, ROM_ASCII16 },   // Androgynus
    { 0xa4996aaa, ROM_ASCII16 },   // Dynamite Bowl
    { 0x7de493ab, ROM_ASCII16 },   // Mask III (Not working)
    { 0xb19e88aa, ROM_ASCII16 },   // Ogre
    { 0x5c4f98d9, ROM_KONAMI5 },
    { 0x41ec9928, ROM_STANDARD },    // War of the Dead
    { 0xd91b4f39, ROM_STANDARD },    // War of the Dead
    { 0xec036e37, ROM_ASCII16 },     // Gall Force
    { 0xf65b5271, ROM_ASCII16 },     // Gall Force
    { 0xc13004c5, ROM_ASCII16 },    // Game World
    { 0xafa0ff77, ROM_ASCII16 },    // Golvellius
    { 0xdbf2f244, ROM_ASCII16 },    // Jagur
    { 0x1d1ec602, ROM_ASCII16 },    // Eggland 2
    { 0x5a090675, ROM_ASCII8 },     // Arsenelupin
    { 0x40d009fe, ROM_ASCII8 },     // Dragon King
    { 0x2023178a, ROM_KONAMI5 },    // Metal Gear 2
    { 0x816d3ce0, ROM_KONAMI4 },    // Shalome
    { 0x5640e6e2, ROM_KONAMI4NF },  // Donkey Kong
    { 0x83f28ad3, ROM_HOLYQURAN },
    { 0xcb935a50, ROM_BASIC }, // 3D golf
    { 0x0c989ad1, ROM_BASIC }, // 3D golf
    { 0xf83d4b0d, ROM_BASIC }, // 3D golf
    { 0xfb00f21b, ROM_BASIC }, // 3D tennis
    { 0x70113f76, ROM_BASIC }, // Break Out
    { 0x74af0726, ROM_BASIC }, // Comet Tail
    { 0xd0e8f418, ROM_BASIC }, // Crazy Bullet
    { 0x571f12fb, ROM_BASIC }, // Danger X4
    { 0x22f5e82a, ROM_BASIC }, // Danger X4
    { 0x2c2b8a0e, ROM_BASIC }, // Exchanger
    { 0xc3ee7e15, ROM_BASIC }, // Gang Master
    { 0x5d88275f, ROM_BASIC }, // Golf Game
    { 0x36a08e0b, ROM_BASIC }, // Iriegas
    { 0x12be29fc, ROM_BASIC }, // Karamaru
    { 0xc6064ce0, ROM_BASIC }, // Ladder Building
    { 0xab5e42fe, ROM_BASIC }, // Lord Over
    { 0x24923bdb, ROM_BASIC }, // Marine battle
    { 0xd0e39c9b, ROM_BASIC }, // Mole
    { 0x8a2c11da, ROM_BASIC }, // Moon Landing
    { 0x268cc301, ROM_BASIC }, // MSX Derby
    { 0x13b8243c, ROM_BASIC }, // MSX21
    { 0xe84994f7, ROM_BASIC }, // Nausiscaa
    { 0x9c1c10ca, ROM_BASIC }, // Paris
    { 0x3f042540, ROM_BASIC }, // Paris
    { 0x82b8f501, ROM_BASIC }, // Pass ball
    { 0x40df2723, ROM_BASIC }, // Renju Ojama Dogs
    { 0x1cdb462e, ROM_BASIC }, // Rotors
    { 0x5b33e583, ROM_BASIC }, // Scope On
    { 0x4579ce12, ROM_BASIC }, // Star Command
    { 0x79bc12b2, ROM_BASIC }, // Super Billiard
    { 0xbb402b47, ROM_BASIC }, // The Brain
    { 0x152bebc2, ROM_BASIC }, // Trial Ski
    { 0x21a0b009, ROM_BASIC }, // Warrior
    { 0x5c7cf05b, ROM_BASIC }, // Ligue-se
    { 0xe74b7943, ROM_BASIC }, // Pasokon Sakkyokuka
    { 0x4651ffd0, ROM_BASIC }, // Crazy bullet
    { 0xab6cd62c, ROM_ASCII16 }, // King Night
    { 0x4d884352, ROM_ASCII16 }, // King Night
    { 0xa330b549, ROM_ASCII16 }, // Light Corridor
    { 0x2a8bbb4d, ROM_ASCII16 }, // Magnum Prohibition
    { 0x68fe9580, ROM_ASCII16 }, // Pinball Blaster
    { 0x704ec575, ROM_ASCII16 }, // Tobin
    { 0x61f41bcd, ROM_0x4000 },  // Green Beret
    { 0x46f5e571, ROM_0x4000 },  // Green Beret
    { 0x3caec828, ROM_0x4000 },  // Green Beret
    { 0x195f5366, ROM_0x4000 },  // Pairs
    { 0x18896882, ROM_0x4000 },  // Sinbad
    { 0x78ab8fd7, ROM_0x4000 },  // Wonder boy
    { 0x0b69dd50, ROM_0x4000 },  // Crusader
    { 0xf7d4e468, ROM_0x4000 },  // Pipi
    { 0xd614d769, ROM_0x4000 },  // Sky Galdo
    { 0x54f84047, ROM_0x4000 },  // Sky Galdo
    { 0xc0a81a48, ROM_0x4000 },  // Tritorn
    { 0x80ecac11, ROM_LODERUNNER },
    { 0xc9651b32, ROM_KANJI },
    { 0xb4fc574d, ROM_KANJI12 },
    { 0xdb46ef0e, ROM_0x4000 },   // Green Baret
    { 0xd8a17006, ROM_0x4000 },   // Philips MSX audio
    { 0xdc405f8c, ROM_0x4000 },   // Toshiba MSX audio
    { 0xd6d01f72, ROM_MSXAUDIO }, // Panasonic MSX audio
    { 0x775fe34e, ROM_0x4000 },   // Bridge
    { 0x0596a961, ROM_0x4000 },   // B-Tower
    { 0xbca34bad, ROM_0x4000 },   // U-Do
    { 0x57efca5a, ROM_0x4000 },   // Bouken Roman
    { 0xab6b56ba, ROM_0x4000 },   // Aramo
    { 0xe74b7943, ROM_BASIC },    // Pasokon Sakkyokuka
    { 0x4a79851b, ROM_STANDARD }, // Alfaroid
    { 0x7306e61d, ROM_STANDARD }, // Bioman (not working)
    { 0x73f3c7e2, ROM_0x4000 },   // Crusader
    { 0x4f195441, ROM_STANDARD }, // Cso
    { 0x4fd42edb, ROM_STANDARD }, // David II
    { 0xc70e3a34, ROM_STANDARD }, // Double Dragon
    { 0x121c7d23, ROM_STANDARD }, // Extra
    { 0x68c2426a, ROM_STANDARD }, // Mr Chin
    { 0x0e989532, ROM_STANDARD }, // Bobble Bouble
    { 0x18510ca7, ROM_STANDARD }, // Korean Ninja
    { 0x37b5d39b, ROM_STANDARD }, // Super Bobble Bouble (MSX1)
    { 0x112abf34, ROM_STANDARD }, // Super Bio Man
    { 0xcb43bfba, ROM_STANDARD }, // Son Son
    { 0x6adeadf5, ROM_STANDARD }, // Spart Tanks
    { 0x62256991, ROM_STANDARD }, // Super Bros World
    { 0x13b34548, ROM_STANDARD }, // Super Boy
    { 0xb358d5a7, ROM_STANDARD }, // Tetris
    { 0x30151594, ROM_STANDARD }, // Korean Tetris
    { 0x6235de29, ROM_STANDARD },
    { 0x2808a3bc, ROM_STANDARD },
    { 0x36ed61af, ROM_STANDARD }, // Wrestle
    { 0xe54ee601, ROM_ASCII8 },   // 1942
    { 0x53fa8533, ROM_KONAMI5 },  // Game collection 1
    { 0x00c15010, ROM_KONAMI5 },  // Game collection 2 
    { 0x84e65759, ROM_KONAMI5 },  // Game collection 3A
    { 0xae5ac7f7, ROM_KONAMI5 },  // Game collection 3B
    { 0x3e19f0a2, ROM_KONAMI5 },  // Game collection 4
    { 0x7faaf10c, ROM_KONAMI5 },  // Game collection 5
    { 0xfee6d0c6, ROM_ASCII8 },   // Minestry
    { 0x334596b3, ROM_SCC },      // Go Othello
    { 0xcf34c1a6, ROM_SCCPLUS },  // Go Othello
    { 0xfd86f5a0, ROM_SCCPLUS },  // Go Othello
    { 0x14ccb2fd, ROM_SCCPLUS },  // Hyper Rally
    { 0x2ca8e6bf, ROM_SCCPLUS },  // Hyper Rally
    { 0x04391bc8, ROM_SCC },      // Hyper Sports 2
    { 0x04a889bf, ROM_SCCPLUS },  // Hyper Sports 2
    { 0x62f1531c, ROM_SCCPLUS },  // Hyper Sports 2
    { 0x9ab41ff5, ROM_SCC },      // Knight Mare
    { 0xdbc8b11b, ROM_SCCPLUS },  // Knight Mare
    { 0xe4a8ee40, ROM_SCC },      // Konami's Boxing
    { 0x408070c4, ROM_SCCPLUS },  // Konami's Boxing
    { 0x0e8efd8b, ROM_SCCPLUS },  // Konami's Boxing
    { 0x1ec24c26, ROM_SCCPLUS },  // Pippols
    { 0x9ea714c2, ROM_SCCPLUS },  // Pippols
    { 0xca823ad9, ROM_SCC },      // Rice 2
    { 0x36f36dcc, ROM_SCCPLUS },  // Rice 2
    { 0x044159ca, ROM_SCCPLUS },  // Rice 2
    { 0x1826474b, ROM_SCC },      // Rice
    { 0x04459be0, ROM_SCCPLUS },  // Rice
    { 0x36f7afe6, ROM_SCCPLUS },  // Rice
    { 0x6616395a, ROM_SCCPLUS },  // Road Fighter
    { 0x39b45611, ROM_SCCPLUS },  // Road Fighter
    { 0x1b47f46e, ROM_SCCPLUS },  // Sky Jaguar
    { 0x37183efd, ROM_SCC },      // Super Cobra
    { 0x8a4b51bc, ROM_SCCPLUS },  // Super Cobra
    { 0x4f834531, ROM_SCC },      // Title Memory
    { 0xb3f21224, ROM_SCCPLUS },  // Title Memory
    { 0x81402622, ROM_SCCPLUS },  // Title Memory
    { 0xb29f137c, ROM_SCC },      // Video Hustler
    { 0x7a5fec5a, ROM_SCCPLUS },  // Video Hustler
    { 0xf338052f, ROM_SCCPLUS },  // Video Hustler
    { 0xe36710ce, ROM_SCC },      // Yi Ar Kung Fu
    { 0x278114cd, ROM_SCCPLUS },  // Yi Ar Kung Fu
    { 0x4cf4d4cf, ROM_SCC },      // Yi Ar Kung Fu 2
    { 0xc080159c, ROM_SCCPLUS },  // Yi Ar Kung Fu 2
    { 0xd9aa0055, ROM_KOREAN80 },
    { 0x8d452fa1, ROM_KOREAN80 },
    { 0xe50bab84, ROM_KOREAN80 },
    { 0x123be4e7, ROM_KOREAN90 },
    { 0x2ee35c23, ROM_KOREAN126 },
    { 0x330f5b0d, ROM_FMPAC },    // Empty FM-PAC
    { 0x00000000, ROM_UNKNOWN },
};


// Static CRC table
static UInt32 crc32Table[256] = {
	0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
	0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
	0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
	0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
	0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
	0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
	0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
	0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
	0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
	0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
	0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
	0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
	0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
	0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
	0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
	0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,

	0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
	0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
	0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
	0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
	0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
	0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
	0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
	0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
	0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
	0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
	0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
	0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
	0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
	0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
	0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
	0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,

	0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
	0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
	0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
	0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
	0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
	0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
	0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
	0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
	0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
	0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
	0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
	0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
	0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
	0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
	0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
	0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,

	0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
	0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
	0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
	0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
	0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
	0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
	0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
	0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
	0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
	0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
	0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
	0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
	0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
	0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
	0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
	0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d,
};

static UInt32 crc32(const void* buffer, int size) {
    const UInt8* ptr = (const UInt8*)buffer;
    UInt32 crc = 0xffffffff;

    while (size--) {
    	crc = (crc >> 8) ^ crc32Table[*ptr++ ^ (crc & 0xff)];
    }

    return ~crc;
}

RomType getRomType(const void* buffer, int size) {
    UInt32 crc = crc32(buffer, size);
    int index;

    for (index = 0; romTable[index].romType != ROM_UNKNOWN; index++) {
        if (crc == romTable[index].crc) {

            // Roms with conflicting CRC's
            if (crc == 0x0e84505d) {
                if (crc32(buffer, 128) == 0xb8a8e218) {
                    return ROM_FMPAC;
                }
            }

            if (crc == 0xe0e72736) {
                if (crc32(buffer, 128) == 0x4cae8359) {
                    return ROM_KONAMI5;
                }
            }

            if (crc == 0x0b69dd50) {
                if (crc32(buffer, 128) == 0x61896ad8) {
                    return ROM_0x4000;
                }
            }

            if (crc == 0x54f84047) {
                if (crc32(buffer, 128) == 0x9c44d314) {
                    return ROM_0x4000;
                }
            }

            if (crc == 0xc0a81a48) {
                if (crc32(buffer, 128) == 0xe6831d78) {
                    return ROM_0x4000;
                }
            }

            if (crc == 0xdc405f8c) {
                if (crc32(buffer, 128) == 0xab630020) {
                    return ROM_0x4000;
                }
            }

            if (crc == 0x7de493ab) {
                if (crc32(buffer, 128) == 0xa6d26b6e) {
                    return ROM_0x4000;
                }
            }

            if (crc == 0xdbdb64ac) {
                if (crc32(buffer, 128) == 0x01255842) {
                    return ROM_0x4000;
                }
            }

            if (crc == 0xe167fede) {
                if (crc32(buffer, 128) == 0x881a87fc) { // Crusader
                    return ROM_0x4000;
                }
            }

            if (crc == 0xb9b0999a) {
                if (crc32(buffer, 128) == 0x7d624d6) { // Konami's Synthesizer (wrong mapper)
                    return ROM_KONAMI5;
                }
            }
            

            return romTable[index].romType;
        }
    }

    return ROM_UNKNOWN;
}

RomType romMapperTypeFromString(const char* name)
{
    if (0 == strcmp(name, "standard"))     return ROM_STANDARD;
    if (0 == strcmp(name, "msxdos2"))      return ROM_MSXDOS2;
    if (0 == strcmp(name, "konami5"))      return ROM_KONAMI5;
    if (0 == strcmp(name, "konami4"))      return ROM_KONAMI4;
    if (0 == strcmp(name, "ascii8"))       return ROM_ASCII8;
    if (0 == strcmp(name, "halnote"))      return ROM_HALNOTE;
    if (0 == strcmp(name, "basic"))        return ROM_BASIC;
    if (0 == strcmp(name, "0x4000"))       return ROM_0x4000;
    if (0 == strcmp(name, "ascii16"))      return ROM_ASCII16;
    if (0 == strcmp(name, "gamemaster2"))  return ROM_GAMEMASTER2;
    if (0 == strcmp(name, "ascii8sram"))   return ROM_ASCII8SRAM;
    if (0 == strcmp(name, "koei"))         return ROM_KOEI;
    if (0 == strcmp(name, "ascii16sram"))  return ROM_ASCII16SRAM;
    if (0 == strcmp(name, "konami4nf"))    return ROM_KONAMI4NF;
    if (0 == strcmp(name, "ascii16nf"))    return ROM_ASCII16NF;
    if (0 == strcmp(name, "snatcher"))     return ROM_SNATCHER;
    if (0 == strcmp(name, "sdsnatcher"))   return ROM_SDSNATCHER;
    if (0 == strcmp(name, "sccmirrored"))  return ROM_SCCMIRRORED;
    if (0 == strcmp(name, "sccexpanded"))  return ROM_SCCEXTENDED;
    if (0 == strcmp(name, "scc"))          return ROM_SCC;
    if (0 == strcmp(name, "sccplus"))      return ROM_SCCPLUS;
    if (0 == strcmp(name, "pac"))          return ROM_PAC;
    if (0 == strcmp(name, "fmpac"))        return ROM_FMPAC;
    if (0 == strcmp(name, "msxaudio"))     return ROM_MSXAUDIO;
    if (0 == strcmp(name, "rtype"))        return ROM_RTYPE;
    if (0 == strcmp(name, "crossblaim"))   return ROM_CROSSBLAIM;
    if (0 == strcmp(name, "harryfox"))     return ROM_HARRYFOX;
    if (0 == strcmp(name, "loderunner"))   return ROM_LODERUNNER;
    if (0 == strcmp(name, "korean80"))     return ROM_KOREAN80;
    if (0 == strcmp(name, "korean90"))     return ROM_KOREAN90;
    if (0 == strcmp(name, "korean126"))    return ROM_KOREAN126;
    if (0 == strcmp(name, "holyquran"))    return ROM_HOLYQURAN;
    if (0 == strcmp(name, "msxaudio"))     return ROM_MSXAUDIO;      

    return ROM_UNKNOWN;
}    


void romMapperSetRomdbFilename(const char* filename)
{
    strcpy(romdbFilename, filename);
}

RomType romMapperRomFromFile(const void *romData, int size)
{
    UInt32 crc = crc32(romData, size);
    char buffer[128];
    RomType romType = ROM_UNKNOWN;
    FILE* file = fopen(romdbFilename, "r");

    if (file == NULL) {
        file = fopen(romdbFilename, "w+");
        if (file == NULL) {
            return ROM_UNKNOWN;
        }

        fwrite(romdbInit, 1, strlen(romdbInit), file);
        fseek(file, 0, SEEK_SET);
    }

    while (romType == ROM_UNKNOWN && fgets(buffer, 128, file)) {
        char* crcBuf  = getCmdArgument(buffer, 0);

        if (crcBuf) {
            int c = 0;

            sscanf(crcBuf, "%x", &c);

            if (c == crc) {
                romType = romMapperTypeFromString(getCmdArgument(buffer, 1));              
            }
        }
    }

    fclose(file);

    return romType;
}


RomType romMapperGuessRom(const void *buffer, int size, int guess)
{
    const UInt8* romData = buffer;
    int i;
    int mapper;
    int counters[6] = { 0, 0, 0, 0, 0, 0 };
    RomType romType;

    if (size < 128) {
        return ROM_UNKNOWN;
    }

    romType = romMapperRomFromFile(romData, size);
    if (romType == ROM_UNKNOWN) {    
        romType = getRomType(romData, size);
    } 

    if (romType == ROM_STANDARD &&
        (size < 0x10000 || (size == 0x10000 && romData[0x4000] == 'A' && romData[0x4001] == 'B'))) {
        romType = ROM_PLAIN;
    }

    if (romType != ROM_UNKNOWN) {    
        return romType;
    }

    if (!guess) {
        return ROM_UNKNOWN;
    }

    counters[0]++; /* Mapper #0 is default */
    counters[4]--; /* #5 preferred over #4 */

    /* Count occurences of characteristic addresses */
    for (i = 0; i < size - 2; i++) {
        UInt8 value = romData[i] + ((int)romData[i + 1] << 8) + ((int)romData[i + 2] << 16);

        switch(value) {
        case 0x400032: 
            counters[3]++;
            break;

        case 0x500032: 
            counters[2]++;
            break;

        case 0x600032: 
            counters[3]++;
            counters[4]++;
            counters[5]++;
            break;

        case 0x680032: 
            counters[4]++;
            break;

        case 0x700032: 
            counters[2]++;
            counters[4]++;
            counters[5]++;
            break;

        case 0x780032: 
            counters[4]++;
            break;

        case 0x77ff32: 
            counters[5]++;
            break;

        case 0x800032: 
            counters[3]++;
            break;

        case 0x900032: 
            counters[2]++;
            break;

        case 0xa00032: 
            counters[3]++;
            break;

        case 0xb00032: 
            counters[2]++;
            break;
        }
    }

    /* Find which mapper type got more hits */
    mapper = 0;

    for (i = 0; i < 6; i++) {
        if (counters[i] > counters[mapper]) {
            mapper = i;
        }
    }

    switch (mapper) {
    default:
    case 0: return ROM_STANDARD;
    case 1: return ROM_MSXDOS2;
    case 2: return ROM_KONAMI5;
    case 3: return ROM_KONAMI4;
    case 4: return ROM_ASCII8;
    case 5: return ROM_ASCII16;
    case 6: return ROM_GAMEMASTER2;
    }
}
