/*****************************************************************************
** File:        Win32Timer.c
**
** Author:      Daniel Vik
**
** Description: Timer service (does currently not support muliple timers)
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include <windows.h>

typedef struct {
    HWND hwnd;
    HANDLE hSemaphore;
    DWORD period;
    DWORD nextTimeout;
    DWORD timeAnchor;
    MMRESULT timerId;
    int running;
} TimerControlBlock;

 
DWORD GetSystemUpTime()
{
    static LONGLONG frequency = 0;
    LARGE_INTEGER li;

    if (!frequency) {
        if (QueryPerformanceFrequency(&li)) {
            frequency = li.QuadPart;
        }
        else {
            return 0;
        }
    }

    QueryPerformanceCounter(&li);

    return (DWORD)(li.QuadPart * 1000 / frequency);
}

static void CheckTimeout(TimerControlBlock* pTCB) {
    DWORD nextTimeout = (DWORD)-1;
    DWORD currentTime = GetSystemUpTime();

    if (pTCB->running) {
        while (pTCB->nextTimeout - pTCB->timeAnchor <= currentTime - pTCB->timeAnchor) {
            ReleaseSemaphore(pTCB->hSemaphore, 1, NULL);
            pTCB->nextTimeout += pTCB->period;
        }
    }

    pTCB->timeAnchor = currentTime;
}

static TimerControlBlock* pTheTCB = NULL;
static void CALLBACK TimerProc(HWND hwnd, UINT uMsg, UINT_PTR idEvent, DWORD dwTime)
{
    if (pTheTCB != NULL) {
        CheckTimeout(pTheTCB);
    }
}

static void CALLBACK timerCallback(unsigned int unused1,
                                   unsigned int unused2,
                                   unsigned long unused3,
                                   unsigned long unused4,
                                   unsigned long unused5)
{
    if (pTheTCB != NULL) {
        CheckTimeout(pTheTCB);
    }
}

void* StartTimer(HWND hwnd, HANDLE hSemaphore, DWORD period, int useHiResTimer) {
    TimerControlBlock* pTCB = (TimerControlBlock*)malloc(sizeof(TimerControlBlock));
    DWORD currentTime = GetSystemUpTime();

    pTCB->hSemaphore  = hSemaphore;
    pTCB->nextTimeout = period + currentTime;
    pTCB->hwnd        = hwnd;
    pTCB->period      = period;
    pTCB->timeAnchor  = currentTime;
    pTCB->running     = 1;
    pTCB->timerId     = 0;
    
    if (useHiResTimer) {
        pTCB->timerId = timeSetEvent(period, period, timerCallback, 0, TIME_PERIODIC);
    }
    if (pTCB->timerId == 0) {
        SetTimer(pTCB->hwnd, 0, pTCB->period, TimerProc);
    }

    pTheTCB = pTCB;

    return pTCB;
}

void StopTimer(void* pTimer) {
    TimerControlBlock* pTCB = (TimerControlBlock*)pTimer;

    if (pTCB->timerId != 0) {
        timeKillEvent(pTCB->timerId);
    }
    else {
        KillTimer(pTCB->hwnd, 0);
    }

    pTheTCB = NULL;
    free(pTCB);
}

void SuspendTimer(void* pTimer) {
    TimerControlBlock* pTCB = (TimerControlBlock*)pTimer;
    pTCB->running = 0;
}

void ResumeTimer(void* pTimer) {
    TimerControlBlock* pTCB = (TimerControlBlock*)pTimer;
    pTCB->running = 1;
}

