/*****************************************************************************
** File:        ym2413.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the YM2413 sound chip. 
**              Wraps the c++ implementation taken from openMSX
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "YM2413.h"
#include "openMsxYM2413.h"
extern "C" {
#include "IoPort.h"
#include "romMapper.h"
#include "SlotManager.h"
}


#define FREQUENCY        3579545
#define SAMPLERATE       44100
#define BUFFER_SIZE      1024
 
#define OFFSETOF(s, a) ((char*)&ym2413->ym2413.a - (char*)&ym2413->ym2413)

struct YM_2413 {
    YM_2413() : ym2413("ym2413", 100, 0), address(0) {
        memset(defaultBuffer, 0, sizeof(defaultBuffer));
    }

    Mixer* mixer;
    int    slotHandle;
    OpenYM2413 ym2413;
    UInt8  address;
    UInt8  registers[256];
    Int32  buffer[BUFFER_SIZE];
    Int32  defaultBuffer[BUFFER_SIZE];
};



extern "C" void ym2413Destroy(void* ref) {
    YM_2413* ym2413 = (YM_2413*)ref;

    slotUnregisterUnslotted(ym2413->slotHandle);

    ioPortUnregister(0x7c);
    ioPortUnregister(0x7d);

    mixerUnregisterChannel(ym2413->mixer, MIXER_CHANNEL_MSX_MUSIC);
    delete ym2413;
}


extern "C" void ym2413WriteAddress(YM_2413* ym2413, UInt16 ioPort, UInt8 address, UInt32 cpuClock)
{
    ym2413->address = address;
}

extern "C" void ym2413WriteData(YM_2413* ym2413, UInt16 ioPort, UInt8 data, UInt32 cpuClock)
{
    mixerSync(ym2413->mixer, cpuClock);
    ym2413->registers[ym2413->address & 0xff] = data;
    ym2413->ym2413.writeReg(ym2413->address, data, cpuClock);
}

extern "C" Int32* ym2413Sync(void* ref, UInt32 count) 
{
    YM_2413* ym2413 = (YM_2413*)ref;
    int* genBuf;
    UInt32 i;

    genBuf = ym2413->ym2413.updateBuffer(count);

    if (genBuf == NULL) {
        return ym2413->defaultBuffer;
    }

    for (i = 0; i < count; i++) {
        ym2413->buffer[i] = genBuf[i];
    }

    return ym2413->buffer;
}

extern "C" int ym2413GetState(void* ref, UInt8* buffer, UInt32 systemTime)
{
    YM_2413* ym2413 = (YM_2413*)ref;

    int offset = OFFSETOF(OpenYM2413, firstMember);
    int size   = sizeof(ym2413->ym2413) - offset;

    memcpy(buffer, (char*)ym2413->registers, 256);
    memcpy(buffer + 256, (char*)&ym2413->ym2413 + offset, size);

    return 256 + size;
}

extern "C" int ym2413SetState(void* ref, UInt8* buffer, UInt32 systemTime)
{
    YM_2413* ym2413 = (YM_2413*)ref;

    int offset = OFFSETOF(OpenYM2413, firstMember);
    int size   = sizeof(ym2413->ym2413) - offset;

    memcpy((char*)ym2413->registers, buffer, 256);
    memcpy((char*)&ym2413->ym2413 + offset, buffer + 256, size);

    return 256 + size;
}

extern "C" int ym2413Create(Mixer* mixer, UInt32 cpuClock)
{
    SlotCallbacks callbacks = { ym2413Destroy, NULL, NULL, ym2413GetState, ym2413SetState };
    YM_2413* ym2413 = new YM_2413;

    ym2413->mixer = mixer;
    ym2413->slotHandle = slotRegisterUnslotted(AUDIO_YM2413, &callbacks, ym2413);

    mixerRegisterChannel(mixer, MIXER_CHANNEL_MSX_MUSIC, 0, ym2413Sync, ym2413);

    ym2413->ym2413.setSampleRate(SAMPLERATE);
	ym2413->ym2413.setVolume(32767);
    
    ioPortRegister(0x7c, NULL, (IoPortWrite)ym2413WriteAddress, ym2413);
    ioPortRegister(0x7d, NULL, (IoPortWrite)ym2413WriteData,    ym2413);

    return 1;
}
