/*****************************************************************************
** File:        audioMixer.c
**
** Author:      Daniel Vik
** 
** Description: Mixer
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "audioMixer.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

extern int MsxFrequency;

#define MSX_FREQUENCY     MsxFrequency
#define SAMPLERATE        44100
#define BITSPERSAMPLE     16
#define CHANNEL_COUNT     6

#define str2ul(s) ((UInt32)s[0]<<0|(UInt32)s[1]<<8|(UInt32)s[2]<<16|(UInt32)s[3]<<24)

#define MIN(a, b) ((a) < (b) ? (a) : (b))
#define MAX(a, b) ((a) > (b) ? (a) : (b))

typedef struct {
    UInt32 riff;
    UInt32 fileSize;
    UInt32 wave;
    struct {
        UInt32 fmt;
        UInt32 chunkSize;
        UInt16 formatType;
        UInt16 channels;
        UInt32 samplesPerSec;
        UInt32 avgBytesPerSec;
        UInt16 blockAlign;
        UInt16 bitsPerSample;
    } wavHeader;
    UInt32 data;
    UInt32 dataSize;
} WavHeader;

typedef struct {
    MixerUpdateCallback     updateCallback;
    void* ref;
    Int32 volume;
    Int32 pan;
    Int32 enable;
    Int32 volumeLeft;
    Int32 volumeRight;
    Int32 stereo;
} Channel;

struct Mixer
{ 
    MixerWriteCallback writeCallback;
    void*  writeRef;
    UInt32 baseClock;
    UInt32 curCount;
    UInt32 index;
    Int16   buffer[1024];
    Channel channels[CHANNEL_COUNT];
    Int32   dummyBuffer[1024];
    int     logging;
    int     stereo;
    Int32   masterVolume;
    FILE*   file;
};

Int32 mixerGetSize() {
    return sizeof(Mixer);
}

Mixer* mixerCreate()
{
    Mixer* mixer = (Mixer*)calloc(1, sizeof(Mixer));

    return mixer;
}

void mixerDestroy(Mixer* mixer)
{
    mixerStopLog(mixer);
    free(mixer);
}


static void mixerSetChannelRecalculate(Mixer* mixer, Channel* channel)
{
    double volume        = pow(10.0, (channel->volume - 100) / 60.0) - pow(10.0, -100 / 60.0);
    double masterVolume  = pow(10.0, (mixer->masterVolume - 100) / 60.0) - pow(10.0, -100 / 60.0);
    double panLeft       = pow(10.0, (MIN(100 - channel->pan, 50) - 50) / 30.0) - pow(10.0, -50 / 30.0);
    double panRight      = pow(10.0, (MIN(channel->pan, 50) - 50) / 30.0) - pow(10.0, -50 / 30.0);

    channel->volumeLeft  = channel->enable * (Int32)(1024 * masterVolume * volume * panLeft);
    channel->volumeRight = channel->enable * (Int32)(1024 * masterVolume * volume * panRight);

    if (!mixer->stereo) {
        Int32 tmp = (channel->volumeLeft + channel->volumeRight) / 2;
        channel->volumeLeft  = tmp;
        channel->volumeRight = tmp;
    }
}

void mixerSetStereo(Mixer* mixer, Int32 stereo)
{
    int i;

    if (mixer->logging == 1) {
        mixerStopLog(mixer);
    }
        
    mixer->stereo = stereo;
    mixer->index = 0;

    for (i = 0; i < CHANNEL_COUNT; i++) {
        mixerSetChannelRecalculate(mixer, mixer->channels + i);
    }
}

void mixerSetMasterVolume(Mixer* mixer, Int32 volume)
{
    int i;

    mixer->masterVolume = volume;
    
    for (i = 0; i < CHANNEL_COUNT; i++) {
        mixerSetChannelRecalculate(mixer, mixer->channels + i);
    }
}

void mixerSetChannelVolume(Mixer* mixer, Int32 channelNumber, Int32 volume)
{
    mixer->channels[channelNumber].volume = volume;
    mixerSetChannelRecalculate(mixer, mixer->channels + channelNumber);
}

void mixerSetChannelPan(Mixer* mixer, Int32 channelNumber, Int32 pan)
{
    mixer->channels[channelNumber].pan = pan;
    mixerSetChannelRecalculate(mixer, mixer->channels + channelNumber);
}

void mixerEnableChannel(Mixer* mixer, Int32 channelNumber, Int32 enable) 
{
    mixer->channels[channelNumber].enable = enable;
    mixerSetChannelRecalculate(mixer, mixer->channels + channelNumber);
}

void mixerSetWriteCallback(Mixer* mixer, MixerWriteCallback callback, void* ref)
{
    mixer->writeCallback = callback;
    mixer->writeRef = ref;
}

void mixerRegisterChannel(Mixer* mixer, Int32 channelNumber, Int32 stereo, MixerUpdateCallback callback, void* ref)
{
    mixer->channels[channelNumber].updateCallback = callback;
    mixer->channels[channelNumber].ref            = ref;
    mixer->channels[channelNumber].stereo         = stereo;
}

void mixerUnregisterChannel(Mixer* mixer, Int32 channelNumber) 
{
    mixer->channels[channelNumber].updateCallback = NULL;
    mixer->channels[channelNumber].ref = NULL;
}

void mixerReset(Mixer* mixer, UInt32 cpuClock)
{
    mixer->baseClock = cpuClock;
    mixer->curCount = 0;
    mixer->index = 0;
}

extern int dx_write(Int16 *, UInt32);

void mixerSync(Mixer* mixer, UInt32 cpuClock)
{
    Int16* buffer   = mixer->buffer;
    Int32* chBuff[CHANNEL_COUNT];
    UInt32 curCount;
    UInt32 count;
    int i;

    if (cpuClock - mixer->baseClock > (UInt32)MSX_FREQUENCY) {
        mixer->baseClock += MSX_FREQUENCY;
    }

    curCount = (UInt64)SAMPLERATE * (cpuClock - mixer->baseClock) / MSX_FREQUENCY;
    count    = (SAMPLERATE + curCount - mixer->curCount) % SAMPLERATE;
    
    if (count == 0 || count > 1024) {
        return;
    }

    mixer->curCount = curCount;

    for (i = 0; i < CHANNEL_COUNT; i++) {
        if (mixer->channels[i].updateCallback != NULL) {
            chBuff[i] = mixer->channels[i].updateCallback(mixer->channels[i].ref, count);
        }
        else {
            chBuff[i] = mixer->dummyBuffer;
        }
    }

    if (mixer->stereo) {
        while (count--) {
            Int32 left = 0;
            Int32 right = 0;

            for (i = 0; i < CHANNEL_COUNT; i++) {
                if (mixer->channels[i].stereo) {
                    left  += mixer->channels[i].volumeLeft * *chBuff[i]++;
                    right += mixer->channels[i].volumeRight * *chBuff[i]++;
                }
                else {
                    Int32 tmp = *chBuff[i]++;
                    left  += mixer->channels[i].volumeLeft * tmp;
                    right += mixer->channels[i].volumeRight * tmp;
                }
            }

            left  /= 4096;
            right /= 4096;

            if (left  >  32767) { left  = 32767; }
            if (left  < -32767) { left  = -32767; }
            if (right >  32767) { right = 32767; }
            if (right < -32767) { right = -32767; }

            buffer[mixer->index++] = (Int16)left;
            buffer[mixer->index++] = (Int16)right;

            if (mixer->index == 0x400) {
                if (mixer->writeCallback != NULL) {
                    mixer->writeCallback(mixer->writeRef, buffer, 0x400);
                }
                if (mixer->logging) {
                    fwrite(buffer, 0x800, 1, mixer->file);
                }
                mixer->index = 0;
            }
        }
    }
    else {
        while (count--) {
            Int32 left = 0;

            for (i = 0; i < CHANNEL_COUNT; i++) {
                if (mixer->channels[i].stereo) {
                    Int32 tmp = *chBuff[i]++;
                    left  += mixer->channels[i].volumeLeft * (tmp + *chBuff[i]++) / 2;
                }
                else {
                    left  += mixer->channels[i].volumeLeft * *chBuff[i]++;
                }
            }

            left  /= 32768;

            if (left  >  32767) left  = 32767;
            if (left  < -32767) left  = -32767;

            buffer[mixer->index++] = (Int16)left;
            
            if (mixer->index == 0x200) {
                if (mixer->writeCallback != NULL) {
                    mixer->writeCallback(mixer->writeRef, buffer, 0x200);
                }
                if (mixer->logging) {
                    fwrite(buffer, 0x400, 1, mixer->file);
                }
                mixer->index = 0;
            }
        }
    }
}

int mixerStartLog(Mixer* mixer, char* fileName) 
{
    if (mixer->logging == 1) {
        mixerStopLog(mixer);
    }
    mixer->file = fopen(fileName, "wb");
    if (mixer->file != NULL) {
        fseek(mixer->file, sizeof(WavHeader), SEEK_SET);
        mixer->logging = 1;
    }

    return mixer->logging;
}

void mixerStopLog(Mixer* mixer) 
{
    WavHeader header;
    int fileSize;

    if (mixer->logging == 0) {
        return;
    }

    mixer->logging = 0;
    
    fileSize = ftell(mixer->file);
    
    header.riff                     = str2ul("RIFF");
    header.fileSize                 = fileSize - 8;
    header.wave                     = str2ul("WAVE");
    header.wavHeader.fmt            = str2ul("fmt ");
    header.wavHeader.chunkSize      = 16;
    header.wavHeader.formatType     = 1;
    header.wavHeader.channels       = (mixer->stereo ? 2 : 1);
    header.wavHeader.samplesPerSec  = SAMPLERATE;
    header.wavHeader.avgBytesPerSec = (mixer->stereo ? 2 : 1) * SAMPLERATE * BITSPERSAMPLE / 8;
    header.wavHeader.blockAlign     = (mixer->stereo ? 2 : 1) * BITSPERSAMPLE / 8;
    header.wavHeader.bitsPerSample  = BITSPERSAMPLE;
    header.data                     = str2ul("data");
    header.dataSize                 = fileSize - sizeof(WavHeader);

    fseek(mixer->file, 0, SEEK_SET);
    fwrite(&header, 1, sizeof(WavHeader), mixer->file);
    fclose(mixer->file);
}
