/*****************************************************************************
** File:
**      Disk.c
**
** Author:
**      Daniel Vik
** 
** Description:
**      Bios patch for disk support
**
******************************************************************************
*/
#include "Disk.h"
#include "ziphelper.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


static char* toLower(const char* str) {
    static char tempname[256];
    char *ptr = tempname;

    while (*str) {
        *ptr++ = tolower(*str++);
    }

    return tempname;
}

static FILE* drives[2];                  /* Disk image files */
static int   RdOnly[2];                  /* 1 = read-only    */
static char* ramImageBuffer[2];          /* 0 = Use file     */
static int   ramImageSize[2];            /* Size of ram buf  */
extern int diskBusy[];


UInt8 DiskPresent(UInt8 driveId)
{
    return driveId < MAXDRIVES && 
           (drives[driveId] != NULL || ramImageBuffer[driveId] != NULL);
}


UInt8 DiskRead(UInt8 driveId, UInt8* buffer, int sector)
{
    if (driveId >= MAXDRIVES) {
        return 0;
    }

    if (ramImageBuffer[driveId] != NULL) {
        int offset = sector * 512;

        if (ramImageSize[driveId] < offset + 512) {
            return 0;
        }

        memcpy(buffer, ramImageBuffer[driveId] + offset, 512);
        diskBusy[driveId] = 1;
        return 1;
    }
    else {
        if ((drives[driveId] != NULL)) {
            if (0 == fseek(drives[driveId], sector * 512, SEEK_SET)) {
                UInt8 success = fread(buffer, 1, 512, drives[driveId]) == 512;
                diskBusy[driveId] = 1;
                return success;
            }
        }
    }
    return 0;
}


UInt8 DiskWrite(UInt8 driveId,UInt8 *buffer,int sector)
{
    if (driveId >= MAXDRIVES) {
        return 0;
    }

    if (ramImageBuffer[driveId] != NULL) {
        int offset = sector * 512;

        if (ramImageSize[driveId] < offset + 512) {
            return 0;
        }

        memcpy(ramImageBuffer[driveId] + offset, buffer, 512);
        diskBusy[driveId] = 1;
        return 1;
    }
    else {
        if (drives[driveId] != NULL && !RdOnly[driveId]) {
            if (0 == fseek(drives[driveId], sector * 512, SEEK_SET)) {
                UInt8 success = fwrite(buffer, 1, 512, drives[driveId]) == 512;
                diskBusy[driveId] = 1;
                return success;
            }
        }
    }
    return 0;
}


UInt8 ChangeDisk(UInt8 driveId,char* fileName, const char* fileInZipFile)
{
    if (driveId >= MAXDRIVES) {
        return 0;
    }

    /* Close previous disk image */
    if(drives[driveId] != NULL) { 
        fclose(drives[driveId]);
        drives[driveId] = NULL; 
    }

    if (ramImageBuffer[driveId] != NULL) {
        // Flush to file??
        free(ramImageBuffer[driveId]);
        ramImageBuffer[driveId] = NULL;
    }

    if(!fileName) {
        return 1;
    }

    if (fileInZipFile != NULL) {
        ramImageBuffer[driveId] = zipLoadFile(fileName, fileInZipFile, &ramImageSize[driveId]);
        return ramImageBuffer[driveId] != NULL;
    }

    drives[driveId] = fopen(fileName, "r+b");
    RdOnly[driveId] = 0;

    if (drives[driveId] == NULL) {
        drives[driveId] = fopen(fileName, "rb");
        RdOnly[driveId] = 1;
    }

    return drives[driveId] != 0;
}

