/*****************************************************************************
** File:        y8950.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the YM2413 sound chip. 
**              Wraps the c++ implementation taken from openMSX
**
** History:     1.0 - 11/5 2003 Initial version
**
******************************************************************************
*/
#include "Y8950.h"
#include "fmopl.h"
#include <stdlib.h>
#include "msx.h"

#define FREQUENCY        GLOBAL_FREQ //3570259 //3568884 //3579545
#define SAMPLERATE       44100
#define BUFFER_SIZE      1024

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))

 
Int16* y8950Sync(void* ref, UInt32 count);

struct Y8950 {
    Mixer* mixer;
    FM_OPL* opl;
    UInt8  address;
    UInt8  registers[256];
    Int16  buffer[BUFFER_SIZE];
};

extern INT32 outd;
extern INT32 ams;
extern INT32 vib;
extern INT32 feedback2;

int y8950LoadState(Y8950* y8950, FILE* file, UInt32 cpuClock) {
    int offset = OFFSETOF(FM_OPL, firstMember);
    int size   = sizeof(FM_OPL) + 9 * sizeof(OPL_CH) + sizeof(YM_DELTAT);
    void* oplMem = y8950->opl->deltat->memory;

    OPLResetChip(y8950->opl);

    if (fread((char*)y8950->registers, 1, 256, file) != 256) {
        return 0;
    }

    if (fread(((char*)y8950->opl) + offset, 1, size - offset, file) != size - offset) {
        return 0;
    }
    y8950->opl->deltat->memory = oplMem;

    if (fread((char*)oplMem, 1, 256 * 1024, file) != 256 * 1024) {
        return 0;
    }

    if (fread((char*)&outd, 1, sizeof(outd), file) != sizeof(outd)) {
        return 0;
    }

    if (fread((char*)&ams, 1, sizeof(ams), file) != sizeof(ams)) {
        return 0;
    }

    if (fread((char*)&vib, 1, sizeof(vib), file) != sizeof(vib)) {
        return 0;
    }

    if (fread((char*)&feedback2, 1, sizeof(feedback2), file) != sizeof(feedback2)) {
        return 0;
    }

    return 1;
}

int y8950SaveState(Y8950* y8950, FILE* file, UInt32 cpuClock) {
    int offset = OFFSETOF(FM_OPL, firstMember);
    int size   = sizeof(FM_OPL) + 9 * sizeof(OPL_CH) + sizeof(YM_DELTAT);
    void* oplMem = y8950->opl->deltat->memory;

    if (fwrite((char*)y8950->registers, 1, 256, file) != 256) {
        return 0;
    }

    if (fwrite(((char*)y8950->opl) + offset, 1, size - offset, file) != size - offset) {
        return 0;
    }

    if (fwrite((char*)oplMem, 1, 256 * 1024, file) != 256 * 1024) {
        return 0;
    }

    if (fwrite((char*)&outd, 1, sizeof(outd), file) != sizeof(outd)) {
        return 0;
    }

    if (fwrite((char*)&ams, 1, sizeof(ams), file) != sizeof(ams)) {
        return 0;
    }

    if (fwrite((char*)&vib, 1, sizeof(vib), file) != sizeof(vib)) {
        return 0;
    }

    if (fwrite((char*)&feedback2, 1, sizeof(feedback2), file) != sizeof(feedback2)) {
        return 0;
    }

    return 1;
}


Y8950* y8950Create(Mixer* mixer, UInt32 cpuClock)
{
    Y8950* y8950 = (Y8950*)calloc(1, sizeof(Y8950));

	memset( y8950, 0, sizeof(Y8950)) ;

    y8950->mixer = mixer;
    mixerRegisterChannel(mixer, MIXER_CHANNEL_MSX_AUDIO, y8950Sync, y8950);

    y8950->opl = OPLCreate(OPL_TYPE_Y8950, FREQUENCY, SAMPLERATE, 256);
    OPLResetChip(y8950->opl);

    return y8950;
}


void y8950Destroy(Y8950* y8950) {
    if (y8950 != NULL) {
        mixerUnregisterChannel(y8950->mixer, MIXER_CHANNEL_MSX_AUDIO);
        OPLDestroy(y8950->opl);
    }
}


UInt8 y8950ReadAddress(Y8950* y8950)
{
    return (UInt8)OPLRead(y8950->opl, 0);
}

UInt8 y8950ReadData(Y8950* y8950)
{
    return (UInt8)OPLRead(y8950->opl, 1);
}

void y8950WriteAddress(Y8950* y8950, UInt8 address)
{
    OPLWrite(y8950->opl, 0, address);
}

void y8950WriteData(Y8950* y8950, UInt8 data, UInt32 cpuClock)
{
    mixerSync(y8950->mixer, cpuClock);
    y8950->registers[y8950->address & 0xff] = data;

    OPLWrite(y8950->opl, 1, data);
}

static Int16* y8950Sync(void* ref, UInt32 count) 
{
    Y8950* y8950 = (Y8950*)ref;
    UInt32 i;

    for (i = 0; i < count; i++) {
        y8950->buffer[i] = Y8950UpdateOne(y8950->opl);
    }

    return y8950->buffer;
}
