/*****************************************************************************
** File:        romMapperFMPAC.c
**
** Author:      Daniel Vik
**
** Description: Rom mapper for FMPAC
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
******************************************************************************
*/
#include "romMapperFMPAC.h"
#include "romMapper.h"
#include <stdlib.h>
#include <string.h>

 
void memMapPage(int slot, int sslot, int page, UInt8* pageData, 
               int noReadCache, int noWriteCache);


typedef void (*SlotWriteFunc)(void*, UInt16, UInt8, UInt32);
typedef void (*SlotReadFunc)(void*, UInt16, UInt8, UInt32);

void slotInfoRegister(int type, int slot, int sslot, int startpage, int pages,
                      SlotWriteFunc wrfunc, SlotWriteFunc rdfunc, void* ref);

char* sramCreateFilename(char* romFilename);
void  sramLoad(char* filename, UInt8* sram, int length, void* header, int headerLength);
void  sramSave(char* filename, UInt8* sram, int length, void* header, int headerLength);

static char pacHeader[] = "PAC2 BACKUP DATA";

UInt8* ramGetEmptyPage();

struct RomMapperFMPAC {
    UInt8 romData[0x10000];
    UInt8 sram[0x2000];
    char sramFilename[512];
    int bankSelect;
    int slot;
    int sslot;
    int page;
    int sramEnabled;
    YM_2413* ym2413;
};

static void romMapperFMPACWrite(void* arg, UInt16 address, UInt8 value, UInt32 systemTime) 
{
    RomMapperFMPAC* rm = (RomMapperFMPAC*)arg;
    int update = 0;

    address &= 0x3fff;

    switch (address) {
    case 0x1ffe:
        rm->romData[0x1ffe] = value;
        rm->romData[0x5ffe] = value;
        rm->romData[0x9ffe] = value;
        rm->romData[0xdffe] = value;
        rm->sramEnabled = rm->romData[0x1ffe] == 0x4d && rm->romData[0x1fff] == 0x69;
        update = 1;
        break;
    case 0x1fff:
        rm->romData[0x1fff] = value;
        rm->romData[0x5fff] = value;
        rm->romData[0x9fff] = value;
        rm->romData[0xdfff] = value;
        rm->sramEnabled = rm->romData[0x1ffe] == 0x4d && rm->romData[0x1fff] == 0x69;
        update = 1;
        break;
	case 0x3ff4:
        if (rm->ym2413 != NULL) ym2413WriteAddress(rm->ym2413, value);
		break;
	case 0x3ff5:
        if (rm->ym2413 != NULL) ym2413WriteData(rm->ym2413, value, systemTime);
		break;
	case 0x3ff6:
        rm->romData[0x3ff6] = value;
        rm->romData[0x7ff6] = value;
        rm->romData[0xbff6] = value;
        rm->romData[0xfff6] = value;
		break;
	case 0x3ff7:
        if ((value & 3) != rm->bankSelect) {
            rm->bankSelect = value & 3;
            rm->romData[0x3ff7] = value;
            rm->romData[0x7ff7] = value;
            rm->romData[0xbff7] = value;
            rm->romData[0xfff7] = value;
            update = 1;
        }
		break;
	default:
		if (rm->sramEnabled && address < 0x1ffe) {
            rm->sram[address] = value;
		}
        break;
    }

    if (update) {
        if (rm->sramEnabled) {
            memMapPage(rm->slot, rm->sslot, rm->page,     rm->sram, 1, 0);
            memMapPage(rm->slot, rm->sslot, rm->page + 1, ramGetEmptyPage(), 1, 0);
        }
        else {
            UInt8* pageData = rm->romData + (rm->bankSelect << 14);

            memMapPage(rm->slot, rm->sslot, rm->page,     pageData, 1, 0);
            memMapPage(rm->slot, rm->sslot, rm->page + 1, pageData + 0x2000, 1, 0);
        }
    }
}

RomMapperFMPAC* romMapperFMPACCreate(char* filename, UInt8* romData, 
                                     int size, int slot, int sslot, int page, int pagecount,
                                     YM_2413* ym2413) 
{
    RomMapperFMPAC* rm;

    if (size != 0x10000 || pagecount != 2) {
        return NULL;
    }

    rm = malloc(sizeof(RomMapperFMPAC));

    slotInfoRegister(ROM_FMPAC, slot, sslot, page, pagecount, romMapperFMPACWrite, NULL, rm);

    memcpy(rm->romData, romData, 0x10000);
    memset(rm->sram, 0xff, 0x2000);
    rm->bankSelect = 0;
    rm->slot  = slot;
    rm->sslot = sslot;
    rm->page  = page;
    rm->sramEnabled = 0;
    rm->ym2413 = ym2413;
    strcpy(rm->sramFilename, sramCreateFilename(filename));

    sramLoad(rm->sramFilename, rm->sram, 0x2000, pacHeader, strlen(pacHeader));

    memMapPage(rm->slot, rm->sslot, rm->page,     rm->romData, 1, 0);
    memMapPage(rm->slot, rm->sslot, rm->page + 1, rm->romData + 0x2000, 1, 0);

    rm->romData[0x1ffe] = 0xff;
    rm->romData[0x5ffe] = 0xff;
    rm->romData[0x9ffe] = 0xff;
    rm->romData[0xdffe] = 0xff;
    rm->romData[0x1fff] = 0xff;
    rm->romData[0x5fff] = 0xff;
    rm->romData[0x9fff] = 0xff;
    rm->romData[0xdfff] = 0xff;
    rm->romData[0x3ff6] = 0;
    rm->romData[0x7ff6] = 0;
    rm->romData[0xbff6] = 0;
    rm->romData[0xfff6] = 0;
    rm->romData[0x3ff7] = 0;
    rm->romData[0x7ff7] = 0;
    rm->romData[0xbff7] = 0;
    rm->romData[0xfff7] = 0;

    return rm;
}

void romMapperFMPACDestroy(RomMapperFMPAC* rm)
{
    sramSave(rm->sramFilename, rm->sram, 0x2000, pacHeader, strlen(pacHeader));
    free(rm);
}

int romMapperFMPACLoadState(RomMapperFMPAC* rm, FILE* file)
{
    UInt8  reg1ffe;
    UInt8  reg1fff;
    UInt8  reg3ff6;
    UInt8  reg3ff7;
    int    slot;
    int    sslot;
    int    page;
    UInt32 data[32];
    int    index = 0;

    if (fread(data, 1, sizeof(data), file) != sizeof(data)) { 
        fclose(file);
        return 0; 
    }

    slot  = data[index++];
    sslot = data[index++];
    page  = data[index++];

    rm->bankSelect  = data[index++];
    rm->sramEnabled = data[index++];
    reg1ffe         = (UInt8)data[index++];
    reg1fff         = (UInt8)data[index++];
    reg3ff6         = (UInt8)data[index++];
    reg3ff7         = (UInt8)data[index++];

    // Verify state. For now memory map has to match
    if (slot != rm->slot || sslot != rm->sslot || page != rm->page) {
        return 0;
    }

    rm->slot  = slot;
    rm->sslot = sslot;
    rm->page  = page;

    rm->romData[0x1FFE] = reg1ffe;
    rm->romData[0x5FFE] = reg1ffe;
    rm->romData[0x9FFE] = reg1ffe;
    rm->romData[0xDFFE] = reg1ffe;
    rm->romData[0x1FFF] = reg1fff;
    rm->romData[0x5FFF] = reg1fff;
    rm->romData[0x9FFF] = reg1fff;
    rm->romData[0xDFFF] = reg1fff;
    rm->romData[0x3FF6] = reg3ff6;
    rm->romData[0x7FF6] = reg3ff6;
    rm->romData[0xBFF6] = reg3ff6;
    rm->romData[0xFFF6] = reg3ff6;
    rm->romData[0x3FF7] = reg3ff7;
    rm->romData[0x7FF7] = reg3ff7;
    rm->romData[0xBFF7] = reg3ff7;
    rm->romData[0xFFF7] = reg3ff7;

    if (rm->sramEnabled) {
        memMapPage(rm->slot, rm->sslot, rm->page,     rm->sram, 1, 0);
        memMapPage(rm->slot, rm->sslot, rm->page + 1, ramGetEmptyPage(), 1, 0);
    }
    else {
        UInt8* pageData = rm->romData + (rm->bankSelect << 14);

        memMapPage(rm->slot, rm->sslot, rm->page,     pageData, 1, 0);
        memMapPage(rm->slot, rm->sslot, rm->page + 1, pageData + 0x2000, 1, 0);
    }

    return 1;
}

int romMapperFMPACSaveState(RomMapperFMPAC* rm, FILE* file) 
{
    UInt32 data[32];
    int    index = 0;

    data[index++] = rm->slot;
    data[index++] = rm->sslot;
    data[index++] = rm->page;
    data[index++] = rm->bankSelect;
    data[index++] = rm->sramEnabled;

    data[index++] = rm->romData[0x1ffe];
    data[index++] = rm->romData[0x1fff];
    data[index++] = rm->romData[0x3ff6];
    data[index++] = rm->romData[0x3ff7];

    if (fwrite(data, 1, sizeof(data), file) != sizeof(data)) { 
        fclose(file);
        return 0; 
    }

    return 1;
}

