/*****************************************************************************
** File:        Win32directXSound.c
**
** Author:      Daniel Vik
**
** Description: Contains DirectX audio metods.
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "Win32directXSound.h"
#include <stdio.h>
#define DIRECTSOUND_VERSION 0x0500
#include <dsound.h>
 
struct DxSound {
    Mixer* mixer;
    UInt32 bufferOffset;
    UInt32 bufferSize;
    UInt32 fragmentSize;
    UInt32 fragmentCount;
    Int16  channels;
    Int16  bitCount;
    Int32  skipCount;
    LPDIRECTSOUNDBUFFER primaryBuffer;
    LPDIRECTSOUNDBUFFER secondaryBuffer;
    LPDIRECTSOUND directSound;
};


static void dxClear(DxSound* dxSound)
{
    void*   audioBuffer1;
    UInt32  audioSize1;
    void*   audioBuffer2;
    UInt32  audioSize2;
    HRESULT result;

    result = IDirectSoundBuffer_Lock(dxSound->primaryBuffer, 0, dxSound->bufferSize * (dxSound->bitCount / 8),
                                     &audioBuffer1, &audioSize1, &audioBuffer2, &audioSize2, 0);
    if (result == DSERR_BUFFERLOST) {
        IDirectSoundBuffer_Restore(dxSound->primaryBuffer);
    } else {
        if (dxSound->bitCount == 16) {
            memset(audioBuffer1, 0, audioSize1);
            if (audioBuffer2) memset(audioBuffer2, 0, audioSize2);
        } else {
            memset(audioBuffer1, 0x80, audioSize1);
            if (audioBuffer2) memset(audioBuffer2, 0x80, audioSize2);
        }
        result = IDirectSoundBuffer_Unlock(dxSound->primaryBuffer, audioBuffer1, audioSize1,
                                           audioBuffer2, audioSize2);
    }
}

static UInt32 dxBufferspace(DxSound* dxSound)
{
    UInt32 readPos;
    UInt32 writePos;

    IDirectSoundBuffer_GetCurrentPosition(dxSound->primaryBuffer, &readPos, &writePos);
    if (readPos > dxSound->bufferOffset) {
        return (readPos - dxSound->bufferOffset) / (dxSound->bitCount / 8);
    }

    return (dxSound->bufferSize - (dxSound->bufferOffset - readPos) / (dxSound->bitCount / 8)) / dxSound->channels;
}

static int dxWriteOne(DxSound* dxSound, Int16 *buffer, UInt32 count)
{
    void*   audioBuffer1;
    UInt32  audioSize1;
    void*   audioBuffer2;
    UInt32  audioSize2;
    HRESULT result;
    UInt32  lockSize;
    UInt32  i;

    count /= dxSound->fragmentSize;
    lockSize = dxSound->fragmentSize * dxSound->bitCount / 8;

    for (i = 0; i < count; i++) {
        do {
            result = IDirectSoundBuffer_Lock(dxSound->primaryBuffer, dxSound->bufferOffset,
                                             lockSize,
                                             &audioBuffer1, &audioSize1, &audioBuffer2,
                                             &audioSize2, 0);

            if (result == DSERR_BUFFERLOST) {
                IDirectSoundBuffer_Restore(dxSound->primaryBuffer);
                result = IDirectSoundBuffer_Lock(dxSound->primaryBuffer, dxSound->bufferOffset,
                                                 lockSize, &audioBuffer1,
                                                 &audioSize1,
                                                 &audioBuffer2, &audioSize2, 0);
            }
        } while ((audioSize1 + audioSize2) < lockSize);
        if (dxSound->bitCount == 16) {
            memcpy(audioBuffer1,buffer,audioSize1);
            if (audioBuffer2)
                memcpy(audioBuffer2,(BYTE *)buffer + audioSize1, audioSize2);
        } else {
            Int16 *copyptr = buffer;
            for (i = 0; i < audioSize1; i++) {
                ((BYTE *)audioBuffer1)[i]=(*(copyptr++) >> 8) + 0x80;
            }
            if (audioBuffer2 != NULL) {
                for (i = 0; i < audioSize2; i++) {
                    ((BYTE *)audioBuffer2)[i]=(*(copyptr++) >> 8) + 0x80;
                }
            }
        }

        result = IDirectSoundBuffer_Unlock(dxSound->primaryBuffer, audioBuffer1, audioSize1,
                                           audioBuffer2, audioSize2);
        dxSound->bufferOffset += lockSize;

        if (dxSound->bufferOffset >= dxSound->bufferSize * (dxSound->bitCount / 8))
            dxSound->bufferOffset = 0;

        buffer += dxSound->fragmentSize;
    }

    return 0;
}

static Int32 dxWrite(DxSound* dxSound, Int16 *buffer, UInt32 count)
{
    UInt32 cnt = dxBufferspace(dxSound) / dxSound->fragmentSize;

    while (dxSound->skipCount > 0) {
        dxSound->skipCount--;
        count -= dxSound->fragmentSize;
        if (count == 0) {
            return 0;
        }
    }

    if (cnt <= 1) {
        if (count > dxSound->fragmentCount / 2) {
            count -= dxSound->fragmentCount / 2;
        }
        else {
            dxSound->skipCount = dxSound->fragmentCount / 2 - count;
            return 0;
        }
    }

    if (cnt >= dxSound->fragmentCount - 2) {
        UInt32 i;
        for (i = 0; i < dxSound->fragmentCount / 2; i++) {
            dxWriteOne(dxSound, buffer, dxSound->fragmentSize);
        }
    }

    return dxWriteOne(dxSound, buffer, count);
}



DxSound* dxSoundCreate(HWND hwnd, Mixer* mixer, UInt32 sampleRate, UInt32 bufferSize, Int16 channels) 
{
    DxSound* dxSound = (DxSound*)calloc(1, sizeof(DxSound));
    HRESULT  result;
    UInt32 fragmentSize = bufferSize <= 10 ? 32 : bufferSize <= 25 ? 64 : bufferSize <= 50 ? 128 : bufferSize <= 100 ? 256 : 512;
    UInt32 fragmentCount = (bufferSize * sampleRate / 1000 + fragmentSize / 2) / fragmentSize;
    
    DSBUFFERDESC desc;
    PCMWAVEFORMAT pcmwf;
    DSCAPS  capabilities;
    WAVEFORMATEX    wfex;
    dxSound->skipCount = 0;

    if (fragmentCount < 8) fragmentCount = 8;
    if (fragmentCount > 32) fragmentCount = 32;

    if (dxSound->directSound == NULL) {
        result = DirectSoundCreate(NULL, &dxSound->directSound, NULL);
        if (result != DS_OK) {
            free(dxSound);
            return NULL;
        }

        result = IDirectSound_SetCooperativeLevel(dxSound->directSound, hwnd, DSSCL_EXCLUSIVE);
        if (result != DS_OK) {
            free(dxSound);
            return NULL;
        }
    }

    memset(&capabilities, 0, sizeof(DSCAPS));
    capabilities.dwSize = sizeof(DSCAPS);

    IDirectSound_GetCaps(dxSound->directSound, &capabilities);
    if ((capabilities.dwFlags & DSCAPS_PRIMARY16BIT) || (capabilities.dwFlags & DSCAPS_SECONDARY16BIT)) {
        dxSound->bitCount = 16;
    } 
    else {
        dxSound->bitCount = 8;
    }

    if (!((capabilities.dwFlags & DSCAPS_PRIMARYSTEREO) || (capabilities.dwFlags & DSCAPS_SECONDARYSTEREO))) {
        channels = 1;
    }

    dxSound->channels = channels;
    fragmentSize *= channels;

    memset(&pcmwf, 0, sizeof(PCMWAVEFORMAT));
    pcmwf.wf.wFormatTag = WAVE_FORMAT_PCM;
    pcmwf.wf.nChannels = channels;
    pcmwf.wf.nSamplesPerSec = sampleRate;
    pcmwf.wBitsPerSample = dxSound->bitCount;
    pcmwf.wf.nBlockAlign = channels * dxSound->bitCount / 8;
    pcmwf.wf.nAvgBytesPerSec = pcmwf.wf.nSamplesPerSec * pcmwf.wf.nBlockAlign;

    memset(&desc, 0, sizeof(DSBUFFERDESC));
    desc.dwSize = sizeof(DSBUFFERDESC);
    desc.dwFlags = DSBCAPS_PRIMARYBUFFER;

    dxSound->fragmentSize = fragmentSize;
    dxSound->fragmentCount = fragmentCount;

    dxSound->bufferSize = fragmentSize * fragmentCount * channels;

    result = IDirectSound_CreateSoundBuffer(dxSound->directSound, &desc, &dxSound->secondaryBuffer, NULL);

    if (result != DS_OK) {
        free(dxSound);
        return NULL;
    }

    memset(&desc, 0, sizeof(DSBUFFERDESC));
    desc.dwSize = sizeof(DSBUFFERDESC);
    desc.dwFlags = DSBCAPS_CTRLPOSITIONNOTIFY | DSBCAPS_GETCURRENTPOSITION2
                   | DSBCAPS_CTRLFREQUENCY | DSBCAPS_CTRLPAN
                   | DSBCAPS_CTRLVOLUME | DSBCAPS_GLOBALFOCUS ;

    desc.dwBufferBytes = dxSound->bufferSize * (dxSound->bitCount / 8);
    desc.lpwfxFormat = (LPWAVEFORMATEX)&pcmwf;

    result = IDirectSound_CreateSoundBuffer(dxSound->directSound, &desc, &dxSound->primaryBuffer, NULL);
    if (result != DS_OK) {
        free(dxSound);
        return NULL;
    }

    memset(&wfex, 0, sizeof(WAVEFORMATEX));
    wfex.wFormatTag = WAVE_FORMAT_PCM;
    wfex.nChannels = channels;
    wfex.nSamplesPerSec = sampleRate;
    wfex.wBitsPerSample = dxSound->bitCount;
    wfex.nBlockAlign = channels * dxSound->bitCount / 8;
    wfex.nAvgBytesPerSec = wfex.nSamplesPerSec * wfex.nBlockAlign;

    result=IDirectSoundBuffer_SetFormat(dxSound->secondaryBuffer, &wfex);
    if (result != DS_OK) {
        free(dxSound);
        return NULL;
    }

    dxSound->bufferOffset= fragmentSize * dxSound->bitCount / 8 * (fragmentCount - 1) * channels;
    dxClear(dxSound);
    
    result = IDirectSoundBuffer_Play(dxSound->primaryBuffer, 0, 0, DSBPLAY_LOOPING);
    if (result == DSERR_BUFFERLOST) {
        result = IDirectSoundBuffer_Play(dxSound->primaryBuffer, 0, 0, DSBPLAY_LOOPING);
    }
    if (result != DS_OK) {
        free(dxSound);
        return NULL;
    }

    dxSound->mixer = mixer;
    dxSound->fragmentSize = fragmentSize;

    mixerSetStereo(mixer, channels == 2);
    mixerSetWriteCallback(mixer, dxWrite, dxSound);

    return dxSound;
}

void dxSoundDestroy(DxSound* dxSound)
{
    if (dxSound == NULL) {
        return;
    }

    mixerSetWriteCallback(dxSound->mixer, NULL, NULL);
    Sleep(200);

    if (dxSound->directSound == NULL) {
        return;
    }

    IDirectSoundBuffer_Stop(dxSound->primaryBuffer);
    IDirectSoundBuffer_Release(dxSound->primaryBuffer);
    IDirectSound_Release(dxSound->directSound);
    dxSound->primaryBuffer = NULL;
    dxSound->directSound = NULL;
}

void dxSoundSuspend(DxSound* dxSound)
{
    Int16* buf;

    if (dxSound == NULL) {
        return;
    }

    Sleep(100);

    buf = (Int16*)calloc(dxSound->bufferSize * (dxSound->bitCount / 8), 1);

    dxWriteOne(dxSound, buf, dxSound->bufferSize);
    free(buf);
}

void dxSoundResume(DxSound* dxSound)
{
    if (dxSound == NULL) {
        return;
    }

    dxSound->bufferOffset = dxSound->bufferSize * (dxSound->bitCount / 8) - dxSound->fragmentSize;
    IDirectSoundBuffer_Play(dxSound->primaryBuffer, 0, 0, DSBPLAY_LOOPING);
}

