/*****************************************************************************
** File:
**      JoystickIo.c
** 
** Author:
**      Daniel Vik
**
** Description:
**      Joystick emulation
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "JoystickIO.h"
#include <stdlib.h>
#include <string.h>

extern UInt8 Joystick(UInt8 no);
extern void mouseEmuGetState(int* dx, int* dy);
extern int  mouseEmuGetButtonState(int checkAlways);
extern void SetKanaLed(int enable);

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))


struct JoystickIO {
    AY8910* ay8910;
    struct {
        JoystickIoType type;
        UInt8 buttons;
        int dx;
        int dy;
        UInt8 count;
        UInt32 joyClock;
    } controls[2];
    int mouseAsJoystick;
    UInt32 mouseClock;
    UInt32 data[4];
    UInt8 registers[2];
};

static JoystickIoType joyTypeConfigured[2];
static int joyTypeConfiguredUserData[2];

static void joystickCheckType(JoystickIO* joyIO) {
    int port;

    for (port = 0; port < 2; port++) {
        if (joyIO->controls[port].type != joyTypeConfigured[port]) {
            joyIO->controls[port].type = joyTypeConfigured[port];
            joyIO->controls[port].count = 0;
        }
    }
}

static UInt8 read(JoystickIO* joyIO, UInt16 address, UInt32 cpuClock) 
{
    UInt8 value;
    int joyId;

    if (address & 1) {
        return joyIO->registers[1] & 0xf0;
    }

    joystickCheckType(joyIO);

    /* Number of a joystick port */
    joyId = joyIO->registers[1] & 0x40 ? 1 : 0;

    /* If no joystick, return dummy value */
    if (joyIO->controls[joyId].type == JOYTYPE_NONE) {
        return 0x7f;
    }

    if (joyIO->controls[joyId].type == JOYTYPE_MOUSE) {
        int buttons = mouseEmuGetButtonState(0);
    
        joyIO->controls[joyId].buttons = (~buttons << 4) & 0x30;

        if (joyIO->controls[joyId].count == 1 || 
            (joyIO->controls[joyId].count == 0 && cpuClock - joyIO->mouseClock > 30000)) 
        {
            int dx;
            int dy;

            mouseEmuGetState(&dx, &dy);
            joyIO->mouseClock = cpuClock;
            joyIO->controls[joyId].dx = (dx > 127 ? 127 : (dx < -127 ? -127 : dx));
            joyIO->controls[joyId].dy = (dy > 127 ? 127 : (dy < -127 ? -127 : dy));
        }
    }

    switch (joyIO->controls[joyId].count) {
    case 0:
        if (joyIO->registers[1] & (joyId ? 0x20 : 0x10)) {
            return 0x7f;
        }

        if(joyIO->controls[joyId].type != JOYTYPE_MOUSE) {
            return 0x40 | (~Joystick((UInt8)joyId) & 0x3f);
        }

        value =  joyIO->controls[joyId].buttons | 
            ((joyIO->controls[joyId].dx / 3)  ? 
                ((joyIO->controls[joyId].dx > 0) ? 0x08 : 0x04) : 0x0c) |
            ((joyIO->controls[joyId].dy / 3) ? 
                ((joyIO->controls[joyId].dy > 0) ? 0x02 : 0x01) : 0x03);
        return 0x40 | value;
    case 1: 
        return 0x40 | ((joyIO->controls[joyId].dx >> 4) & 0x0f) | joyIO->controls[joyId].buttons;
    case 2: 
        return 0x40 | (joyIO->controls[joyId].dx & 0x0f) | joyIO->controls[joyId].buttons;
    case 3: 
        return 0x40 | ((joyIO->controls[joyId].dy >> 4) & 0x0f) | joyIO->controls[joyId].buttons;
    case 4: 
        return 0x40 | (joyIO->controls[joyId].dy & 0x0f) | joyIO->controls[joyId].buttons;
    }

    return 0x7f;
}


static void write(JoystickIO* joyIO, UInt16 address, UInt8 value, UInt32 cpuClock) 
{
    if (address & 1) {
        joystickCheckType(joyIO);
        
        if (joyIO->controls[1].type == JOYTYPE_MOUSE && !joyIO->mouseAsJoystick) {
            if (cpuClock - joyIO->controls[1].joyClock > 1000) {
                joyIO->controls[1].count = 0;
            }
            joyIO->controls[1].joyClock = cpuClock;

            if ((value ^ joyIO->registers[1]) & 0x20) {
                joyIO->controls[1].count = 1 + (joyIO->controls[1].count & 3);
            }
        }
        else {
            if ((value & 0x0c) == 0x0c) {
                joyIO->controls[1].count = 0;
            }
        }

        if (joyIO->controls[0].type == JOYTYPE_MOUSE && !joyIO->mouseAsJoystick) {
            if ((value ^ joyIO->registers[1]) & 0x10) {
                if (cpuClock - joyIO->controls[0].joyClock > 1000) {
                    joyIO->controls[0].count = 0;
                }
                joyIO->controls[0].joyClock = cpuClock;

                joyIO->controls[0].count = 1 + (joyIO->controls[0].count & 3);
            }
        }
        else {
            if ((value & 0x03) == 0x03) {
                joyIO->controls[0].count = 0;
            }
        }

        SetKanaLed(0 == (joyIO->registers[1] & 0x80));
    }

    joyIO->registers[address & 1] = value;
}

JoystickIO* joystickIoCreate(AY8910* ay8910)
{
    JoystickIO* joyIO;
    int buttons = mouseEmuGetButtonState(1);

    joyIO = calloc(1, sizeof(JoystickIO));

    joyIO->ay8910 = ay8910;
    joyIO->mouseAsJoystick = buttons & 1;

    ay8910SetIoPort(ay8910, read, write, joyIO);

    SetKanaLed(0);

    return joyIO;
}

void joystickIoDestroy(JoystickIO* joyIO)
{
    ay8910SetIoPort(joyIO->ay8910, NULL, NULL, NULL);

    free(joyIO);
}

void joystickIoSetType(int port, JoystickIoType type, int userData) {
    joyTypeConfiguredUserData[port] = userData;
    joyTypeConfigured[port] = type;
}

JoystickIoType joystickIoGetType(int port, int* userData) {
    *userData = joyTypeConfiguredUserData[port];
    return joyTypeConfigured[port];
}

int joystickIoGetState(JoystickIO* joyIO, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(JoystickIO, controls);
    int size   = sizeof(JoystickIO) - offset;

    joyIO->data[0] = joyTypeConfigured[0];
    joyIO->data[1] = joyTypeConfigured[1];
    joyIO->data[2] = joyTypeConfiguredUserData[0];
    joyIO->data[3] = joyTypeConfiguredUserData[1];

    memcpy(buffer, (char*)joyIO + offset, size);

    return size;
}

int joystickIoSetState(JoystickIO* joyIO, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(JoystickIO, controls);
    int size   = sizeof(JoystickIO) - offset;

    memcpy((char*)joyIO + offset, buffer, size);

    joyTypeConfigured[0]         = joyIO->data[0];
    joyTypeConfigured[1]         = joyIO->data[1];
    joyTypeConfiguredUserData[0] = joyIO->data[2];
    joyTypeConfiguredUserData[1] = joyIO->data[3];

    SetKanaLed(0 == (joyIO->registers[1] & 0x80));

    return size;
}
