/*****************************************************************************
** File:
**      Disk.c
**
** Author:
**      Daniel Vik
** 
** Description:
**      Bios patch for disk support
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "Disk.h"
#include "ziphelper.h"
#include "msx.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


static FILE* drives[2];                  /* Disk image files */
static int   RdOnly[2];                  /* 1 = read-only    */
static char* ramImageBuffer[2];          /* 0 = Use file     */
static int   ramImageSize[2];            /* Size of ram buf  */
extern int   diskBusy[];


UInt8 DiskPresent(int driveId)
{
    return driveId < MAXDRIVES && 
           (drives[driveId] != NULL || ramImageBuffer[driveId] != NULL);
}


UInt8 DiskRead(int driveId, UInt8* buffer, int sector)
{
    if (driveId >= MAXDRIVES) {
        return 0;
    }

    if (ramImageBuffer[driveId] != NULL) {
        int offset = sector * 512;

        if (ramImageSize[driveId] < offset + 512) {
            return 0;
        }

        memcpy(buffer, ramImageBuffer[driveId] + offset, 512);
        diskBusy[driveId] = 1;
        return 1;
    }
    else {
        if ((drives[driveId] != NULL)) {
            if (0 == fseek(drives[driveId], sector * 512, SEEK_SET)) {
                UInt8 success = fread(buffer, 1, 512, drives[driveId]) == 512;
                diskBusy[driveId] = 1;
                return success;
            }
        }
    }
    return 0;
}


UInt8 DiskWrite(int driveId, UInt8 *buffer, int sector)
{
    if (driveId >= MAXDRIVES) {
        return 0;
    }

    if (ramImageBuffer[driveId] != NULL) {
        int offset = sector * 512;

        if (ramImageSize[driveId] < offset + 512) {
            return 0;
        }

        memcpy(ramImageBuffer[driveId] + offset, buffer, 512);
        diskBusy[driveId] = 1;
        return 1;
    }
    else {
        if (drives[driveId] != NULL && !RdOnly[driveId]) {
            if (0 == fseek(drives[driveId], sector * 512, SEEK_SET)) {
                UInt8 success = fwrite(buffer, 1, 512, drives[driveId]) == 512;
                diskBusy[driveId] = 1;
                return success;
            }
        }
    }
    return 0;
}


UInt8 ChangeDisk(int driveId, char* fileName, const char* fileInZipFile)
{
    if (driveId >= MAXDRIVES) {
        return 0;
    }

    /* Close previous disk image */
    if(drives[driveId] != NULL) { 
        fclose(drives[driveId]);
        drives[driveId] = NULL; 
    }

    if (ramImageBuffer[driveId] != NULL) {
        // Flush to file??
        free(ramImageBuffer[driveId]);
        ramImageBuffer[driveId] = NULL;
    }

    if( (!fileName) || ( strlen(fileName)==0) ) {
        return 1;
    }

    if (fileInZipFile != NULL) {
        ramImageBuffer[driveId] = zipLoadFile(fileName, fileInZipFile, &ramImageSize[driveId]);
        return ramImageBuffer[driveId] != NULL;
    }

    drives[driveId] = fopen(fileName, "r+b");
    RdOnly[driveId] = 0;

    if (drives[driveId] == NULL) {
        drives[driveId] = fopen(fileName, "rb");
        RdOnly[driveId] = 1;
    }

    return drives[driveId] != 0;
}

void ReopenDisks( DeviceInfo *devInfo )
{
	int id ;

	for ( id = 0 ; id < MAXDRIVES ; id++ )
	{
        ChangeDisk( id, devInfo->diskette[id].name, NULL ) ;
	}

}