/*****************************************************************************
** File:
**      MachineDefaults.c
**
** Author:
**      Daniel Vik
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "MachineDefaults.h"
#include "Machine.h"
#include <stdlib.h>
#include <xtl.h>
#include <direct.h>

static void machineNameInit(Machine* machine, char* machineName)
{
    strcpy(machine->name, machineName);
}

static void subslottedInit(Machine* machine, int slot0, int slot1, int slot2, int slot3) 
{
    machine->slot[0].subslotted = slot0;
    machine->slot[1].subslotted = slot1;
    machine->slot[2].subslotted = slot2;
    machine->slot[3].subslotted = slot3;
}

static void externalSlotInit(Machine* machine, int slotA, int subslotA, int slotB, int subslotB) 
{
    machine->cart[0].slot    = slotA;
    machine->cart[0].subslot = subslotA;
    machine->cart[1].slot    = slotB;
    machine->cart[1].subslot = subslotB;
}

static void videoInit(Machine* machine, VdpVersion vdpVersion, int vramSize)
{
    machine->video.vdpVersion = vdpVersion;
    machine->video.vramSize   = vramSize;
}

static void audioInit(Machine* machine, int enableYM2413, int enableY8950, int enableMoonsound)
{
    machine->audio.enableYM2413    = enableYM2413;
    machine->audio.enableY8950     = enableY8950;
    machine->audio.enableMoonsound = enableMoonsound;
    machine->audio.moonsoundSRAM   = 640;
}

static void slotInit(Machine* machine, int config, RomType romType,
                     int slot, int subslot, int startPage, int pageCount, char* romName)
{
    machine->slotInfo[config].romType   = romType;
    machine->slotInfo[config].slot      = slot;
    machine->slotInfo[config].subslot   = subslot;
    machine->slotInfo[config].startPage = startPage;
    machine->slotInfo[config].pageCount = pageCount;
    strcpy(machine->slotInfo[config].name, romName);
    strcpy(machine->slotInfo[config].inZipName, "");

    if (config >= machine->slotInfoCount) {
        machine->slotInfoCount = config + 1;
    }
}

Machine* createMachineMSX() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX");
    subslottedInit(machine, 0, 0, 0, 0);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_TMS9929A, 0x4000);
    audioInit(machine, 0, 0, 0);

    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX.rom");
    slotInit(machine, config++, ROM_DISK,   2, 0, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, RAM_NORMAL, 3, 0, 0, 8, "");

    return machine;
}


Machine* createMachineMSXBR() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX - Brazilian");
    subslottedInit(machine, 0, 0, 0, 0);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_TMS99x8A, 0x4000);
    audioInit(machine, 0, 0, 0);

    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSXBR.rom");
    slotInit(machine, config++, ROM_DISK,   2, 0, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, RAM_NORMAL, 3, 0, 0, 8, "");

    return machine;
}


Machine* createMachineMSXJ() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX - Japanese");
    subslottedInit(machine, 0, 0, 0, 0);
    externalSlotInit(machine, 2, 0, 3, 0);
    videoInit(machine, VDP_TMS99x8A, 0x4000);
    audioInit(machine, 0, 0, 0);

    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSXJ.rom");
    slotInit(machine, config++, RAM_NORMAL, 1, 0, 0, 8, "");
    slotInit(machine, config++, ROM_DISK,   3, 0, 2, 2, "internal/DISK.rom");

    return machine;
}


Machine* createMachineMSXKR() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX - Korean");
    subslottedInit(machine, 0, 0, 0, 0);
    externalSlotInit(machine, 1, 0, 3, 0);
    videoInit(machine, VDP_TMS99x8A, 0x4000);
    audioInit(machine, 0, 0, 0);

    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSXKR.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 4, 1, "internal/MSXHAN.rom");
    slotInit(machine, config++, RAM_NORMAL, 2, 0, 0, 8, "");
    slotInit(machine, config++, ROM_DISK,   3, 0, 2, 2, "internal/DISK.rom");

    return machine;
}

Machine* createMachineMSX2() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2");
    subslottedInit(machine, 0, 0, 0, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2.rom");
    slotInit(machine, config++, ROM_FMPAC,  3, 0, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 0, 2, "internal/MSX2EXT.rom");
    slotInit(machine, config++, ROM_DISK,   3, 1, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, RAM_MAPPER, 3, 2, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");

    return machine;
}

Machine* createMachineMSX2A() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2 - Arabic");
    subslottedInit(machine, 1, 0, 1, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_KANJI,  0, 0, 0, 0, "internal/KANJI.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 1, 0, 2, "internal/MSX2EXT.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 1, 2, 4, "internal/ARABIC.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 2, 2, 4, "internal/SWP.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 3, 0, 8, "internal/PAINT.rom");
    slotInit(machine, config++, ROM_FMPAC,  2, 2, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, ROM_NORMAL, 2, 3, 2, 2, "internal/XBASIC2.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 2, 4, "internal/MSXKANJI.rom");
    slotInit(machine, config++, RAM_MAPPER, 3, 2, 0, 32, "");
    slotInit(machine, config++, ROM_DISK,   3, 3, 2, 2, "internal/DISK.rom");

    return machine;
}

Machine* createMachineMSX2BR() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2 - Brazilian");
    subslottedInit(machine, 0, 0, 0, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2BR.rom");
    slotInit(machine, config++, ROM_FMPAC,  3, 0, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 0, 2, "internal/MSX2BREXT.rom");
    slotInit(machine, config++, ROM_DISK,   3, 1, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, RAM_MAPPER, 3, 2, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");

    return machine;
}

Machine* createMachineMSX2FR() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2 - French");
    subslottedInit(machine, 0, 0, 0, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2FR.rom");
    slotInit(machine, config++, ROM_FMPAC,  3, 0, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 0, 2, "internal/MSX2FREXT.rom");
    slotInit(machine, config++, ROM_DISK,   3, 1, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, RAM_MAPPER, 3, 2, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");

    return machine;
}

Machine* createMachineMSX2G() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2 - German");
    subslottedInit(machine, 0, 0, 0, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2G.rom");
    slotInit(machine, config++, ROM_FMPAC,  3, 0, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 0, 2, "internal/MSX2GEXT.rom");
    slotInit(machine, config++, ROM_DISK,   3, 1, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, RAM_MAPPER, 3, 2, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");

    return machine;
}

Machine* createMachineMSX2J() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2 - Japanese");
    subslottedInit(machine, 1, 0, 0, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2J.rom");
    slotInit(machine, config++, ROM_KANJI,  0, 0, 0, 0, "internal/KANJI.rom");
    slotInit(machine, config++, ROM_FMPAC,  0, 2, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, RAM_MAPPER, 3, 0, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 0, 2, "internal/MSX2JEXT.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 2, 4, "internal/MSXKANJI.rom");
    slotInit(machine, config++, ROM_DISK,   3, 2, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");

    return machine;
}

Machine* createMachineMSX2KR() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2 - Korean");
    subslottedInit(machine, 1, 0, 0, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2KR.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 1, 2, 4, "internal/MSX2HAN.rom");
    slotInit(machine, config++, RAM_MAPPER, 0, 2, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 0, 3, 0, 4, "internal/MSX2KREXT.rom");
    slotInit(machine, config++, ROM_FMPAC,  3, 1, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, ROM_DISK,   3, 2, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");


    return machine;
}

Machine* createMachineMSX2R() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2 - Russian");
    subslottedInit(machine, 0, 0, 0, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9938, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2R.rom");
    slotInit(machine, config++, ROM_FMPAC,  3, 0, 2, 2, "internal/FMPAC.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 0, 2, "internal/MSX2REXT.rom");
    slotInit(machine, config++, ROM_DISK,   3, 1, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, RAM_MAPPER, 3, 2, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");

    return machine;
}

Machine* createMachineMSX2P() 
{
    Machine* machine = calloc(1, sizeof(Machine));
    int config = 0;

    machineNameInit(machine, "MSX2+");
    subslottedInit(machine, 1, 0, 1, 1);
    externalSlotInit(machine, 1, 0, 2, 0);
    videoInit(machine, VDP_V9958, 0x20000);
    audioInit(machine, 1, 1, 1);
    
    slotInit(machine, config++, ROM_NORMAL, 0, 0, 0, 4, "internal/MSX2P.rom");
    slotInit(machine, config++, ROM_F4INVERTED,  0, 0, 0, 0, "");
    slotInit(machine, config++, ROM_KANJI,  0, 0, 0, 0, "internal/KANJI.rom");
    slotInit(machine, config++, ROM_NORMAL, 0, 2, 2, 2, "internal/MSX2PMUS.rom");
    slotInit(machine, config++, ROM_PAC,    2, 2, 2, 2, "");
    slotInit(machine, config++, RAM_MAPPER, 3, 0, 0, 32, "");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 0, 2, "internal/MSX2PEXT.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 1, 2, 4, "internal/MSXKANJI.rom");
    slotInit(machine, config++, ROM_DISK,   3, 2, 2, 2, "internal/DISK.rom");
    slotInit(machine, config++, ROM_NORMAL, 3, 3, 2, 2, "internal/XBASIC2.rom");

    return machine;
}


int isDefaultMSX2Machine(char* machineName)
{
    return 0 == strcmp(machineName, "MSX2") ||
           0 == strcmp(machineName, "MSX2 - Arabic") ||
           0 == strcmp(machineName, "MSX2 - Brazilian") ||
           0 == strcmp(machineName, "MSX2 - Japanese") ||
           0 == strcmp(machineName, "MSX2 - Korean") ||
           0 == strcmp(machineName, "MSX2 - Russian") ||
           0 == strcmp(machineName, "MSX2+");
}
