/*****************************************************************************
** File:
**      Machine.c
**
** Author:
**      Daniel Vik
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "Machine.h"
#include "MachineDefaults.h"
#include <stdlib.h>
#include <xtl.h>
#include <direct.h>


extern char* getCmdArgument(char* szLine, int argNo);
int toint(char* buffer) 
{
    int i;

    if (buffer == NULL) {
        return -1;
    }

    for (i = 0; buffer[i]; i++) {
        if (!isdigit(buffer[i])) return -1;
    }

    return atoi(buffer);
}

int readMachine(Machine* machine, char* machineName, char* file)
{
#if 0
    static char buffer[10000];
    char* slotBuf;
    int value;
    int i = 0;

    strcpy(machine->name, machineName);

    // Read audio info
    GetPrivateProfileString("Audio", "YM2413", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "0")) machine->audio.enableYM2413 = 0;
    else if (0 == strcmp(buffer, "1")) machine->audio.enableYM2413 = 1;
    else return 0;

    GetPrivateProfileString("Audio", "Y8950", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "0")) machine->audio.enableY8950 = 0;
    else if (0 == strcmp(buffer, "1")) machine->audio.enableY8950 = 1;
    else return 0;
 
    GetPrivateProfileString("Audio", "Moonsound", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "none")) machine->audio.enableMoonsound = isDefaultMSX2Machine(machineName);
    if (0 == strcmp(buffer, "0")) machine->audio.enableMoonsound = 0;
    else if (0 == strcmp(buffer, "1")) machine->audio.enableMoonsound = 1;
    else return 0;

    GetPrivateProfileString("Audio", "Moonsound SRAM", "none", buffer, 10000, file);
    if (0 == sscanf(buffer, "%dkB", &value)) {
        value = 640;
    }

    machine->audio.moonsoundSRAM = value;

    // Read video info
    GetPrivateProfileString("Video", "version", "none", buffer, 10000, file);
    if      (0 == strcmp(buffer, "V9938"))    machine->video.vdpVersion = VDP_V9938;
    else if (0 == strcmp(buffer, "V9958"))    machine->video.vdpVersion = VDP_V9958;
    else if (0 == strcmp(buffer, "TMS9929A")) machine->video.vdpVersion = VDP_TMS9929A;
    else if (0 == strcmp(buffer, "TMS99x8A")) machine->video.vdpVersion = VDP_TMS99x8A;
    else return 0;

    GetPrivateProfileString("Video", "vram size", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "16kB")) machine->video.vramSize = 16 * 1024;
    else if (0 == strcmp(buffer, "64kB")) machine->video.vramSize = 64 * 1024;
    else if (0 == strcmp(buffer, "128kB")) machine->video.vramSize = 128 * 1024;
    else return 0;

    // Read subslot info
    GetPrivateProfileString("Subslotted Slots", "slot 0", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "0")) machine->slot[0].subslotted = 0;
    else if (0 == strcmp(buffer, "1")) machine->slot[0].subslotted = 1;
    else return 0;
    
    GetPrivateProfileString("Subslotted Slots", "slot 1", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "0")) machine->slot[1].subslotted = 0;
    else if (0 == strcmp(buffer, "1")) machine->slot[1].subslotted = 1;
    else return 0;
    
    GetPrivateProfileString("Subslotted Slots", "slot 2", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "0")) machine->slot[2].subslotted = 0;
    else if (0 == strcmp(buffer, "1")) machine->slot[2].subslotted = 1;
    else return 0;
    
    GetPrivateProfileString("Subslotted Slots", "slot 3", "none", buffer, 10000, file);
    if (0 == strcmp(buffer, "0")) machine->slot[3].subslotted = 0;
    else if (0 == strcmp(buffer, "1")) machine->slot[3].subslotted = 1;
    else return 0;

    // Read external slot info
    GetPrivateProfileString("External Slots", "slot A", "none", buffer, 10000, file);
    machine->cart[0].slot = toint(getCmdArgument(buffer, 0));        
    machine->cart[0].subslot = toint(getCmdArgument(buffer, 1));    
    if (machine->cart[0].slot < 0 || machine->cart[0].slot >= 4) return 0;
    if (machine->cart[0].subslot < 0 || machine->cart[0].subslot >= 4) return 0;    

    GetPrivateProfileString("External Slots", "slot B", "none", buffer, 10000, file);
    machine->cart[1].slot = toint(getCmdArgument(buffer, 0));        
    machine->cart[1].subslot = toint(getCmdArgument(buffer, 1));    
    if (machine->cart[1].slot < 0 || machine->cart[1].slot >= 4) return 0;
    if (machine->cart[1].subslot < 0 || machine->cart[1].subslot >= 4) return 0;    

    // Read slots
    GetPrivateProfileSection("Slots", buffer, 10000, file);

    slotBuf = buffer;
    for (i = 0; i < sizeof(machine->slotInfo) / sizeof(SlotInfo) && *slotBuf; i++) {
        char* arg;

        machine->slotInfo[i].slot = toint(getCmdArgument(slotBuf, 0));        
        machine->slotInfo[i].subslot = toint(getCmdArgument(slotBuf, 1));
        machine->slotInfo[i].startPage = toint(getCmdArgument(slotBuf, 2));
        machine->slotInfo[i].pageCount = toint(getCmdArgument(slotBuf, 3));
        machine->slotInfo[i].romType = toint(getCmdArgument(slotBuf, 4));
        arg = getCmdArgument(slotBuf, 5);
        strcpy(machine->slotInfo[i].name, arg ? arg : "");
        arg = getCmdArgument(slotBuf, 6);
        strcpy(machine->slotInfo[i].inZipName, arg ? arg : "");

        if (machine->slotInfo[i].slot < 0 || machine->slotInfo[i].slot >= 4) return 0;
        if (machine->slotInfo[i].subslot < 0 || machine->slotInfo[i].subslot >= 4) return 0;
        if (machine->slotInfo[i].startPage < 0 || machine->slotInfo[i].startPage >= 8) return 0;
        if (machine->slotInfo[i].pageCount == -1) return 0;
        if (machine->slotInfo[i].romType < 1 || machine->slotInfo[i].romType > ROM_MAXROMID) return 0;

        slotBuf += strlen(slotBuf) + 1;
    }

    machine->slotInfoCount = i;
#endif 
    return 1;

}

void writeMachine(Machine* machine)
{
#if 0
    char dir[128];
    char file[128];
    char buffer[10000];
    int size = 0;
    int i;

	sprintf(dir, "X:\\Config\\%s", machine->name);

	CreateDirectory( dir, NULL ) ;

    //mkdir(dir);

	sprintf(file, "X:\\Config\\%s\\config.ini", machine->name);

    // Write audio info
    WritePrivateProfileString("Audio", "YM2413", machine->audio.enableYM2413 ? "1" : "0", file);
    WritePrivateProfileString("Audio", "Y8950", machine->audio.enableY8950 ? "1" : "0", file);
    WritePrivateProfileString("Audio", "Moonsound", machine->audio.enableMoonsound ? "1" : "0", file);

    if (machine->audio.moonsoundSRAM < 1000) {
        sprintf(buffer, "%dkB", machine->audio.moonsoundSRAM);
    }
    else {
        sprintf(buffer, "%dMB", machine->audio.moonsoundSRAM / 1024);
    }
    WritePrivateProfileString("Audio", "Moonsound SRAM", buffer, file);

    // Write video info
    switch (machine->video.vdpVersion) {
        case VDP_V9958:     WritePrivateProfileString("Video", "version", "V9958", file); break;
        case VDP_V9938:     WritePrivateProfileString("Video", "version", "V9938", file); break;
        case VDP_TMS9929A:  WritePrivateProfileString("Video", "version", "TMS9929A", file); break;
        case VDP_TMS99x8A:  WritePrivateProfileString("Video", "version", "TMS99x8A", file); break;
    }

    sprintf(buffer, "%dkB", machine->video.vramSize / 0x400);
    WritePrivateProfileString("Video", "vram size", buffer, file);

    // Write subslot info
    WritePrivateProfileString("Subslotted Slots", "slot 0", machine->slot[0].subslotted ? "1" : "0", file);
    WritePrivateProfileString("Subslotted Slots", "slot 1", machine->slot[1].subslotted ? "1" : "0", file);
    WritePrivateProfileString("Subslotted Slots", "slot 2", machine->slot[2].subslotted ? "1" : "0", file);
    WritePrivateProfileString("Subslotted Slots", "slot 3", machine->slot[3].subslotted ? "1" : "0", file);

    // Write external slot info
    sprintf(buffer, "%d %d", machine->cart[0].slot, machine->cart[0].subslot);
    WritePrivateProfileString("External Slots", "slot A", buffer, file);
    sprintf(buffer, "%d %d", machine->cart[1].slot, machine->cart[1].subslot);
    WritePrivateProfileString("External Slots", "slot B", buffer, file);

    // Write slots
    for (i = 0; i < machine->slotInfoCount; i++) {
        size += sprintf(buffer + size, "%d %d %d %d %d \"%s\" \"%s\"",
                        machine->slotInfo[i].slot,
                        machine->slotInfo[i].subslot,
                        machine->slotInfo[i].startPage,
                        machine->slotInfo[i].pageCount,
                        machine->slotInfo[i].romType,
                        machine->slotInfo[i].name,
                        machine->slotInfo[i].inZipName);
        buffer[size++] = 0;
    }

    buffer[size++] = 0;
    buffer[size++] = 0;

    WritePrivateProfileString("Slots", NULL, NULL, file);
    WritePrivateProfileSection("Slots", buffer, file);
#endif
}

void initDefaultMachines()
{
#ifndef NO_BIOS_ROMS_INCLUDE
#if 0
	CreateDirectory( "X:\\Config", NULL ) ;
    writeMachine(createMachineMSX());
    writeMachine(createMachineMSXBR());
    writeMachine(createMachineMSXJ());
    writeMachine(createMachineMSXKR());
    writeMachine(createMachineMSX2());
    writeMachine(createMachineMSX2A());
    writeMachine(createMachineMSX2BR());
    writeMachine(createMachineMSX2FR());
    writeMachine(createMachineMSX2G());
    writeMachine(createMachineMSX2J());
    writeMachine(createMachineMSX2KR());
    writeMachine(createMachineMSX2R());
    writeMachine(createMachineMSX2P());
#endif
#endif
}

Machine* createMachine(char* machineName)
{
    char fileName[512];
    Machine* machine;
    int success;

	if ( strcmp( machineName, "MSX" ) == 0 )
		return createMachineMSX() ;
	if ( strcmp( machineName, "MSX - Brazilian" ) == 0 )
		return createMachineMSXBR();
	if ( strcmp( machineName, "MSX - Japanese" ) == 0 )
		return createMachineMSXJ();
	if ( strcmp( machineName, "MSX - Korean" ) == 0 )
		return createMachineMSXKR();
	if ( strcmp( machineName, "MSX2" ) == 0 )
		return createMachineMSX2();
	if ( strcmp( machineName, "MSX2 - Arabic" ) == 0 )
		return createMachineMSX2A();
	if ( strcmp( machineName, "MSX2 - Brazilian" ) == 0 )
		return createMachineMSX2BR();
	if ( strcmp( machineName, "MSX2 - French" ) == 0 )
		return createMachineMSX2FR();
	if ( strcmp( machineName, "MSX2 - German" ) == 0 )
		return createMachineMSX2G();
	if ( strcmp( machineName, "MSX2 - Japanese" ) == 0 )
		return createMachineMSX2J();
	if ( strcmp( machineName, "MSX2 - Korean" ) == 0 )
		return createMachineMSX2KR();
	if ( strcmp( machineName, "MSX2 - Russian" ) == 0 )
		return createMachineMSX2R();
	if ( strcmp( machineName, "MSX2+" ) == 0 )
		return createMachineMSX2P();

	return NULL ;



#if 0
    machine = malloc(sizeof(Machine));

	sprintf(fileName, "X:\\Config\\%s\\config.ini", machineName);
    success = readMachine(machine, machineName, fileName);
    if (!success) {
        free(machine);
        return NULL;
    }

    machineUpdate(machine);

    return machine;
#endif
}

void destroyMachine(Machine* machine)
{
    free(machine);
}


char** machineGetAvailable()
{
    static char* machineNames[64];
    static char  names[64][64];
	HANDLE          handle;
	WIN32_FIND_DATA wfd;
    int index = 0;
    BOOL cont = TRUE;

    handle = FindFirstFile("Config/*", &wfd);

    if (handle == INVALID_HANDLE_VALUE) {
        machineNames[0] = NULL;
        return machineNames;
    }

    while (cont) {
        char fileName[128];

		DWORD fa = GetFileAttributes(wfd.cFileName);
        if (fa & FILE_ATTRIBUTE_DIRECTORY) {
            FILE* file;
		    sprintf(fileName, "Config/%s/config.ini", wfd.cFileName);
            file = fopen(fileName, "rb");
            if (file != NULL) {
                strcpy(names[index], wfd.cFileName);
                machineNames[index] = names[index];
                fclose(file);
                index++;
            }
            
        }
        cont = FindNextFile(handle, &wfd);
    }

	FindClose(handle);
    
    machineNames[index] = NULL;

    return machineNames;
}


int machineIsValid(char* machineName)
{
    char ** machineNames = machineGetAvailable();

    while (machineNames != NULL) {
        if (0 == strcmp(*machineNames, machineName)) {
            return 1;
        }
        machineNames++;
    }
    return 0;
}

void machineUpdate(Machine* machine)
{
    int entry;
    int i;

    for (entry = 0; entry < machine->slotInfoCount; entry++) {
        machine->slotInfo[entry].error = 0;
        if (machine->slotInfo[entry].subslot && !machine->slot[machine->slotInfo[entry].slot].subslotted) {
            machine->slotInfo[entry].error = 1;
        }
        if (machine->slotInfo[entry].pageCount > 0) {
            for (i = 0; i < entry; i++) {
                if (machine->slotInfo[i].slot    == machine->slotInfo[entry].slot &&
                    machine->slotInfo[i].subslot == machine->slotInfo[entry].subslot &&
                    machine->slotInfo[i].pageCount > 0)
                {
                    int tstStart = machine->slotInfo[entry].startPage;
                    int tstEnd   = tstStart + machine->slotInfo[entry].pageCount;
                    int start    = machine->slotInfo[i].startPage;
                    int end      = start + machine->slotInfo[i].pageCount;
                    if (tstStart >= start && tstStart < end) {
                        machine->slotInfo[entry].error = 1;
                    }
                    if (tstEnd > start && tstEnd <= end) {
                        machine->slotInfo[entry].error = 1;
                    }
                }
            }
        }
    }

    for (i = 0; i < machine->slotInfoCount; i++) {
        for (entry = 0; entry < machine->slotInfoCount - 1; entry++) {
            SlotInfo* si1 = &machine->slotInfo[entry];
            SlotInfo* si2 = &machine->slotInfo[entry + 1];
            int tst1 = (si1->slot << 24) + (si1->subslot << 20) + 
                       (si1->startPage << 12) + si1->pageCount;
            int tst2 = (si2->slot << 24) + (si2->subslot << 20) + 
                       (si2->startPage << 12) + si2->pageCount;

            if (tst2 < tst1) {
                SlotInfo tmp;
                memcpy(&tmp, si1, sizeof(SlotInfo));
                memcpy(si1, si2, sizeof(SlotInfo));
                memcpy(si2, &tmp, sizeof(SlotInfo));
            }
        }
    }

    // Check VRAM size
    if (machine->video.vdpVersion == VDP_V9938) {
        if (machine->video.vramSize >= 128 * 1024) {
            machine->video.vramSize = 128 * 1024;
        }
        else {
            machine->video.vramSize = 64 * 1024;
        }
    }
    else if (machine->video.vdpVersion == VDP_V9958) {
        machine->video.vramSize = 128 * 1024;
    }
    else {
        machine->video.vramSize = 16 * 1024;
    }

    // Check Moonsound SRAM size
    if      (machine->audio.moonsoundSRAM <=  128) machine->audio.moonsoundSRAM = 128;
    else if (machine->audio.moonsoundSRAM <=  256) machine->audio.moonsoundSRAM = 256;
    else if (machine->audio.moonsoundSRAM <=  640) machine->audio.moonsoundSRAM = 640;
    else if (machine->audio.moonsoundSRAM <= 1024) machine->audio.moonsoundSRAM = 1024;
    else                                           machine->audio.moonsoundSRAM = 2048;
}
