/*****************************************************************************
** File:
**      VDP.c
**
** Author:
**      Daniel Vik
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "VDP.h"
#include "V9938.h"
#include "IoPort.h"
#include <string.h>


#define HPERIOD      1368
#define HREFRESH_240 960           /* 240dot scanline refresh  */
#define HREFRESH_256 1024          /* 256dot scanline refresh  */

#define CPU_HPERIOD  (HPERIOD/6)
#define CPU_H240     (10+HREFRESH_240/6)
#define CPU_H256     (8+HREFRESH_256/6)

#define INT_IE0     0x01
#define INT_IE1     0x02
#define INT_IE2     0x04

#define VRAM_SIZE (128 * 1024)

static int vramAddr;
#define MAP_VRAM(addr) (VRAM + (vramAddr = addr, ScrMode >= 7 && ScrMode <= 8 ? (vramAddr >> 1 | ((vramAddr & 1) << 16)) : vramAddr))
#define MAP_VRAM_16(addr) (VRAM + (addr))
#define MAP_VRAM_78(addr) (VRAM + (vramAddr = addr, vramAddr >> 1 | ((vramAddr & 1) << 16)))
#define IS_VRAM_78() (ScrMode >= 7 && ScrMode <= 8)

#define vdpIsSpritesBig(regs)        (regs[1]  & 0x01)
#define vdpIsSprites16x16(regs)      (regs[1]  & 0x02)
#define vdpIsSpritesOff(regs)        (regs[8]  & 0x02)
#define vdpIsScreenOn(regs)          (regs[1]  & 0x40)
#define vdpIsColor0Solid(regs)       (regs[8]  & 0x20)
#define vdpIsVideoPal(regs)          (regs[9]  & 0x02)
#define vdpIsOddPage(regs)           (regs[9]  & 0x04)
#define vdpIsInterlaceOn(regs)       (regs[9]  & 0x08)
#define vdpIsScanLines212(regs)      (regs[9]  & 0x80)
#define vdpIsEdgeMasked(regs)        (regs[25] & 0x02)
#define vdpIsModeYJK(regs)           (regs[25] & 0x08)
#define vdpIsModeYAE(regs)           (regs[25] & 0x10)
#define VScroll       VDP[23]
#define HScroll       ((((int)(VDP[26]&0x3F&~(~HScroll512<<5)))<<3)-(int)(VDP[27]&0x07))
#define HScroll512    (VDP[25]&(VDP[2]>>5)&0x1)
#define VAdjust       (-((signed char)(VDP[18])>>4))
#define HAdjust       (-((signed char)(VDP[18]<<4)>>4))

#define MAXSCREEN   12      /* Highest screen mode supported */
#define MAXSPRITE1  4       /* Sprites/line in SCREEN 1-3    */
#define MAXSPRITE2  8       /* Sprites/line in SCREEN 4-8    */


/** VDP Address Register Masks *******************************/
static struct { UInt8 R2,R3,R4,R5,M2,M3,M4,M5; } VDPmask[MAXSCREEN + 2] =
{
    { 0x7F,0x00,0x3F,0x00,0x00,0x00,0x00,0x00 }, /* SCR 0:  TEXT 40x24  */
    { 0x7F,0xFF,0x3F,0xFF,0x00,0x00,0x00,0x00 }, /* SCR 1:  TEXT 32x24  */
    { 0x7F,0x80,0x3C,0xFF,0x00,0x7F,0x03,0x00 }, /* SCR 2:  BLK 256x192 */
    { 0x7F,0x00,0x3F,0xFF,0x00,0x00,0x00,0x00 }, /* SCR 3:  64x48x16    */
    { 0x7F,0x80,0x3C,0xFC,0x00,0x7F,0x03,0x03 }, /* SCR 4:  BLK 256x192 */
    { 0x60,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 5:  256x192x16  */
    { 0x60,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 6:  512x192x4   */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 7:  512x192x16  */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 8:  256x192x256 */
    { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 }, /* SCR 9:  NONE        */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 10: YAE 256x192 */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 11: YAE 256x192 */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 12: YJK 256x192 */
    { 0x7C,0xF8,0x3F,0x00,0x03,0x07,0x00,0x00 }  /* SCR 0:  TEXT 80x24  */
};

static const byte registerMaskMSX1[8] = {
	0x03, 0xfb, 0x0f, 0xff, 0x07, 0x7f, 0x07, 0xff
};

static const byte registerMaskMSX2[64] = {
	0x7e, 0x7b, 0x7f, 0xff, 0x3f, 0xff, 0x3f, 0xff,
	0xfb, 0xbf, 0x07, 0x03, 0xff, 0xff, 0x07, 0x0f,
	0x0f, 0xbf, 0xff, 0xff, 0x3f, 0x3f, 0x3f, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

static const byte registerMaskMSX2p[64] = {
	0x7e, 0x7b, 0x7f, 0xff, 0x3f, 0xff, 0x3f, 0xff,
	0xfb, 0xbf, 0x07, 0x03, 0xff, 0xff, 0x07, 0x0f,
	0x0f, 0xbf, 0xff, 0xff, 0x3f, 0x3f, 0x3f, 0xff,
    0x00, 0x7f, 0x3f, 0x07, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

static UInt32 msx1Palette[16] = {
    0x000000, 0x000000, 0x21c842, 0x5edc78, 0x5455ed, 0x7d76fc, 0xd4524d, 0x42ebf5,
    0xfc5554, 0xff7978, 0xd4c154, 0xe6ce80, 0x21b03b, 0xc95bba, 0xcccccc, 0xffffff
};

static UInt32 defaultPalette[16] = {
    0x000000, 0x000000, 0x24da24, 0x68ff68, 0x2424ff, 0x4868ff, 0xb62424, 0x48daff,
    0xff2424, 0xff6868, 0xdada24, 0xdada91, 0x249124, 0xda48b6, 0xb6b6b6, 0xffffff
};


static void RefreshLineTx80(int, int);
static void RefreshLine0(int, int);
static void RefreshLine1(int, int);
static void RefreshLine2(int, int);
static void RefreshLine3(int, int);
static void RefreshLine4(int, int);
static void RefreshLine5(int, int);
static void RefreshLine6(int, int);
static void RefreshLine7(int, int);
static void RefreshLine8(int, int);
static void RefreshLine10(int, int);
static void RefreshLine12(int, int);
static void SetNewDrawPage();

/** Screen Mode Handlers [number of screens + 1] *************/
static void (*RefreshLine[MAXSCREEN+2])(int, int) =
{
    RefreshLine0,   /* SCR 0:  TEXT 40x24  */
    RefreshLine1,   /* SCR 1:  TEXT 32x24  */
    RefreshLine2,   /* SCR 2:  BLK 256x192 */
    RefreshLine3,   /* SCR 3:  64x48x16    */
    RefreshLine4,   /* SCR 4:  BLK 256x192 */
    RefreshLine5,   /* SCR 5:  256x192x16  */
    RefreshLine6,   /* SCR 6:  512x192x4   */
    RefreshLine7,   /* SCR 7:  512x192x16  */
    RefreshLine8,   /* SCR 8:  256x192x256 */
    0,              /* SCR 9:  NONE        */
    RefreshLine10,  /* SCR 10: YAE 256x192 */
    RefreshLine10,  /* SCR 11: YAE 256x192 */
    RefreshLine12,  /* SCR 12: YJK 256x192 */
    RefreshLineTx80 /* SCR 0:  TEXT 80x24  */
};


static int    scr0splitLine;
static int    sprOffset;
static int    scrLine;
static UInt32 curSystemTime;
static int    VRAMpage;
static int    ChrTabO;
static int    ColTabO;
static int    ChrGenO;
static int    SprGenO;
static int    SprTabO;
static int    ChrGenM;
static int    ChrTabM;
static int    ColTabM;
static int    SprTabM;
static UInt16 VAddr;
static UInt8  VKey;
static UInt8  PKey;
static UInt8  WKey;
static UInt8  FGColor;
static UInt8  BGColor;
static UInt8  XFGColor;
static UInt8  XBGColor;
static int    ScanLine;
static UInt8  VDPData;
static UInt8  PLatch;
static UInt8  ALatch;
static UInt8  BFlag;
static UInt8  BCount;
static UInt8  Drawing;
static int    Palette[16];
static UInt32 evenOddPage;
static int    VRAMPages;
static UInt8  pendingIRQ;
static int    vdpMSX1;
static int    PALVideo;
static Z80*   z80;
static int    lineOffset;
static UInt32 lineStartTime;
static int firstLine;
static int firstLineOffset;
int loopTime;
int lineTime;
int phase;

UInt8  VRAM[VRAM_SIZE];
UInt8  VDP[64];
UInt8  VDPStatus[16];
int    ScrMode;


void SetIRQ(UInt8 IRQ)
{
    if (IRQ & 0x80) {
        pendingIRQ &= IRQ; 
    }
    else {
        pendingIRQ |= IRQ;
    }
    
    z80SetInt(z80, pendingIRQ);
}


static UInt8 SetScreen(void)
{
    switch (((VDP[0] & 0x0E)>>1) | (VDP[1] & 0x18)) {
    case 0x10: ScrMode = 0; break;
    case 0x00: ScrMode = 1; break;
    case 0x01: ScrMode = 2; break;
    case 0x08: ScrMode = 3; break;
    case 0x02: ScrMode = 4; break;
    case 0x03: ScrMode = 5; break;
    case 0x04: ScrMode = 6; break;
    case 0x05: ScrMode = 7; break;
    case 0x07: ScrMode = 8; break;
    case 0x12: ScrMode = MAXSCREEN + 1; break;
    }

    ChrTabO = (int)(VDP[2] & VDPmask[ScrMode].R2) << 10;
    ChrTabM = (((int)(VDP[2] | ~VDPmask[ScrMode].M2) << 10) | ((1 << 10) - 1)) & 0x7FFF;

    scr0splitLine = (ScanLine - firstLine) & ~7;
    sprOffset = ScrMode == 5 ? 0 : 1;
    scrLine = 0;

    ChrGenO = (int)(VDP[4] & VDPmask[ScrMode].R4) << 11;
    ChrGenM = ((int)(VDP[4] | ~VDPmask[ScrMode].M4) << 11) | 0x007FF;

    ColTabO = ((int)(VDP[3] & VDPmask[ScrMode].R3) << 6) + ((int)VDP[10] << 14);
    ColTabM = ((int)(VDP[3] | ~VDPmask[ScrMode].M3) << 6) | 0x1C03F;

    SprTabO = ((VDP[11] << 15) | (VDP[5] << 7) | ~(-1 << 7)) & ((VRAMPages << 14) - 1);
    SprGenO = ((VDP[6] << 11) | ~(-1 << 11)) & ((VRAMPages << 14) - 1);

    return ScrMode;
}

static void VDPOut(UInt8 reg, UInt8 value)
{ 
    if (vdpMSX1) {
        reg &= 0x07;
        value &= registerMaskMSX1[reg];
    }
    else {
        reg &= 0x3f;
        value &= registerMaskMSX2p[reg];
    }

    switch (reg) {
    case 0: 
        /* Reset HBlank interrupt if disabled */
        if ((VDPStatus[1] & 0x01) && !(value & 0x10)) {
            VDPStatus[1] &= 0xFE;
            SetIRQ(~INT_IE1);
        }

        /* Set screen mode */
        if ((VDP[0] ^ value)) {
            VDP[0] = value;
            SetScreen(); 
        }
        break;

    case 1: 
        /* Set/Reset VBlank interrupt if enabled or disabled */
        if (VDPStatus[0] & 0x80) {
            SetIRQ(value & 0x20 ? INT_IE0 : ~INT_IE0);
        }

        /* Set screen mode */
        if (VDP[1] != value) { 
            VDP[1] = value;
            SetScreen(); 
        }
        break;

    case 2: 
        ChrTabO = (int)(value & VDPmask[ScrMode].R2) << 10;
        ChrTabM = (((int)(value | ~VDPmask[ScrMode].M2) << 10) | ((1 << 10) - 1)) & 0x7FFF;
        break;

    case 3: 
        ColTabO = ((int)(value & VDPmask[ScrMode].R3) << 6) + ((int)VDP[10] << 14);
        ColTabM = ((int)(value | ~VDPmask[ScrMode].M3) << 6) | 0x1C03F;
        break;

    case 4: 
        ChrGenO = (int)(value & VDPmask[ScrMode].R4) << 11;
        ChrGenM = ((int)(value | ~VDPmask[ScrMode].M4) << 11) | 0x007FF;
        break;

    case 5: 
        SprTabO = ((VDP[11] << 15) | (value << 7) | ~(-1 << 7)) & ((VRAMPages << 14) - 1);
        break;

    case 6: 
    	SprGenO = ((value << 11) | ~(-1 << 11)) & ((VRAMPages << 14) - 1);
        break;

    case 7: 
        FGColor = value >> 4;
        BGColor = value & 0x0F;
        break;

    case 10: 
        value &= 0x07;
        ColTabO = ((int)(VDP[3] & VDPmask[ScrMode].R3) << 6) + ((int)value << 14);
        break;

    case 11: 
        SprTabO = ((value << 15) | (VDP[5] << 7) | ~(-1 << 7)) & ((VRAMPages << 14) - 1);
        break;

    case 14: 
        value &= VRAMPages - 1;
        VRAMpage = (int)value << 14;
        break;

    case 15: 
        value &= 0x0F;
        break;

    case 16: 
        value &= 0x0F;
        PKey = 1;
        break;

    case 17: 
        value &= 0xBF;
        break;

    case 25: 
        VDP[25] = value;
        SetScreen();
        break;

    case 44: 
        vdpCmdWrite(value);
        break;

    case 46: 
        vdpCmdSetCommand(value);
        break;
    }

    VDP[reg] = value;
} 



void vdpInit(Z80* z80arg, int vramPages, UInt32 systemTime, VdpVersion version) 
{
    int i;

    vdpMSX1    = version == VDP_TMS9929A || version == VDP_TMS99x8A;
    curSystemTime = systemTime;
    loopTime   = 0;
    lineTime   = 0;
    phase      = 0;
    ChrGenO    = 0;
    ChrTabO    = 0;
    ColTabO    = 0;
    SprGenO    = 0;
    SprTabO    = 0;
    ChrGenM    = ~0;
    ChrTabM    = 0x7FFF;
    ColTabM    = ~0;
    SprTabM    = ~0;
    VRAMpage   = 0;
    BFlag      = 0;
    BCount     = 0;
    Drawing    = 0;
    VAddr      = 0;
    VKey       = 1;
    PKey       = 1;
    WKey       = 0;
    FGColor    = 0;
    BGColor    = 0;
    XFGColor   = 0;
    XBGColor   = 0;
    ScrMode    = 1;
    ScanLine   = 0;
    VDPData    = 0xff;
    PLatch     = 0;
    ALatch     = 0;
    VRAMPages  = vramPages;
    pendingIRQ = 0;
    z80        = z80arg;
    PALVideo   = 2;
    lineOffset    = 32;
    lineStartTime = 0;
    firstLine = 1;
    firstLineOffset = 0;
    scr0splitLine = 0;
    sprOffset = 1;
    scrLine = 0;

    memset(Palette, 0, sizeof(Palette));
    memset(VRAM, 0, VRAM_SIZE);
    memset(VDPStatus, 0, sizeof(VDPStatus));
    memset(VDP, 0, sizeof(VDP));

    vdpCmdInit();

    VDPStatus[0] = 0x9f;
    VDPStatus[2] = 0x6c;
        
    VDP[1] = 0x10;
    VDP[2] = 0xff;
    VDP[3] = 0xff;
    VDP[4] = 0xff;
    VDP[5] = 0xff;
    VDP[9] = 0x02;

    if (version == VDP_V9958) {
        VDPStatus[1] |= 0x04;
    }
    
    if (version == VDP_TMS99x8A) {
        VDP[9] &= ~0x02;
    }

    ioPortRegister(0x98, vdpRead,       vdpWrite,             NULL);
    ioPortRegister(0x99, vdpReadStatus, vdpWriteLatch,        NULL);

    if (vdpMSX1) {
        memcpy(Palette, msx1Palette, sizeof(Palette));
    }
    else {
        memcpy(Palette, defaultPalette, sizeof(Palette));
        ioPortRegister(0x9a, NULL, vdpWritePaletteLatch, NULL);
        ioPortRegister(0x9b, NULL, vdpWriteRegister,     NULL);
    }

    for (i = 0; i < 16; i++) {
        SetColor(i, Palette[i]);
    }
}

void vdpDestroy() 
{
    ioPortUnregister(0x98);
    ioPortUnregister(0x99);
    ioPortUnregister(0x9a);
    ioPortUnregister(0x9b);
}

UInt8 vdpRead(void* dummy, UInt16 ioPort, UInt32 systemTime) 
{
    UInt8 vdpData;

    vdpSync(systemTime);

    /* Read from VRAM data buffer */
    vdpData = VDPData;

    /* Reset VAddr latch sequencer */
    VKey = 1;

    /* Fill data buffer with a new value */
    VDPData = *MAP_VRAM(VRAMpage + VAddr);

    /* Increment VRAM address */
    VAddr=(VAddr+1)&0x3FFF;

    /* If rolled over, modify VRAM page# */
    if (VAddr == 0 && ScrMode > 3) {
        VDP[14] = (VDP[14] + 1) & (VRAMPages - 1);
        VRAMpage = (int)VDP[14] << 14;
    }
    return vdpData;
}

UInt8 vdpReadStatus(void* dummy, UInt16 ioPort, UInt32 systemTime)
{
    UInt8 vdpStatus;

    vdpSync(systemTime);

    if (vdpMSX1) {
        vdpStatus = VDPStatus[0];
        VDPStatus[0] &= 0x5f;
        SetIRQ(~INT_IE0);
        return vdpStatus;
    }

    /* Read an appropriate status register */
    vdpStatus = VDPStatus[VDP[15]];
    
    /* Reset VAddr latch sequencer */
    VKey=1;

    /* Update status register's contents */
    switch(VDP[15]) {
    case 0: 
        VDPStatus[0] &= 0x5f;
        SetIRQ(~INT_IE0);
        break;

    case 1: 
        VDPStatus[1] &= 0xfe;
        SetIRQ(~INT_IE1);
        break;

    case 7: 
        VDPStatus[7] = VDP[44] = vdpCmdRead();
        break;
    }

    /* Return the status register value */
    return vdpStatus;
}

void vdpWrite(void* dummy, UInt16 ioPort, UInt8 value, UInt32 systemTime)
{
    vdpSync(systemTime);

    VKey=1;
    if (WKey) {
        /* VDP set for writing */
        VDPData = *MAP_VRAM(VRAMpage + VAddr) = value;
        VAddr = (VAddr + 1) & 0x3FFF;
    }
    else {
        /* VDP set for reading */
        VDPData = *MAP_VRAM(VRAMpage + VAddr);
        VAddr = (VAddr + 1) & 0x3FFF;
        *MAP_VRAM(VRAMpage + VAddr) = value;
    }

    /* If VAddr rolled over, modify VRAM page# */
    if (VAddr == 0 && ScrMode > 3) {
        VDP[14] = (VDP[14] + 1 )& (VRAMPages - 1);
        VRAMpage = (int)VDP[14] << 14;
    }

    return;
}

void vdpWriteLatch(void* dummy, UInt16 ioPort, UInt8 value, UInt32 systemTime)
{
    vdpSync(systemTime);

    if (VKey) { 
        ALatch=value;
        VKey=0; 
    }
    else if (value & 0x80) {
        VKey=1;
        /* Writing into VDP registers */
        VDPOut(value&0x3F,ALatch);
    }
    else {
        VKey = 1;
        
        /* Set the VRAM access address */
        VAddr = (((UInt16)value << 8) + ALatch) & 0x3FFF;
        
        /* WKey=1 when VDP set for writing into VRAM */
        WKey=value&0x40;
        
        /* When set for reading, perform first read */
        if (!WKey) {
            VDPData = *MAP_VRAM(VRAMpage + VAddr);
            VAddr = (VAddr + 1) & 0x3FFF;
            
            if (VAddr == 0 && ScrMode > 3) {
                VDP[14] = (VDP[14] + 1) & (VRAMPages - 1);
                VRAMpage = (int)VDP[14] << 14;
            }
        }
    }
    return;
}

void vdpWritePaletteLatch(void* dummy, UInt16 ioPort, UInt8 value, UInt32 systemTime)
{
    vdpSync(systemTime);
    if (PKey) { 
        PLatch = value;
        PKey = 0; 
    }
    else {
        int palEntry = VDP[16];
        PKey = 1;

        /* Set new color for palette entry J */
        Palette[palEntry] = (((UInt32)(PLatch & 0x70) * 255 / 112) << 16) |
                            (((UInt32)(value & 0x07) * 255 / 7) << 8) |
                            ((UInt32)(PLatch & 0x07) * 255 / 7);

        SetColor(palEntry, Palette[palEntry]);
        /* Next palette entry */
        VDP[16] = (palEntry + 1) & 0x0F;
    }
    return;
}

void vdpWriteRegister(void* dummy, UInt16 ioPort, UInt8 value, UInt32 systemTime)
{
    int J;

    vdpSync(systemTime);

    J=VDP[17]&0x3F;
    if(J!=17) VDPOut(J,value);
    if(!(VDP[17]&0x80)) VDP[17]=(J+1)&0x3F;

    return;
}

int vdpRefreshLine(UInt32 systemTime) 
{
    int J;
    int maxLine   = PALVideo ? 313 : 262;

    if (ScanLine < firstLine) {
        if (phase == 0) { // Left border
            loopTime = (CPU_HPERIOD - (ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256)) / 2;
            lineTime = CPU_HPERIOD - loopTime;
            phase = 1;

            if (ScanLine == 0) {
                int screenMode = 0;
                VDPStatus[2] ^= 0x02;
                PALVideo = vdpIsVideoPal(VDP);

                firstLineOffset = PALVideo ? 27 : 0;
                firstLine = firstLineOffset + (vdpIsScanLines212(VDP) ? 8 : 18) + VAdjust + 5;

                if (!vdpIsModeYJK(VDP) || ScrMode < 7 || ScrMode > 8) {
                    screenMode = ScrMode;
                }
                else if (vdpIsModeYAE(VDP)) {
                    screenMode = 10;
                }
                else {
                    screenMode = 12;
                }

                RefreshScreen(screenMode, evenOddPage, vdpIsInterlaceOn(VDP));
                
                SetNewDrawPage();

                if(BCount) {
                    BCount--;
                }
                else {
                    BFlag=!BFlag;
                    if(!VDP[13]) { 
                        XFGColor=FGColor;
                        XBGColor=BGColor; 
                    }
                    else {
                        BCount=(BFlag? VDP[13]&0x0F:VDP[13]>>4)*10;
                        if(BCount)
                        {
                            if(BFlag) { XFGColor=FGColor;XBGColor=BGColor; }
                            else      { XFGColor=VDP[12]>>4;XBGColor=VDP[12]&0x0F; }
                        }
                    }
                }
            }

            if (ScanLine == firstLine - 1) {
                VDPStatus[2]&=0xBF;
            }
        }
        else if (phase == 1) { // draw area
            phase = 2;
            lineOffset    = 0;
            lineStartTime = systemTime;
            VDPStatus[2] &= ~0x20;
            loopTime = ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256;
            lineTime -= loopTime;
        }
        else if (phase == 2) { // Right border
            loopTime = lineTime;
            phase = 0;
            VDPStatus[2] |= 0x20;
            
            ScanLine++;
            scrLine++;
            
            if (ScanLine == firstLine) {
                Drawing = 1;
            }
        }
    }
    else {
        int curLine = ScanLine - firstLine;

        if (phase == 0) { // Left border
            phase = 1;
            loopTime = (CPU_HPERIOD - (ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256)) / 2;
            lineTime = CPU_HPERIOD - loopTime;
            
            J = vdpIsScanLines212(VDP) ? 212 : 192;
            if (curLine == J) {
                VDPStatus[0] |= 0x80;
                VDPStatus[2] |= 0x40;
                if (VDP[1] & 0x20) {
                    SetIRQ(INT_IE0);
                }
            }
        }
        else if (phase == 2) { // Right border
            phase = 0;
            loopTime = lineTime;
            VDPStatus[2] |= 0x20;
            
            ScanLine++; 
            scrLine++;
            if (ScanLine == maxLine) {
                scr0splitLine = 0;
                sprOffset = 1;
                ScanLine = 0;
                scrLine = 0;
            }
        }
        else if (phase == 1) { // Right part of draw area
            phase = 2;
            VDPStatus[2] &= ~0x20;

            loopTime = (ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256);
            lineTime -= loopTime;
            lineOffset    = 0;
            lineStartTime = systemTime;
            phase = 2;

            if (curLine == (vdpIsScanLines212(VDP)? 212:192)) {
                Drawing = 0;
            }

            J=PALVideo ? 256 : vdpIsScanLines212(VDP) ? 245 : 235;

            if(curLine == J) {
                VDPStatus[1] &= 0xFE;
                SetIRQ(~INT_IE1);
            }
            if(curLine < J) {
                J = (((curLine+VScroll)&0xFF)-VDP[19])&0xFF;
                if(J == 1) {
                    VDPStatus[1]|=0x01;
                    if(VDP[0]&0x10) {
                        SetIRQ(INT_IE1);
                    }
                }
                else {
                    if(!(VDP[0]&0x10)) VDPStatus[1]&=0xFE;
                }
            }
        }
    }
    
    return loopTime;
}

int vdpGetRefreshRate() {
    return vdpIsVideoPal(VDP) ? 50 : 60;
}

void vdpSync(UInt32 systemTime) {
    int curLine = ScanLine - firstLineOffset;
    int curLineOffset;
    if (!vdpMSX1) {
        vdpCmdExecute(36 + systemTime - curSystemTime);
        curSystemTime = systemTime;
    }

    if (curLine < 0) {
        return;
    }

    curLineOffset = 32 * (systemTime - lineStartTime) / CPU_H256;
    if (curLineOffset < 32) curLineOffset -= 10;
    if (curLineOffset > 32) curLineOffset = 32;

    if (curLine < 256) {
        if(!vdpIsModeYJK(VDP) || ScrMode < 7 || ScrMode > 8) {
            while (lineOffset < curLineOffset) (RefreshLine[ScrMode])(curLine, lineOffset++);
        }
        else if (vdpIsModeYAE(VDP)) {
            while (lineOffset < curLineOffset) RefreshLine10(curLine, lineOffset++);
        }
        else {
            while (lineOffset < curLineOffset) RefreshLine12(curLine, lineOffset++);
        }
    }
}

int vdpGetState(void* ref, UInt8* buffer, UInt32 systemTime)
{
    UInt32 state[48];
    int size = 0;
    int index = 0;

    state[index++] = VDPData;
    state[index++] = PLatch;
    state[index++] = ALatch;
    state[index++] = VAddr;
    state[index++] = VKey;
    state[index++] = PKey;
    state[index++] = WKey;
    state[index++] = ScanLine;
    state[index++] = VRAMpage;
    state[index++] = ChrTabO;
    state[index++] = ColTabO;
    state[index++] = ChrGenO;
    state[index++] = SprGenO;
    state[index++] = SprTabO;
    state[index++] = ChrGenM;
    state[index++] = ChrTabM;
    state[index++] = ColTabM;
    state[index++] = SprTabM;
    state[index++] = FGColor;
    state[index++] = BGColor;
    state[index++] = XFGColor;
    state[index++] = XBGColor;
    state[index++] = ScrMode;
    state[index++] = curSystemTime;
    state[index++] = loopTime;
    state[index++] = lineTime;
    state[index++] = phase;
    state[index++] = pendingIRQ;
    state[index++] = PALVideo;
    state[index++] = lineOffset;
    state[index++] = lineStartTime;
    state[index++] = firstLine;
    state[index++] = firstLineOffset;

    memcpy(buffer + size, state, sizeof(state));
    size += sizeof(state);

    memcpy(buffer + size, VDP, sizeof(VDP));
    size += sizeof(VDP);

    size += vdpCmdGetState(buffer + size);

    memcpy(buffer + size, VDPStatus, sizeof(VDPStatus));
    size += sizeof(VDPStatus);

    memcpy(buffer + size, Palette, sizeof(Palette));
    size += sizeof(Palette);

    memcpy(buffer + size, VRAM, VRAM_SIZE);
    size += VRAM_SIZE;

    return size;
}

int vdpSetState(void* ref, UInt8* buffer, UInt32 systemTime)
{
    UInt32 state[48];
    int size = 0;
    int index = 0;

    memcpy(state, buffer + size, sizeof(state));
    size += sizeof(state);

    memcpy(VDP, buffer + size, sizeof(VDP));
    size += sizeof(VDP);

    size += vdpCmdSetState(buffer + size);

    memcpy(VDPStatus, buffer + size, sizeof(VDPStatus));
    size += sizeof(VDPStatus);

    memcpy(Palette, buffer + size, sizeof(Palette));
    size += sizeof(Palette);

    memcpy(VRAM, buffer + size, VRAM_SIZE);
    size += VRAM_SIZE;

    VDPData       = (UInt8)state[index++];
    PLatch        = (UInt8)state[index++];
    ALatch        = (UInt8)state[index++];
    VAddr         = (UInt16)state[index++];
    VKey          = (UInt8)state[index++];
    PKey          = (UInt8)state[index++];
    WKey          = (UInt8)state[index++];
    ScanLine      = state[index++];
    VRAMpage      = state[index++];
    ChrTabO       = state[index++];
    ColTabO       = state[index++];
    ChrGenO       = state[index++];
    SprGenO       = state[index++];
    SprTabO       = state[index++];
    ChrGenM       = state[index++];
    ChrTabM       = state[index++];
    ColTabM       = state[index++];
    SprTabM       = state[index++];
    FGColor       = (UInt8)state[index++];
    BGColor       = (UInt8)state[index++];
    XFGColor      = (UInt8)state[index++];
    XBGColor      = (UInt8)state[index++];
    ScrMode       = state[index++];
    curSystemTime = state[index++];
    loopTime      = state[index++];
    lineTime      = state[index++];
    phase         = state[index++];
    pendingIRQ    = (UInt8)state[index++];
    PALVideo      = state[index++];
    lineOffset    = state[index++];
    lineStartTime = state[index++];
    firstLine     = state[index++];
    firstLineOffset = state[index++];


    /* Set palette */
    for (index = 0; index < 16; index++) {
        SetColor(index, Palette[index]);
    }

    /* Set screen mode and VRAM table addresses */
    SetScreen();

    return size;
}


#include "common.h"

