/*****************************************************************************
** File:        ay8910.c
**
** Author:      Daniel Vik
** 
** Description: Emulation of the AY8910 sound chip
**
** History:     1.0 - 10/17 2003 Initial version
**              1.1 - 11/21 2003 Updated phase handling (bug in Chuckie Egg)
**              1.2 - 12/6  2003 Fixed problem with playing samples
**              1.3 - 1/18  2004 Added 16x oversampling to make HF audio sound good
**              1.4 - 2/29  2004 Fixed bug that made samples not play ok
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "AY8910.h"
#include "IoPort.h"
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define BASE_PHASE_STEP 0x28959becUL  /* = (1 << 28) * 3579545 / 32 / 44100 */
#define BUFFER_SIZE     10000

static Int16 voltTable[16] = {
    0x0000, 0x004f, 0x00b4, 0x0133, 0x01d4, 0x029f, 0x03a1, 0x04e7, 
    0x0683, 0x088c, 0x0b1f, 0x0e62, 0x1281, 0x17b8, 0x1e50, 0x26a9
};

static Int32* ay8910Sync(void* ref, UInt32 count);

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))

struct AY8910 {
    Mixer* mixer;

    AY8910ReadCb  ioPortReadCb;
    AY8910WriteCb ioPortWriteCb;
    void*         ioPortArg;

    UInt8  address;
    UInt8  regs[16];

    UInt32 tonePhase[3];
    UInt32 toneStep[3];

    UInt32 noisePhase;
    UInt32 noiseStep;
    UInt32 noiseRand;
    Int16  noiseVolume;

    UInt8  envShape;
    UInt32 envStep;
    UInt32 envPhase;

    UInt8  enable;
    UInt8  ampVolume[3];
    Int32 ctrlVolume;
    Int32 oldSampleVolume;
    Int32 daVolume;

    Int32  buffer[BUFFER_SIZE];
};

int ay8910GetState(AY8910* ay8910, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(AY8910, address);
    int size   = sizeof(AY8910) - offset;

    memcpy(buffer, (char*)ay8910 + offset, size);

    return size;
}

int ay8910SetState(AY8910* ay8910, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(AY8910, address);
    int size   = sizeof(AY8910) - offset;

    memcpy((char*)ay8910 + offset, buffer, size);

    return size;
}

AY8910* ay8910Create(Mixer* mixer, UInt32 cpuClock)
{
    AY8910* ay8910 = (AY8910*)calloc(1, sizeof(AY8910));
    int i;

    ay8910->mixer = mixer;
    ay8910->noiseRand = 1;
    ay8910->noiseVolume = 1;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_PSG, 0, ay8910Sync, ay8910);

    for (i = 0; i < 16; i++) {
        ay8910WriteAddress(ay8910, 0xa0, i, cpuClock);
        ay8910WriteData(ay8910, 0xa1, 0, cpuClock);
    }
    
    ioPortRegister(0xa0, NULL,           ay8910WriteAddress, ay8910);
    ioPortRegister(0xa1, NULL,           ay8910WriteData,    ay8910);
    ioPortRegister(0xa2, ay8910ReadData, NULL,               ay8910);

    return ay8910;
}


void ay8910SetIoPort(AY8910* ay8910, AY8910ReadCb readCb, AY8910WriteCb writeCb, void* arg)
{
    ay8910->ioPortReadCb  = readCb;
    ay8910->ioPortWriteCb = writeCb;
    ay8910->ioPortArg     = arg;
}

void ay8910Destroy(AY8910* ay8910)
{
    ioPortUnregister(0xa0);
    ioPortUnregister(0xa1);
    ioPortUnregister(0xa2);

    mixerUnregisterChannel(ay8910->mixer, MIXER_CHANNEL_PSG);
    free(ay8910);
}

void ay8910WriteAddress(AY8910* ay8910, UInt16 ioPort, UInt8 address, UInt32 systemTime)
{
    ay8910->address = address & 0xf;
}

UInt8 ay8910ReadData(AY8910* ay8910, UInt16 ioPort, UInt32 cpuClock)
{
    UInt8  address = ay8910->address;

    if (address >= 14) {
        int port = address - 14;
        if (ay8910->ioPortReadCb != NULL){// && !(ay8910->regs[7] & (1 << (port + 6)))) {
            ay8910->regs[address] = ay8910->ioPortReadCb(ay8910->ioPortArg, port, cpuClock);
        }
    }
    return ay8910->regs[address];
}

void ay8910WriteData(AY8910* ay8910, UInt16 ioPort, UInt8 data, UInt32 cpuClock)
{
    UInt8  address = ay8910->address;
    UInt32 period;
    int port;

    if (address < 14) {
        mixerSync(ay8910->mixer, cpuClock);
    }

    ay8910->regs[address] = data;

    switch (address) {
    case 0:
    case 1:
    case 2:
    case 3:
    case 4:
    case 5:
        period = ay8910->regs[address & 6] | ((Int32)(ay8910->regs[address | 1] & 0x0f) << 8);
        ay8910->toneStep[address >> 1] = period > 4 ? BASE_PHASE_STEP / period : 1 << 31;
        break;
        
    case 6:
        period = data & 0x1f ? data & 0x1f : 1;
        ay8910->noiseStep = period > 3 ? BASE_PHASE_STEP / period : 1 << 31;
        break;
        
    case 7:
        ay8910->enable = data & 0x3f;
        break;
        
    case 8:
    case 9:
    case 10:
        ay8910->ampVolume[address - 8] = data & 0x1f;
        break;

    case 11:
    case 12:
        period = 16 * (ay8910->regs[11] | ((UInt32)ay8910->regs[12] << 8));
        ay8910->envStep = BASE_PHASE_STEP / (period ? period : 8);
        break;
        
    case 13:
        if (data < 4) data = 0x09;
        if (data < 8) data = 0x0f;
        ay8910->envShape = data;
        ay8910->envPhase = 0;
        break;

    case 14:
    case 15:
        port = address - 14;
        if (ay8910->ioPortWriteCb != NULL){// && (ay8910->regs[7] & (1 << (port + 6)))) {
            ay8910->ioPortWriteCb(ay8910->ioPortArg, port, data, cpuClock);
        }
    }
}

static Int32* ay8910Sync(void* ref, UInt32 count)
{
    AY8910* ay8910 = (AY8910*)ref;
    Int32   channel;
    UInt32  index;

    for (index = 0; index < count; index++) {
        Int32 sampleVolume = 0;
        Int16 envVolume;

        /* Update noise generator */
        ay8910->noisePhase += ay8910->noiseStep;
        while (ay8910->noisePhase >> 28) {
            ay8910->noisePhase  -= 0x10000000;
            ay8910->noiseVolume ^= ((ay8910->noiseRand + 1) >> 1) & 1;
            ay8910->noiseRand    = (ay8910->noiseRand ^ (0x28000 * (ay8910->noiseRand & 1))) >> 1;
        }

        /* Update envelope phase */
        ay8910->envPhase += ay8910->envStep;
        if ((ay8910->envShape & 1) && (ay8910->envPhase >> 28)) {
            ay8910->envPhase = 0x10000000;
        }
 
        /* Calculate envelope volume */
        envVolume = (Int16)((ay8910->envPhase >> 24) & 0x0f);
        if (((ay8910->envPhase >> 27) & (ay8910->envShape + 1) ^ (~ay8910->envShape >> 1)) & 2) {
            envVolume ^= 0x0f;
        }

        /* Calculate and add channel samples to buffer */
        for (channel = 0; channel < 3; channel++) {
            UInt32 enable = ay8910->enable >> channel;
            UInt32 noiseEnable = ((enable >> 3) | ay8910->noiseVolume) & 1;
            UInt32 phaseStep = (~enable & 1) * ay8910->toneStep[channel];
            UInt32 tonePhase = ay8910->tonePhase[channel];
            UInt32 tone = 0;
            Int32  count = 16;

            /* Perform 16x oversampling */
            while (count--) {
                /* Update phase of tone */
                tonePhase += phaseStep;
     
                /* Calculate if tone is on or off */
                tone += (enable | (tonePhase >> 31)) & noiseEnable;
            }

            /* Store phase */
            ay8910->tonePhase[channel] = tonePhase;

            /* Amplify sample using either envelope volume or channel volume */
            if (ay8910->ampVolume[channel] & 0x10) {
                sampleVolume += (Int16)tone * voltTable[envVolume] / 16;
            }
            else {
                sampleVolume += (Int16)tone * voltTable[ay8910->ampVolume[channel]] / 16;
            }
        }

        /* Perform DC offset filtering */
        ay8910->ctrlVolume = sampleVolume - ay8910->oldSampleVolume + 0x3fe7 * ay8910->ctrlVolume / 0x4000;
        ay8910->oldSampleVolume = sampleVolume;

        /* Perform simple 1 pole low pass IIR filtering */
        ay8910->daVolume += 2 * (ay8910->ctrlVolume - ay8910->daVolume) / 3;
        
        /* Store calclulated sample value */
        ay8910->buffer[index] = 8 * ay8910->daVolume;
    }

    return ay8910->buffer;
}
