/*****************************************************************************
** File:        ramNormal.c
**
** Author:      Daniel Vik
**
** Description: Ram mapper
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "ramNormal.h"
#include "romMapper.h"
#include "SlotManager.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


static UInt8 ramData[0x10000];


typedef struct {
    int slot;
    int sslot;
    int startPage;
    int pages;
} RamNormal;

static int getState(RamNormal* rm, UInt8* buffer, UInt32 systemTime)
{
    int ramSize = rm->pages * 0x2000;

    memcpy(buffer, ramData, ramSize);
    return ramSize;
}

static int setState(RamNormal* rm, UInt8* buffer, UInt32 systemTime)
{
    int ramSize = rm->pages * 0x2000;
    int i;

    memcpy(ramData, buffer, ramSize);

    for (i = 0; i < rm->pages; i++) {
        slotMapPage(rm->slot, rm->sslot, i + rm->startPage, ramData + 0x2000 * i, 1, 1);
    }

    return ramSize;
}

static void destroy(void* ref)
{
    RamNormal* rm = (RamNormal*)ref;

    slotUnregister(rm->slot, rm->sslot, 0);

    free(rm);
}

int ramNormalCreate(int size, int slot, int sslot, int startPage, int clearRAM, UInt8** ramPtr, UInt32* ramSize) 
{
    SlotCallbacks callbacks = { destroy, NULL, NULL, getState, setState };
    RamNormal* rm;
    int pages = size / 0x2000;
    int i;

    if (size > 0x10000 || (size & 0x1fff)) {
        return 0;
    }

    // Start page must be zero (only full slot allowed)
    if (startPage + pages > 8) {
        return 0;
    }

    if (clearRAM) {
        memset(ramData, 0xff, sizeof(ramData));
    }

    rm = malloc(sizeof(RamNormal));

    rm->slot      = slot;
    rm->sslot     = sslot;
    rm->startPage = startPage;
    rm->pages     = pages;

    for (i = 0; i < pages; i++) {
        slotMapPage(slot, sslot, i + startPage, ramData + 0x2000 * i, 1, 1);
    }

    slotRegister(RAM_NORMAL, slot, sslot, startPage, pages, &callbacks, rm);

    *ramPtr = ramData;
    *ramSize = pages * 0x2000;

    return 1;
}

