/*****************************************************************************
** File:
**      romMapper.h
**
** Author:
**      Daniel Vik
**
** Description:
**      Determines the type of a rom image.
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#ifndef ROM_MAPPER_H
#define ROM_MAPPER_H

typedef enum {
    ROM_UNKNOWN     = 0,
    ROM_STANDARD    = 1,
    ROM_MSXDOS2     = 2,
    ROM_KONAMI5     = 3,
    ROM_KONAMI4     = 4,
    ROM_ASCII8      = 5,
    ROM_ASCII16     = 6,
    ROM_GAMEMASTER2 = 7,
    ROM_ASCII8SRAM  = 8,
    ROM_ASCII16SRAM = 9,
    ROM_RTYPE       = 10,
    ROM_CROSSBLAIM  = 11,
    ROM_HARRYFOX    = 12,
    ROM_KOREAN80    = 13,
    ROM_KOREAN126   = 14,
    ROM_SCCEXTENDED = 15,
    ROM_FMPAC       = 16,
    ROM_KONAMI4NF   = 17,
    ROM_ASCII16NF   = 18,
    ROM_PLAIN       = 19,
    ROM_NORMAL      = 20,
    ROM_DISK        = 21,
    RAM_MAPPER      = 22,
    RAM_NORMAL      = 23,
    ROM_KANJI       = 24,
    ROM_HOLYQURAN   = 25,
    SRAM_MATSUCHITA = 26,
    ROM_PANASONIC   = 27,
    ROM_BUNSETU     = 28,
    ROM_JISYO       = 29,
    ROM_KANJI12     = 30,
    ROM_NATIONAL    = 31,
    SRAM_S1985      = 32,
    ROM_F4DEVICE    = 33,
    ROM_F4INVERTED  = 34,
    AUDIO_MOONSOUND = 35,
    AUDIO_Y8950     = 36,
    AUDIO_YM2413    = 37,
    ROM_KOEI        = 38,
    ROM_BASIC       = 39,
    ROM_HALNOTE     = 40,
    ROM_LODERUNNER  = 41,
    ROM_0x4000      = 42,
    ROM_PAC         = 43,
    ROM_MEGARAM     = 44,
    ROM_MEGARAM128  = 45,
    ROM_MEGARAM256  = 46,
    ROM_MEGARAM512  = 47,
    ROM_MEGARAM768  = 48,
    ROM_MEGARAM2M   = 49,
    ROM_MSXAUDIO    = 50,
    ROM_KOREAN90    = 51,
    ROM_SNATCHER    = 52,
    ROM_SDSNATCHER  = 53,
    ROM_SCCMIRRORED = 54,
    ROM_SCC         = 55,
    ROM_SCCPLUS     = 56,
    ROM_MAXROMID    = 56,
    // Not used yet
    ROM_KONAMISYNTH = 57,
    ROM_KONAMIMJ2   = 58
} RomType;

static char* romTypeToString(RomType romType) {
    switch (romType) {    
    case ROM_STANDARD:    return "Standard";
    case ROM_MSXDOS2:     return "MSXDOS 2";
    case ROM_KONAMI5:     return "Konami 5";
    case ROM_KONAMI4:     return "Konami 4";
    case ROM_ASCII8:      return "ASCII 8";
    case ROM_ASCII16:     return "ASCII 16";
    case ROM_GAMEMASTER2: return "Game Master 2 (SRAM)";
    case ROM_ASCII8SRAM:  return "ASCII 8 (SRAM)";
    case ROM_ASCII16SRAM: return "ASCII 16 (SRAM)";
    case ROM_RTYPE:       return "R-Type";
    case ROM_CROSSBLAIM:  return "Cross Blaim";
    case ROM_HARRYFOX:    return "Harry Fox";
    case ROM_KOREAN80:    return "Korean 80 in 1";
    case ROM_KOREAN126:   return "Korean 126 in 1";
    case ROM_SCC:         return "SCC";
    case ROM_SCCPLUS:     return "SCC+";
    case ROM_SNATCHER:    return "The Snatcher";
    case ROM_SDSNATCHER:  return "SD Snatcher";
    case ROM_SCCMIRRORED: return "SCC mirrored";
    case ROM_SCCEXTENDED: return "SCC extended";
    case ROM_FMPAC:       return "FMPAC (SRAM)";
    case ROM_KONAMI4NF:   return "Konami4 alt.";
    case ROM_ASCII16NF:   return "ASCII 16 alt.";
    case ROM_PLAIN:       return "Mirrored ROM";
    case ROM_NORMAL:      return "Normal ROM";
    case ROM_DISK:        return "Disk controller";
    case RAM_MAPPER:      return "Mapped RAM";
    case RAM_NORMAL:      return "Normal RAM";
    case ROM_KANJI:       return "Kanji";
    case ROM_HOLYQURAN:   return "Holy Quran";
    case SRAM_MATSUCHITA: return "Matsushita SRAM";
    case ROM_PANASONIC:   return "Panasonic";
    case ROM_BUNSETU:     return "Bunsetsu";
    case ROM_JISYO:       return "Jisyo";
    case ROM_KANJI12:     return "Kanji12";
    case ROM_NATIONAL:    return "National (SRAM)";
    case SRAM_S1985:      return "S1985";
    case ROM_F4DEVICE:    return "F4 Device Normal";
    case ROM_F4INVERTED:  return "F4 Device Inverted";
    case AUDIO_MOONSOUND: return "Moonsound Audio";
    case AUDIO_Y8950:     return "Y8950 Audio";
    case AUDIO_YM2413:    return "YM2413 Audio";
    case ROM_KOEI:        return "Koei (SRAM)";
    case ROM_BASIC:       return "Basic ROM";
    case ROM_HALNOTE:     return "Halnote";
    case ROM_LODERUNNER:  return "Lode Runner";
    case ROM_0x4000:      return "Normal 0x4000";
    case ROM_PAC:         return "PAC (SRAM)";
    case ROM_MEGARAM:     return "MegaRAM";
    case ROM_MEGARAM128:  return "128kB MegaRAM";
    case ROM_MEGARAM256:  return "256kB MegaRAM";
    case ROM_MEGARAM512:  return "512kB MegaRAM";
    case ROM_MEGARAM768:  return "768kB MegaRAM";
    case ROM_MEGARAM2M:   return "2MB MegaRAM";
    case ROM_MSXAUDIO:    return "MSX Audio";
    case ROM_KOREAN90:    return "Korean 90 in 1";

    case ROM_KONAMISYNTH: return "Konami synth";
    case ROM_KONAMIMJ2:   return "Konami J2";
    case ROM_UNKNOWN:     return "Unknown";
    }

    return "unknown";
}

extern RomType getRomType(const void* buffer, int size);
extern RomType romMapperGuessRom(const void *buffer, int size, int guess);
extern RomType romMapperTypeFromString(const char* name);
extern void    romMapperSetRomdbFilename(const char* filename);

#endif /*ROM_MAPPER_H*/

