/*****************************************************************************
** File:        romMapperSCCplus.c
**
** Author:      Daniel Vik
**
** Description: Rom mapper for SCC plus
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "romMapperSCCplus.h"
#include "romMapper.h"
#include "SlotManager.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


typedef struct {
    UInt8 romData[0x20000];
    int slot;
    int sslot;
    int startPage;
    UInt8 modeRegister;
    int isRamSegment[4];
    int romMapper[4];
    SccMode sccMode;
    SCC* scc;
} RomMapperSCCplus;


static void destroy(RomMapperSCCplus* rm)
{
    slotUnregister(rm->slot, rm->sslot, rm->startPage);

    free(rm);
}

static int getState(RomMapperSCCplus* rm, UInt8* buffer, UInt32 systemTime)
{
    int size = 0;
    
    memcpy(buffer + size, &rm->modeRegister, sizeof(rm->modeRegister));
    size += sizeof(rm->modeRegister);
    
    memcpy(buffer + size, rm->isRamSegment, sizeof(rm->isRamSegment));
    size += sizeof(rm->isRamSegment);
    
    memcpy(buffer + size, rm->romMapper, sizeof(rm->romMapper));
    size += sizeof(rm->romMapper);
    
    memcpy(buffer + size, &rm->sccMode, sizeof(rm->sccMode));
    size += sizeof(rm->sccMode);

    return size;
}

static int setState(RomMapperSCCplus* rm, UInt8* buffer, UInt32 systemTime)
{
    int size = 0;
    
    memcpy(&rm->modeRegister, buffer + size, sizeof(rm->modeRegister));
    size += sizeof(rm->modeRegister);
    
    memcpy(rm->isRamSegment, buffer + size, sizeof(rm->isRamSegment));
    size += sizeof(rm->isRamSegment);
    
    memcpy(rm->romMapper, buffer + size, sizeof(rm->romMapper));
    size += sizeof(rm->romMapper);
    
    memcpy(&rm->sccMode, buffer + size, sizeof(rm->sccMode));
    size += sizeof(rm->sccMode);

    if (rm->sccMode == SCC_PLUS) {
        slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 1, 0);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 0, 0);
    }
    else if (rm->sccMode = SCC_COMPATIBLE) {
        slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 0, 0);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 1, 0);
    }
    else {
        slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 1, 0);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 1, 0);
    }

    return size;
}

static UInt8 read(void* arg, UInt16 address, UInt32 systemTime) 
{
    RomMapperSCCplus* rm = (RomMapperSCCplus*)arg;
    int bank;
    
    address += 0x4000;

    if (rm->sccMode == SCC_COMPATIBLE && address >= 0x9800 && address < 0xa000) {
        return sccRead(rm->scc, (UInt8)(address & 0xff), systemTime);
    }
    if (rm->sccMode == SCC_PLUS && address >= 0xb800 && address < 0xc000) {
        return sccRead(rm->scc, (UInt8)(address & 0xff), systemTime);
    }

    bank = (address - 0x4000) >> 13;

    if (rm->isRamSegment[bank]) {
    	return rm->romData[0x2000 * (rm->romMapper[bank] & 0x0f) + (address & 0x1fff)];
    }

    return 0xff;
}

static void write(void* arg, UInt16 address, UInt8 value, UInt32 systemTime) 
{
    RomMapperSCCplus* rm = (RomMapperSCCplus*)arg;
    int bank;

    address += 0x4000;

    if (address < 0x4000 && address >= 0xc000) {       
        return;
    }

    if ((address | 1) == 0x7fff) {
        rm->modeRegister = value;
        rm->isRamSegment[0] = (value & 0x10) | (value & 0x01);
        rm->isRamSegment[1] = (value & 0x10) | (value & 0x02);
        rm->isRamSegment[2] = (value & 0x10) | ((value & 0x24) == 0x24);
		rm->isRamSegment[3] = (value & 0x10);
        
        if ((rm->modeRegister & 0x20) && (rm->romMapper[3] & 0x80)) {
            slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 1, 0);
            slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 0, 0);
            sccSetMode(rm->scc, SCC_PLUS);
            rm->sccMode = SCC_PLUS;
        }
        else if (!(rm->modeRegister & 0x20) && (rm->romMapper[2] & 0x3f) == 0x3f) {
            slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 0, 0);
            slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 1, 0);
            sccSetMode(rm->scc, SCC_COMPATIBLE);
            rm->sccMode = SCC_COMPATIBLE;
        }
        else {
            slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 1, 0);
            slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 1, 0);
            rm->sccMode = SCC_NONE;
        }
        return;
    }

    bank = (address - 0x4000) >> 13;

    if (rm->isRamSegment[bank]) {
    	rm->romData[0x2000 * (rm->romMapper[bank] & 0x0f) + (address & 0x1fff)] = value;
        return;
    }

    if ((address & 0x1800) == 0x1000) {
        if (rm->romMapper[bank] != value) {
            rm->romMapper[bank] = value;
            slotMapPage(rm->slot, rm->sslot, rm->startPage + bank, rm->romData + 0x2000 * (value & 0x0f), 1, 0);

            if ((rm->modeRegister & 0x20) && (rm->romMapper[3] & 0x80)) {
                if (rm->sccMode = SCC_PLUS) {
                    slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 1, 0);
                    slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 0, 0);
                    sccSetMode(rm->scc, SCC_PLUS);
                    rm->sccMode = SCC_PLUS;
                }
            }
            else if (!(rm->modeRegister & 0x20) && (rm->romMapper[2] & 0x3f) == 0x3f) {
                if (rm->sccMode = SCC_COMPATIBLE) {
                    slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 0, 0);
                    slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 1, 0);
                    sccSetMode(rm->scc, SCC_COMPATIBLE);
                    rm->sccMode = SCC_COMPATIBLE;
                }
            }
            else {
                if (rm->sccMode = SCC_NONE) {
                    slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, NULL, 1, 0);
                    slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, NULL, 1, 0);
                    rm->sccMode = SCC_NONE;
                }
            }
        }

        return;
    }

    if (rm->sccMode == SCC_COMPATIBLE && address >= 0x9800 && address < 0xa000) {
        sccWrite(rm->scc, address & 0xff, value, systemTime);
    }
    else if (rm->sccMode == SCC_PLUS && address >= 0xb800 && address < 0xc000) {
        sccWrite(rm->scc, address & 0xff, value, systemTime);
    }
}

int romMapperSCCplusCreate(char* filename, UInt8* romData, 
                           int size, int slot, int sslot, int startPage, SCC* scc) 
{
    SlotCallbacks callbacks = { destroy, read, write, getState, setState };
    RomMapperSCCplus* rm;

    rm = malloc(sizeof(RomMapperSCCplus));

    slotRegister(ROM_SCCPLUS, slot, sslot, startPage, 4, &callbacks, rm);

    memset(rm->romData, 0xff, 0x20000);
    rm->slot  = slot;
    rm->sslot = sslot;
    rm->startPage  = startPage;
    rm->modeRegister = 0;
    rm->isRamSegment[0] = 0;
    rm->isRamSegment[1] = 0;
    rm->isRamSegment[2] = 0;
    rm->isRamSegment[3] = 0;
    rm->scc = scc;
    rm->sccMode = SCC_NONE;

    rm->romMapper[0] = 0;
    rm->romMapper[1] = 1;
    rm->romMapper[2] = 2;
    rm->romMapper[3] = 3;

    slotMapPage(rm->slot, rm->sslot, rm->startPage,     rm->romData, 1, 0);
    slotMapPage(rm->slot, rm->sslot, rm->startPage + 1, rm->romData + 0x2000, 1, 0);
    slotMapPage(rm->slot, rm->sslot, rm->startPage + 2, rm->romData + 0x4000, 1, 0);
    slotMapPage(rm->slot, rm->sslot, rm->startPage + 3, rm->romData + 0x6000, 1, 0);

    return 1;
}

