/*****************************************************************************
** File:        romMapperLodeRunner.c
**
** Author:      Daniel Vik
**
** Description: Rom mapper for Super Lode Runner cartridge
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "romMapperLodeRunner.h"
#include "romMapper.h"
#include "SlotManager.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


typedef struct {
    UInt8* romData;
    int slot;
    int sslot;
    int startPage;
    int size;
    int romMapper;
} RomMapperLodeRunner;

static void destroy(RomMapperLodeRunner* rm)
{
    free(rm->romData);
    free(rm);
}

static int getState(RomMapperLodeRunner* rm, UInt8* buffer, UInt32 systemTime)
{
    memcpy(buffer, &rm->romMapper, sizeof(rm->romMapper));
    return sizeof(rm->romMapper);
}

static int setState(RomMapperLodeRunner* rm, UInt8* buffer, UInt32 systemTime)
{
    UInt8* bankData;

    memcpy(&rm->romMapper, buffer, sizeof(rm->romMapper));

    bankData = rm->romData + (rm->romMapper << 14);
    slotMapPage(rm->slot, rm->sslot, rm->startPage,     bankData, 1, 0);
    slotMapPage(rm->slot, rm->sslot, rm->startPage + 1, bankData + 0x2000, 1, 0);

    return sizeof(rm->romMapper);
}

static void write(RomMapperLodeRunner* rm, UInt16 address, UInt8 value, UInt32 systemTime) 
{
    value &= (rm->size / 0x4000 - 1);

    if (rm->romMapper != value) {
        UInt8* bankData = rm->romData + ((int)value << 14);
        
        rm->romMapper = value;

        slotMapPage(rm->slot, rm->sslot, rm->startPage,     bankData, 1, 0);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + 1, bankData + 0x2000, 1, 0);
    }
}

int romMapperLodeRunnerCreate(char* filename, UInt8* romData, 
                          int size, int slot, int sslot, int startPage) 
{
    SlotCallbacks callbacks1 = { destroy, NULL, NULL, getState, setState };
    SlotCallbacks callbacks2 = { NULL, NULL, write, NULL, NULL };
    RomMapperLodeRunner* rm;

    if (size != 0x20000) {
        return 0;
    }

    rm = malloc(sizeof(RomMapperLodeRunner));

    slotRegister(ROM_LODERUNNER, slot, sslot, startPage, 4, &callbacks1, rm);
    slotRegisterWrite0(&callbacks2, rm);

    rm->romData = malloc(size);
    memcpy(rm->romData, romData, size);
    rm->size = size;
    rm->slot  = slot;
    rm->sslot = sslot;
    rm->startPage  = startPage;

    rm->romMapper = 0;

    slotMapPage(rm->slot, rm->sslot, rm->startPage,     rm->romData + rm->romMapper * 0x2000, 1, 0);
    slotMapPage(rm->slot, rm->sslot, rm->startPage + 1, rm->romData + rm->romMapper * 0x2000 + 0x2000, 1, 0);

    return 1;
}

