/*****************************************************************************
** File:        romMapperCrossBlaim.c
**
** Author:      Daniel Vik
**
** Description: Rom mapper for CrossBlaim cartridges
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "romMapperCrossBlaim.h"
#include "romMapper.h"
#include "SlotManager.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


typedef struct {
    UInt8* romData;
    int slot;
    int sslot;
    int startPage;
    int size;
    int romMapper[4];
} RomMapperCrossBlaim;

static void destroy(RomMapperCrossBlaim* rm)
{
    slotUnregister(rm->slot, rm->sslot, rm->startPage);

    free(rm->romData);
    free(rm);
}

static int getState(RomMapperCrossBlaim* rm, UInt8* buffer, UInt32 systemTime)
{
    memcpy(buffer, rm->romMapper, sizeof(rm->romMapper));
    return sizeof(rm->romMapper);
}

static int setState(RomMapperCrossBlaim* rm, UInt8* buffer, UInt32 systemTime)
{
    int i;

    memcpy(rm->romMapper, buffer, sizeof(rm->romMapper));

    for (i = 0; i < 4; i += 2) {
        UInt8* bankData = rm->romData + (rm->romMapper[i] << 14);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + i,     bankData, 1, 0);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + i + 1, bankData + 0x2000, 1, 0);
    }

    return sizeof(rm->romMapper);
}

static void write(RomMapperCrossBlaim* rm, UInt16 address, UInt8 value, UInt32 systemTime) 
{
    int bank;

    address += 0x4000;

    if (address != 0x4045) {
        return;
    }

    bank = 2;

    if (rm->romMapper[bank] != value) {
        UInt8* bankData = rm->romData + ((int)value << 14);
        
        rm->romMapper[bank] = value;

        slotMapPage(rm->slot, rm->sslot, rm->startPage + bank,     bankData, 1, 0);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + bank + 1, bankData + 0x2000, 1, 0);
    }
}

int romMapperCrossBlaimCreate(char* filename, UInt8* romData, 
                              int size, int slot, int sslot, int startPage) 
{
    SlotCallbacks callbacks = { destroy, NULL, write, getState, setState };
    RomMapperCrossBlaim* rm;
    int i;

    if (size < 0x8000) {
        return 0;
    }

    rm = malloc(sizeof(RomMapperCrossBlaim));

    slotRegister(ROM_CROSSBLAIM, slot, sslot, startPage, 4, &callbacks, rm);

    rm->romData = malloc(size);
    memcpy(rm->romData, romData, size);
    rm->size = size;
    rm->slot  = slot;
    rm->sslot = sslot;
    rm->startPage  = startPage;

    rm->romMapper[0] = 0;
    rm->romMapper[2] = 0;

    for (i = 0; i < 4; i += 2) {   
        slotMapPage(rm->slot, rm->sslot, rm->startPage + i,     rm->romData + rm->romMapper[i] * 0x2000, 1, 0);
        slotMapPage(rm->slot, rm->sslot, rm->startPage + i + 1, rm->romData + rm->romMapper[i] * 0x2000 + 0x2000, 1, 0);
    }

    return 1;
}

