/*****************************************************************************
** File:        RTC.c
**
** Author:      Daniel Vik
** 
** Description: Emulation of the Real Time Clock
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "RTC.h"
#include "IoPort.h"
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))

struct RTC {
    char cmosName[512];
    UInt8 address;
    UInt8 bank;
    UInt8 registers[4][13]; 
};

int rtcGetState(RTC* rtc, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(RTC, address);
    int size   = sizeof(RTC) - offset;

    memcpy(buffer, (char*)rtc + offset, size);

    return size;
}

int rtcSetState(RTC* rtc, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(RTC, address);
    int size   = sizeof(RTC) - offset;

    memcpy((char*)rtc + offset, buffer, size);

    return size;
}

RTC* rtcCreate(char* cmosName)
{
    RTC* rtc = (RTC*)calloc(1, sizeof(RTC));
    FILE* file;

    rtc->registers[2][4] = 40;
    rtc->registers[2][5] = 80;
    rtc->registers[2][6] = 15;
    rtc->registers[2][7] = 4;
    rtc->registers[2][8] = 4;

    strcpy(rtc->cmosName, cmosName);

    file = fopen(cmosName, "r");

    if (file != NULL) {
        UInt8 registers[4][13]; 

        int size = fread(registers, 1, sizeof(registers), file);
    
        if (size == sizeof(registers)) {
            memcpy(rtc->registers, registers, sizeof(registers));
        }

        fclose(file);
    }
    
    ioPortRegister(0xb4, NULL,        rtcWriteAddress, rtc);
    ioPortRegister(0xb5, rtcReadData, rtcWriteData,    rtc);

    return rtc;
}

void rtcDestroy(RTC* rtc)
{
    FILE* file = fopen(rtc->cmosName, "w");

    ioPortUnregister(0xb4);
    ioPortUnregister(0xb5);

    fwrite(rtc->registers, 1, sizeof(rtc->registers), file);

    fclose(file);
}

void rtcWriteAddress(RTC* rtc, UInt16 ioPort, UInt8 value, UInt32 systemTime)
{
    rtc->address = value & 0x0f;
}

void rtcWriteData(RTC* rtc, UInt16 ioPort, UInt8 value, UInt32 systemTime)
{
    if (rtc->address < 13) {
        rtc->registers[rtc->bank & 0x03][rtc->address] = value;
    }
    if(rtc->address == 13) {
        rtc->bank = value;
    }
}

UInt8 rtcReadData(RTC* rtc, UInt16 ioPort, UInt32 systemTime)
{
    static time_t prevTime;
    static struct tm TM;
    time_t currTime;

    if (rtc->address == 13) {
        return 0xf0 | rtc->bank;
    }

    if(rtc->address > 13) {
        return 0xff;
    }

    if (rtc->bank & 0x03) {
        return 0xf0 | rtc->registers[rtc->bank & 0x03][rtc->address];
    }

    // Update time structure only if needed
    currTime = time(NULL);
    if (currTime != prevTime) {
        TM = *localtime(&currTime);
        prevTime = currTime;
    }

    // Parse contents of last retrieved TM
    switch(rtc->address) {
    case 0:  return 0xf0 | (TM.tm_sec % 10);
    case 1:  return 0xf0 | (TM.tm_sec / 10);
    case 2:  return 0xf0 | (TM.tm_min % 10);
    case 3:  return 0xf0 | (TM.tm_min / 10);
    case 4:  return 0xf0 | (TM.tm_hour % 10);
    case 5:  return 0xf0 | (TM.tm_hour / 10);
    case 6:  return 0xf0 | (TM.tm_wday);
    case 7:  return 0xf0 | (TM.tm_mday % 10);
    case 8:  return 0xf0 | (TM.tm_mday / 10);
    case 9:  return 0xf0 | ((TM.tm_mon + 1) % 10);
    case 10: return 0xf0 | ((TM.tm_mon + 1) / 10);
    case 11: return 0xf0 | ((TM.tm_year - 80) % 10);
    case 12: return 0xf0 | (((TM.tm_year - 80) / 10) % 10);
    } 

    return 0xff;
}

