/*****************************************************************************
** File:        Win32joystick.c
**
** Author:      Daniel Vik
**
** Description: Joystick input methods
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
******************************************************************************
*/
#include "Win32joystick.h"
#include "Win32SystemTime.h"
#include <windows.h>
#include <mmsystem.h>
 
static JoyType joyType[2] = { JOY_NONE, JOY_NONE };
static DWORD autofire[2] = { 0, 0 };
int keySet0[6] = { -1, -1, -1, -1, -1, -1 };
int keySet1[6] = { -1, -1, -1, -1, -1, -1 };

static BYTE joystickHwEx(int hwId) {
    JOYINFOEX joyinfo; 
    JOYCAPS joycaps;
    BOOL attached = FALSE;
    BYTE value = 0;
 
    joyinfo.dwFlags = JOY_RETURNBUTTONS | JOY_RETURNCENTERED | JOY_RETURNX | JOY_RETURNY;
    joyinfo.dwSize = sizeof(JOYINFOEX);

    attached = joyGetPosEx(hwId, &joyinfo) == JOYERR_NOERROR;
    attached &= joyGetDevCaps(hwId, &joycaps, sizeof(JOYCAPS)) == JOYERR_NOERROR;

    if (attached) {
        if (joyinfo.dwXpos <= joycaps.wXmin + (joycaps.wXmax - joycaps.wXmin) / 4) {
            value |= 0x04;
        }
        if (joyinfo.dwXpos >= joycaps.wXmin + (joycaps.wXmax - joycaps.wXmin) / 4 * 3) {
            value |= 0x08;
        }
        if (joyinfo.dwYpos <= joycaps.wYmin + (joycaps.wYmax - joycaps.wYmin) / 4) {
            value |= 0x01;
        }
        if (joyinfo.dwYpos >= joycaps.wYmin + (joycaps.wYmax - joycaps.wYmin) / 4 * 3) {
            value |= 0x02;
        }
        if (joyinfo.dwButtons & 0x0001) {
            value |= 0x10;
        }
        if (joyinfo.dwButtons & 0x0002) {
            value |= 0x20;
        }
    }

    return value;
}

static BYTE joystickHW(int hwId) {
    UINT wNumDevs = 1504;
    JOYINFOEX joyinfo; 
    JOYCAPS joycaps;
    BOOL attached = FALSE;
    BYTE value = 0;
 
    if (wNumDevs == 1504) {
        wNumDevs = joyGetNumDevs();
    }

    joyinfo.dwFlags = JOY_RETURNBUTTONS | JOY_RETURNCENTERED | JOY_RETURNX | JOY_RETURNY;
    joyinfo.dwSize = sizeof(JOYINFOEX);

    if (hwId == 0 && wNumDevs > 0) {
        attached = joyGetPosEx(JOYSTICKID1, &joyinfo) == JOYERR_NOERROR;
        attached &= joyGetDevCaps(JOYSTICKID1, &joycaps, sizeof(JOYCAPS)) == JOYERR_NOERROR;
    }
    else if (hwId == 1 && wNumDevs > 1) {
        attached  = joyGetPosEx(JOYSTICKID2,&joyinfo) == JOYERR_NOERROR; 
        attached &= joyGetDevCaps(JOYSTICKID2, &joycaps, sizeof(JOYCAPS)) == JOYERR_NOERROR;
    }

    if (attached) {
        if (joyinfo.dwXpos <= joycaps.wXmin + (joycaps.wXmax - joycaps.wXmin) / 4) {
            value |= 0x04;
        }
        if (joyinfo.dwXpos >= joycaps.wXmin + (joycaps.wXmax - joycaps.wXmin) / 4 * 3) {
            value |= 0x08;
        }
        if (joyinfo.dwYpos <= joycaps.wYmin + (joycaps.wYmax - joycaps.wYmin) / 4) {
            value |= 0x01;
        }
        if (joyinfo.dwYpos >= joycaps.wYmin + (joycaps.wYmax - joycaps.wYmin) / 4 * 3) {
            value |= 0x02;
        }
        if (joyinfo.dwButtons & 0x0001) {
            value |= 0x10;
        }
        if (joyinfo.dwButtons & 0x0002) {
            value |= 0x20;
        }
    }

    return value;
}

static BYTE joystickNumpad() {
    BYTE value = 0;
    if (GetKeyState(VK_NUMPAD8) > 1UL) value |= 0x01;
    if (GetKeyState(VK_NUMPAD2) > 1UL) value |= 0x02;
    if (GetKeyState(VK_NUMPAD4) > 1UL) value |= 0x04;
    if (GetKeyState(VK_NUMPAD6) > 1UL) value |= 0x08;
    if (GetKeyState(VK_NUMPAD0) > 1UL) value |= 0x10;
    if (GetKeyState(VK_DECIMAL) > 1UL) value |= 0x20;

    return value;
}

static BYTE joystickKeyset(int keyset) {
    int* pKeyset = keyset == 1 ? keySet0 : keySet1;
    BYTE value = 0;

    if (pKeyset[0] != 0xff && GetAsyncKeyState(pKeyset[0]) > 1UL) value |= 0x01;
    if (pKeyset[1] != 0xff && GetAsyncKeyState(pKeyset[1]) > 1UL) value |= 0x02;
    if (pKeyset[2] != 0xff && GetAsyncKeyState(pKeyset[2]) > 1UL) value |= 0x04;
    if (pKeyset[3] != 0xff && GetAsyncKeyState(pKeyset[3]) > 1UL) value |= 0x08;
    if (pKeyset[4] != 0xff && GetAsyncKeyState(pKeyset[4]) > 1UL) value |= 0x10;
    if (pKeyset[5] != 0xff && GetAsyncKeyState(pKeyset[5]) > 1UL) value |= 0x20;

    return value;
}

void JoystickSetKeyStateKey(int keyset, JoyAction key, int value) {
    int* pKeyset = keyset == 1 ? keySet0 : keySet1;
    pKeyset[key] = value;
}

void JoystickSetType(int joyId, JoyType type) {
    joyType[joyId - 1] = type;
}

static int joyHwIndex[2] = { -1, -1 };

void JoystickSetHwType(int joyId, int hwId) {
    joyHwIndex[joyId - 1] = hwId;
}

void JoystickSetAutofire(int joyId, JoyAutofire joyAutofire) {
    switch (joyAutofire) {
    case JOY_AF_OFF:
        autofire[joyId - 1] = 0;
        break;
    case JOY_AF_SLOW:
        autofire[joyId - 1] = 0x80000UL;
        break;
    case JOY_AF_MEDIUM:
        autofire[joyId - 1] = 0x20000UL;
        break;
    case JOY_AF_FAST:
        autofire[joyId - 1] = 0x10000UL;
        break;
    }
}

BYTE JoystickGetState(int joyId) {
    BYTE value = 0;

    switch (joyType[joyId - 1]) {
    case JOY_NUMPAD:
        value = joystickNumpad();
        break;
    case JOY_HW1:
        value = joystickHW(0);
        break;
    case JOY_HW2:
        value = joystickHW(1);
        break;
    case JOY_KEYSETA:
        value = joystickKeyset(1);
        break;
    case JOY_KEYSETB:
        value = joystickKeyset(2);
        break;
    case JOY_HW:
        value = joystickHwEx(joyHwIndex[joyId - 1]);
        break;
    }

    if ((DWORD)GetMSXSystemTime() & autofire[joyId - 1]) {
        value &= 0x0f;
    }

    return value;
}

static char* joyDevList[17];
static char joyString[17][MAXPNAMELEN];

char** JoystickGetDevList() {
    int i;
    int cnt = 0;
    JOYCAPS jc;
    
    for (i = 0; i < 16; i++) {
        HRESULT rv = joyGetDevCaps(i, &jc, sizeof(jc));
        if (rv == JOYERR_NOERROR) {
            strcpy(joyString[cnt], jc.szPname);
            joyDevList[cnt] = joyString[cnt];
            cnt++;
        }
    }
    joyDevList[cnt] = NULL;

    return joyDevList;
}
