#include "VDP.h"
#include <string.h>

#define RGB2INT(R, G, B) ((B) | ((int)(G) << 8) | ((int)(R) << 16))

#define VRAM_SIZE (128 * 1024)

static int vramAddr;
#define MAP_VRAM(addr) (VRAM + (vramAddr = addr, ScrMode >= 7 && ScrMode <= 8 ? (vramAddr >> 1 | ((vramAddr & 1) << 16)) : vramAddr))
#define MAP_VRAM_16(addr) (VRAM + (addr))
#define MAP_VRAM_78(addr) (VRAM + (vramAddr = addr, vramAddr >> 1 | ((vramAddr & 1) << 16)))
#define IS_VRAM_78() (ScrMode >= 7 && ScrMode <= 8)

/** Following macros can be used in screen drivers ***********/
#define BigSprites    (VDP[1]&0x01)   /* Zoomed sprites      */
#define Sprites16x16  (VDP[1]&0x02)   /* 16x16/8x8 sprites   */
#define ScreenON      (VDP[1]&0x40)   /* Show screen         */
#define SpritesOFF    (VDP[8]&0x02)   /* Don't show sprites  */
#define SolidColor0   (VDP[8]&0x20)   /* Solid/Tran. COLOR 0 */
#define PALVideo      (VDP[9]&0x02)   /* PAL/NTSC video      */
#define FlipEvenOdd   (VDP[9]&0x04)   /* Flip even/odd pages */
#define InterlaceON   (VDP[9]&0x08)   /* Interlaced screen   */
#define ScanLines212  (VDP[9]&0x80)   /* 212/192 scanlines   */
#define MaskEdges     (VDP[25]&0x02)  /* Mask 8-pixel edges  */
#define ModeYJK       (VDP[25]&0x08)  /* YJK screen mode     */
#define ModeYAE       (VDP[25]&0x10)  /* YJK/YAE screen mode */
#define VScroll       VDP[23]
#define HScroll       (((int)(VDP[26]&0x3F&~(~HScroll512<<5)))<<3)-(int)(VDP[27]&0x07)
#define HScroll512    (VDP[25]&(VDP[2]>>5)&0x1)
#define VAdjust       (-((signed char)(VDP[18])>>4))
#define HAdjust       (-((signed char)(VDP[18]<<4)>>4))

#define MAXSCREEN   12      /* Highest screen mode supported */
#define MAXSPRITE1  4       /* Sprites/line in SCREEN 1-3    */
#define MAXSPRITE2  8       /* Sprites/line in SCREEN 4-8    */


/** VDP Address Register Masks *******************************/
static struct { UInt8 R2,R3,R4,R5,M2,M3,M4,M5; } VDPmask[MAXSCREEN + 2] =
{
    { 0x7F,0x00,0x3F,0x00,0x00,0x00,0x00,0x00 }, /* SCR 0:  TEXT 40x24  */
    { 0x7F,0xFF,0x3F,0xFF,0x00,0x00,0x00,0x00 }, /* SCR 1:  TEXT 32x24  */
    { 0x7F,0x80,0x3C,0xFF,0x00,0x7F,0x03,0x00 }, /* SCR 2:  BLK 256x192 */
    { 0x7F,0x00,0x3F,0xFF,0x00,0x00,0x00,0x00 }, /* SCR 3:  64x48x16    */
    { 0x7F,0x80,0x3C,0xFC,0x00,0x7F,0x03,0x03 }, /* SCR 4:  BLK 256x192 */
    { 0x60,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 5:  256x192x16  */
    { 0x60,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 6:  512x192x4   */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 7:  512x192x16  */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 8:  256x192x256 */
    { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 }, /* SCR 9:  NONE        */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 10: YAE 256x192 */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 11: YAE 256x192 */
    { 0x20,0x00,0x00,0xFC,0x1F,0x00,0x00,0x03 }, /* SCR 12: YJK 256x192 */
    { 0x7C,0xF8,0x3F,0x00,0x03,0x07,0x00,0x00 }  /* SCR 0:  TEXT 80x24  */
};

static UInt32 defaultPalette[16] = {
    0x000000, 0x000000, 0x20c020, 0x60e060, 0x2020e0, 0x4060e0, 0xa02020, 0x40c0e0,
    0xe02020, 0xe06060, 0xc0c020, 0xc0c080, 0x208020, 0xc040a0, 0xa0a0a0, 0xe0e0e0
};

/*** Initial palette: ***/
static const UInt8 PalInit[16][3] =
{
    {0x00,0x00,0x00},{0x00,0x00,0x00},{0x20,0xC0,0x20},{0x60,0xE0,0x60},
    {0x20,0x20,0xE0},{0x40,0x60,0xE0},{0xA0,0x20,0x20},{0x40,0xC0,0xE0},
    {0xE0,0x20,0x20},{0xE0,0x60,0x60},{0xC0,0xC0,0x20},{0xC0,0xC0,0x80},
    {0x20,0x80,0x20},{0xC0,0x40,0xA0},{0xA0,0xA0,0xA0},{0xE0,0xE0,0xE0}
};


void RefreshLineTx80(byte Y);
void RefreshLine0(byte Y);
void RefreshLine1(byte Y);
void RefreshLine2(byte Y);
void RefreshLine3(byte Y);
void RefreshLine4(byte Y);
void RefreshLine5(byte Y);
void RefreshLine6(byte Y);
void RefreshLine7(byte Y);
void RefreshLine8(byte Y);
void RefreshLine10(byte Y);
void RefreshLine12(byte Y);

/** Screen Mode Handlers [number of screens + 1] *************/
static void (*RefreshLine[MAXSCREEN+2])(UInt8 Y) =
{
    RefreshLine0,   /* SCR 0:  TEXT 40x24  */
    RefreshLine1,   /* SCR 1:  TEXT 32x24  */
    RefreshLine2,   /* SCR 2:  BLK 256x192 */
    RefreshLine3,   /* SCR 3:  64x48x16    */
    RefreshLine4,   /* SCR 4:  BLK 256x192 */
    RefreshLine5,   /* SCR 5:  256x192x16  */
    RefreshLine6,   /* SCR 6:  512x192x4   */
    RefreshLine7,   /* SCR 7:  512x192x16  */
    RefreshLine8,   /* SCR 8:  256x192x256 */
    0,              /* SCR 9:  NONE        */
    RefreshLine10,  /* SCR 10: YAE 256x192 */
    RefreshLine10,  /* SCR 11: YAE 256x192 */
    RefreshLine12,  /* SCR 12: YJK 256x192 */
    RefreshLineTx80 /* SCR 0:  TEXT 80x24  */
};


static UInt32 curSystemTime;
static int    VRAMpage;
static int    ChrTabO;
static int    ColTabO;
static int    ChrGenO;
static int    SprGenO;
static int    SprTabO;
static int    ChrGenM;
static int    ChrTabM;
static int    ColTabM;
static int    SprTabM;
static UInt16 VAddr;
static UInt8  VKey;
static UInt8  PKey;
static UInt8  WKey;
static UInt8  FGColor;
static UInt8  BGColor;
static UInt8  XFGColor;
static UInt8  XBGColor;
static int    ScanLine;
static UInt8  VDPData;
static UInt8  PLatch;
static UInt8  ALatch;
static UInt8  BFlag;
static UInt8  BCount;
static UInt8  Drawing;
static int    Palette[16];
static UInt32 evenOddPage = 0;
int loopTime = 0;
int lineTime = 0;
int phase = 0;


UInt8  VRAM[VRAM_SIZE];
UInt8  VDP[64];
UInt8  VDPStatus[16];
int    ScrMode;


static UInt8 SetScreen(void)
{
    switch (((VDP[0] & 0x0E)>>1) | (VDP[1] & 0x18)) {
    case 0x10: ScrMode = 0; break;
    case 0x00: ScrMode = 1; break;
    case 0x01: ScrMode = 2; break;
    case 0x08: ScrMode = 3; break;
    case 0x02: ScrMode = 4; break;
    case 0x03: ScrMode = 5; break;
    case 0x04: ScrMode = 6; break;
    case 0x05: ScrMode = 7; break;
    case 0x07: ScrMode = 8; break;
    case 0x12: ScrMode = MAXSCREEN + 1; break;
    }

    ChrTabO = (int)(VDP[2] & VDPmask[ScrMode].R2) << 10;
    ChrTabM = ((int)(VDP[2] | ~VDPmask[ScrMode].M2) << 10) | ((1 << 10) - 1);

    ChrGenO = (int)(VDP[4] & VDPmask[ScrMode].R4) << 11;
    ChrGenM = ((int)(VDP[4] | ~VDPmask[ScrMode].M4) << 11) | 0x007FF;

    ColTabO = ((int)(VDP[3] & VDPmask[ScrMode].R3) << 6) + ((int)VDP[10] << 14);
    ColTabM = ((int)(VDP[3] | ~VDPmask[ScrMode].M3) << 6) | 0x1C03F;

    SprTabO = ((int)(VDP[5] & VDPmask[ScrMode].R5) << 7) + ((int)VDP[11] << 15);
    SprTabM = ((int)(VDP[5] | ~VDPmask[ScrMode].M5) << 7) | 0x1807F;

    SprGenO = (int)VDP[6] << 11;

    return ScrMode;
}

static void VDPOut(UInt8 R, UInt8 V)
{ 
    switch (R) {
    case 0: 
        /* Reset HBlank interrupt if disabled */
        if ((VDPStatus[1] & 0x01) && !(V & 0x10)) {
            VDPStatus[1] &= 0xFE;
            SetIRQ(~INT_IE1);
        }

        /* Set screen mode */
        if (VDP[0] != V) {
            VDP[0] = V;
            SetScreen(); 
        }
        break;

    case 1: 
        /* Set/Reset VBlank interrupt if enabled or disabled */
        if (VDPStatus[0] & 0x80) {
            SetIRQ(V & 0x20? INT_IE0:~INT_IE0);
        }

        /* Set screen mode */
        if (VDP[1] != V) { 
            VDP[1] = V;
            SetScreen(); 
        }
        break;

    case 2: 
        ChrTabO = (int)(V & VDPmask[ScrMode].R2) << 10;
        ChrTabM = ((int)(V | ~VDPmask[ScrMode].M2) << 10) | ((1 << 10) - 1);
        break;

    case 3: 
        ColTabO = ((int)(V & VDPmask[ScrMode].R3) << 6) + ((int)VDP[10] << 14);
        ColTabM = ((int)(V | ~VDPmask[ScrMode].M3) << 6) | 0x1C03F;
        break;

    case 4: 
        ChrGenO = (int)(V & VDPmask[ScrMode].R4) << 11;
        ChrGenM = ((int)(V | ~VDPmask[ScrMode].M4) << 11) | 0x007FF;
        break;

    case 5: 
        SprTabO = ((int)(V & VDPmask[ScrMode].R5) << 7) + ((int)VDP[11] << 15);
        SprTabM = ((int)(V | ~VDPmask[ScrMode].M5) << 7) | 0x1807F;
        break;

    case 6: 
        V &= 0x3F;
        SprGenO = (int)V << 11;
        break;

    case 7: 
        FGColor=V>>4;
        BGColor=V&0x0F;
        break;

    case 9:
        if (VPeriod == CPU_VPERIOD_PAL) {
            V |= 0x02;
        }
        else {
            V &= ~0x02;
        }
        break;

    case 10: 
        V &= 0x07;
        ColTabO = ((int)(VDP[3] & VDPmask[ScrMode].R3) << 6) + ((int)V << 14);
        break;

    case 11: 
        V &= 0x03;
        SprTabO = ((int)(VDP[5] & VDPmask[ScrMode].R5) << 7) + ((int)V << 15);
        break;

    case 14: 
        V &= VRAMPages - 1;
        VRAMpage = (int)V << 14;
        break;

    case 15: 
        V &= 0x0F;
        break;

    case 16: 
        V &= 0x0F;
        PKey = 1;
        break;

    case 17: 
        V &= 0xBF;
        break;

    case 25: 
        VDP[25] = V;
        SetScreen();
        break;

    case 44: 
        VDPWrite(V);
        break;

    case 46: 
        VDPDraw(V);
        break;
    }

    VDP[R] = V;
} 



void vdpInit(UInt32 systemTime, VdpVersion version) 
{
    int i;

    curSystemTime = systemTime;
    loopTime = 0;
    lineTime = 0;
    phase    = 0;
    ChrGenO  = 0;
    ChrTabO  = 0;
    ColTabO  = 0;
    SprGenO  = 0;
    SprTabO  = 0;
    ChrGenM  = ~0;
    ChrTabM  = ~0;
    ColTabM  = ~0;
    SprTabM  = ~0;
    VRAMpage = 0;
    BFlag    = 0;
    BCount   = 0;
    Drawing  = 0;
    VAddr    = 0;
    VKey     = 1;
    PKey     = 1;
    WKey     = 0;
    FGColor  = 0;
    BGColor  = 0;
    XFGColor = 0;
    XBGColor = 0;
    ScrMode  = 1;
    ScanLine = 0;
    VDPData  = 0xff;
    PLatch   = 0;
    ALatch   = 0;

    memset(Palette, 0, sizeof(Palette));
    memset(VRAM, 0, VRAM_SIZE);
    memset(VDPStatus, 0, sizeof(VDPStatus));
    memset(VDP, 0, sizeof(VDP));

    memset(&MMC, 0, sizeof(MMC));

    VDPStatus[0] = 0x9f;
    VDPStatus[2] = 0x6c;
        
    VDP[1] = 0x10;
    VDP[2] = 0xff;
    VDP[3] = 0xff;
    VDP[4] = 0xff;
    VDP[5] = 0xff;

    if (version == VDP_V9958) {
        VDPStatus[1] |= 0x04;
    }

    /* Update VDP register 9 */
    if (VPeriod == CPU_VPERIOD_PAL) {
        VDP[9] |= 0x02;
    }
    else {
        VDP[9] &= ~0x02;
    }

    memcpy(Palette, defaultPalette, sizeof(Palette));

    for (i = 0; i < 16; i++) {
        SetColor(i, Palette[i]);
    }
}

void vdpDestroy() {
}

UInt8 vdpRead(UInt32 systemTime) {
    UInt8 vdpData;

    vdpSync(systemTime);

    /* Read from VRAM data buffer */
    vdpData = VDPData;

    /* Reset VAddr latch sequencer */
    VKey = 1;

    /* Fill data buffer with a new value */
    VDPData = *MAP_VRAM(VRAMpage + VAddr);

    /* Increment VRAM address */
    VAddr=(VAddr+1)&0x3FFF;

    /* If rolled over, modify VRAM page# */
    if (VAddr == 0 && ScrMode > 3) {
        VDP[14] = (VDP[14] + 1) & (VRAMPages - 1);
        VRAMpage = (int)VDP[14] << 14;
    }
    return vdpData;
}

UInt8 vdpReadStatus(UInt32 systemTime) {
    UInt8 vdpStatus;

    vdpSync(systemTime);

    /* Read an appropriate status register */
    vdpStatus=VDPStatus[VDP[15]];
    /* Reset VAddr latch sequencer */
    VKey=1;
    /* Update status register's contents */
    switch(VDP[15])
    {
    case 0: VDPStatus[0]&=0x5F;SetIRQ(~INT_IE0);break;
    case 1: VDPStatus[1]&=0xFE;SetIRQ(~INT_IE1);break;
    case 7: VDPStatus[7]=VDP[44]=VDPRead();break;
    }

    /* Return the status register value */
    return vdpStatus;
}

void vdpWrite(UInt32 systemTime, UInt8 Value)
{
    vdpSync(systemTime);

    VKey=1;
    if (WKey) {
        /* VDP set for writing */
        VDPData = *MAP_VRAM(VRAMpage + VAddr) = Value;
        VAddr = (VAddr + 1) & 0x3FFF;
    }
    else {
        /* VDP set for reading */
        VDPData = *MAP_VRAM(VRAMpage + VAddr);
        VAddr = (VAddr + 1) & 0x3FFF;
        *MAP_VRAM(VRAMpage + VAddr) = Value;
    }

    /* If VAddr rolled over, modify VRAM page# */
    if (VAddr == 0 && ScrMode > 3) {
        VDP[14] = (VDP[14] + 1 )& (VRAMPages - 1);
        VRAMpage = (int)VDP[14] << 14;
    }

    return;
}

void vdpWriteLatch(UInt32 systemTime, UInt8 Value) 
{
    vdpSync(systemTime);

    if (VKey) { 
        ALatch=Value;
        VKey=0; 
    }
    else if (Value & 0x80) {
        VKey=1;
        /* Writing into VDP registers */
        VDPOut(Value&0x3F,ALatch);
    }
    else {
        VKey = 1;
        
        /* Set the VRAM access address */
        VAddr = (((UInt16)Value << 8) + ALatch) & 0x3FFF;
        
        /* WKey=1 when VDP set for writing into VRAM */
        WKey=Value&0x40;
        
        /* When set for reading, perform first read */
        if (!WKey) {
            VDPData = *MAP_VRAM(VRAMpage + VAddr);
            VAddr = (VAddr + 1) & 0x3FFF;
            
            if (VAddr == 0 && ScrMode > 3) {
                VDP[14] = (VDP[14] + 1) & (VRAMPages - 1);
                VRAMpage = (int)VDP[14] << 14;
            }
        }
    }
    return;
}

void vdpWritePaletteLatch(UInt32 systemTime, UInt8 Value) {
    vdpSync(systemTime);
    if (PKey) { 
        PLatch = Value;
        PKey = 0; 
    }
    else {
        int palEntry = VDP[16];
        PKey = 1;

        /* Set new color for palette entry J */
        Palette[palEntry] = (((UInt32)(PLatch & 0x70) * 255 / 112) << 16) |
                            (((UInt32)(Value & 0x07) * 255 / 7) << 8) |
                            ((UInt32)(PLatch & 0x07) * 255 / 7);
        SetColor(palEntry, Palette[palEntry]);
        /* Next palette entry */
        VDP[16] = (palEntry + 1) & 0x0F;
    }
    return;
}

void vdpWriteRegister(UInt32 systemTime, UInt8 Value) 
{
    int J;

    vdpSync(systemTime);

    J=VDP[17]&0x3F;
    if(J!=17) VDPOut(J,Value);
    if(!(VDP[17]&0x80)) VDP[17]=(J+1)&0x3F;

    return;
}

int vdpRefreshLine(Z80 *R) 
{
    int firstLine = (PALVideo ? (ScanLines212 ? 35 : 45) : (ScanLines212 ? 8 : 18)) + VAdjust;
    int maxLine   = PALVideo ? 313 : 262;
    int J;

    if (ScanLine < firstLine) {
        if (phase == 0) { // Left border
            loopTime = (HPeriod - (ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256)) / 2;
            lineTime = HPeriod - loopTime;
            phase = 1;

            if (ScanLine == 0) {
                int screenMode = 0;
                VDPStatus[2] ^= 0x02;
                Drawing = 1;

                if(!ModeYJK||(ScrMode<7)||(ScrMode>8)) {
                    screenMode = ScrMode;
                }
                else if(ModeYAE) screenMode = 10;
                else screenMode = 12;

                RefreshScreen(screenMode, evenOddPage, InterlaceON);

                if(BCount) {
                    BCount--;
                }
                else {
                    BFlag=!BFlag;
                    if(!VDP[13]) { 
                        XFGColor=FGColor;
                        XBGColor=BGColor; 
                    }
                    else {
                        BCount=(BFlag? VDP[13]&0x0F:VDP[13]>>4)*10;
                        if(BCount)
                        {
                            if(BFlag) { XFGColor=FGColor;XBGColor=BGColor; }
                            else      { XFGColor=VDP[12]>>4;XBGColor=VDP[12]&0x0F; }
                        }
                    }
                }
            }

            if (ScanLine == firstLine - 1) {
                VDPStatus[2]&=0xBF;
            }
        }
        else if (phase == 1) { // draw area
            phase = 2;
            VDPStatus[2] &= ~0x20;
            loopTime = ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256;
            lineTime -= loopTime;
        }
        else if (phase == 2) { // Right border
            loopTime = lineTime;
            phase = 0;
            VDPStatus[2] |= 0x20;
            
            ScanLine++;
        }
    }
    else {
        int curLine = ScanLine - firstLine;

        if (phase == 0) { // Left border
            phase = 1;
            loopTime = (HPeriod - (ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256)) / 2;
            lineTime = HPeriod - loopTime;
            
            J = ScanLines212 ? 212 : 192;
            if (curLine == J) {
                VDPStatus[0] |= 0x80;
                VDPStatus[2] |= 0x40;
                if (VDP[1] & 0x20) {
                    SetIRQ(INT_IE0);
                }
            }
        }
        else if (phase == 3) { // Right border
            phase = 0;
            loopTime = lineTime;
            VDPStatus[2] |= 0x20;
            
            ScanLine++; 
            if (ScanLine == maxLine) {
                ScanLine = 0;
            }
        }
        else if (phase == 1) { // Right part of draw area
            phase = 2;
            VDPStatus[2] &= ~0x20;

            loopTime = (ScrMode == 0 || ScrMode == MAXSCREEN + 1 ? CPU_H240 : CPU_H256) - 80;
            lineTime -= loopTime;

            J=PALVideo? 256:ScanLines212? 245:235;

            if(curLine==J) {
                VDPStatus[1]&=0xFE;
                SetIRQ(~INT_IE1);
            }
            if(curLine<J) {
                J=(((curLine+VScroll)&0xFF)-VDP[19])&0xFF;
                if(J==1) {
                    VDPStatus[1]|=0x01;
                    if(VDP[0]&0x10) {
                        SetIRQ(INT_IE1);
                    }
                }
                else {
                    if(!(VDP[0]&0x10)) VDPStatus[1]&=0xFE;
                }
            }
        }
        else { // Left part of draw area
            phase = 3;
            loopTime = 80;
            lineTime -= loopTime;

            if (curLine == (ScanLines212? 212:192)) Drawing = 0;
            if (Drawing && curLine < 256) {
                if(!ModeYJK||(ScrMode<7)||(ScrMode>8)) {
                    (RefreshLine[ScrMode])(curLine);
                }
                else {
                    if(ModeYAE) RefreshLine10(curLine);
                    else RefreshLine12(curLine);
                }
            }
        }
    }
    
    return loopTime;
}

void vdpSync(UInt32 systemTime) {
    LoopVDP(37+systemTime - curSystemTime);
    curSystemTime = systemTime;
}

int vdpSaveState(FILE* F) {
    unsigned int State[32];
    int J = 0;

    State[J++] = VDPData;
    State[J++] = PLatch;
    State[J++] = ALatch;
    State[J++] = VAddr;
    State[J++] = VKey;
    State[J++] = PKey;
    State[J++] = WKey;
    State[J++] = ScanLine;
    State[J++] = VRAMpage;
    State[J++] = ChrTabO;
    State[J++] = ColTabO;
    State[J++] = ChrGenO;
    State[J++] = SprGenO;
    State[J++] = SprTabO;
    State[J++] = ChrGenM;
    State[J++] = ChrTabM;
    State[J++] = ColTabM;
    State[J++] = SprTabM;
    State[J++] = FGColor;
    State[J++] = BGColor;
    State[J++] = XFGColor;
    State[J++] = XBGColor;
    State[J++] = ScrMode;
    State[J++] = curSystemTime;
    State[J++] = loopTime;
    State[J++] = lineTime;
    State[J++] = phase;
 
    if (fwrite(VDP, 1, sizeof(VDP), F) != sizeof(VDP)) { 
        fclose(F);
        return 0; 
    }

    if (fwrite(&MMC, 1, sizeof(MMC), F) != sizeof(MMC)) { 
        fclose(F);
        return 0; 
    }

    if (fwrite(VDPStatus, 1, sizeof(VDPStatus), F) != sizeof(VDPStatus)) { 
        fclose(F);
        return 0; 
    }
    if (fwrite(Palette, 1, sizeof(Palette), F) != sizeof(Palette)) { 
        fclose(F);
        return 0; 
    }
    if (fwrite(VRAM, 1, VRAM_SIZE, F) != VRAM_SIZE) { 
        fclose(F);
        return 0; 
    }
    if (fwrite(State, 1, sizeof(State), F) != sizeof(State)) { 
        fclose(F);
        return 0; 
    }
    return 1;
}

int vdpLoadState(FILE* F) {
    unsigned int State[32];
    int J = 0;
    int I;

    if (fread(VDP, 1, sizeof(VDP), F) != sizeof(VDP)) { 
        fclose(F);
        return 0; 
    }

    if (fread(&MMC, 1, sizeof(MMC), F) != sizeof(MMC)) { 
        fclose(F);
        return 0; 
    }

    if (fread(VDPStatus, 1, sizeof(VDPStatus), F) != sizeof(VDPStatus)) { 
        fclose(F);
        return 0; 
    }

    if (fread(Palette, 1, sizeof(Palette), F) != sizeof(Palette)) { 
        fclose(F);
        return 0; 
    }

    if (fread(VRAM, 1, VRAM_SIZE, F) != VRAM_SIZE) { 
        fclose(F);
        return 0; 
    }

    if (fread(State, 1, sizeof(State), F) != sizeof(State)) { 
        fclose(F);
        return 0; 
    }

    VDPData    = State[J++];
    PLatch     = State[J++];
    ALatch     = State[J++];
    VAddr      = State[J++];
    VKey       = State[J++];
    PKey       = State[J++];
    WKey       = State[J++];
    ScanLine   = State[J++];
    VRAMpage   = State[J++];
    ChrTabO    = State[J++];
    ColTabO    = State[J++];
    ChrGenO    = State[J++];
    SprGenO    = State[J++];
    SprTabO    = State[J++];
    ChrGenM    = State[J++];
    ChrTabM    = State[J++];
    ColTabM    = State[J++];
    SprTabM    = State[J++];
    FGColor    = State[J++];
    BGColor    = State[J++];
    XFGColor   = State[J++];
    XBGColor   = State[J++];
    ScrMode    = State[J++];
    curSystemTime = State[J++];
    loopTime   = State[J++];
    lineTime   = State[J++];
    phase      = State[J++];

    /* Set palette */
    for (I = 0; I < 16; I++)
        SetColor(I, Palette[I]);

    /* Set screen mode and VRAM table addresses */
    SetScreen();

    return 1;
}

#include "common.h"

