/** fMSX: portable MSX emulator ******************************/
/**                                                         **/
/**                          MSX.c                          **/
/**                                                         **/
/** This file contains implementation for the MSX-specific  **/
/** hardware: slots, memory mapper, PPIs, PSG, clock,       **/
/** etc. Initialization code and definitions needed for the **/
/** machine-dependent drivers are also here.                **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-2003                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/
  
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <direct.h>
#include <time.h>

#include "MSX.h"
#include "AY8910.h"
#include "YM2413.h"
#include "Y8950.h"
#include "SCC.h"
#include "KeyClick.h"
#include "audioMixer.h"
#include "romMapper.h"
#include "VDP.h"
#include "Casette.h"
#include "Disk.h"
#include "ziphelper.h"

#include "romMapperFMPAC.h"

#include "romDISK.h"
#include "romFMPAC.h"
#include "romKANJI.h"
//#include "romRS232.h"

#include <direct.h>

static void LoopZ80(register Z80 *R);

int syncCount;
extern int WaitForSync(void);

static int isRamSegment[2][4];
static byte modeRegister[2];
static byte memoryBank[2][0x20000];
static byte* internalMemoryBank[2][4];
static int loopTime;


typedef enum { EN_NONE = 0, EN_SCC, EN_SCCPLUS } SccEnable;


/** User-defined parameters for fMSX *************************/
int  SyncPeriod    = 3579545;
int MsxFrequency   = 3579545;
int enableYM2413   = 1;
int enableY8950    = 1;
int  VPeriod       = CPU_VPERIOD_PAL;  /* Cycles per VBlank  */
int  HPeriod       = CPU_HPERIOD;  /* CPU cycles per HBlank  */
byte SaveCMOS      = 0;            /* Save CMOS.ROM on exit  */
byte MSXVersion    = 1;            /* 0=MSX1,1=MSX2,2=MSX2+  */
char* CMOSname     = "DEFAULT.cmos";
byte* MSXSysRom    = NULL;
byte* MSXSysRomExt = NULL;
byte* KoreaMSX1    = NULL;
byte* KoreaMSX2    = NULL;
byte JoyTypeA      = 0;            /* 0=None,1=Joystick,     */
byte JoyTypeB      = 0;            /* 2=MouseAsJstk,3=Mouse  */
RomType ROMTypeA   = ROM_UNKNOWN;  /* MegaROM types          */
RomType ROMTypeB   = ROM_UNKNOWN;
int  RAMPages      = 4;            /* Number of RAM pages    */
int  VRAMPages     = 2;            /* Number of VRAM pages   */

/** Main hardware: CPU, RAM, VRAM, mappers *******************/
Z80 CPU;                           /* Z80 CPU state and regs */

byte *RAM[8];                      /* Main RAM (8x8kB pages) */
byte *EmptyRAM;                    /* Empty RAM page (8kB)   */
byte *SRAM[2];                     /* SRAM (battery backed)  */
byte *MemMap[4][4][8];             /* Memory maps:           */
                                   /* [PPage][SPage][Addr]   */
byte IRQPending;                   /* Pending interrupts     */

byte RAMData[256 * 0x4000];        /* RAM Mapper contents    */
byte RAMMapper[4];                 /* RAM Mapper state       */
byte RAMMask;                      /* RAM Mapper mask        */

byte *ROMData[2];                  /* ROM Mapper contents    */
byte ROMMapper[2][4];              /* ROM Mappers state      */
byte ROMMask[2];                   /* ROM Mapper masks       */
int  ROMSize[2];
word SRAMenabled[2];
byte EnWrite[8];                   /* 1 if write enabled     */
byte PSL[4],SSL[4];                /* Lists of current slots */
byte PSLReg,SSLReg;      /* Storage for A8h port and (FFFFh) */

/** Working directory names **********************************/
char *ProgDir    = NULL;           /* Program directory      */
char *WorkDir;                     /* Working directory      */

/** Cartridge files used         *****************************/
char *CartA      = "CARTA.ROM";    /* Cartridge A ROM file   */
char *CartB      = "CARTB.ROM";    /* Cartridge B ROM file   */
char *CartAZip   = NULL;
char *CartBZip   = NULL;

/** Disk images used by      *********************************/
char *DiskA      = "DRIVEA.DSK";   /* Drive A disk image  */
char *DiskB      = "DRIVEB.DSK";   /* Drive B disk image  */
char *DiskAZip   = NULL;
char *DiskBZip   = NULL;

/** Emulation state saving ***********************************/
char *StateName  = 0;              /* State file (autogen-d) */

/** Printer **************************************************/
char *PrnName    = NULL;           /* Printer redirect. file */
FILE *PrnStream;

/** Cassette tape ********************************************/
char *CasName    = "DEFAULT.CAS";  /* Tape image file        */
char *CasNameZip = NULL;
int   CasPos = 0;
/** Serial port **********************************************/
char *ComName    = NULL;           /* Serial redirect. file  */
FILE *ComIStream;
FILE *ComOStream;

// Use static FMPACs for now. Change when memory model changes
RomMapperFMPAC* fmpac;
RomMapperFMPAC* fmpacA;
RomMapperFMPAC* fmpacB;

/** Kanji font ROM *******************************************/
byte *Kanji;                       /* Kanji ROM 4096x32      */
int  KanLetter;                    /* Current letter index   */
byte KanCount;                     /* Byte count 0..31       */

/** Keyboard and mouse ***************************************/
byte KeyMap[16];                   /* Keyboard map           */
byte Buttons[2];                   /* Mouse button states    */
byte MouseDX[2],MouseDY[2];        /* Mouse offsets          */
byte OldMouseX[2],OldMouseY[2];    /* Old mouse coordinates  */
byte MCount[2];                    /* Mouse nibble counter   */

/** General I/O registers: i8255 *****************************/
I8255 PPI;                         /* i8255 PPI at A8h-ABh   */
byte IOReg;                        /* Storage for AAh port   */

/** Sound hardware: PSG, SCC, OPLL ***************************/
AY8910*         ay8910;
YM_2413*        ym2413;
Y8950*          y8950;
SCC*            scc;
AudioKeyClick*  keyClick;
Mixer*          TheMixer;

int SCCOn[2];                      /* 1 = SCC page active    */

/** Serial I/O hardware: i8251+i8253 *************************/
I8251 SIO;                         /* SIO registers & state  */

/** Real-time clock ******************************************/
byte RTCReg,RTCMode;               /* RTC register numbers   */
byte RTC[4][13];                   /* RTC registers          */

UInt32 systemTime;
UInt32 nextSyncTime;


/** Places in DiskROM to be patched with ED FE C9 ************/
word DiskPatches[] = { 0x4010,0x4013,0x4016,0x401C,0x401F,0 };

/** Places in BIOS to be patched with ED FE C9 ***************/
word BIOSPatches[] = { 0x00E1,0x00E4,0x00E7,0x00EA,0x00ED,0x00F0,0x00F3,0 };

/** Internal Functions ***************************************/
/** These functions are defined and internally used by the  **/
/** code in MSX.c.                                          **/
/*************************************************************/
byte* loadRomImage(const char *fileName, const char *fileInZipFile, int* size);
int   mapRom(int slot, byte* buf, int size);
void mapSccPlusRom(int slot);

int  GuessROM(const byte *Buf,int Size);
void SetMegaROM(int Slot,byte P0,byte P1,byte P2,byte P3);
void MapROM(void* arg, UInt16 A, UInt8 V, UInt32 systemTime);
void SSlot(byte V);               /* Switch secondary slots          */
void Printer(byte V);             /* Send a character to a printer   */
void PPIOut(byte New,byte Old);   /* Set PPI bits (key click, etc.)  */
byte RTCIn(byte R);               /* Read RTC registers              */
word StateID(void);               /* Compute emulation state ID      */

// The Slot* methods and structures will in time replace the old memory mapper.

typedef void (*SlotWriteFunc)(void*, UInt16, UInt8, UInt32);
typedef void (*SlotReadFunc)(void*, UInt16, UInt8, UInt32);

typedef struct SlotInfo {
    int           type;
    UInt16        startpage;
    UInt16        pages;
    int           writeCache;
    int           readCache;
    SlotWriteFunc wrfunc;
    SlotWriteFunc rdfunc;
    void*         ref;
} SlotInfo;

SlotInfo slotInfoTable[4][4][8];


void slotInfoRegister(int type, int slot, int sslot, int startpage, int pages,
                      SlotWriteFunc wrfunc, SlotWriteFunc rdfunc, void* ref)
{
    SlotInfo* slotInfo = &slotInfoTable[slot][sslot][startpage];

    slotInfo->type  = type;
    slotInfo->pages = pages;

    while (pages--) {
        slotInfo->startpage = startpage;
        slotInfo->wrfunc    = wrfunc;
        slotInfo->rdfunc    = rdfunc;
        slotInfo->ref       = ref;
        slotInfo++;
    }
}

char* sramCreateFilename(char* romFilename) {
    static char SRAMfileName[512];
    char* dst = SRAMfileName + 512;
    char* src;
    
    src = romFilename + strlen(romFilename);

    *--dst = '\0';
    *--dst = 'm';
    *--dst = 'a';
    *--dst = 'r';
    *--dst = 's';

    while (*src != '.' && src >= romFilename) {
        src--;
    }

    while (*src != '/' && *src != '\\' && src >= romFilename) {
        *--dst = *src--;
    }

    *--dst = '/';
    *--dst = 'M';
    *--dst = 'A';
    *--dst = 'R';
    *--dst = 'S';

    return dst;
}

// Temporary methods for mapping slots. They will be updated shortly
void memMapPage(int slot, int sslot, int page, UInt8* pageData, 
               int readCache, int writeCache) 
{
    MemMap[slot][sslot][page] = pageData;
    EnWrite[page] = writeCache;
//    if (PSL[1] == slot) {
        RAM[page] = pageData;
//    }
}

void sramLoad(char* filename, UInt8* sram, int length, void* header, int headerLength) {
    FILE* file;

    memset(sram, 0xff, length);

    file = fopen(filename, "rb");
    if (file != NULL) {
        if (headerLength > 0) {
            char* readHeader[256];

            fread(readHeader, 1, headerLength, file);
            if (memcmp(readHeader, header, headerLength)) {
                fclose(file);
                return;
            }
        }
        fread(sram, 1, length, file);
        fclose(file);
    }
}

void sramSave(char* filename, UInt8* sram, int length, void* header, int headerLength) {
    FILE* file;

    file = fopen(filename, "wb");
    if (file != NULL) {
        if (headerLength > 0) {
            fwrite(header, 1, headerLength, file);
        }
        fwrite(sram, 1, length, file);
        fclose(file);
    }
}

UInt8* ramGetEmptyPage() {
    return EmptyRAM;
}

static byte* tryLoadSRAM(RomType romType, char* fileName) {
    FILE* file;
    byte* SRAM = NULL;
    int offset;

    switch (romType) {
    case ROM_ASCII8SRAM:
        SRAM = malloc(0x2000);
        memset(SRAM, NORAM, 0x2000);

        file = fopen(sramCreateFilename(fileName), "rb");
        if (file != NULL) {
            fread(SRAM, 1, 0x2000, file);
            fclose(file);
        }

        break;

    case ROM_ASCII16SRAM:
        SRAM = malloc(0x4000);
        memset(SRAM, NORAM, 0x4000);

        file = fopen(sramCreateFilename(fileName), "rb");
        if (file != NULL) {
            fread(SRAM, 1, 0x800, file);
            for (offset = 0x800; offset < 0x4000; offset += 0x800) {
                memcpy(SRAM + offset, SRAM, 0x800);
            }
            fclose(file);
        }
        break;

    case ROM_GAMEMASTER2:
        SRAM = malloc(0x4000);
        memset(SRAM, NORAM, 0x4000);

        file = fopen(sramCreateFilename(fileName), "rb");
        if (file != NULL) {
            if (fread(SRAM, 1, 0x2000, file) == 0x2000) {
                /* Mirror GameMaster2 SRAM as needed */
                memcpy(SRAM + 0x2000, SRAM + 0x1000, 0x1000);
                memcpy(SRAM + 0x3000, SRAM + 0x1000, 0x1000);
                memcpy(SRAM + 0x1000, SRAM, 0x1000);
            }
            fclose(file);
        }
        break;
    }

    return SRAM;
}

static void trySaveSRAM(RomType romType, byte* SRAM, char* fileName) {
    FILE* file;

    switch (romType) {
    case ROM_ASCII8SRAM:
        file = fopen(sramCreateFilename(fileName),"wb");
        if (file != NULL) {
            fwrite(SRAM, 1, 0x2000, file);
            fclose(file);
        }
        break;

    case ROM_ASCII16SRAM:
        file = fopen(sramCreateFilename(fileName),"wb");
        if (file != NULL) {
            fwrite(SRAM, 1, 0x800, file);
            fclose(file);
        }
        break;

    case ROM_GAMEMASTER2:
        file = fopen(sramCreateFilename(fileName),"wb");
        if (file != NULL) {
            fwrite(SRAM, 1, 0x1000, file);
            fwrite(SRAM + 0x2000, 1, 0x1000, file);
            fclose(file);
        }
    }
}



void ResetMSX(int hard) {
    VPeriod     = CPU_VPERIOD_PAL;
    HPeriod     = CPU_HPERIOD;
    SaveCMOS    = 0;
    MSXVersion  = 1;
    JoyTypeA    = 0;
    JoyTypeB    = 0;
    ROMTypeA    = ROM_UNKNOWN;
    ROMTypeB    = ROM_UNKNOWN;
    RAMPages    = 4;
    VRAMPages   = 2;

    if (hard) {
        memset(RAMData, NORAM, sizeof(RAMData));
    }

    memset(RAM, sizeof(RAM), 0);

    EmptyRAM = NULL;
    SRAM[0] = NULL;
    SRAM[1] = NULL;
    memset(MemMap, sizeof(MemMap), 0);

    memset(RAMMapper, sizeof(RAMMapper), 0);
    RAMMask = 0;

    memset(ROMData, sizeof(ROMData), 0);
    memset(ROMMapper, sizeof(ROMMapper), 0);
    memset(ROMMask, sizeof(ROMMask), 0);

    memset(EnWrite, sizeof(EnWrite), 0);
    memset(PSL, sizeof(PSL), 0);
    memset(SSL, sizeof(SSL), 0);
    PSLReg = 0;
    SSLReg = 0;

    ProgDir = NULL;
    WorkDir = NULL;

    CartA = "CARTA.ROM";
    CartB = "CARTB.ROM";

    DiskA = "DRIVEA.DSK";
    DiskB = "DRIVEB.DSK";

    StateName = 0;

    PrnName = NULL;
    PrnStream = NULL;

    CasName = "DEFAULT.CAS";
    
    ComName = NULL;
    ComIStream = NULL;
    ComOStream = NULL;

    Kanji = NULL;
    KanLetter = 0;
    KanCount = 0;
    loopTime = 0;

    fmpac = NULL;
    fmpacA = NULL;
    fmpacB = NULL;

    memset(KeyMap, sizeof(KeyMap), 0);
    memset(Buttons, sizeof(Buttons), 0);
    memset(MouseDX, sizeof(MouseDX), 0);
    memset(MouseDY, sizeof(MouseDY), 0);
    memset(OldMouseX, sizeof(OldMouseX), 0);
    memset(OldMouseY, sizeof(OldMouseY), 0);
    memset(MCount, sizeof(MCount), 0);

    memset(slotInfoTable, 0, sizeof(slotInfoTable));

    memset(&PPI, sizeof(PPI), 0);
    IOReg = 0;

    SCCOn[0] = EN_NONE;
    SCCOn[1] = EN_NONE;

    memset(&SIO, sizeof(SIO), 0);

    RTCReg = 0;
    RTCMode = 0;
    memset(RTC, sizeof(RTC), 0);
}

void ChangeCartridge(int drive, RomType romType, char* cart, char* cartZip) 
{
    byte* buf;
    int size;
    int J;

    for (J = 2; J < 6; J++) {
        if (RAM[J] == MemMap[drive+1][0][J]) RAM[J] = EmptyRAM;
    }


    ROMSize[drive] = 0;
    ROMMask[drive] = 0;
    ROMData[drive] = 0;
    ROMMapper[drive][0] = 0;
    ROMMapper[drive][1] = 0;
    ROMMapper[drive][2] = 0;
    ROMMapper[drive][3] = 0;
    SCCOn[drive] = EN_NONE;

    sccReset(scc);

    if (drive == 0) {
        ROMTypeA = romType;
        CartA    = cart;
        CartAZip = cartZip;

        switch (ROMTypeA) {
        case ROM_SCCPLUS:
            mapSccPlusRom(0);
            slotInfoRegister(ROM_ANY_MEGAROM, 1, 0, 2, 4, MapROM, NULL, (void*)1);
            break;

        case ROM_FMPAC:
            fmpacA = romMapperFMPACCreate("FmPacA.rom", romFMPAC, 0x10000, 1, 0, 2, 2, 
                                          (enableYM2413 ? ym2413 : NULL));
            break;

        default:
            /* Loading cartridge into slot A... */
            buf = loadRomImage(CartA, CartAZip, &size);
            if (buf != NULL) {
                ROMTypeA = GuessROM(buf, size);
                if (ROMTypeA == ROM_FMPAC) {
                    fmpacA = romMapperFMPACCreate(CartAZip != NULL ? CartAZip : CartA, buf, size, 1, 0, 2, 2,
                                                (enableYM2413 ? ym2413 : NULL));
                    break;
                }
                ROMSize[0] = size;
                mapRom(0, buf, size);
                free(buf);
            }

            J = (ROMMask[0] + 1) * 8192;

            if (J <= 0x8000) {
                ROMTypeA = ROM_UNKNOWN;
            }

            /* Guess mapper is not given */
            if (ROMTypeA != ROM_UNKNOWN) {
                SRAMenabled[0] = 0;
                SRAM[0] = tryLoadSRAM(ROMTypeA, CartAZip != NULL ? CartAZip : CartA);
                slotInfoRegister(ROM_ANY_MEGAROM, 1, 0, 2, 4, MapROM, NULL, (void*)1);
            }

            /* For Generic/16kB carts, set ROM pages as 0:1:N-2:N-1 */
            if((ROMTypeA == ROM_MSXDOS2) && (J > 0x8000)) {
                SetMegaROM(0, 0, 1, ROMMask[0] - 1, ROMMask[0]);
            }

            if (ROMTypeA == ROM_ASCII16 || ROMTypeA == ROM_ASCII16SRAM) {
                SetMegaROM(0, 0, 1, 0, 1);
            }

            if (ROMTypeA == ROM_ASCII8 || ROMTypeA == ROM_ASCII8SRAM) {
                SetMegaROM(0, 0, 0, 0, 0);
            }

            /* For Rtype cart, set ROM pages as 2E:2F:0:1 */
            if (ROMTypeA == ROM_RTYPE) {
                SetMegaROM(0, 0x2E, 0x2F, 0, 1);
            }
            if (ROMTypeA == ROM_CROSSBLAIM) {
                SetMegaROM(0, 0, 1, 0, 1);
            }
        }
    }
    else if (drive == 1) {
        ROMTypeB = romType;
        CartB    = cart;
        CartBZip = cartZip;

        switch (ROMTypeB) {
        case ROM_SCCPLUS:
            mapSccPlusRom(1);
            slotInfoRegister(ROM_ANY_MEGAROM, 2, 0, 2, 4, MapROM, NULL, (void*)2);
            break;

        case ROM_FMPAC:
            fmpacB = romMapperFMPACCreate("FmPacB.rom", romFMPAC, 0x10000, 2, 0, 2, 2, 
                                          (enableYM2413 ? ym2413 : NULL));
            break;

        default:
            /* Loading cartridge into slot B... */
            buf = loadRomImage(CartB, CartBZip, &size);
            if (buf != NULL) {
                ROMTypeB = GuessROM(buf, size);
                if (ROMTypeB == ROM_FMPAC) {
                    fmpacB = romMapperFMPACCreate(CartBZip != NULL ? CartBZip : CartB, buf, size, 2, 0, 2, 2,
                                                (enableYM2413 ? ym2413 : NULL));
                    break;
                }
                ROMSize[1] = size;
                mapRom(1, buf, size);
                free(buf);
            }

            J=(ROMMask[1]+1)*8192;

            if (J <= 0x8000) {
                ROMTypeB = ROM_UNKNOWN;
            }

            /* Guess mapper is not given */
            if (ROMTypeB != ROM_UNKNOWN) {
                SRAMenabled[1] = 0;
                SRAM[1] = tryLoadSRAM(ROMTypeB, CartBZip != NULL ? CartBZip : CartB);
                slotInfoRegister(ROM_ANY_MEGAROM, 2, 0, 2, 4, MapROM, NULL, (void*)2);
            }

            /* For Generic/16kB carts, set ROM pages as 0:1:N-2:N-1 */
            if ((ROMTypeB == ROM_MSXDOS2) && (J > 0x8000)) {
                SetMegaROM(1, 0, 1, ROMMask[1] - 1, ROMMask[1]);
            }

            if (ROMTypeB == ROM_ASCII16 || ROMTypeB == ROM_ASCII16SRAM) {
                SetMegaROM(1, 0, 1, 0, 1);
            }

            if (ROMTypeB == ROM_ASCII8 || ROMTypeB == ROM_ASCII8SRAM) {
                SetMegaROM(1, 0, 0, 0, 0);
            }

            /* For Rtype cart, set ROM pages as 2E:2F:0:1 */
            if (ROMTypeB == ROM_RTYPE) {
                SetMegaROM(1, 0x2E, 0x2F, 0, 1);
            }
            if (ROMTypeB == ROM_CROSSBLAIM) {
                SetMegaROM(1, 0, 1, 0, 1);
            }
        }
    }
}


/** StartMSX() ***********************************************/
/** Allocate memory, load ROM images, initialize hardware,  **/
/** CPU and start the emulation. This function returns 0 in **/
/** the case of failure.                                    **/
/*************************************************************/
int StartMSX(void)
{
    /*** CMOS ROM default values: ***/
    static const byte RTCInit[4][13]  =
    {
        {  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
        {  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
        {  0, 0, 0, 0,40,80,15, 4, 4, 0, 0, 0, 0 },
        {  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
    };

    int *T,I,J,K;
    byte *P;
    byte* buf;
    int size;

    /*** STARTUP CODE starts here: ***/

    T=(int *)"\01\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0";

    systemTime = 0;
    nextSyncTime = 0;
    
    /* Zero everyting */
    syncCount = 0;
    PrnStream=ComIStream=ComOStream=NULL;
    ROMData[0]=ROMData[1]=NULL;
    SRAM[0]  = NULL;
    SRAM[1]  = NULL;
    Kanji    = NULL;
    WorkDir  = NULL;
    SaveCMOS = 0;
    IRQPending = 0;

    /* Calculate IPeriod from VPeriod/HPeriod */
    if(HPeriod<CPU_HPERIOD) HPeriod=CPU_HPERIOD;
    if(VPeriod<CPU_VPERIOD_NTSC) VPeriod=CPU_VPERIOD_NTSC;

    /* Check parameters for validity */
    if(MSXVersion>2) MSXVersion=2;
    if((RAMPages<(MSXVersion? 8:4))||(RAMPages>256))
        RAMPages=MSXVersion? 8:4;
    if((VRAMPages<(MSXVersion? 8:2))||(VRAMPages>8))
        VRAMPages=MSXVersion? 8:2;

    /* Number of RAM pages should be power of 2 */
    /* Calculate RAMMask=(2^RAMPages)-1 */
    for(J=1;J<RAMPages;J<<=1);
    RAMPages=J;
    RAMMask=J-1;

    /* Initialize ROMMasks to zeroes for now */
    ROMMask[0]=ROMMask[1]=0;

    /* Joystick types are in 0..3 range */
    JoyTypeA&=3;
    JoyTypeB&=3;

    /* Allocate 16kB for the empty space (scratch RAM) */
    EmptyRAM=malloc(0x4000);
    memset(EmptyRAM,NORAM,0x4000);

    /* Reset memory map to the empty space */
    for(I=0;I<4;I++)
        for(J=0;J<4;J++)
            for(K=0;K<8;K++)
                MemMap[I][J][K]=EmptyRAM;

    /* Save current directory and cd to wherever system ROMs are */
    if(ProgDir) {
        if(WorkDir=getcwd(NULL,1024)) {
            chdir(ProgDir);
        }
    }

    /* Reset sound chips */
    mixerReset(TheMixer, systemTime);
    ay8910   = ay8910Create(TheMixer, systemTime);
    ym2413   = ym2413Create(TheMixer, systemTime);
    y8950    = y8950Create(TheMixer, systemTime);
    scc      = sccCreate(TheMixer, systemTime);
    keyClick = audioKeyClickCreate(TheMixer, systemTime);

    /* Initialize VDP */
    vdpInit(systemTime, MSXVersion > 0 ? VDP_V9958 : VDP_V9938);

    /* Map system roms */
    if (MSXSysRom != NULL) {
        MemMap[0][0][0]=MSXSysRom;
        MemMap[0][0][1]=MSXSysRom+0x2000;
        MemMap[0][0][2]=MSXSysRom+0x4000;
        MemMap[0][0][3]=MSXSysRom+0x6000;
    }

    if (MSXSysRomExt != NULL) {
        MemMap[3][1][0]=MSXSysRomExt;
        MemMap[3][1][1]=MSXSysRomExt+0x2000;
    }

    /* Korean Bios MSX 1 */
    if (KoreaMSX1 != NULL) {
        MemMap[0][0][4]=KoreaMSX1;
    }

    /* Korean Bios MSX 2 */
    if (KoreaMSX2 != NULL) {
        MemMap[0][1][2]=KoreaMSX2;
        MemMap[0][1][3]=KoreaMSX2+0x2000;
        MemMap[0][1][4]=KoreaMSX2+0x4000;
        MemMap[0][1][5]=KoreaMSX2+0x6000;
    }

    /* Map DISK rom */
    MemMap[3][1][2]=romDISK;
    MemMap[3][1][3]=romDISK+0x2000;

    /* Apply patches to BIOS */
    for (J=0; BIOSPatches[J]; J++) {
        P = MemMap[0][0][0] + BIOSPatches[J];
        P[0] = 0xED;
        P[1] = 0xFE;
        P[2] = 0xC9;
    }

    /* Apply patches to BDOS */
    for (J=0; DiskPatches[J]; J++) {
        P = MemMap[3][1][2] + DiskPatches[J] - 0x4000;
        P[0] = 0xED;
        P[1] = 0xFE;
        P[2] = 0xC9;
    }

    buf = loadRomImage(CMOSname, NULL, &size);
    if (buf != NULL && size == sizeof(RTC)) {
        memcpy(RTC, buf, sizeof(RTC));
    }
    else {
        memcpy(RTC, RTCInit, sizeof(RTC));
    }
    if (buf != NULL) {
        free(buf);
    }

    /* Kanji Support */
    Kanji=romKANJI;

// RS232 Not Supported Yet once BIOS is Empty
//    if (romRS232 != NULL) {
//        MemMap[3][0][2]=romRS232;
//        MemMap[3][0][3]=romRS232 + 0x2000;
//    }

    if (MSXVersion > 4) {
        fmpac = romMapperFMPACCreate("FMPAC.rom", romFMPAC, 0x10000, 3, 3, 2, 2, 
                                    (enableYM2413 ? ym2413 : NULL));
    }

    /* Map Cartridge rom */
    ChangeCartridge(0, ROMTypeA, CartA, CartAZip);
    ChangeCartridge(1, ROMTypeB, CartB, CartBZip);

    /* We are now back to working directory */
    if (WorkDir) chdir(WorkDir);

    /* Open stream for a printer */
    if (!PrnName) {
        PrnStream = stdout;
    }
    else {
        if (!(PrnStream = fopen(PrnName, "wb"))) {
            PrnStream = stdout;
        }
    }

    /* Open streams for serial IO */
    if (!ComName) { 
        ComIStream = stdin;
        ComOStream = stdout; 
    }
    else {
        if (!(ComOStream = ComIStream = fopen(ComName, "r+b"))) { 
            ComIStream = stdin;
            ComOStream = stdout; 
        }
    }

    /* Open disk images */

    ChangeDisk(0, DiskA, DiskAZip);
    ChangeDisk(1, DiskB, DiskBZip);

    tapeInsert(CasName, CasNameZip);
    tapeSetCurrentPos(CasPos);

    for (J = 0; J < 4; J++) {
        EnWrite[2*J]=0;                        /* Write protect ON for all slots */
        EnWrite[2*J+1]=0;                      /* Write protect ON for all slots */
        PSL[J]=SSL[J]=0;                   /* PSL=0:0:0:0, SSL=0:0:0:0       */
        MemMap[3][2][J*2]   = RAMData+(3-J)*0x4000;        /* RAMMap=3:2:1:0 */
        MemMap[3][2][J*2+1] = MemMap[3][2][J*2]+0x2000;
        RAMMapper[J]        = 3-J;
        RAM[J*2]            = MemMap[0][0][J*2];           /* Setting RAM    */
        RAM[J*2+1]          = MemMap[0][0][J*2+1];
    }

    /* Reset mouse coordinates/counters */
    for(J=0;J<2;J++) {
        Buttons[J]=MouseDX[J]=MouseDY[J]=OldMouseX[J]=OldMouseY[J]=MCount[J]=0;
    }

    /* Reset serial I/O */
    Reset8251(&SIO,ComIStream,ComOStream);

    /* Reset PPI chips and slot selectors */
    Reset8255(&PPI);
    PPI.Rout[0]=PSLReg=0x00;
    PPI.Rout[2]=IOReg=0x00;
    SSLReg=0x00;

    memset(KeyMap,0xFF,16);               /* Keyboard         */
    IRQPending=0x00;                      /* No IRQs pending  */
    SCCOn[0]=SCCOn[1]=EN_NONE;           /* SCCs off for now */
    RTCReg=RTCMode=0;                     /* Clock registers  */

    /* Reset CPU */
    z80Reset(&CPU, 0);
    
    /* Try loading emulation state */
    if(StateName) {
        J = LoadState(StateName);
        if (J == 0) {
            return 0;
        }
    }

    mixerReset(TheMixer, systemTime);

    /* Start execution of the code */
    LoopZ80(&CPU);

    return 1;
}

/** TrashMSX() ***********************************************/
/** Free resources allocated by StartMSX().                 **/
/*************************************************************/
void TrashMSX(void)
{
    int J;
    FILE *F;

    /* CMOS.ROM and GMASTER2.RAM are saved in the program directory */
    if(ProgDir) chdir(ProgDir);

    /* Save CMOS RAM, if present */
    if(SaveCMOS)
    {
        if(!(F=fopen(CMOSname,"wb"))) {
            SaveCMOS=0;
        }
        else {
            if(fwrite(RTC,1,sizeof(RTC),F)!=sizeof(RTC)) SaveCMOS=0;
            fclose(F);
        }
    }

    ay8910Destroy(ay8910);
    ym2413Destroy(ym2413);
    y8950Destroy(y8950);
    sccDestroy(scc);
    audioKeyClickDestroy(keyClick);

    if (fmpac)  romMapperFMPACDestroy(fmpac);
    if (fmpacA) romMapperFMPACDestroy(fmpacA);
    if (fmpacB) romMapperFMPACDestroy(fmpacB);

    trySaveSRAM(ROMTypeA, SRAM[0], CartAZip != NULL ? CartAZip : CartA);
    trySaveSRAM(ROMTypeB, SRAM[1], CartBZip != NULL ? CartBZip : CartB);

    /* Change back to working directory */
    if(WorkDir) chdir(WorkDir);

    /* Close all IO streams */
    if(PrnStream&&(PrnStream!=stdout))   fclose(PrnStream);
    if(ComOStream&&(ComOStream!=stdout)) fclose(ComOStream);
    if(ComIStream&&(ComIStream!=stdin))  fclose(ComIStream);

    /* Close disk and cas images, if present */

    for(J=0;J<MAXDRIVES;J++) ChangeDisk(J,0, 0);

    tapeInsert(0, 0);

    vdpDestroy();

    if (EmptyRAM   != NULL) { free(EmptyRAM);   EmptyRAM   = NULL; }
    if (SRAM[0]    != NULL) { free(SRAM[0]);    SRAM[0]    = NULL; }
    if (SRAM[1]    != NULL) { free(SRAM[1]);    SRAM[1]    = NULL; }
    if (ROMData[0] != NULL) { free(ROMData[0]); ROMData[0] = NULL; }
    if (ROMData[1] != NULL) { free(ROMData[1]); ROMData[1] = NULL; }
    if (WorkDir    != NULL) { free(WorkDir);    WorkDir    = NULL; }
}

/** RdZ80() **************************************************/
/** Z80 emulation calls this function to read a byte from   **/
/** address A of Z80 address space. Now moved to Z80.c and  **/
/** made inlined to speed things up.                        **/
/*************************************************************/
byte RdZ80(word A)
{
    int sccType;

    if (A == 0xFFFF) {
        return PSL[3] == 3 ? ~SSLReg:RAM[7][0x1FFF];
    }

    sccType = SCCOn[0];
    if (sccType && PSL[A >> 14] == 1) {
        if (sccType == EN_SCC && A >= 0x9800 && A < 0xA000) {
            return sccRead(scc, (UInt8)(A & 0xff));
        }
        if (sccType == EN_SCCPLUS && A >= 0xB800 && A < 0xC000) {
            return sccPlusRead(scc, (UInt8)(A & 0xff));
        }
    }
    
    sccType = SCCOn[1];
    if (sccType && PSL[A >> 14] == 2) {
        if (sccType == EN_SCC && A >= 0x9800 && A < 0xA000) {
            return sccRead(scc, (UInt8)(A & 0xff));
        }
        if (sccType == EN_SCCPLUS && A >= 0xB800 && A < 0xC000) {
            return sccPlusRead(scc, (UInt8)(A & 0xff));
        }
    }

    return RAM[A>>13] ? RAM[A>>13][A&0x1FFF] : 0xFF;    
}

/** WrZ80() **************************************************/
/** Z80 emulation calls this function to write byte V to    **/
/** address A of Z80 address space.                         **/
/*************************************************************/
void WrZ80(UInt16 address, UInt8 value)
{
    SlotInfo* slotInfo;
    int ssl;

    if (address == 0xFFFF) {
        if (PSL[3] == 3) {
            SSlot(value);
        }
        else if (EnWrite[7]) {
            RAM[7][address & 0x1FFF] = value;
        }
        return;
    }

    if (EnWrite[address >> 13]) {
        RAM[address >> 13][address & 0x1FFF] = value;
        return;
    }

    ssl = PSL[address >> 14] == 3 ? SSL[address >> 14] : 0;
    slotInfo = &slotInfoTable[PSL[address >> 14]][ssl][address >> 13];

    if (slotInfo->wrfunc != NULL) {
        address -= slotInfo->startpage << 13;
        slotInfo->wrfunc(slotInfo->ref, address, value, systemTime);
    }
}

/** InZ80() **************************************************/
/** Z80 emulation calls this function to read a byte from   **/
/** a given I/O port.                                       **/
/*************************************************************/
byte InZ80(word Port)
{
    /* MSX only uses 256 IO ports */
    Port&=0xFF;

    /* Return an appropriate port value */
    switch(Port)
    {

    case 0x90: return(0xFD);                   /* Printer READY signal */
    case 0xB5: return(RTCIn(RTCReg));          /* RTC registers        */

    case 0xA8: /* Primary slot state   */
    case 0xA9: /* Keyboard port        */
    case 0xAA: /* General IO register  */
    case 0xAB: /* PPI control register */
        PPI.Rin[1]=KeyMap[PPI.Rout[2]&0x0F];
        return(Read8255(&PPI,Port-0xA8));

    case 0xFC: /* Mapper page at 0000h */
    case 0xFD: /* Mapper page at 4000h */
    case 0xFE: /* Mapper page at 8000h */
    case 0xFF: /* Mapper page at C000h */
        return(RAMMapper[Port-0xFC]|~RAMMask);

    case 0xD9: /* Kanji support */
        Port=Kanji? Kanji[KanLetter+KanCount]:NORAM;
        KanCount=(KanCount+1)&0x1F;
        return((byte)Port);

    case 0x80: /* SIO data */
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
        return NORAM;
        /*return(Rd8251(&SIO,Port&0x07));*/

    case 0x98: return vdpRead(systemTime);

    case 0x99: return vdpReadStatus(systemTime);

    case 0x04: return 2;
    case 0x05: return 0;

    case 0xC0: return enableY8950 ? y8950ReadAddress(y8950) : NORAM;
    case 0xC1: return enableY8950 ? y8950ReadData(y8950) : NORAM;

    case 0xA2: /* PSG input port */
        /* PSG[14] returns joystick/mouse data */
        if (ay8910GetLatch(ay8910) == 14) {
            int DX,DY,L;

            /* Number of a joystick port */
            Port = ay8910GetRegister(ay8910, 15) & 0x40 ? 1 : 0;
            L = Port ? JoyTypeB : JoyTypeA;

            /* If no joystick, return dummy value */
            if (!L) return(0x7F);

            /* Poll mouse position, if needed */
            if ((L == 2) || (MCount[Port] == 1)) {
                int buttons;
                Mouse(&DX, &DY, &buttons);
                Buttons[Port] = (~buttons >> 12) & 0x30;

                /* Adjust offsets */
                MouseDX[Port] = (DX > 127 ? 127 : (DX < -127 ? -127 : DX)) & 0xFF;
                MouseDY[Port] = (DY > 127 ? 127 : (DY < -127 ? -127 : DY)) & 0xFF;
            }

            /* Determine return value */
            switch (MCount[Port]) {
            case 0: /* Normal joystick */
                if (ay8910GetRegister(ay8910, 15) & (Port ? 0x20 : 0x10)) {
                    Port = 0x3F;
                }
                else if((Port ? JoyTypeB : JoyTypeA) < 2) {
                    Port = ~Joystick((byte)Port) & 0x3F;
                }
                else {
                    Port = Buttons[Port] |(MouseDX[Port] ? (MouseDX[Port] < 128 ? 0x08 : 0x04) : 0x0C) |
                        (MouseDY[Port] ? (MouseDY[Port] < 128 ? 0x02 : 0x01) : 0x03);
                }
                break;

            case 1: 
                Port = (MouseDX[Port] >> 4) | Buttons[Port];
                break;
            case 2: 
                Port = (MouseDX[Port] & 0x0F) | Buttons[Port];
                break;
            case 3: 
                Port = (MouseDY[Port] >> 4) | Buttons[Port];
                break;
            case 4: 
                Port = (MouseDY[Port] & 0x0F) | Buttons[Port];
                break;
            }

            return Port | 0x40;
        }

        /* PSG[15] resets mouse counters */
        if(ay8910GetLatch(ay8910) == 15) {
            return ay8910GetRegister(ay8910, 15) & 0xF0;
        }

        /* Return PSG[0-13] as they are */
        return ay8910ReadData(ay8910);
    }

    /* Return NORAM for non-existing ports */
    return(NORAM);
}

/** OutZ80() *************************************************/
/** Z80 emulation calls this function to write byte V to a  **/
/** given I/O port.                                         **/
/*************************************************************/
void OutZ80(word Port,byte Value)
{
    register byte I,J,K;  

    Port&=0xFF;
    switch(Port)
    {

    case 0x7C: if (enableYM2413) ym2413WriteAddress(ym2413, Value);return;        /* OPLL Register# */
    case 0x7D: if (enableYM2413) ym2413WriteData(ym2413, Value, systemTime);return;        /* OPLL Data      */

    case 0xC0: if (enableY8950) y8950WriteAddress(y8950, Value);return;        /* OPLL Register# */
    case 0xC1: if (enableY8950) y8950WriteData(y8950, Value, systemTime);return;        /* OPLL Data      */

    case 0x91: Printer(Value);return;                 /* Printer Data   */
    case 0xA0: ay8910WriteAddress(ay8910, Value);return;  /* PSG Register#  */
    case 0xB4: RTCReg=Value&0x0F;return;              /* RTC Register#  */ 

    case 0xD8: /* Upper bits of Kanji ROM address */
        KanLetter=(KanLetter&0x1F800)|((int)(Value&0x3F)<<5);
        KanCount=0;
        return;

    case 0xD9: /* Lower bits of Kanji ROM address */
        KanLetter=(KanLetter&0x007E0)|((int)(Value&0x3F)<<11);
        KanCount=0;
        return;

    case 0x80: /* SIO data */
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
        return;
        /*Wr8251(&SIO,Port&0x07,Value);
        return;*/

    case 0x98: /* VDP Data */
        vdpWrite(systemTime, Value);
        return;

    case 0x99: /* VDP Address Latch */
        vdpWriteLatch(systemTime, Value);
        return;

    case 0x9A: /* VDP Palette Latch */
        vdpWritePaletteLatch(systemTime, Value);
        return;

    case 0x9B: /* VDP Register Access */
        vdpWriteRegister(systemTime, Value);
        return;

    case 0xA1: /* PSG Data */
        /* PSG[15] is responsible for joystick/mouse */
        if (ay8910GetLatch(ay8910) == 15) {
            UInt8 R15 = ay8910GetRegister(ay8910, 15);

            /* For mouse, update nibble counter      */
            /* For joystick, set nibble counter to 0 */
            if ((Value & 0x0C) == 0x0C) {
                MCount[1] = 0;
            }
            else if ((JoyTypeB > 2) && ((Value ^ R15) & 0x20)) {
                MCount[1] += MCount[1] == 4 ? -3 : 1;
            }
            /* For mouse, update nibble counter      */
            /* For joystick, set nibble counter to 0 */
            if ((Value & 0x03) == 0x03) {
                MCount[0] = 0;
            }
            else if ((JoyTypeA > 2)&& ((Value ^ R15) & 0x10)) {
                MCount[0] += MCount[0] == 4 ? -3 : 1;
            }

            SetKanaLed(0 == (R15 & 0x80));
        }

        /* Put value into a register */
        ay8910WriteData(ay8910, Value, systemTime);
        return;

    case 0xB5: /* RTC Data */
        if(RTCReg<13)
        {
            /* J = register bank# now */
            J=RTCMode&0x03;
            /* Store the value */
            RTC[J][RTCReg]=Value;
            /* If CMOS modified, we need to save it */
            if(J>1) SaveCMOS=1;
            return;
        }
        /* RTC[13] is a register bank# */
        if(RTCReg==13) RTCMode=Value;
        return;

    case 0xA8: /* Primary slot state   */
    case 0xA9: /* Keyboard port        */
    case 0xAA: /* General IO register  */
    case 0xAB: /* PPI control register */
        /* Write to PPI */
        Write8255(&PPI,Port-0xA8,Value);
        /* If general I/O register has changed... */
        if(PPI.Rout[2]!=IOReg) { PPIOut(PPI.Rout[2],IOReg);IOReg=PPI.Rout[2]; }
        /* If primary slot state has changed... */
        if(PPI.Rout[0]!=PSLReg) {
            for(J=0,PSLReg=Value=PPI.Rout[0];J<4;J++,Value>>=2)
            {
                PSL[J]=Value&3;I=J<<1;
                K=PSL[J]==3? SSL[J]:0;
                EnWrite[I]=(K==2)&&(MemMap[3][2][I]!=EmptyRAM);
                EnWrite[I+1]=(K==2)&&(MemMap[3][2][I+1]!=EmptyRAM);
                RAM[I]=MemMap[PSL[J]][K][I];
                RAM[I+1]=MemMap[PSL[J]][K][I+1];
            }
        }
        /* Done */  
        return;

    case 0xFC: /* Mapper page at 0000h */
    case 0xFD: /* Mapper page at 4000h */
    case 0xFE: /* Mapper page at 8000h */
    case 0xFF: /* Mapper page at C000h */
        J=Port-0xFC;
        Value&=RAMMask;
        if(RAMMapper[J]!=Value) {
            I=J<<1;
            RAMMapper[J]      = Value;
            MemMap[3][2][I]   = RAMData+((int)Value<<14);
            MemMap[3][2][I+1] = MemMap[3][2][I]+0x2000;
            if((PSL[J]==3)&&(SSL[J]==2))
            {
                EnWrite[I] = 1;
                EnWrite[I+1] = 1;
                RAM[I]     = MemMap[3][2][I];
                RAM[I+1]   = MemMap[3][2][I+1];
            }
        }
        return;

    }
}

/** MapROM() *************************************************/
/** Switch ROM Mapper pages. This function is supposed to   **/
/** be called when ROM page registers are written to.       **/
/*************************************************************/
void MapROM(void* arg, UInt16 A, UInt8 V, UInt32 systemTime)
{
    byte J, romType, *P;
    int slot = (int)arg;

    slot -= 1;
    A += 0x4000;

    romType = slot ? ROMTypeB : ROMTypeA;

    /* If no cartridge or no mapper, exit */
    if (romType != ROM_SCCPLUS && (!ROMData[slot] || !ROMMask[slot])) {
        return;
    }

    switch(romType) {
    case ROM_STANDARD:
        if ((A & 0xFF00) == 0x9800 && SCCOn[slot] == EN_SCC) {
            sccWrite(scc, A & 0x00FF, V, systemTime);
            return;
        }
        else if ((A >= 0x4000) && (A < 0xC000)) {
            J = (A - 0x4000) >> 13;

            /* Turn SCC on/off on writes to 8000h-9FFFh */
            if (J==2) {
                SCCOn[slot] = ((V & 0x3F) == 0x3F) ? EN_SCC : EN_NONE;
            }

            /* Switch ROM pages */
            V &= ROMMask[slot];
            if (V != ROMMapper[slot][J]) {
                RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 13);
                ROMMapper[slot][J] = V;
            }
        }
        return;

    case ROM_KONAMI4:
        /* Page at 4000h is fixed */
        if ((A < 0x6000) || (A > 0xA000) || (A & 0x1FFF)) return;

        J = (A - 0x4000) >> 13;
        V &= ROMMask[slot];

        if(V != ROMMapper[slot][J]) {
            RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 13);
            ROMMapper[slot][J] = V;
        }
        break;

    case ROM_KONAMI5:
        if ((A & 0xFF00) == 0x9800 && SCCOn[slot] == EN_SCC) {
            sccWrite(scc, A & 0x00FF, V, systemTime);
            return;
        }
        else if ((A < 0x5000) || (A > 0xB7FF) || ((A & 0x1FFF) != 0x1000)) return;
        J = (A - 0x5000) >> 13;
        /* Turn SCC on/off on writes to 9000h */
    	if (J == 2) SCCOn[slot] = ((V & 0x3F) == 0x3F) ? EN_SCC : EN_NONE;
        /* Switch ROM pages */
        V&=ROMMask[slot];
        if(V!=ROMMapper[slot][J]) {
            RAM[J+2]=MemMap[slot+1][0][J+2]=ROMData[slot]+((int)V<<13);
            ROMMapper[slot][J]=V;
        }
        break;

    case ROM_ASCII8:
        if (( A < 0x6000) || (A > 0x7FFF)) return;
        J=(A&0x1800)>>11;
        V&=ROMMask[slot];
        if(V!=ROMMapper[slot][J])
        {
            P=ROMData[slot]+((int)V<<13);
            MemMap[slot+1][0][J+2]=P;
            ROMMapper[slot][J]=V;
            /* Only update memory when cartridge's slot selected */
            if(PSL[(J>>1)+1]==slot+1) RAM[J+2]=P;
        }
        break;

    case ROM_ASCII16:
        if ((A >= 0x6000) && (A < 0x7800) && !(A & 0x0800)) {
            J = (A & 0x1000) >> 11;
            V = (V << 1) & ROMMask[slot];
            if(V!=ROMMapper[slot][J]) {
                P=ROMData[slot]+((int)V<<13);
                MemMap[slot+1][0][J+2]=P;
                MemMap[slot+1][0][J+3]=P+0x2000;
                ROMMapper[slot][J]=V;
                /* Only update memory when cartridge's slot selected */
                if(PSL[(J>>1)+1]==slot+1) {
                    RAM[J + 2] = P;
                    RAM[J + 3] = P + 0x2000;
                }
            }
        }
        break;

    case ROM_ASCII8SRAM:
        if ((A >= 0x6000) && (A < 0x8000)) {
            J=(A&0x1800)>>11;
            if (V & (ROMSize[slot] / 8192)) {
                P = SRAM[slot];
            	SRAMenabled[slot] |= (1 << (J + 2));
            }
            else {
                P=ROMData[slot]+((int)V<<13);
            	SRAMenabled[slot] &= ~(1 << (J + 2));
            }
            if(V!=ROMMapper[slot][J]) {
                MemMap[slot+1][0][J+2]=P;
                ROMMapper[slot][J]=V;
                /* Only update memory when cartridge's slot selected */
                if(PSL[(J>>1)+1]==slot+1) RAM[J+2]=P;
            }
        }
        else if ((1 << (A >> 13)) & SRAMenabled[slot] & 0x30) {
            SRAM[slot][A & 0x1fff] = V;
        }
        break;

    case ROM_ASCII16SRAM:
        if ((A >= 0x6000) && (A < 0x7800) && !(A & 0x0800)) {
            J=(A&0x1000)>>11;

            V <<= 1;

            if (V & 0x20) {
                P = SRAM[slot];
            	SRAMenabled[slot] |= 1 << ((J >> 1) + 1);
            }
            else {
                P=ROMData[slot]+((int)V<<13);
            	SRAMenabled[slot] &= ~(1 << ((J >> 1) + 1));
            }

            if (V!=ROMMapper[slot][J]) {
                MemMap[slot+1][0][J + 2] = P;
                MemMap[slot+1][0][J + 3] = P + 0x2000;
                ROMMapper[slot][J] = V;
                /* Only update memory when cartridge's slot selected */
                if(PSL[(J >> 1) + 1] == slot + 1) {
                    RAM[J + 2] = P;
                    RAM[J + 3] = P + 0x2000;
                }
            }
        }
        else if ((1 << (A >> 14)) & SRAMenabled[slot] & 0x4) {
            word offset;
            for (offset = A & 0x7ff; offset < 0x4000; offset += 0x800) {
                SRAM[slot][offset] = V;
            }
        }
        break;

    case ROM_KONAMI4NF:
//  Uses Konami 4 but in different adresses :
//  Know Games : Animal Wars 2 , Ashguine 3 and Ashguine T&E
//      if ((A < 0x6000) || (A > 0xA000) || (A & 0x1FFF)) return;
        J = (A - 0x4000) >> 13;
        V &= ROMMask[slot];
        if(V != ROMMapper[slot][J]) {
            RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 13);
            ROMMapper[slot][J] = V;
        }
        break;

    case ROM_ASCII16NF:
//  Uses ASCII16 but in different adresses :
//  Know Game : Super Pierrot
//      if ((A < 0x6000) || ( A > 0x77FF)) return;
        J = (A & 0x1000) >> 11;
        V = (V << 1) & ROMMask[slot];
        if(V!=ROMMapper[slot][J]) {
            P=ROMData[slot]+((int)V<<13);
            MemMap[slot+1][0][J+2]=P;
            MemMap[slot+1][0][J+3]=P+0x2000;
            ROMMapper[slot][J]=V;
            /* Only update memory when cartridge's slot selected */
            if(PSL[(J>>1)+1]==slot+1) {
                RAM[J + 2] = P;
                RAM[J + 3] = P + 0x2000;
            }
        }
        break;

    case ROM_GAMEMASTER2:
        /* Switch ROM and SRAM pages, page at 4000h is fixed */
        if ((A >= 0x6000) && (A <= 0xA000) && !(A & 0x1FFF)) {
            /* If changing SRAM page... */
            if (V & 0x10) {
                /* Select SRAM page */
                RAM[5] = MemMap[slot + 1][0][5] = (SRAM[slot] ? SRAM[slot] : EmptyRAM) + (V & 0x20 ? 0x2000 : 0);
                /* SRAM is now on */
                ROMMapper[slot][3] = 0xFF;
            }
            else {
                /* Figure out which ROM page gets switched */
                J = (A - 0x4000) >> 13;
                /* Compute new ROM page number */
                V&=ROMMask[slot];
                /* If ROM page number has changed... */
                if(V != ROMMapper[slot][J]) {
                    RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 13);
                    ROMMapper[slot][J] = V;
                }
            }
            /* Done with page switch */
            break;
        }
        /* Write to SRAM */
        if ((A >= 0xB000) && (A < 0xC000) && (ROMMapper[slot][3] == 0xFF)) {
            RAM[5][(A & 0x0FFF) | 0x1000] = RAM[5][A & 0x0FFF] = V;
            break;
        }
        /* Done */
        return;

    case ROM_MSXDOS2: 
        if ((A >= 0x4000) && (A < 0xC000)) {
            J = (A & 0x8000) >> 14;
            V = (V << 1) & ROMMask[slot];
            if (V != ROMMapper[slot][J]) {
                RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 13);
                RAM[J + 3] = MemMap[slot + 1][0][J + 3] = RAM[J + 2] + 0x2000;
                ROMMapper[slot][J] = V;
            }
        }
        return;
        
    case ROM_RTYPE:
        if ((A >= 0x7000) && (A < 0x8000)) {
            J = 2;
		    V &= (V & 0x10) ? 0x17 : 0x0F;
            if (V != ROMMapper[slot][J]) {
                RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 14);
                RAM[J + 3] = MemMap[slot + 1][0][J + 3] = RAM[J + 2] + 0x2000;
                ROMMapper[slot][J] = V;
            }
        }
        break;

    case ROM_CROSSBLAIM:
        if (A == 0x4045) {
            J = 2;
            if (V != ROMMapper[slot][J]) {
                RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 14);
                RAM[J + 3] = MemMap[slot + 1][0][J + 3] = RAM[J + 2] + 0x2000;
                ROMMapper[slot][J] = V;
            }
        }
        break;
    case ROM_HARRYFOX:
        if ((A == 0x6000) || (A == 0x7000)) {
            J = (A >> 12) & 1;
            V = ((V & 1) << 1) + J;
            J <<= 1;
            if (V != ROMMapper[slot][J]) {
                RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 14);
                RAM[J + 3] = MemMap[slot + 1][0][J + 3] = RAM[J + 2] + 0x2000;
                ROMMapper[slot][J] = V;
            }
        }
        break;
    case ROM_KOREAN80:
        if ((A >= 0x4000) && (A < 0x4004)) {
            J = A - 0x4000;
            V &= ROMMask[slot];
            if(V != ROMMapper[slot][J]) {
                RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 13);
                ROMMapper[slot][J] = V;
            }
        }
        break;
    case ROM_KOREAN126:
        if ((A >= 0x4000) && (A < 0x4001)) {
            J = (A - 0x4000) << 1;
            V = (V << 1) & ROMMask[slot];
            if(V != ROMMapper[slot][J]) {
                RAM[J + 2] = MemMap[slot + 1][0][J + 2] = ROMData[slot] + ((int)V << 13);
                RAM[J + 3] = MemMap[slot + 1][0][J + 3] = RAM[J + 2] + 0x2000;
                ROMMapper[slot][J] = V;
            }
        }
        break;

    case ROM_SCCPLUS:
        {
            if ((A >= 0x4000) && (A < 0xC000)) {         
                if ((A | 1) == 0xBFFF) {
                    modeRegister[slot] = V;
                    isRamSegment[slot][0] = (V & 0x10) | (V & 0x01);
                    isRamSegment[slot][1] = (V & 0x10) | (V & 0x02);
                    isRamSegment[slot][2] = (V & 0x10) | ((V & 0x24) == 0x24);
			        isRamSegment[slot][3] = (V & 0x10);
                    if ((modeRegister[slot] & 0x20) && (ROMMapper[slot][3] & 0x80)) SCCOn[slot] = EN_SCCPLUS;
                    else if (!(modeRegister[slot] & 0x20) && (ROMMapper[slot][2] & 0x3F) == 0x3F) SCCOn[slot] = EN_SCC;
                    else SCCOn[slot] = EN_NONE;
                    return;
                }

                J = (A - 0x4000) >> 13;

                if (isRamSegment[slot][J]) {
    			    internalMemoryBank[slot][J][A & 0x1FFF] = V;
                    return;
                }

                if ((A & 0x1800) == 0x1000) {
                    if(V !=ROMMapper[slot][J]) {
                        internalMemoryBank[slot][J] = MemMap[slot + 1][0][J + 2] = memoryBank[slot] + (0x2000 * (V & 0x0F));
                        ROMMapper[slot][J] = V;
                        
                        if(PSL[(J >> 1) + 1] == slot + 1) {
                            RAM[J + 2] = internalMemoryBank[slot][J];
                        }

                        if ((modeRegister[slot] & 0x20) && (ROMMapper[slot][3] & 0x80)) SCCOn[slot] = EN_SCCPLUS;
                        else if (!(modeRegister[slot] & 0x20) && (ROMMapper[slot][2] & 0x3F) == 0x3F) SCCOn[slot] = EN_SCC;
                        else SCCOn[slot] = EN_NONE;
                    }

                    return;
                }

                if (SCCOn[slot] == EN_SCC && A >= 0x9800 && A < 0xA000) {
                    sccWrite(scc, A & 0xFF, V, systemTime);
                }
                else if (SCCOn[slot] == EN_SCCPLUS && A >= 0xB800 && A < 0xC000) {
                    sccPlusWrite(scc, A & 0xFF, V, systemTime);
                }
            }
        }
        break;
 
    default:
        return; 
    }
}

/** SSlot() **************************************************/
/** Switch secondary memory slots. This function is called  **/
/** when value in (FFFFh) changes.                          **/
/*************************************************************/
void SSlot(register byte V)
{
    register byte I,J;

    if(SSLReg!=V)
    {
        SSLReg=V;

        for(J=0;J<4;J++,V>>=2)
        {
            SSL[J]=V&3;
            if(PSL[J]==3)
            {
                I=J<<1;
                EnWrite[I]=(SSL[J]==2)&&(MemMap[3][2][I]!=EmptyRAM);
                EnWrite[I+1]=(SSL[J]==2)&&(MemMap[3][2][I+1]!=EmptyRAM);
                RAM[I]=MemMap[3][SSL[J]][I];
                RAM[I+1]=MemMap[3][SSL[J]][I+1];
            }
        }
    }
}


/** LoadCart() ***********************************************/
/** Load a cartridge ROM from .ROM file into given slot.    **/
/*************************************************************/
byte* loadRomImage(const char *fileName, const char *fileInZipFile, int* size)
{
    byte* buf;
    FILE *file;

    if (fileName == NULL) {
        return NULL;
    }

    if (fileInZipFile != NULL) {
        return zipLoadFile(fileName, fileInZipFile, size);
    }

    file = fopen(fileName, "rb");
    if (file == NULL) {
        return NULL;
    }

    fseek(file, 0, SEEK_END);
    *size = ftell(file);

    fseek(file, 0, SEEK_SET);

    buf = malloc(*size);
    
    *size = fread(buf, 1, *size, file);
    fclose(file);

    return buf;
}

void mapSccPlusRom(int slot) 
{
    memset(memoryBank[slot], 0xff, 0x20000);

    RAM[2] = internalMemoryBank[slot][0] = MemMap[1][0][2] = memoryBank[slot] + 0x2000 * 0;
    RAM[3] = internalMemoryBank[slot][1] = MemMap[1][0][3] = memoryBank[slot] + 0x2000 * 1;
    RAM[4] = internalMemoryBank[slot][2] = MemMap[1][0][4] = memoryBank[slot] + 0x2000 * 2;
    RAM[5] = internalMemoryBank[slot][3] = MemMap[1][0][5] = memoryBank[slot] + 0x2000 * 3;
    ROMMapper[slot][0] = 0;
    ROMMapper[slot][1] = 1;
    ROMMapper[slot][2] = 2;
    ROMMapper[slot][3] = 3;
    modeRegister[slot] = 0;
    isRamSegment[slot][0] = 0;
    isRamSegment[slot][1] = 0;
    isRamSegment[slot][2] = 0;
    isRamSegment[slot][3] = 0;

    ROMMask[slot] = 0x0f;
}

int mapRom(int slot, byte* buf, int size)
{
    int regular   = (size & 0x1fff) == 0 && (buf[0] == 'A' || buf[1] == 'B');
    int ROM64     = !regular && (size & 0x1fff) == 0 && size >= 0xa000 && size <= 0x10000 && (buf[0x4000] == 'A' || buf[0x4001] == 'B');
    int lastFirst = !regular && !ROM64 && (size & 0x1fff) == 0 && size > 0x4000 && (buf[size - 0x4000] == 'A' || buf[size - 0x3fff] == 'B');
    int pages;
    int maxPages;

    if (slot < 0 ||  slot > 1) {
        return 0;
    }

    if (!regular && !ROM64 && !lastFirst) {
        return 0;
    }

    pages = size >> 13;
    maxPages = 1;

    while (maxPages < pages) {
        maxPages <<= 1;
    }

    /* Assign ROMMask for MegaROMs */
    ROMMask[slot] = !ROM64 && (pages > 4) ? maxPages - 1 : 0x00;

    /* Allocate space for the ROM */
    ROMData[slot] = malloc(maxPages * 0x2000);

    memcpy(ROMData[slot], buf, size);
    memcpy(ROMData[slot] + size, buf, (maxPages - pages) * 0x2000);

    /* Set memory map depending on the ROM size */
    switch (pages) {
    case 1:
        /* 8kB ROMs are mirrored 8 times: 0:0:0:0:0:0:0:0 */
        MemMap[slot+1][0][0]=ROMData[slot];
        MemMap[slot+1][0][1]=ROMData[slot];
        MemMap[slot+1][0][2]=ROMData[slot];
        MemMap[slot+1][0][3]=ROMData[slot];
        MemMap[slot+1][0][4]=ROMData[slot];
        MemMap[slot+1][0][5]=ROMData[slot];
        MemMap[slot+1][0][6]=ROMData[slot];
        MemMap[slot+1][0][7]=ROMData[slot];
        break;

    case 2:
        /* 16kB ROMs are mirrored 4 times: 0:1:0:1:0:1:0:1 */
        MemMap[slot+1][0][0]=ROMData[slot];
        MemMap[slot+1][0][1]=ROMData[slot]+0x2000;
        MemMap[slot+1][0][2]=ROMData[slot];
        MemMap[slot+1][0][3]=ROMData[slot]+0x2000;
        MemMap[slot+1][0][4]=ROMData[slot];
        MemMap[slot+1][0][5]=ROMData[slot]+0x2000;
        MemMap[slot+1][0][6]=ROMData[slot];
        MemMap[slot+1][0][7]=ROMData[slot]+0x2000;
        break;

    case 3:
    case 4:
        /* 24kB and 32kB ROMs are mirrored twice: 0:1:0:1:2:3:2:3 */
        MemMap[slot+1][0][0]=ROMData[slot];
        MemMap[slot+1][0][1]=ROMData[slot]+0x2000;
        MemMap[slot+1][0][2]=ROMData[slot];
        MemMap[slot+1][0][3]=ROMData[slot]+0x2000;
        MemMap[slot+1][0][4]=ROMData[slot]+0x4000;
        MemMap[slot+1][0][5]=ROMData[slot]+0x6000;
        MemMap[slot+1][0][6]=ROMData[slot]+0x4000;
        MemMap[slot+1][0][7]=ROMData[slot]+0x6000;
        break;

    default:
        if(ROM64) {
            MemMap[slot+1][0][0]=ROMData[slot];
            MemMap[slot+1][0][1]=ROMData[slot]+0x2000;
            MemMap[slot+1][0][2]=ROMData[slot]+0x4000;
            MemMap[slot+1][0][3]=ROMData[slot]+0x6000;
            MemMap[slot+1][0][4]=ROMData[slot]+0x8000;
            MemMap[slot+1][0][5]=ROMData[slot]+0xA000;
            MemMap[slot+1][0][6]=ROMData[slot]+0xC000;
            MemMap[slot+1][0][7]=ROMData[slot]+0xE000;
        }
        else if(lastFirst) { 
            SetMegaROM(slot,pages - 2, pages - 1, pages - 2, pages - 1);
        }
        else {
            SetMegaROM(slot, 0, 1, 2, 3);
        }
        break;
    }

    /* Done loading cartridge */
    return(1);
}


void SetIRQ(register byte IRQ)
{
    if (IRQ & 0x80) {
        IRQPending &= IRQ; 
    }
    else {
        IRQPending |= IRQ;
    }
    
    z80SetInt(&CPU, IRQPending);
}

/** SetMegaROM() *********************************************/
/** Set MegaROM pages for a given slot. SetMegaROM() always **/
/** assumes 8kB pages.                                      **/
/*************************************************************/
void SetMegaROM(int Slot,byte P0,byte P1,byte P2,byte P3)
{
    P0&=ROMMask[Slot];
    P1&=ROMMask[Slot];
    P2&=ROMMask[Slot];
    P3&=ROMMask[Slot]; 
    MemMap[Slot+1][0][2]=ROMData[Slot]+P0*0x2000;
    MemMap[Slot+1][0][3]=ROMData[Slot]+P1*0x2000;
    MemMap[Slot+1][0][4]=ROMData[Slot]+P2*0x2000;
    MemMap[Slot+1][0][5]=ROMData[Slot]+P3*0x2000;
    ROMMapper[Slot][0]=P0;
    ROMMapper[Slot][1]=P1;
    ROMMapper[Slot][2]=P2;
    ROMMapper[Slot][3]=P3;
}

/** Printer() ************************************************/
/** Send a character to the printer.                        **/
/*************************************************************/
void Printer(byte V) { fputc(V,PrnStream); }

/** PPIOut() *************************************************/
/** This function is called on each write to PPI to make    **/
/** key click sound, motor relay clicks, and so on.         **/
/*************************************************************/
void PPIOut(register byte New,register byte Old)
{
    if((New & 0x80) && !(Old & 0x80)) {
        audioKeyClick(keyClick, systemTime);
    }

    SetCapslockLed(0 == (New & 0x40));
}

/** RTCIn() **************************************************/
/** Read value from a given RTC register.                   **/
/*************************************************************/
byte RTCIn(register byte R)
{
    static time_t PrevTime;
    static struct tm TM;
    register byte J;
    time_t CurTime;

    /* Only 16 registers/mode */
    R&=0x0F;

    /* Bank mode 0..3 */
    J=RTCMode&0x03;

    if(R>12) {
        J=R==13? RTCMode:NORAM;
    }
    else {
        if(J) {
            J=RTC[J][R];
        }
        else {
            /* Retrieve system time if any time passed */
            CurTime=time(NULL);
            if(CurTime!=PrevTime) {
                TM=*localtime(&CurTime);
                PrevTime=CurTime;
            }

            /* Parse contents of last retrieved TM */
            switch(R)
            {
            case 0:  J=TM.tm_sec%10;break;
            case 1:  J=TM.tm_sec/10;break;
            case 2:  J=TM.tm_min%10;break;
            case 3:  J=TM.tm_min/10;break;
            case 4:  J=TM.tm_hour%10;break;
            case 5:  J=TM.tm_hour/10;break;
            case 6:  J=TM.tm_wday;break;
            case 7:  J=TM.tm_mday%10;break;
            case 8:  J=TM.tm_mday/10;break;
            case 9:  J=(TM.tm_mon+1)%10;break;
            case 10: J=(TM.tm_mon+1)/10;break;
            case 11: J=(TM.tm_year-80)%10;break;
            case 12: J=((TM.tm_year-80)/10)%10;break;
            default: J=0x0F;break;
            } 
        }
    }
    /* Four upper bits are always high */
    return(J|0xF0);
}

static void LoopZ80(Z80 *R)
{
    for (;;) {
        if (syncCount >= SyncPeriod) {
            int rv = WaitForSync();
            if (rv) {
                return;
            }
            syncCount -= SyncPeriod;
            mixerSync(TheMixer, systemTime);    
            Keyboard(KeyMap);
        }

        nextSyncTime += loopTime;
        syncCount += loopTime;

        z80execute(R, nextSyncTime);
        vdpSync(nextSyncTime);

        loopTime = vdpRefreshLine(R);
    }
}

/** GuessROM() ***********************************************/
/** Guess MegaROM mapper of a ROM.                          **/
/*************************************************************/

int GuessROM(const byte *Buf,int Size)
{
    int J,I,ROMCount[6];

    RomType romType = getRomType(Buf, Size);

    switch (romType) {
    case ROM_STANDARD:
    case ROM_MSXDOS2:
    case ROM_KONAMI5:
    case ROM_KONAMI4:
    case ROM_ASCII8:
    case ROM_ASCII16:
    case ROM_GAMEMASTER2:
    case ROM_ASCII8SRAM:
    case ROM_ASCII16SRAM:
    case ROM_RTYPE:
    case ROM_CROSSBLAIM:
    case ROM_HARRYFOX:
    case ROM_KOREAN80:
    case ROM_KOREAN126:
    case ROM_FMPAC:
    case ROM_KONAMI4NF:
    case ROM_ASCII16NF:
        return romType;

    default:
        break;
    }

    /* Clear all counters */
    for(J=0;J<6;J++) ROMCount[J]=1;
    ROMCount[0]+=1; /* Mapper #0 is default */
    ROMCount[4]-=1; /* #5 preferred over #4 */

    /* Count occurences of characteristic addresses */
    for(J=0;J<Size-2;J++)
    {
        I=Buf[J]+((int)Buf[J+1]<<8)+((int)Buf[J+2]<<16);
        switch(I)
        {
        case 0x500032: ROMCount[2]++;break;
        case 0x900032: ROMCount[2]++;break;
        case 0xB00032: ROMCount[2]++;break;
        case 0x400032: ROMCount[3]++;break;
        case 0x800032: ROMCount[3]++;break;
        case 0xA00032: ROMCount[3]++;break;
        case 0x680032: ROMCount[4]++;break;
        case 0x780032: ROMCount[4]++;break;
        case 0x600032: ROMCount[3]++;ROMCount[4]++;ROMCount[5]++;break;
        case 0x700032: ROMCount[2]++;ROMCount[4]++;ROMCount[5]++;break;
        case 0x77FF32: ROMCount[5]++;break;
        }
    }

    /* Find which mapper type got more hits */
    for(I=0,J=0;J<6;J++) {
        if(ROMCount[J]>ROMCount[I]) {
            I=J;
        }
    }

    switch (I) {
    default:
    case 0: return ROM_STANDARD;
    case 1: return ROM_MSXDOS2;
    case 2: return ROM_KONAMI5;
    case 3: return ROM_KONAMI4;
    case 4: return ROM_ASCII8;
    case 5: return ROM_ASCII16;
    case 6: return ROM_GAMEMASTER2;
    }
}

/** SaveState() **********************************************/
/** Save emulation state to a .STA file.                    **/
/*************************************************************/
int SaveState(const char *FileName)
{
    static byte Header[16] = "STA03\0\0\0\0\0\0\0\0\0\0\0";
    unsigned int State[256],J,I;
    FILE *F;

    /* Open state file */
    if(!(F=fopen(FileName,"wb"))) return(0);

    /* Prepare the header */
    J=StateID();
    Header[5] = RAMPages;
    Header[6] = 0;
    Header[7] = J&0x00FF;
    Header[8] = J>>8;

    /* Write out the header */
    if(fwrite(Header,1,sizeof(Header),F)!=sizeof(Header))
    { fclose(F);return(0); }

    /* Fill out hardware state */
    J=0;
    State[J++] = systemTime;
    State[J++] = nextSyncTime;
    State[J++] = IRQPending;
    State[J++] = RTCReg;
    State[J++] = RTCMode;
    State[J++] = KanLetter;
    State[J++] = KanCount;
    State[J++] = IOReg;
    State[J++] = PSLReg;
    State[J++] = SSLReg;
    State[J++] = ROMSize[0];
    State[J++] = ROMSize[1];
    State[J++] = SRAMenabled[0];
    State[J++] = SRAMenabled[1];
    State[J++] = loopTime;
    State[J++] = ROMTypeA;
    State[J++] = ROMTypeB;
    State[J++] = MSXVersion;

    for (I = 0; I < 4; I++) {
        State[J++] = PSL[I];
        State[J++] = SSL[I];
        State[J++] = EnWrite[2*I];
        State[J++] = EnWrite[2*I+1];
        State[J++] = RAMMapper[I];
        State[J++] = ROMMapper[0][I];
        State[J++] = ROMMapper[1][I];
    }  

    /* Write out hardware state */
    if (fwrite(&CPU, 1, sizeof(CPU), F) != sizeof(CPU)) { 
        fclose(F);
        return 0; 
    }
    if (fwrite(&PPI, 1, sizeof(PPI), F) != sizeof(PPI)) { 
        fclose(F);
        return 0; 
    }

    if (0 == ay8910SaveState(ay8910, F, systemTime)) { 
        fclose(F);
        return 0; 
    }

    if (0 == y8950SaveState(y8950, F, systemTime)) { 
        fclose(F);
        return 0; 
    }

    if (0 == ym2413SaveState(ym2413, F, systemTime)) { 
        fclose(F);
        return 0; 
    }

    if (0 == sccSaveState(scc, F, systemTime)) { 
        fclose(F);
        return 0; 
    }

    if (0 == vdpSaveState(F)) {
        fclose(F);
        return 0;
    }

    if (fwrite(State, 1, sizeof(State), F) != sizeof(State)) { 
        fclose(F);
        return 0; 
    }

    /* Save memory contents */
    if (fwrite(RAMData, 1, RAMPages * 0x4000, F) != RAMPages * 0x4000) { 
        fclose(F);
        return 0;
    }

    if (MSXVersion > 4) {
        if (0 == romMapperFMPACSaveState(fmpac, F)) {
            fclose(F);
            return 0;
        }
    }

    if (ROMTypeA == ROM_FMPAC) {
        if (0 == romMapperFMPACSaveState(fmpacA, F)) {
            fclose(F);
            return 0;
        }
    }

    if (ROMTypeB == ROM_FMPAC) {
        if (0 == romMapperFMPACSaveState(fmpacB, F)) {
            fclose(F);
            return 0;
        }
    }

    /* Done */
    fclose(F);
    
    return 1;
}

/** LoadState() **********************************************/
/** Load emulation state from a .STA file.                  **/
/*************************************************************/
int LoadState(const char *FileName)
{
    unsigned int State[256],J,I;
    byte Header[16];
    int romtypeA;
    int romtypeB;
    int msxver;
    FILE *F;

    /* Open state file */
    if(!(F=fopen(FileName,"rb"))) return(0);

    /* Read the header */
    if(fread(Header,1,sizeof(Header),F)!=sizeof(Header))
    { fclose(F);return(0); }

    /* Verify the header */
    if (memcmp(Header, "STA03", 5)) { 
        fclose(F);
        return -1; 
    }

    if (strlen(FileName) < 8 || strcmp("temp.sta", FileName + strlen(FileName) - 8)) {
        if (Header[7] + Header[8] * 256 != StateID()) { 
            fclose(F);
            return 0; 
        }
    }

    if (Header[5] != (UInt8)RAMPages) { 
        fclose(F);
        return 0; 
    }

    /* Read the hardware state */
    if (fread(&CPU, 1, sizeof(CPU), F) != sizeof(CPU)) { 
        fclose(F);
        return 0; 
    }

    if (fread(&PPI, 1, sizeof(PPI), F) != sizeof(PPI)) { 
        fclose(F);
        return 0; 
    }

    if (0 == ay8910LoadState(ay8910, F, systemTime)) { 
        fclose(F);
        return 0; 
    }
    
    if (0 == y8950LoadState(y8950, F, systemTime)) { 
        fclose(F);
        return 0; 
    }

    if (0 == ym2413LoadState(ym2413, F, systemTime)) { 
        fclose(F);
        return 0; 
    }
    
    if (0 == sccLoadState(scc, F, systemTime)) { 
        fclose(F);
        return 0; 
    }

    if (0 == vdpLoadState(F)) {
        fclose(F);
        return 0;
    }

    if (fread(State, 1, sizeof(State), F) != sizeof(State)) { 
        fclose(F);
        return 0; 
    }

    /* Load memory contents */
    if (fread(RAMData, 1, Header[5] * 0x4000, F) != Header[5] * 0x4000) { 
        fclose(F);
        return 0; 
    }

    /* Parse hardware state */
    J=0;
    systemTime     = State[J++];
    nextSyncTime   = State[J++];
    IRQPending     = State[J++];
    RTCReg         = State[J++];
    RTCMode        = State[J++];
    KanLetter      = State[J++];
    KanCount       = State[J++];
    IOReg          = State[J++];
    PSLReg         = State[J++];
    SSLReg         = State[J++];
    ROMSize[0]     = State[J++];
    ROMSize[1]     = State[J++];
    SRAMenabled[0] = State[J++];
    SRAMenabled[1] = State[J++];
    loopTime       = State[J++];
    romtypeA       = State[J++];
    romtypeB       = State[J++];
    msxver         = State[J++];

    if (MSXVersion != msxver) {
        fclose(F);
        return 0; 
    }

    if (MSXVersion > 4) {
        if (0 == romMapperFMPACLoadState(fmpac, F)) {
            fclose(F);
            return 0;
        }
    }

    for(I=0;I<4;I++) {
        PSL[I]          = State[J++];
        SSL[I]          = State[J++];
        EnWrite[2*I]    = State[J++];
        EnWrite[2*I+1]  = State[J++];
        RAMMapper[I]    = State[J++];
        ROMMapper[0][I] = State[J++];
        ROMMapper[1][I] = State[J++];
    }  

    /* Set RAM mapper pages */
    if(RAMMask) {
        for(I=0;I<4;I++) {
            RAMMapper[I]       &= RAMMask;
            MemMap[3][2][I*2]   = RAMData+RAMMapper[I]*0x4000;
            MemMap[3][2][I*2+1] = MemMap[3][2][I*2]+0x2000;
        }
    }

    /* Set ROM mapper pages */
    if(ROMMask[0])
        SetMegaROM(0,ROMMapper[0][0],ROMMapper[0][1],ROMMapper[0][2],ROMMapper[0][3]);
    if(ROMMask[1])
        SetMegaROM(1,ROMMapper[1][0],ROMMapper[1][1],ROMMapper[1][2],ROMMapper[1][3]);

    // The following code is a bit messy but it is needed for old vs new memory model
    // Load and map rom if in saved state and if configured
    if (romtypeA == ROM_FMPAC) {
        if (ROMTypeA == ROM_FMPAC) {
            romMapperFMPACLoadState(fmpacA, F);
        }
        else {
            UInt32 data[32];
            if (fread(data, 1, sizeof(data), F) != sizeof(data)) { 
                fclose(F);
                return 0; 
            }
        }
    }

    if (romtypeB == ROM_FMPAC) {
        if (ROMTypeB == ROM_FMPAC) {
            romMapperFMPACLoadState(fmpacB, F);
        }
        else {
            UInt32 data[32];
            if (fread(data, 1, sizeof(data), F) != sizeof(data)) { 
                fclose(F);
                return 0; 
            }
        }
    }

    /* Set main address space pages */
    for(I=0;I<4;I++) {
        RAM[2*I]   = MemMap[PSL[I]][PSL[I]==3? SSL[I]:0][2*I];
        RAM[2*I+1] = MemMap[PSL[I]][PSL[I]==3? SSL[I]:0][2*I+1];
    }

    /* Done with the file */
    fclose(F);

    /* Done */
    return(1);
}

/** StateID() ************************************************/
/** Compute 16bit emulation state ID used to identify .STA  **/
/** files.                                                  **/
/*************************************************************/
word StateID(void)
{
    word ID;
    int J;

    ID=0x0000;

    /* Add up cartridge ROM, BIOS, BASIC, and ExtBIOS bytes */
    if(ROMData[0]) for(J=0;J<(ROMMask[0]+1)*0x2000;J++) ID+=ROMData[0][J];
    if(ROMData[1]) for(J=0;J<(ROMMask[1]+1)*0x2000;J++) ID+=ROMData[1][J];
    if(MemMap[0][0][0]) for(J=0;J<0x8000;J++) ID+=MemMap[0][0][0][J];
    if(MemMap[3][1][0]) for(J=0;J<0x4000;J++) ID+=MemMap[3][1][0][J];

    return(ID);
}
