#include <xtl.h>
#include "MSX.h"
#include "audioMixer.h"
#include "videoRender.h"
#include "ziphelper.h"
#include "Casette.h"
#include "Disk.h"
#include "romMSX.h"
#include "romMSXBR.h"
#include "romMSXJ.h"
#include "romMSX2.h"
#include "romMSX2EXT.h"
#include "romMSX2J.h"
#include "romMSX2EXTJ.h"
#include "romMSX2P.h"
#include "romMSX2PEXT.h"
#include "romMSXKR.h"
#include "romMSXHAN.h"
#include "romMSX2KR.h"
#include "romMSX2HAN.h"
#include "romMSX2EXTKR.h"

#define UInt32 unsigned int
#define UInt8 unsigned char

UInt32  emuFixedPalette[256];
UInt32* emuFrameBuffer;
int*    emuLineWidth;
UInt32  emuPalette0;
UInt32  emuPalette[300];
int diskBusy[2];
int casBusy;
int doquit = 0 ;

char *xbox_save_path();
void emulatorSetFrequency(int logFrequency);
void emulatorSuspend();
void emulatorResume();
void emulatorStart(int hard);
void emulatorStop();

static int          emuExitFlag = 0 ;
extern Mixer* TheMixer;
extern int  SyncPeriod;
extern int MsxFrequency;

#define CARTNAME_SCCPLUS "SCC+ Cartridge"
#define CARTNAME_FMPAC   "FM-PAC Cartridge"

typedef enum { P_EMU_MSX = 0, P_EMU_MSXBR, P_EMU_MSXJ, P_EMU_MSXKR, P_EMU_MSX2, P_EMU_MSX2J, P_EMU_MSX2KR, P_EMU_MSX2P } PropEmuFamily;
typedef enum { P_EMU_SPEEDX03 = 0, P_EMU_SPEEDX05, P_EMU_SPEEDX1, P_EMU_SPEEDX2, P_EMU_SPEEDX3 } PropEmuSpeed;
typedef enum { P_EMU_RAM16 = 0, P_EMU_RAM32, P_EMU_RAM64, P_EMU_RAM128, P_EMU_RAM256, P_EMU_RAM512, P_EMU_RAM1024, P_EMU_RAM2048, P_EMU_RAM4096 } PropEmuRAM;
typedef enum { P_EMU_VRAM16 = 0, P_EMU_VRAM32, P_EMU_VRAM64, P_EMU_VRAM128 } PropEmuVRAM;
typedef enum { P_EMU_SYNC1MS = 0, P_EMU_SYNCAUTO } PropEmuSync;
typedef enum { P_VIDEO_COLOR = 0, P_VIDEO_BW, P_VIDEO_GREEN } PropVideoMon;
typedef enum { P_VIDEO_PALNONE = 0, P_VIDEO_PALYC, P_VIDEO_PALNYC, P_VIDEO_PALCOMP, P_VIDEO_PALNCOMP, P_VIDEO_PALSCALE2X } PropVideoPalEmu;
typedef enum { P_VIDEO_PAL, P_VIDEO_NTSC } PropVideoType;
typedef enum { P_VIDEO_SIZEX1 = 0, P_VIDEO_SIZEX2, P_VIDEO_SIZEFULLSCREEN } PropVideoSize;
typedef enum { P_VIDEO_DRVDIRECTX = 0, P_VIDEO_DRVGDI } PropVideoDriver;
typedef enum { P_VIDEO_FSKIP0 = 0, P_VIDEO_FSKIP1, P_VIDEO_FSKIP2, P_VIDEO_FSKIP3, P_VIDEO_FSKIP4, P_VIDEO_FSKIP5 } PropVideoFrameSkip;
typedef enum { P_VIDEO_FRES320X240_16 = 0, P_VIDEO_FRES320X240_32, P_VIDEO_FRES640X480_16, P_VIDEO_FRES640X480_32 } PropVideoFullRes;
typedef enum { P_SOUND_DRVNONE = 0, P_SOUND_DRVWMM, P_SOUND_DRVDIRECTX } PropSoundDriver;
typedef enum { P_SOUND_FREQ11 = 0, P_SOUND_FREQ22, P_SOUND_FREQ44 } PropSoundFrequency;
typedef enum { P_SOUND_BUF100 = 0, P_SOUND_BUF150, P_SOUND_BUF200, P_SOUND_BUF250, P_SOUND_BUF300, P_SOUND_BUF350 } PropSoundBufSize;
typedef enum { P_SOUND_SYNCFAST = 0, P_SOUND_SYNCQADJUST, P_SOUND_SYNCEADJUST } PropSoundSync;
typedef enum { P_JOY_NONE = 0, P_JOY_NUMPAD, P_JOY_KEYSETA, P_JOY_KEYSETB, P_JOY_HW1, P_JOY_HW2, P_JOY_MOUSE } PropControlsJoy;
typedef enum { P_JOY_AFOFF = 0, P_JOY_AFSLOW, P_JOY_AFMEDIUM, P_JOY_AFFAST } PropControlsAutofire;
typedef enum { P_CHAR_EUROPEAN = 0, P_CHAR_RUSSIAN, P_CHAR_JAPANESE, P_CHAR_KOREAN } PropControlsLanguage;
typedef enum { EMU_RUNNING, EMU_PAUSED, EMU_STOPPED, EMU_SUSPENDED } EmuState;

#define MAX_HISTORY 10

typedef struct {
    char                statsDefDir[MAX_PATH];
    PropEmuFamily       family;
    PropEmuSpeed        speed;
    PropEmuRAM          RAMsize;
    PropEmuVRAM         VRAMsize;
    PropEmuSync         syncMethod;
} EmulationProperties;

typedef struct {
    PropVideoMon        monType;
    PropVideoPalEmu     palEmu;
    PropVideoType       videoType;
    PropVideoSize       size;
    PropVideoDriver     driver;
    PropVideoFrameSkip  frameSkip;
    PropVideoFullRes    fullRes;
} VideoProperties;

typedef struct {
    int enable;
    int volume;
    int pan;
} MixerChannel;

enum MixerChannelType {
    MCT_PSG = 0,
    MCT_SCC,
    MCT_MSXMUSIC,
    MCT_MSXAUDIO,
    MCT_KEYBOARD,
    MCT_CASSETTE,
    MCT_NUMENTRIES
};

typedef struct {
    int msxaudio;
    int msxmusic;
} SoundChip;

typedef struct {
    PropSoundDriver     driver;
    PropSoundFrequency  frequency;
    PropSoundBufSize    bufSize;
    PropSoundSync       syncMethod;
    SoundChip           chip;
    int                 stereo;
    int                 masterVolume;
    MixerChannel        mixerChannel[6]; // See MixerChannelType
    char                log[MAX_PATH];
} SoundProperties;

typedef struct {
    int id;
    PropControlsJoy      type;
    PropControlsAutofire autofire;
    int keyUp;
    int keyDown;
    int keyLeft;
    int keyRight;
    int button1;
    int button2;
}  JoystickProperties;

typedef struct {
    PropControlsLanguage keySet;
} KeyboardProperties;

typedef struct {
    char defDir[MAX_PATH];
    char slotA[MAX_PATH];
    char slotB[MAX_PATH];
    char slotAZip[MAX_PATH];
    char slotBZip[MAX_PATH];
    int  autoReset;
} CartridgeProperties;

typedef struct {
    char defDir[MAX_PATH];
    char slotA[MAX_PATH];
    char slotB[MAX_PATH];
    char slotAZip[MAX_PATH];
    char slotBZip[MAX_PATH];
    int autostartA;
} DiskdriveProperties;

typedef struct {
    char defDir[MAX_PATH];
    char tape[MAX_PATH];
    char tapeZip[MAX_PATH];
} CassetteProperties;

typedef struct {
    char cartridgeA[MAX_HISTORY][MAX_PATH];
    char cartridgeB[MAX_HISTORY][MAX_PATH];
    char diskdriveA[MAX_HISTORY][MAX_PATH];
    char diskdriveB[MAX_HISTORY][MAX_PATH];
    char cassette[MAX_HISTORY][MAX_PATH];
} FileHistory;
typedef enum { 
    EMU_LANG_ENGLISH    = 0, 
    EMU_LANG_SWEDISH    = 1, 
    EMU_LANG_JAPANESE   = 2, 
    EMU_LANG_PORTUGUESE = 3, 
    EMU_LANG_FRENCH     = 4, 
    EMU_LANG_DUTCH      = 5,
    EMU_LANG_SPANISH    = 6,
    EMU_LANG_COUNT      = 7,
    EMU_LANG_UNKNOWN    = -1 
} EmuLanguageType;

typedef struct {
    EmulationProperties emulation;
    VideoProperties     video;
    SoundProperties     sound;
    JoystickProperties  joy1;
    JoystickProperties  joy2;
    KeyboardProperties  keyboard;
    CartridgeProperties cartridge;
    DiskdriveProperties diskdrive;
    CassetteProperties  cassette;
    FileHistory         filehistory;
    EmuLanguageType     language;
} Properties;

Properties* pProperties = 0;

void propInitDefaults(Properties* pProperties) ;

typedef struct {
    int showMenu;
    void* bmBits;
    int* srcArr;
    void* bmBitsGDI;
    UInt8 keyMap[16];
    EmuState emuState;
    int updatePending;
    char pStoredState[MAX_PATH];
    char pCurDir[MAX_PATH];
    Video* pVideo;
    int logSound;
    Mixer* mixer;
    int enteringFullscreen;
    int screenMode;
    int evenOdd;
    int interlace;
    int autostart;
    int noautostart;
    UInt32 ledState;
    int tapeCurPos;
    int X;
    int Y;
    int DX;
    int DY;
    int DDY;
    int DTY;
} Port ;

#define WIDTH  320
#define HEIGHT 240

static Port Prt;

void global_init()
{
	memset( &Prt, 0, sizeof(Port) ) ;
}

void xbox_sound( unsigned char *buf, unsigned int bufsize );

static Int32 xboxWrite(void* dummy, Int16 *buffer, UInt32 count)
{
	xbox_sound( (unsigned char*)buffer, count ) ;
}

void init_xmsx()
{
	int i ;

	doquit = 0 ;

	if ( pProperties )
	{
		free( pProperties ) ;
		pProperties = NULL ;
	}

	pProperties = (Properties*)malloc( sizeof(Properties) ) ;
	memset( pProperties, 0, sizeof(Properties) ) ;

    pProperties->joy1.id = 1;
    pProperties->joy2.id = 2;
    propInitDefaults(pProperties);

	if ( Prt.pVideo )
	{
		free( Prt.pVideo ) ;
		Prt.pVideo = NULL ;
	}
	if ( Prt.mixer )
	{
		free( Prt.mixer) ;
		Prt.mixer = NULL ;
	}
	if ( Prt.bmBits )
	{
		free( Prt.bmBits ) ;
		Prt.bmBits = NULL ;
	}

	if ( Prt.srcArr )
	{
		free( Prt.srcArr ) ;
		Prt.srcArr = NULL ;
	}


	memset( &Prt, 0, sizeof(Port) ) ;
    memset(Prt.keyMap, 0xff, 16);

    Prt.ledState = 0;
    Prt.tapeCurPos = 0;
    Prt.autostart = 0;
    Prt.noautostart = 0;
    Prt.screenMode = 0;
    Prt.emuState = EMU_STOPPED;
    Prt.updatePending = 0;
    Prt.enteringFullscreen = 0;
    Prt.pVideo = videoCreate();
    Prt.mixer  = mixerCreate();

    for (i = 0; i < MIXER_CHANNEL_COUNT; i++) {
        mixerSetChannelVolume(Prt.mixer, i, pProperties->sound.mixerChannel[i].volume);
        mixerSetChannelPan(Prt.mixer, i, pProperties->sound.mixerChannel[i].pan);
        mixerEnableChannel(Prt.mixer, i, pProperties->sound.mixerChannel[i].enable);
    }
    
    mixerSetMasterVolume(Prt.mixer, pProperties->sound.masterVolume);

    mixerSetStereo(Prt.mixer, 1);
    mixerSetWriteCallback(Prt.mixer, xboxWrite, NULL);

	strcpy( Prt.pCurDir, xbox_save_path() ) ;
    /* Set SCREEN8 colors */

    //Prt.X = CW_USEDEFAULT;
    //Prt.Y = CW_USEDEFAULT;


	Prt.bmBits = calloc(4 * WIDTH * HEIGHT, sizeof(UInt32));
    Prt.srcArr = calloc(256, sizeof(int));
    Prt.logSound = 0;

}
void emulatorSetFrequency(int logFrequency) {
    SyncPeriod = (int)(3580 * emulatorGetSyncPeriod() * pow(2.0, (logFrequency - 50) / 15.0515));
    //MsxFrequency = (int)(3579545 * pow(2.0, (logFrequency - 50) / 15.0515));
    //MsxFrequency = (int)(3568884 * pow(2.0, (logFrequency - 50) / 15.0515));
    MsxFrequency = (int)(GLOBAL_FREQ * pow(2.0, (logFrequency - 50) / 15.0515));
}

char* getCmdArgument(char* szLine, int argNo) {
    static char argBuf[512];
    int i;

    for (i = 0; i <= argNo; i++) {
        char* arg = argBuf;

        while (*szLine == ' ') szLine++;

        if (*szLine == 0) return NULL;

        if (*szLine == '\"') {
            szLine++;
            while (*szLine != '\"' && *szLine != 0) {
                *arg++ = *szLine++;
            }
            *arg = 0;
            if (*szLine != 0) szLine++;
        }
        else {
            do {
                *arg++ = *szLine++;
            } while (*szLine != ' ' && *szLine != 0);
            *arg = 0;
            if (*szLine != 0) szLine++;
        }
    }
    return argBuf;
}

static int insertCartridge(int drive, char* fname, char* inZipFile) {
    int autostart = Prt.autostart | pProperties->cartridge.autoReset;
    char romName[512] = "";
    char filename[512] = "";
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;
	/*
    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(romName, inZipFile);
        }
        else {
            int count;
            char* fileList = zipGetFileList(filename, ".rom", &count);

            if (fileList == NULL) {
                MessageBox(NULL, langErrorNoRomInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(romName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadRom());
                _stprintf(dlgInfo.description, "%s", langDlgLoadRomDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                strcpy(dlgInfo.selectFile, drive == 0 ? pProperties->cartridge.slotAZip : pProperties->cartridge.slotBZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                autostart = dlgInfo.autoReset;
                strcpy(romName, dlgInfo.selectFile);
            }
            free(fileList);
        }
    }
*/
    if (drive == 0) {
        strcpy(pProperties->cartridge.slotA, filename);
        strcpy(pProperties->cartridge.slotAZip, romName);
        //updateFileHistory(*pProperties->filehistory.cartridgeA, filename);
    }
    else {
        strcpy(pProperties->cartridge.slotB, filename);
        strcpy(pProperties->cartridge.slotBZip, romName);
        //updateFileHistory(*pProperties->filehistory.cartridgeB, filename);
    }

    if (autostart && !Prt.noautostart) {
        emulatorStop();
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        //emulatorSuspend();
        ChangeCartridge(drive, strcmp(CARTNAME_SCCPLUS, filename) ? strcmp(CARTNAME_FMPAC, filename) ? ROM_UNKNOWN : ROM_FMPAC : ROM_SCCPLUS, filename, isZip ? romName : NULL);
        //emulatorResume();
    }

    return 1;
}


void xboxRender( unsigned char *dstbits, unsigned int dstpitch )
{
		videoRender(Prt.pVideo, 16, 1, Prt.evenOdd, Prt.interlace,
					Prt.bmBits, WIDTH, HEIGHT, Prt.srcArr, dstbits, sizeof(DWORD) * WIDTH, dstpitch);
}

int emulatorGetSyncPeriod() {
    return pProperties->emulation.syncMethod == P_EMU_SYNC1MS ? 1 : 10;
}

static int insertDiskette(int drive, char* fname, char* inZipFile) {
    char diskName[512] = "";
    char filename[512] = "";
    int autostart = Prt.autostart | (drive == 0 ? pProperties->diskdrive.autostartA : 0);
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;
	/*
    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(diskName, inZipFile);
        }
        else {
            int count;
            char* fileList = zipGetFileList(filename, ".dsk", &count);

            if (fileList == NULL) {
                MessageBox(NULL, langErrorNoDskInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(diskName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadDsk());
                _stprintf(dlgInfo.description, "%s", langDlgLoadDskDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                strcpy(dlgInfo.selectFile, drive == 0 ? pProperties->diskdrive.slotAZip : pProperties->diskdrive.slotBZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                strcpy(diskName, dlgInfo.selectFile);
                autostart = dlgInfo.autoReset;
            }
            free(fileList);
        }
    }
*/
    if (drive == 0) {
        strcpy(pProperties->diskdrive.slotA, filename);
        strcpy(pProperties->diskdrive.slotAZip, diskName);
        //updateFileHistory(*pProperties->filehistory.diskdriveA, filename);
    }
    else {
        strcpy(pProperties->diskdrive.slotB, filename);
        strcpy(pProperties->diskdrive.slotBZip, diskName);
        //updateFileHistory(*pProperties->filehistory.diskdriveB, filename);
    }

    if (autostart && !Prt.noautostart) {
        emulatorStop();
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        //emulatorSuspend();
        ChangeDisk(drive, filename, isZip ? diskName : NULL);
        //emulatorResume();
    }

    return 1;
}

static int insertCassette(char* fname, char* inZipFile) {
    int autostart = Prt.autostart;
    char tapeName[512] = "";
    char filename[512] = "";
    int isZip = toupper(fname[strlen(fname) - 3]) == 'Z' &&
                toupper(fname[strlen(fname) - 2]) == 'I' &&
                toupper(fname[strlen(fname) - 1]) == 'P';

    if (fname) strcpy(filename, fname);

    Prt.autostart = 0;
/*
    if (isZip) {
        if (inZipFile != NULL) {
            strcpy(tapeName, inZipFile);
        }
        else {
            int count;
            char* fileList = zipGetFileList(filename, ".cas", &count);

            if (fileList == NULL) {
                MessageBox(NULL, langErrorNoCasInZip(), langErrorTitle(), MB_OK);
                return 0;
            }

            if (count == 1) {
                strcpy(tapeName, fileList);
            }
            else {
                ZipFileDlgInfo dlgInfo;

                _stprintf(dlgInfo.title, "%s", langDlgLoadCas());
                _stprintf(dlgInfo.description, "%s", langDlgLoadCasDesc());
                dlgInfo.fileList = fileList;
                dlgInfo.fileListCount = count;
                dlgInfo.autoReset = autostart;

                strcpy(dlgInfo.selectFile, pProperties->cassette.tapeZip);

                DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ZIPDSK), Prt.hwnd, dskZipDlgProc, (LPARAM)&dlgInfo);

                autostart = dlgInfo.autoReset;
                if (dlgInfo.selectFile[0] == '\0') {
                    free(fileList);
                    return 0;
                }
                strcpy(tapeName, dlgInfo.selectFile);
            }
            free(fileList);
        }
    }
*/
    strcpy(pProperties->cassette.tape, filename);
    strcpy(pProperties->cassette.tapeZip, tapeName);
    //updateFileHistory(*pProperties->filehistory.cassette, filename);

    if (autostart && !Prt.noautostart) {
        emulatorStart(1);
    }
    else if (Prt.emuState != EMU_STOPPED) {
        //emulatorSuspend();
        tapeInsert(filename, isZip ? tapeName : NULL);
        //emulatorResume();
    }

    return 1;
}




static RomType romNameToType(char* name) {
    RomType romType = ROM_UNKNOWN;

    if (name == NULL) {
        return ROM_UNKNOWN;
    }

    if (0 == strcmp(name, "standard"))    romType = ROM_STANDARD;
    if (0 == strcmp(name, "msxdos2"))     romType = ROM_MSXDOS2;
    if (0 == strcmp(name, "konami5"))     romType = ROM_KONAMI5;
    if (0 == strcmp(name, "konami4"))     romType = ROM_KONAMI4;
    if (0 == strcmp(name, "ascii8"))      romType = ROM_ASCII8;
    if (0 == strcmp(name, "ascii16"))     romType = ROM_ASCII16;
    if (0 == strcmp(name, "gamemaster2")) romType = ROM_GAMEMASTER2;
    if (0 == strcmp(name, "ascii8sram"))  romType = ROM_ASCII8SRAM;
    if (0 == strcmp(name, "ascii16sram")) romType = ROM_ASCII16SRAM;
    if (0 == strcmp(name, "konami4nf"))   romType = ROM_KONAMI4NF;
    if (0 == strcmp(name, "ascii16nf"))   romType = ROM_ASCII16NF;
    if (0 == strcmp(name, "sccplus"))     romType = ROM_SCCPLUS;
    if (0 == strcmp(name, "fmpac"))       romType = ROM_FMPAC;
    if (0 == strcmp(name, "rtype"))       romType = ROM_RTYPE;
    if (0 == strcmp(name, "crossblaim"))  romType = ROM_CROSSBLAIM;
    if (0 == strcmp(name, "harryfox"))    romType = ROM_HARRYFOX;
    if (0 == strcmp(name, "korean80"))    romType = ROM_KOREAN80;
    if (0 == strcmp(name, "korean126"))   romType = ROM_KOREAN126;

    if (romType == ROM_UNKNOWN) {
        romType = atoi(name);
        if (romType < ROM_STANDARD || romType > ROM_KOREAN126) {
            romType = ROM_UNKNOWN;
        }
    }

    return romType;
}


int isRomFileType(char* fname) {
    int len = strlen(fname);

//    if (fname[len-1] == '\"') len--;

    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'R' &&
            toupper(fname[len - 2]) == 'O' &&
            toupper(fname[len - 1]) == 'M')
        {
            return 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            return 1;
        }
    }
    return 0;
}

int isDskFileType(char* fname) {
    int len = strlen(fname);

    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'D' &&
            toupper(fname[len - 2]) == 'S' &&
            toupper(fname[len - 1]) == 'K')
        {
            return 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            return 1;
        }
    }
    return 0;
}
int isCasFileType(char* fname) {
    int len = strlen(fname);

    if (len > 4) {
        if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'C' &&
            toupper(fname[len - 2]) == 'A' &&
            toupper(fname[len - 1]) == 'S')
        {
            return 1;
        }
        else if (toupper(fname[len - 4]) == '.' &&
            toupper(fname[len - 3]) == 'Z' &&
            toupper(fname[len - 2]) == 'I' &&
            toupper(fname[len - 1]) == 'P')
        {
            return 1;
        }
    }
    return 0;
}

static PropEmuFamily famNameToType(char* name) {
    PropEmuFamily family = -1;

    if (name == NULL) {
        return family;
    }

    if (0 == strcmp(name, "msx"))    family = P_EMU_MSX;
    if (0 == strcmp(name, "msxbr"))  family = P_EMU_MSXBR;
    if (0 == strcmp(name, "msxj"))   family = P_EMU_MSXJ;
    if (0 == strcmp(name, "msxkr"))  family = P_EMU_MSXKR;
    if (0 == strcmp(name, "msx2"))   family = P_EMU_MSX2;
    if (0 == strcmp(name, "msx2j"))  family = P_EMU_MSX2J;
    if (0 == strcmp(name, "msx2kr")) family = P_EMU_MSX2KR;
    if (0 == strcmp(name, "msx2p"))  family = P_EMU_MSX2P;

    return family;
}
BOOL propSetVideoSize(Properties* pProperties, PropVideoSize size) {
    BOOL changed = pProperties->video.size != size;
    pProperties->video.size = size;
    return changed;
}



void updateVideoRender() {
    switch (pProperties->video.monType) {
    case P_VIDEO_COLOR:
        videoSetColorMode(Prt.pVideo, VIDEO_COLOR);
        break;
    case P_VIDEO_BW:
        videoSetColorMode(Prt.pVideo, VIDEO_BLACKWHITE);
        break;
    case P_VIDEO_GREEN:
        videoSetColorMode(Prt.pVideo, VIDEO_GREEN);
        break;
    }

    switch (pProperties->video.palEmu) {
    case P_VIDEO_PALNONE:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_FAST);
        break;
    case P_VIDEO_PALYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP);
        break;
    case P_VIDEO_PALNYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP_NOISE);
        break;
    case P_VIDEO_PALCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR);
        break;
    case P_VIDEO_PALNCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR_NOISE);
        break;
	case P_VIDEO_PALSCALE2X:
		videoSetPalMode(Prt.pVideo, VIDEO_PAL_SCALE2X);
		break;
    }

    videoSetFrameSkip(Prt.pVideo, pProperties->video.frameSkip);
}

void emulatorStart(int hard) {
    int i;

    emuExitFlag = 0;

    ResetMSX(hard);

    for(i=0; i<256; i++) {
        emuFixedPalette[i] = videoGetColor(Prt.pVideo, (i & 0x1c) << 3, (i & 0xe0), ((i & 3) == 3 ? 7 : 2 * (i & 3)) << 5);
    }

    ProgDir        = Prt.pCurDir;
    emuFrameBuffer = (UInt32*)Prt.bmBits;
    emuLineWidth           = (int*)Prt.srcArr;

    emulatorSetFrequency(pProperties->emulation.speed);

    enableYM2413 = pProperties->sound.chip.msxmusic;
    enableY8950  = pProperties->sound.chip.msxaudio;
    VPeriod    = pProperties->video.videoType == P_VIDEO_PAL ? CPU_VPERIOD_PAL : CPU_VPERIOD_NTSC;
    if (0 == strcmp(CARTNAME_SCCPLUS, pProperties->cartridge.slotA)) ROMTypeA = ROM_SCCPLUS;
    if (0 == strcmp(CARTNAME_SCCPLUS, pProperties->cartridge.slotB)) ROMTypeB = ROM_SCCPLUS;
    if (0 == strcmp(CARTNAME_FMPAC, pProperties->cartridge.slotA)) ROMTypeA = ROM_FMPAC;
    if (0 == strcmp(CARTNAME_FMPAC, pProperties->cartridge.slotB)) ROMTypeB = ROM_FMPAC;
    CartA      = strlen(pProperties->cartridge.slotA) ? pProperties->cartridge.slotA : NULL;
    CartB      = strlen(pProperties->cartridge.slotB) ? pProperties->cartridge.slotB : NULL;
    CartAZip   = strlen(pProperties->cartridge.slotAZip) ? pProperties->cartridge.slotAZip : NULL;
    CartBZip   = strlen(pProperties->cartridge.slotBZip) ? pProperties->cartridge.slotBZip : NULL;
    DiskA      = strlen(pProperties->diskdrive.slotA) ? pProperties->diskdrive.slotA : NULL;
    DiskB      = strlen(pProperties->diskdrive.slotB) ? pProperties->diskdrive.slotB : NULL;
    DiskAZip   = strlen(pProperties->diskdrive.slotAZip) ? pProperties->diskdrive.slotAZip : NULL;
    DiskBZip   = strlen(pProperties->diskdrive.slotBZip) ? pProperties->diskdrive.slotBZip : NULL;
    CasName    = strlen(pProperties->cassette.tape) ? pProperties->cassette.tape : NULL;
    CasNameZip = strlen(pProperties->cassette.tapeZip) ? pProperties->cassette.tapeZip : NULL;
    CasPos     = Prt.tapeCurPos;
    JoyTypeA   = pProperties->joy1.type == P_JOY_NONE  ? 0 : 
                 pProperties->joy1.type == P_JOY_MOUSE ? 3 : 1;
    JoyTypeB   = pProperties->joy2.type == P_JOY_NONE  ? 0 : 
                 pProperties->joy2.type == P_JOY_MOUSE ? 3 : 1;
    StateName  = strlen(Prt.pStoredState) ? Prt.pStoredState : NULL;
    switch (pProperties->emulation.family) {
    case P_EMU_MSX:
        MSXVersion   = 0;
        MSXSysRom    = romMSX;
        MSXSysRomExt = NULL;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX.cmos";
        break;
    case P_EMU_MSXBR:
        MSXVersion   = 0;
        MSXSysRom    = romMSXBR;
        MSXSysRomExt = NULL;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSXBR.cmos";
        break;
    case P_EMU_MSXJ:
        MSXVersion   = 0;
        MSXSysRom    = romMSXJ;
        MSXSysRomExt = NULL;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSXJ.cmos";
        break;
    case P_EMU_MSXKR:
        MSXVersion   = 0;
        MSXSysRom    = romMSXKR;
        KoreaMSX1    = romMSXHAN;
        MSXSysRomExt = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSXKR.cmos";
        break;
    case P_EMU_MSX2:
        MSXVersion   = 1;
        MSXSysRom    = romMSX2;
        MSXSysRomExt = romMSX2EXT;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX2.cmos";
        break;
    case P_EMU_MSX2J:
        MSXVersion   = 1;
        MSXSysRom    = romMSX2J;
        MSXSysRomExt = romMSX2EXTJ;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX2J.cmos";
        break;
    case P_EMU_MSX2KR:
        MSXVersion   = 0;
        MSXSysRom    = romMSX2KR;
        KoreaMSX2    = romMSX2HAN;
        MSXSysRomExt = romMSX2EXTKR;
        KoreaMSX1    = NULL;
        CMOSname     = "MSX2KR.cmos";
        break;
    case P_EMU_MSX2P:
        MSXVersion   = 2;
        MSXSysRom    = romMSX2P;
        MSXSysRomExt = romMSX2PEXT;
        KoreaMSX1    = NULL;
        KoreaMSX2    = NULL;
        CMOSname     = "MSX2P.cmos";
        break;
    }
    RAMPages   = 1 << pProperties->emulation.RAMsize;
    VRAMPages  = 1 << pProperties->emulation.VRAMsize;
    ComName    = NULL;
    PrnName    = NULL;
    TheMixer   = Prt.mixer;

    //emuSyncSem = CreateSemaphore(NULL, 0, 1000, NULL);
    //emuTimer   = StartTimer(Prt.hwnd, emuSyncSem, emulatorGetSyncPeriod(), pProperties->emulation.syncMethod == P_EMU_SYNC1MS);

    SetCapslockLed(0);
    SetKanaLed(0);

    //emuThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)emulatorThread, 0, 0, &id);
    
    //soundResume();

    Prt.emuState = EMU_RUNNING;
    //mouseEmuUpdateRunState(1);
}

void emulatorStop() {
    if (Prt.emuState == EMU_STOPPED) {
        return;
    }

    Prt.emuState = EMU_STOPPED;
    //mouseEmuUpdateRunState(0);
    emuExitFlag = 1;
    //ReleaseSemaphore(emuSyncSem, 100, NULL);
    //WaitForSingleObject(emuThread, INFINITE);
    Prt.pStoredState[0] = 0;
    //soundSuspend();
    //CloseHandle(emuThread);
    //CloseHandle(emuSyncSem);
}


int emuStartWithArguments(PSTR szLine) {
    int i;
    char*   argument;
    char    rom1[512] = "";
    char    rom2[512] = "";
    RomType romType1  = ROM_UNKNOWN;
    RomType romType2  = ROM_UNKNOWN;
    PropEmuFamily family = -1;
    char    diskA[512] = "";
    char    diskB[512] = "";
	char    tape[512]  = "" ;
    int     fullscreen = 0;
    int     startEmu = 0;

	init_xmsx() ;

	updateVideoRender() ;
    
	// If one argument, assume it is a rom or disk to run
	/*
    if (!getCmdArgument(szLine, 1)) {
        argument = getCmdArgument(szLine, 0);
        
        if (*argument != '/') {
            if (*argument == '\"') argument++;

            if (*argument) {
                Prt.autostart = 1;
                return tryLaunchUnknownFile(argument);
            }
            return 0;
        }
    }*/

    // If more than one argument, check arguments,
    // set configuration and then run

    for (i = 0; argument = getCmdArgument(szLine, i); i++) {
        if (strcmp(argument, "/rom1") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isRomFileType(argument)) return 0; // Invaid argument
            strcpy(rom1, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/romtype1") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            romType1 = romNameToType(argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/rom2") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isRomFileType(argument)) return 0; // Invaid argument
            strcpy(rom2, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/romtype2") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            romType2 = romNameToType(argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/diskA") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isDskFileType(argument)) return 0; // Invaid argument
            strcpy(diskA, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/tape") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isCasFileType(argument)) return 0; // Invaid argument
            strcpy(tape, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/diskB") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL || !isDskFileType(argument)) return 0; // Invaid argument
            strcpy(diskB, argument);
            startEmu = 1;
        }
        if (strcmp(argument, "/family") == 0) {
            argument = getCmdArgument(szLine, ++i);
            if (argument == NULL) return 0; // Invaid argument
            family = famNameToType(argument);
            if (family == -1) return 0;
            startEmu = 1;
        }
        if (strcmp(argument, "/fullscreen") == 0) {
            fullscreen = 1;
        }
    }

    if (!startEmu) {
        return 1;
    }

    pProperties->cassette.tape[0] = 0;
    pProperties->cassette.tapeZip[0] = 0;

    pProperties->cartridge.slotA[0] = 0;
    pProperties->cartridge.slotAZip[0] = 0;
    pProperties->cartridge.slotB[0] = 0;
    pProperties->cartridge.slotBZip[0] = 0;

    pProperties->diskdrive.slotA[0] = 0;
    pProperties->diskdrive.slotAZip[0] = 0;
    pProperties->diskdrive.slotB[0] = 0;
    pProperties->diskdrive.slotBZip[0] = 0;
    
    Prt.noautostart = 1;

    if (strlen(rom1)  && !insertCartridge(0, rom1, NULL)) return 0;
    if (strlen(rom2)  && !insertCartridge(1, rom2, NULL)) return 0;
    if (strlen(diskA) && !insertDiskette(0, diskA, NULL)) return 0;
    if (strlen(diskB) && !insertDiskette(1, diskB, NULL)) return 0;
    if (strlen(tape)  && !insertCassette(tape, NULL)) return 0;

    if (family != -1) pProperties->emulation.family = family;

    //propSetVideoSize(pProperties, fullscreen ? P_VIDEO_SIZEFULLSCREEN : P_VIDEO_SIZEX2);

    Prt.noautostart = 0;

    emulatorStop();
    emulatorStart(1);

    return 1;
}



UInt8 Joystick(register UInt8 N) {
    //return JoystickGetState(N + 1);
	return 0 ;
}

void Mouse(int* dx, int* dy, int* buttons) {
    //mouseEmuGetState(dx, dy, buttons);
	return  ;
}

void Keyboard(unsigned char* keybardMap) {
    //memcpy(keybardMap, Prt.keyMap, 16);
}

void SetCapslockLed(int enable) {
    //if (enable) Prt.ledState |= 1;
    //else        Prt.ledState &= ~1;
}

void SetKanaLed(int enable) {
    //if (enable) Prt.ledState |= 2;
    //else        Prt.ledState &= ~2;
}

int WaitForSync(void) {
    //do {
        //emuSuspendFlag = 1;
        //WaitForSingleObject(emuSyncSem, INFINITE);
    //} while (!emuExitFlag && Prt.emuState != EMU_RUNNING);
    ///emuSuspendFlag = 0;

	return doquit ;
    //return emuExitFlag;
}

void RefreshScreen(int screenMode, int evenOdd, int interlace) {
    static int frameSkip = 0;
    if (frameSkip++ < pProperties->video.frameSkip) {
        return;
    }
    frameSkip = 0;


    Prt.updatePending++;
    if (Prt.updatePending <= 1) {
        Prt.screenMode = screenMode;
        Prt.evenOdd = evenOdd;
        Prt.interlace = interlace;
    }

	xbox_frame() ;
	if ( xbox_check_events( ) )
		doquit = 1 ;
	
	Prt.updatePending = 0;
}
void SetColor(int palEntry, unsigned int rgbColor) {
    UInt32 color = videoGetColor(Prt.pVideo, ((rgbColor >> 16) & 0xff), ((rgbColor >> 8) & 0xff), rgbColor & 0xff);
    //UInt32 color = videoGetColor(Prt.pVideo, ((rgbColor >> 16) & 0xff)>>3, ((rgbColor >> 8) & 0xff)>>2, (rgbColor & 0xff)>>3);
    if (palEntry == 0) {
        emuPalette0 = color;
    }
    else {
        emuPalette[palEntry] = color;
    }
}

int msx_refreshrate()
{
	if (     pProperties->video.videoType        == P_VIDEO_PAL )
		return 50 ;
	else 
		return 60 ;
}

void propInitDefaults(Properties* pProperties) 
{
    int i;
    pProperties->language                 = EMU_LANG_ENGLISH;

    pProperties->emulation.statsDefDir[0] = 0;
    pProperties->emulation.family         = P_EMU_MSX2;
    pProperties->emulation.speed          = 50;
    pProperties->emulation.RAMsize        = P_EMU_RAM2048;
    pProperties->emulation.VRAMsize       = P_EMU_VRAM128;
    pProperties->emulation.syncMethod     = P_EMU_SYNCAUTO ; //P_EMU_SYNC1MS ; //P_EMU_SYNCAUTO;
    
    pProperties->video.monType          = P_VIDEO_COLOR;
    pProperties->video.palEmu           = P_VIDEO_PALNONE;
    pProperties->video.videoType        = P_VIDEO_PAL ; //P_VIDEO_NTSC ; //P_VIDEO_PAL;
    pProperties->video.size             = P_VIDEO_SIZEX2;
    pProperties->video.driver           = P_VIDEO_DRVDIRECTX;
    pProperties->video.frameSkip        = P_VIDEO_FSKIP0;
    pProperties->video.fullRes          = P_VIDEO_FRES640X480_32;

    pProperties->sound.driver           = P_SOUND_DRVDIRECTX;
    pProperties->sound.frequency        = P_SOUND_FREQ44;
    pProperties->sound.bufSize          = P_SOUND_BUF150;
    pProperties->sound.syncMethod       = P_SOUND_SYNCQADJUST;

    pProperties->sound.stereo = 1;
    pProperties->sound.masterVolume = 92;
    pProperties->sound.chip.msxmusic = 1;
    pProperties->sound.chip.msxaudio = 1;


    pProperties->sound.mixerChannel[MCT_PSG].enable = 1;
    pProperties->sound.mixerChannel[MCT_PSG].pan = 42;
    pProperties->sound.mixerChannel[MCT_PSG].volume = 100;

    pProperties->sound.mixerChannel[MCT_SCC].enable = 1;
    pProperties->sound.mixerChannel[MCT_SCC].pan = 58;
    pProperties->sound.mixerChannel[MCT_SCC].volume = 86;

    pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable = 1;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan = 58;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume = 100;

    pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable = 1;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan = 50;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume = 100;

    pProperties->sound.mixerChannel[MCT_KEYBOARD].enable = 1;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].pan = 54;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].volume = 55;

    pProperties->sound.mixerChannel[MCT_CASSETTE].enable = 0;
    pProperties->sound.mixerChannel[MCT_CASSETTE].pan = 50;
    pProperties->sound.mixerChannel[MCT_CASSETTE].volume = 0;
    
    pProperties->joy1.type              = P_JOY_NONE;
    pProperties->joy1.autofire          = P_JOY_AFOFF;
    pProperties->joy1.keyUp             = 0xff;
    pProperties->joy1.keyDown           = 0xff;
    pProperties->joy1.keyLeft           = 0xff;
    pProperties->joy1.keyRight          = 0xff;
    pProperties->joy1.button1           = 0xff;
    pProperties->joy1.button2           = 0xff;
    
    pProperties->joy2.type              = P_JOY_NONE;
    pProperties->joy2.autofire          = P_JOY_AFOFF;
    pProperties->joy2.keyUp             = 0xff;
    pProperties->joy2.keyDown           = 0xff;
    pProperties->joy2.keyLeft           = 0xff;
    pProperties->joy2.keyRight          = 0xff;
    pProperties->joy2.button1           = 0xff;
    pProperties->joy2.button2           = 0xff;
    
    pProperties->keyboard.keySet        = P_CHAR_EUROPEAN;
    
    pProperties->cartridge.defDir[0]    = 0;
    pProperties->cartridge.slotA[0]     = 0;
    pProperties->cartridge.slotB[0]     = 0;
    pProperties->cartridge.slotAZip[0]  = 0;
    pProperties->cartridge.slotBZip[0]  = 0;
    pProperties->cartridge.autoReset    = 1;
    
    pProperties->diskdrive.defDir[0]    = 0;
    pProperties->diskdrive.slotA[0]     = 0;
    pProperties->diskdrive.slotB[0]     = 0;
    pProperties->diskdrive.slotAZip[0]  = 0;
    pProperties->diskdrive.slotBZip[0]  = 0;
    pProperties->diskdrive.autostartA   = 0;

    pProperties->cassette.defDir[0]     = 0;
    pProperties->cassette.tape[0]       = 0;
    pProperties->cassette.tapeZip[0]    = 0;

    for (i = 0; i < MAX_HISTORY; i++) {
        pProperties->filehistory.cartridgeA[i][0] = 0;
        pProperties->filehistory.cartridgeB[i][0] = 0;
        pProperties->filehistory.diskdriveA[i][0] = 0;
        pProperties->filehistory.diskdriveB[i][0] = 0;
        pProperties->filehistory.cassette[i][0] = 0;
    }

}
