/*****************************************************************************
** File:        Win32properties.c
**
** Author:      Daniel Vik
**
** Description: blueMSX properties dialog and save/load methods
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
** Note:        The following macros can be defined to change the behaviour
**              of the properties:
**
**              PROPERTIES_DEFAULTS_ALT_1
**                  Alternative default settings
**
**              PROPERTIES_NO_REGISTRY
**                  Dont save settings in registry. Uses a windows ini file.
**
**              PROPERTIES_LOCAL_INI_FILE
**                  Stores the windows ini file in same directory as the
**                  executable.
**
******************************************************************************
*/
#include <windows.h>
#include <tchar.h>
#include <math.h>
#include <commctrl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
 
#include "Win32Properties.h"
#include "resource.h"
#include "Language.h"

static int propModified = 0;
static int canUseRegistry = 0;
static Mixer* theMixer;
extern void emulatorRestartSound(int stereo);

static char virtualKeys[256][32] = {
    "",
    "", //"LBUTTON", 
    "", //"RBUTTON",
    "CANCEL",
    "", //"MBUTTON",
    "", //"XBUTTON1",
    "", //"XBUTTON2",
    "",
    "BACKSPACE",
    "", //"TAB",
    "",
    "",
    "CLEAR",
    "ENTER",
    "",
    "",
    "SHIFT",
    "CTRL",
    "ALT",
    "PAUSE",
    "CAPS LOCK",
    "KANA",
    "",
    "JUNJA",
    "FINAL",
    "KANJI",
    "",
    "ESC",
    "CONVERT",
    "NONCONVERT",
    "ACCEPT",
    "MODECHANGE",
    "SPACE",
    "PAGE UP",
    "PAGE DOWN",
    "END",
    "HOME",
    "LEFT ARROW",
    "UP ARROW",
    "RIGHT ARROW",
    "DOWN ARROW",
    "SELECT",
    "PRINT",
    "EXECUTE",
    "PRINT SCREEN",
    "INS",
    "DEL",
    "HELP",
    "0",
    "1",
    "2",
    "3",
    "4",
    "5",
    "6",
    "7",
    "8",
    "9",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "A",
    "B",
    "C",
    "D",
    "E",
    "F",
    "G",
    "H",
    "I",
    "J",
    "K",
    "L",
    "M",
    "N",
    "O",
    "P",
    "Q",
    "R",
    "S",
    "T",
    "U",
    "V",
    "W",
    "X",
    "Y",
    "Z",
    "LEFT WIN",
    "RIGHT WIN",
    "APPS",
    "",
    "SLEEP",
    "NUMPAD 0",
    "NUMPAD 1",
    "NUMPAD 2",
    "NUMPAD 3",
    "NUMPAD 4",
    "NUMPAD 5",
    "NUMPAD 6",
    "NUMPAD 7",
    "NUMPAD 8",
    "NUMPAD 9",
    "NUMPAD *",
    "NUMPAD +",
    "NUMPAD ,",
    "NUMPAD -",
    "NUMPAD .",
    "NUMPAD /",
    "F1",
    "F2",
    "F3",
    "F4",
    "F5",
    "F6",
    "F7",
    "F8",
    "F9",
    "F10",
    "F11",
    "F12",
    "F13",
    "F14",
    "F15",
    "F16",
    "F17",
    "F18",
    "F19",
    "F20",
    "F21",
    "F22",
    "F23",
    "F24",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "NUM LOCK",
    "SCROLL LOCK",
    "OEM 1",
    "OEM 2",
    "OEM 3",
    "OEM 4",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "LEFT SHIFT",
    "RIGHT SHIFT",
    "LEFT CONTROL",
    "RIGHT CONTROL",
    "LEFT ALT",
    "RIGHT ALT",
    "BROWSER BACK",
    "BROWSER FORWARD",
    "BROWSER REFRESH",
    "BROWSER STOP",
    "BROWSER SEARCH",
    "BROWSER FAVORITES",
    "BROWSER HOME",
    "VOLUME MUTE",
    "VOLUME DOWN",
    "VOLUME UP",
    "NEXT TRACK",
    "PREV TRACK",
    "MEDIA STOP",
    "MEDIA PLAY",
    "LAUNCH MAIL",
    "LAUNCH MEDIA SELECT",
    "LAUNCH APP 1",
    "LAUNCH APP 2",
    "",
    "",
    ";",
    "+",
    ",",
    "-",
    ".",
    "?",
    "~",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "[",
    "\\",
    "]",
    "\"",
    "OEM 5",
    "",
    "OEM 6",
    "OEM 7",
    "OEM 8",
    "OEM 9",
    "PROCESS",
    "OEM 10",
    "",
    "",
    "OEM 11",
    "OEM 12",
    "OEM 13",
    "OEM 14",
    "OEM 15",
    "OEM 16",
    "OEM 17",
    "OEM 18",
    "OEM 19",
    "OEM 20",
    "OEM 21",
    "OEM 22",
    "OEM 23",
    "ATTN",
    "CRSEL",
    "EXSEL",
    "EREOF",
    "PLAY",
    "ZOOM",
    "",
    "PA1",
    "CLEAR",
    ""
};

static char* keyPath = NULL;
static char registryKey[] = "blueMSX";

static _TCHAR* pEmuFamily[] = {
    "MSX, Basic 1.0",
    "MSX, Basic 1.0 - Brazilian",
    "MSX, Basic 1.0 - Japanese",
    "MSX, Basic 1.0 - Korean",
    "MSX 2, Basic 2.1",
    "MSX 2, Basic 2.0 - Japanese", 
    "MSX 2, Basic 2.0 - Korean", 
    "MSX 2+, Basic 3.0",
    NULL
};

static _TCHAR* pEmuRAM[] = {
    "16 kBytes",
    "32 kBytes",
    "64 kBytes",
    "128 kBytes",
    "256 kBytes",
    "512 kBytes",
    "1 MBytes",
    "2 MBytes",
    "4 MBytes",
    NULL
};

static _TCHAR* pEmuVRAM[] = {
    "16 kBytes",
    "32 kBytes",
    "64 kBytes",
    "128 kBytes",
    NULL
};

static _TCHAR pVideoMonData[3][64];
static _TCHAR* pVideoMon[] = {
    pVideoMonData[0],
    pVideoMonData[1],
    pVideoMonData[2],
    NULL
};

static _TCHAR pVideoTypeData[2][64];
static _TCHAR* pVideoVideoType[] = {
    pVideoTypeData[0], 
    pVideoTypeData[1],
    NULL
};

static _TCHAR pVideoEmuData[6][64];
static _TCHAR* pVideoPalEmu[] = {
    pVideoEmuData[0],
    pVideoEmuData[1],
    pVideoEmuData[2],
    pVideoEmuData[3],
    pVideoEmuData[4],
    pVideoEmuData[5],
    NULL
};

static _TCHAR pVideoSizeData[6][64];
static _TCHAR* pVideoMonSize[] = {
    pVideoSizeData[0],
    pVideoSizeData[1],
    pVideoSizeData[2],
    NULL
};

static _TCHAR pVideoDriverData[2][64];
static _TCHAR* pVideoDriver[] = {
    pVideoDriverData[0],
    pVideoDriverData[1],
    NULL
};

static _TCHAR pVideoFrameSkipData[6][64];
static _TCHAR* pVideoFrameSkip[] = {
    pVideoFrameSkipData[0],
    pVideoFrameSkipData[1],
    pVideoFrameSkipData[2],
    pVideoFrameSkipData[3],
    pVideoFrameSkipData[4],
    pVideoFrameSkipData[5],
    NULL
};

static _TCHAR pSoundDriverData[3][64];
static _TCHAR* pSoundDriver[] = {
    pSoundDriverData[0],
    pSoundDriverData[1],
    pSoundDriverData[2],
    NULL
};

static _TCHAR pEmuSyncData[2][64];
static _TCHAR* pEmuSync[] = {
    pEmuSyncData[0],
    pEmuSyncData[1],
    NULL
};

static _TCHAR* pSoundBufferSize[] = {
    "100 ms",
    "150 ms",
    "200 ms",
    "250 ms",
    "300 ms",
    "350 ms",
    NULL
};

static _TCHAR pControlsJoyData[7][64];
static _TCHAR* pControlsJoy[] = {
    pControlsJoyData[0], 
    pControlsJoyData[1],
    pControlsJoyData[2], 
    pControlsJoyData[3],
    pControlsJoyData[4], 
    pControlsJoyData[5],
    pControlsJoyData[6], 
    NULL
};

static _TCHAR pControlsAutofireData[4][64];
static _TCHAR* pControlsAutofire[] = {
    pControlsAutofireData[0], 
    pControlsAutofireData[1],
    pControlsAutofireData[2], 
    pControlsAutofireData[3], 
    NULL
};

static void getRegStrValue(char* keyDir, char* keyStr, char* returnValue) {  
    char value[1024];
    LONG rv;
    HKEY hKey;
    DWORD length = 1024;
    DWORD type;
    char directory[64];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegOpenKeyEx(HKEY_CURRENT_USER, directory, 0, KEY_QUERY_VALUE, &hKey);    
    if (rv != ERROR_SUCCESS) {
        return;
    }

    rv = RegQueryValueEx(hKey, keyStr, NULL, &type, (BYTE*)value, &length);
    RegCloseKey(hKey);
    if (rv != ERROR_SUCCESS) {
        return;
    }

    strcpy(returnValue, value);
}

static void getRegIntValue(char* keyDir, char* keyStr, DWORD* returnValue) {  
    LONG rv;
    HKEY hKey;
    DWORD length;
    DWORD type;
    DWORD value;
    char directory[32];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegOpenKeyEx(HKEY_CURRENT_USER, directory, 0, KEY_QUERY_VALUE, &hKey);  
    if (rv != ERROR_SUCCESS) {
        return;
    }

    rv = RegQueryValueEx(hKey, keyStr, NULL, &type, (BYTE *)&value, &length);

    RegCloseKey(hKey);

    if (rv != ERROR_SUCCESS) {
        return;
    }

    *returnValue = value;
}

static BOOL setRegIntValue(char* keyDir, char* keyStr, DWORD value) {
    HKEY hKey;
    DWORD exist;
    DWORD rv;
    DWORD dwValue = (DWORD)value;
    char directory[64];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegCreateKeyEx(HKEY_CURRENT_USER, directory, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    if (rv != ERROR_SUCCESS) {
        return FALSE;
    }

    rv = RegSetValueEx(hKey, keyStr, 0, REG_DWORD, (BYTE *)&dwValue, sizeof(DWORD));

    RegCloseKey(hKey);

    return TRUE;
}

static BOOL setRegStrValue(char* keyDir, char* keyStr, char* value) {
    HKEY hKey;
    DWORD exist;
    DWORD rv;
    DWORD dwValue = (DWORD)value;
    char directory[64];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegCreateKeyEx(HKEY_CURRENT_USER, directory, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    if (rv != ERROR_SUCCESS) {
        return FALSE;
    }

    rv = RegSetValueEx(hKey, keyStr, 0, REG_SZ, (BYTE *)value, strlen(value) + 1);

    RegCloseKey(hKey);

    return TRUE;
}

static void getIniStrValue(char* keyDir, char* keyStr, char* returnValue) {  
    char file[1024];
    char defStr[128];

    if (keyPath != NULL) sprintf(file, "%s\\%s.ini", keyPath, keyDir);
    else                 sprintf(file, "%s.ini", keyDir);

    strcpy(defStr, returnValue);

    GetPrivateProfileString("General", keyStr, defStr, returnValue, 64, file);      
}

static void getIniIntValue(char* keyDir, char* keyStr, DWORD* returnValue) {  
    char file[1024];
    DWORD def = *returnValue;

    if (keyPath != NULL) sprintf(file, "%s\\%s.ini", keyPath, keyDir);
    else                 sprintf(file, "%s.ini", keyDir);

    *returnValue = GetPrivateProfileInt("General", keyStr, def, file);                           
}

static BOOL setIniIntValue(char* keyDir, char* keyStr, DWORD value) {
    char file[1024];
    char buf[30];

    sprintf(buf, "%d", value);

    if (keyPath != NULL) sprintf(file, "%s\\%s.ini", keyPath, keyDir);
    else                 sprintf(file, "%s.ini", keyDir);

    return WritePrivateProfileString("General", keyStr, buf, file);
}

static BOOL setIniStrValue(char* keyDir, char* keyStr, char* value) {
    char file[1024];

    if (keyPath != NULL) sprintf(file, "%s\\%s.ini", keyPath, keyDir);
    else                 sprintf(file, "%s.ini", keyDir);

    return WritePrivateProfileString("General", keyStr, value, file);
}
  
static void getStrValue(char* keyDir, char* keyStr, char* returnValue) {
    if (canUseRegistry) getRegStrValue(keyDir, keyStr, returnValue);
    else                getIniStrValue(keyDir, keyStr, returnValue);
}

static void getIntValue(char* keyDir, char* keyStr, DWORD* returnValue) {
    if (canUseRegistry) getRegIntValue(keyDir, keyStr, returnValue);
    else                getIniIntValue(keyDir, keyStr, returnValue);
}   
    
static BOOL setStrValue(char* keyDir, char* keyStr, char* value) {
    if (canUseRegistry) return setRegStrValue(keyDir, keyStr, value);
    else                return setIniStrValue(keyDir, keyStr, value);
}

static BOOL setIntValue(char* keyDir, char* keyStr, DWORD value) {
    if (canUseRegistry) return setRegIntValue(keyDir, keyStr, value);
    else                return setIniIntValue(keyDir, keyStr, value);
}

static void setButtonCheck(HWND hDlg, int id, int check, int enable) {
    HWND hwnd = GetDlgItem(hDlg, id);

    if (check) {
        SendMessage(hwnd, BM_SETCHECK, BST_CHECKED, 0);
    }
    else {
        SendMessage(hwnd, BM_SETCHECK, BST_UNCHECKED, 0);
    }
    if (!enable) {
        SendMessage(hwnd, BM_SETCHECK, BST_INDETERMINATE, 0);
    }
}

static int getButtonCheck(HWND hDlg, int id) {
    HWND hwnd = GetDlgItem(hDlg, id);

    return BST_CHECKED == SendMessage(hwnd, BM_GETCHECK, 0, 0) ? 1 : 0;
}

static void initDropList(HWND hDlg, int id, _TCHAR** pList, int index) {
    while (*pList != NULL) {
        SendDlgItemMessage(hDlg, id, CB_ADDSTRING, 0, (LPARAM)*pList);
        pList++;
    }

    SendDlgItemMessage(hDlg, id, CB_SETCURSEL, index, 0);
}

static int getDropListIndex(HWND hDlg, int id, _TCHAR** pList) {
    int index = 0;
    _TCHAR s[64];

    GetDlgItemText(hDlg, id, s, 63);
    
    while (*pList != NULL) {
        if (0 == _tcscmp(s, *pList)) {
            return index;
        }
        index++;
        pList++;
    }

    return -1;
}

static char* strEmuSpeed(int logFrequency) {
    UInt32 frequency = (UInt32)(3579545 * pow(2.0, (logFrequency - 50) / 15.0515));
    static char buffer[32];

    sprintf(buffer, "%d.%03dMHz (%d%%)", frequency / 1000000, (frequency / 1000) % 1000, frequency * 10 / 357954);
    return buffer;

}

static BOOL CALLBACK emulationDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static Properties* pProperties;
    static int curSpeed;

    switch (iMsg) {
    case WM_INITDIALOG:    
        SendMessage(GetDlgItem(hDlg, IDC_EMUGENERALGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropEmuGeneralGB());
        SendMessage(GetDlgItem(hDlg, IDC_EMUFAMILYTEXT), WM_SETTEXT, 0, (LPARAM)langPropEmuFamilyText());
        SendMessage(GetDlgItem(hDlg, IDC_EMUMEMORYGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropEmuMemoryGB());
        SendMessage(GetDlgItem(hDlg, IDC_EMURAMSIZETEXT), WM_SETTEXT, 0, (LPARAM)langPropEmuRamSizeText());
        SendMessage(GetDlgItem(hDlg, IDC_EMUVRAMSIZETEXT), WM_SETTEXT, 0, (LPARAM)langPropEmuVramSizeText());
        SendMessage(GetDlgItem(hDlg, IDC_EMUSPEEDGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropEmuSpeedGB());

        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        curSpeed = pProperties->emulation.speed;

        initDropList(hDlg, IDC_EMUFAMILY, pEmuFamily, pProperties->emulation.family);

        SendMessage(GetDlgItem(hDlg, IDC_EMUSPEEDTEXT), WM_SETTEXT, 0, (LPARAM)strEmuSpeed(curSpeed));

        SendMessage(GetDlgItem(hDlg, IDC_EMUSPEED), TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
        SendMessage(GetDlgItem(hDlg, IDC_EMUSPEED), TBM_SETPOS,   1, (LPARAM)curSpeed);

        initDropList(hDlg, IDC_EMURAM, pEmuRAM, pProperties->emulation.RAMsize);
        initDropList(hDlg, IDC_EMUVRAM, pEmuVRAM, pProperties->emulation.VRAMsize);

        return FALSE;

    case WM_NOTIFY:
        if (wParam == IDC_EMUSPEED) {
            curSpeed = SendMessage(GetDlgItem(hDlg, IDC_EMUSPEED), TBM_GETPOS, 0, 0);
            SendMessage(GetDlgItem(hDlg, IDC_EMUSPEEDTEXT), WM_SETTEXT, 0, (LPARAM)strEmuSpeed(curSpeed));
            return TRUE;
        }

        if ((((NMHDR FAR *)lParam)->code) != PSN_APPLY) {
            return FALSE;
        }
            
        pProperties->emulation.family       = getDropListIndex(hDlg, IDC_EMUFAMILY, pEmuFamily);
        pProperties->emulation.speed        = curSpeed;
        pProperties->emulation.RAMsize      = getDropListIndex(hDlg, IDC_EMURAM, pEmuRAM);
        pProperties->emulation.VRAMsize     = getDropListIndex(hDlg, IDC_EMUVRAM, pEmuVRAM);

        propModified = 1;
        
        return TRUE;
    }

    return FALSE;
}

static BOOL CALLBACK performanceDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static Properties* pProperties;

    switch (iMsg) {
    case WM_INITDIALOG:
        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        /* Init language specific dropdown list data */
        _stprintf(pVideoDriver[0], "%s", langEnumVideoDrvDirectDraw());
        _stprintf(pVideoDriver[1], "%s", langEnumVideoDrvGDI());

        _stprintf(pVideoFrameSkip[0], "%s", langEnumVideoFrameskip0());
        _stprintf(pVideoFrameSkip[1], "%s", langEnumVideoFrameskip1());
        _stprintf(pVideoFrameSkip[2], "%s", langEnumVideoFrameskip2());
        _stprintf(pVideoFrameSkip[3], "%s", langEnumVideoFrameskip3());
        _stprintf(pVideoFrameSkip[4], "%s", langEnumVideoFrameskip4());
        _stprintf(pVideoFrameSkip[5], "%s", langEnumVideoFrameskip5());

        _stprintf(pSoundDriver[0], "%s", langEnumSoundDrvNone());
        _stprintf(pSoundDriver[1], "%s", langEnumSoundDrvWMM());
        _stprintf(pSoundDriver[2], "%s", langEnumSoundDrvDirectX());

        _stprintf(pEmuSync[0], "%s", langEnumEmuSync1ms());
        _stprintf(pEmuSync[1], "%s", langEnumEmuSyncAuto());

        /* Init language specific dialog items */
        SendMessage(GetDlgItem(hDlg, IDC_PERFVIDEODRVGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropPerfVideoDrvGB());
        SendMessage(GetDlgItem(hDlg, IDC_PERFDISPDRVTEXT), WM_SETTEXT, 0, (LPARAM)langPropPerfVideoDispDrvText());
        SendMessage(GetDlgItem(hDlg, IDC_PERFFRAMESKIPTEXT), WM_SETTEXT, 0, (LPARAM)langPropPerfFrameSkipText());
        SendMessage(GetDlgItem(hDlg, IDC_AUDIODRVGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropPerfAudioDrvGB());
        SendMessage(GetDlgItem(hDlg, IDC_PERFSNDDRVTEXT), WM_SETTEXT, 0, (LPARAM)langPropPerfAudioDrvText());
        SendMessage(GetDlgItem(hDlg, IDC_PERFSNDBUFSZTEXT), WM_SETTEXT, 0, (LPARAM)langPropPerfAudioBufSzText());
        SendMessage(GetDlgItem(hDlg, IDC_PERFEMUGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropPerfEmuGB());
        SendMessage(GetDlgItem(hDlg, IDC_PERFSYNCMODETEXT), WM_SETTEXT, 0, (LPARAM)langPropPerfSyncModeText());

        initDropList(hDlg, IDC_SNDDRIVER, pSoundDriver, pProperties->sound.driver);
        initDropList(hDlg, IDC_SNDBUFSZ, pSoundBufferSize, pProperties->sound.bufSize);
        initDropList(hDlg, IDC_VIDEODRV, pVideoDriver, pProperties->video.driver);
        initDropList(hDlg, IDC_FRAMESKIP, pVideoFrameSkip, pProperties->video.frameSkip);
        initDropList(hDlg, IDC_EMUSYNC, pEmuSync, pProperties->emulation.syncMethod);

        return FALSE;

    case WM_NOTIFY:
        if ((((NMHDR FAR *)lParam)->code) != PSN_APPLY) {
            return FALSE;
        }
            
        pProperties->sound.driver           = getDropListIndex(hDlg, IDC_SNDDRIVER, pSoundDriver);
        pProperties->sound.bufSize          = getDropListIndex(hDlg, IDC_SNDBUFSZ, pSoundBufferSize);
        pProperties->video.driver           = getDropListIndex(hDlg, IDC_VIDEODRV, pVideoDriver);
        pProperties->video.frameSkip        = getDropListIndex(hDlg, IDC_FRAMESKIP, pVideoFrameSkip);
        pProperties->emulation.syncMethod   = getDropListIndex(hDlg, IDC_EMUSYNC, pEmuSync);

        propModified = 1;
        
        return TRUE;
    }

    return FALSE;
}

static BOOL CALLBACK videoDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static Properties* pProperties;

    switch (iMsg) {
    case WM_INITDIALOG:
        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        /* Init language specific dropdown list data */
        _stprintf(pVideoMon[0], "%s", langEnumVideoMonColor());
        _stprintf(pVideoMon[1], "%s", langEnumVideoMonGrey());
        _stprintf(pVideoMon[2], "%s", langEnumVideoMonGreen());

        _stprintf(pVideoVideoType[0], "%s", langEnumVideoTypePAL());
        _stprintf(pVideoVideoType[1], "%s", langEnumVideoTypeNTSC());

        _stprintf(pVideoPalEmu[0], "%s", langEnumVideoEmuNone());
        _stprintf(pVideoPalEmu[1], "%s", langEnumVideoEmuYc());
        _stprintf(pVideoPalEmu[2], "%s", langEnumVideoEmuYcBlur());
        _stprintf(pVideoPalEmu[3], "%s", langEnumVideoEmuComp());
        _stprintf(pVideoPalEmu[4], "%s", langEnumVideoEmuCompBlur());
        _stprintf(pVideoPalEmu[5], "%s", langEnumVideoEmuScale2x());

        _stprintf(pVideoMonSize[0], "%s", langEnumVideoSize1x());
        _stprintf(pVideoMonSize[1], "%s", langEnumVideoSize2x());
        _stprintf(pVideoMonSize[2], "%s", langEnumVideoSizeFullscreen());

        /* Init language specific dialog items */
        SendMessage(GetDlgItem(hDlg, IDC_MONGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropMonMonGB());
        SendMessage(GetDlgItem(hDlg, IDC_MONTYPETEXT), WM_SETTEXT, 0, (LPARAM)langPropMonTypeText());
        SendMessage(GetDlgItem(hDlg, IDC_MONEMUTEXT), WM_SETTEXT, 0, (LPARAM)langPropMonEmuText());
        SendMessage(GetDlgItem(hDlg, IDC_MONVIDEOTYPETEXT), WM_SETTEXT, 0, (LPARAM)langPropVideoYypeText());
        SendMessage(GetDlgItem(hDlg, IDC_MONWINDOWSIZETEXT), WM_SETTEXT, 0, (LPARAM)langPropWindowSizeText());

        /* Init dropdown lists */
        initDropList(hDlg, IDC_MONTYPE, pVideoMon, pProperties->video.monType);
        initDropList(hDlg, IDC_PALEMU, pVideoPalEmu, pProperties->video.palEmu);
        initDropList(hDlg, IDC_VIDEOTYPE, pVideoVideoType, pProperties->video.videoType);
        initDropList(hDlg, IDC_MONSIZE, pVideoMonSize, pProperties->video.size);

        return FALSE;

    case WM_NOTIFY:
        if ((((NMHDR FAR *)lParam)->code) != PSN_APPLY) {
            return FALSE;
        }

        pProperties->video.monType          = getDropListIndex(hDlg, IDC_MONTYPE, pVideoMon);
        pProperties->video.palEmu           = getDropListIndex(hDlg, IDC_PALEMU, pVideoPalEmu);
        pProperties->video.videoType        = getDropListIndex(hDlg, IDC_VIDEOTYPE, pVideoVideoType);
        pProperties->video.size             = getDropListIndex(hDlg, IDC_MONSIZE, pVideoMonSize);

        propModified = 1;
        
        return TRUE;
    }

    return FALSE;
}

static void CreateToolTip(HWND hwndCtrl, HWND* hwndTT, TOOLINFO* ti)
{
    *hwndTT = CreateWindow(TOOLTIPS_CLASS, TEXT(""), WS_POPUP, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
                          NULL, (HMENU)NULL, GetModuleHandle(NULL), NULL);

    ti->cbSize = sizeof(TOOLINFO);
    ti->uFlags = TTF_IDISHWND | TTF_CENTERTIP | TTF_ABSOLUTE;
    ti->hwnd   = hwndCtrl;
    ti->uId    = (UINT)hwndCtrl;
    ti->hinst  = GetModuleHandle(NULL);
    ti->lpszText  = "";
    ti->rect.left = ti->rect.top = ti->rect.bottom = ti->rect.right = 60; 

    SendMessage(*hwndTT, TTM_ADDTOOL, 0, (LPARAM)ti);
    SendMessage(*hwndTT, TTM_TRACKACTIVATE, (WPARAM)TRUE, (LPARAM)ti);
    SendMessage(hwndCtrl, TBM_SETTOOLTIPS, (WPARAM)*hwndTT, 0);
}

static void UpdateVolumeToolTip(HWND hCtrl, HWND hwndTT, TOOLINFO* ti)
{
    static char str[32];
    int val = SendMessage(hCtrl, TBM_GETPOS,   0, 0);
    if (val == 100) {
        sprintf(str, "-inf.");
    }
    else {
        sprintf(str, "%.1f dB", -30. * val / 100);
    }
    ti->lpszText  = str;
    SendMessage(hwndTT, TTM_UPDATETIPTEXT, 0, (LPARAM)ti);  
}

static void UpdatePanToolTip(HWND hCtrl, HWND hwndTT, TOOLINFO* ti)
{
    static char str[32];
    int val = SendMessage(hCtrl, TBM_GETPOS,   0, 0);
    if (val == 50) {
        sprintf(str, "    C    ");
    }
    else if (val < 50) {
        sprintf(str, "L: %d", 2 * (50 - val));
    }
    else {
        sprintf(str, "R: %d", 2 * (val - 50));
    }
    ti->lpszText  = str;
    SendMessage(hwndTT, TTM_UPDATETIPTEXT, 0, (LPARAM)ti);  
}

static BOOL CALLBACK soundDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static HWND hwndVolTT[MCT_NUMENTRIES];
    static TOOLINFO tiVol[MCT_NUMENTRIES];
    static HWND hwndPanTT[MCT_NUMENTRIES];
    static TOOLINFO tiPan[MCT_NUMENTRIES];
    static HWND hwndMasterTT[2];
    static TOOLINFO tiMaster[2];
    static Properties* pProperties;
    static int stereo;
    int updateAudio = FALSE;
    int i;

    switch (iMsg) {
    case WM_INITDIALOG:
        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        SendMessage(GetDlgItem(hDlg, IDC_SNDMIXERMASTERTEXT), WM_SETTEXT, 0, (LPARAM)langPropSndMasterText());

        SendMessage(GetDlgItem(hDlg, IDC_SNDCHIPEMUGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropSndChipEmuGB());
        SendMessage(GetDlgItem(hDlg, IDC_SNDMIXERGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropSndMixerGB());
        SetWindowText(GetDlgItem(hDlg, IDC_ENABLEMSXMUSIC), langPropSndMsxMusicText());
        SetWindowText(GetDlgItem(hDlg, IDC_ENABLEMSXAUDIO), langPropSndMsxAudioText());

        setButtonCheck(hDlg, IDC_ENABLEMSXMUSIC, pProperties->sound.chip.msxmusic, 1);
        setButtonCheck(hDlg, IDC_ENABLEMSXAUDIO, pProperties->sound.chip.msxaudio, 1);

        for (i = 0; i < MCT_NUMENTRIES; i++) {
            CreateToolTip(GetDlgItem(hDlg, IDC_VOLUME1 + i), &hwndVolTT[i], &tiVol[i]);
            CreateToolTip(GetDlgItem(hDlg, IDC_PAN1 + i), &hwndPanTT[i], &tiPan[i]);

            SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i),    TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
            SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i),    TBM_SETPOS,   1, (LPARAM)(100 - pProperties->sound.mixerChannel[i].volume));
            SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i),       TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
            SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i),       TBM_SETPOS,   1, (LPARAM)pProperties->sound.mixerChannel[i].pan);
            SendMessage(GetDlgItem(hDlg, IDC_VOLENABLE1 + i), BM_SETCHECK, pProperties->sound.mixerChannel[i].enable ? BST_CHECKED : BST_UNCHECKED, 0);
        }

        CreateToolTip(GetDlgItem(hDlg, IDC_MASTERL), &hwndMasterTT[0], &tiMaster[0]);
        CreateToolTip(GetDlgItem(hDlg, IDC_MASTERR), &hwndMasterTT[1], &tiMaster[1]);

        SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
        SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETPOS,   1, (LPARAM)(100 - pProperties->sound.masterVolume));
        SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
        SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETPOS,   1, (LPARAM)(100 - pProperties->sound.masterVolume));

        stereo = pProperties->sound.stereo;
        SetWindowText(GetDlgItem(hDlg, IDC_STEREO), stereo ? langPropSndStereoText() : langPropSndMonoText());

        return FALSE;

    case WM_COMMAND:
        if (LOWORD(wParam) >= IDC_VOLENABLE1 && LOWORD(wParam) < IDC_VOLENABLE1 + MCT_NUMENTRIES) {
            updateAudio = TRUE;
        }

        switch(LOWORD(wParam)) {
        case IDC_STEREO:
            stereo = !stereo;
            SetWindowText(GetDlgItem(hDlg, IDC_STEREO), stereo ? langPropSndStereoText() : langPropSndMonoText());
            emulatorRestartSound(stereo);
            updateAudio = TRUE;
            break;
        case IDC_MASTERL:
            SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0));
            updateAudio = TRUE;
            break;
        case IDC_MASTERR:
            SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_GETPOS,   0, 0));
            updateAudio = TRUE;
            break;
        }
        break;

    case WM_NOTIFY:
        switch (((NMHDR FAR *)lParam)->code) {
        case PSN_APPLY:

            pProperties->sound.chip.msxmusic = getButtonCheck(hDlg, IDC_ENABLEMSXMUSIC);
            pProperties->sound.chip.msxaudio = getButtonCheck(hDlg, IDC_ENABLEMSXAUDIO);


            for (i = 0; i < MCT_NUMENTRIES; i++) {
                pProperties->sound.mixerChannel[i].volume = 100 - SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i),    TBM_GETPOS,   0, 0);
                pProperties->sound.mixerChannel[i].pan    = SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i),       TBM_GETPOS,   0, 0);
                pProperties->sound.mixerChannel[i].enable = SendMessage(GetDlgItem(hDlg, IDC_VOLENABLE1 + i), BM_GETCHECK, 0, 0) == BST_CHECKED;
            }

            pProperties->sound.masterVolume = 100 - SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0);
            pProperties->sound.stereo = stereo;

            for (i = 0; i < MCT_NUMENTRIES; i++) {
                DestroyWindow(hwndVolTT[i]);
                DestroyWindow(hwndPanTT[i]);
            }
            DestroyWindow(hwndMasterTT[0]);
            DestroyWindow(hwndMasterTT[1]);

            propModified = 1;
            return TRUE;

        case PSN_QUERYCANCEL:
            emulatorRestartSound(pProperties->sound.stereo);

            for (i = 0; i < MCT_NUMENTRIES; i++) {
                mixerSetChannelVolume(theMixer, i, pProperties->sound.mixerChannel[i].volume);
                mixerSetChannelPan(theMixer, i, pProperties->sound.mixerChannel[i].pan);
                mixerEnableChannel(theMixer, i, pProperties->sound.mixerChannel[i].enable);
            }
            mixerSetMasterVolume(theMixer, pProperties->sound.masterVolume);

            for (i = 0; i < MCT_NUMENTRIES; i++) {
                DestroyWindow(hwndVolTT[i]);
                DestroyWindow(hwndPanTT[i]);
            }
            DestroyWindow(hwndMasterTT[0]);
            DestroyWindow(hwndMasterTT[1]);

            return FALSE;

        default:
            if (wParam == IDC_MASTERL) {
                SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0));
                UpdateVolumeToolTip(GetDlgItem(hDlg, IDC_MASTERL), hwndMasterTT[0], &tiMaster[0]);
                updateAudio = TRUE;
            }
            if (wParam == IDC_MASTERR) {
                SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_GETPOS,   0, 0));
                UpdateVolumeToolTip(GetDlgItem(hDlg, IDC_MASTERR), hwndMasterTT[1], &tiMaster[1]);
                updateAudio = TRUE;
            }

            if (wParam >= IDC_VOLUME1 && wParam < IDC_VOLUME1 + MCT_NUMENTRIES) {  
                UpdateVolumeToolTip(GetDlgItem(hDlg, wParam), hwndVolTT[wParam - IDC_VOLUME1], &tiVol[wParam - IDC_VOLUME1]);
                updateAudio = TRUE;
            }

            if (wParam >= IDC_PAN1 && wParam < IDC_PAN1 + MCT_NUMENTRIES) {    
                UpdatePanToolTip(GetDlgItem(hDlg, wParam), hwndPanTT[wParam - IDC_PAN1], &tiPan[wParam - IDC_PAN1]);
                updateAudio = TRUE;
            }

            break;
        }

        break;
    }

    if (updateAudio) {
        for (i = 0; i < MCT_NUMENTRIES; i++) {
            mixerSetChannelVolume(theMixer, i, 100 - SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i), TBM_GETPOS,   0, 0));
            mixerSetChannelPan(theMixer, i, SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i), TBM_GETPOS,   0, 0));
            mixerEnableChannel(theMixer, i, SendMessage(GetDlgItem(hDlg, IDC_VOLENABLE1 + i), BM_GETCHECK, 0, 0) == BST_CHECKED);
        }
        mixerSetMasterVolume(theMixer, 100 - SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0));
    }

    return FALSE;
}

static BOOL CALLBACK joykeyDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static JoystickProperties* joyCtrl;
    static JoystickProperties tmpJoyCtrl;
    static int id = 0;

    switch (iMsg) {
    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDOK:
            KillTimer(hDlg, 0);
            *joyCtrl = tmpJoyCtrl;
            EndDialog(hDlg, TRUE);
            return TRUE;
        case IDCANCEL:
            KillTimer(hDlg, 0);
            EndDialog(hDlg, FALSE);
            return TRUE;
        case IDC_KEYUP:
            break;
        }
        switch (HIWORD(wParam)) {
        case EN_KILLFOCUS:
            id = 0;
            break;
        case EN_SETFOCUS:
            id = LOWORD(wParam);
            break;
        }
        break;
    case WM_TIMER:
        if (id != 0) {
            int i;
            for (i = 0; i < 256; i++) {
                if (virtualKeys[i][0]) {
                    SHORT state = GetAsyncKeyState(i);
                    if (state & 1) {
                        SendMessage(GetDlgItem(hDlg, id), WM_SETTEXT, 0, (LPARAM)virtualKeys[i]);
                        switch (id) {
                        case IDC_KEYUP:      tmpJoyCtrl.keyUp    = i; break;
                        case IDC_KEYDOWN:    tmpJoyCtrl.keyDown  = i; break;
                        case IDC_KEYLEFT:    tmpJoyCtrl.keyLeft  = i; break;
                        case IDC_KEYRIGHT:   tmpJoyCtrl.keyRight = i; break;
                        case IDC_KEYBUTTON1: tmpJoyCtrl.button1  = i; break;
                        case IDC_KEYBUTTON2: tmpJoyCtrl.button2  = i; break;
                        }
                    }
                }
            }
        }
        break;

    case WM_INITDIALOG:
        joyCtrl = (JoystickProperties*)lParam;

        SetWindowText(GetDlgItem(hDlg, IDOK), langDlgOK());
        SetWindowText(GetDlgItem(hDlg, IDCANCEL), langDlgCancel());

        SendMessage(GetDlgItem(hDlg, IDC_JOYUPTEXT), WM_SETTEXT, 0, (LPARAM)langDlgJoyUpText());
        SendMessage(GetDlgItem(hDlg, IDC_JOYDOWNTEXT), WM_SETTEXT, 0, (LPARAM)langDlgJoyDownText());
        SendMessage(GetDlgItem(hDlg, IDC_JOYLEFTTEXT), WM_SETTEXT, 0, (LPARAM)langDlgJoyLeftText());
        SendMessage(GetDlgItem(hDlg, IDC_JOYRIGHTTEXT), WM_SETTEXT, 0, (LPARAM)langDlgJoyRightText());
        SendMessage(GetDlgItem(hDlg, IDC_JOYBUTTON1TEXT), WM_SETTEXT, 0, (LPARAM)langDlgJoyButton1Text());
        SendMessage(GetDlgItem(hDlg, IDC_JOYBUTTON2TEXT), WM_SETTEXT, 0, (LPARAM)langDlgJoyButton2Text());
        SendMessage(GetDlgItem(hDlg, IDC_JOYGROUPBOX), WM_SETTEXT, 0, (LPARAM)langDlgJoyGB());

        SetWindowText(hDlg, joyCtrl->id == 1 ? langDlgJoyTitle1() : langDlgJoyTitle2());

        tmpJoyCtrl = *joyCtrl;
        SendMessage(GetDlgItem(hDlg, IDC_KEYUP),      WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyUp]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYDOWN),    WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyDown]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYLEFT),    WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyLeft]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYRIGHT),   WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyRight]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYBUTTON1), WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.button1]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYBUTTON2), WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.button2]);

        SetTimer(hDlg, 0, 1, NULL);
        return FALSE;
    }

    return FALSE;
}

static BOOL CALLBACK controlsDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static Properties* pProperties;

    switch (iMsg) {
    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDC_KEYSET1:
            DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_JOYKEYS), hDlg, joykeyDlgProc, (LPARAM)&pProperties->joy1);
            return TRUE;
        case IDC_KEYSET2:
            DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_JOYKEYS), hDlg, joykeyDlgProc, (LPARAM)&pProperties->joy2);
            return TRUE;
        }
        break;

    case WM_INITDIALOG:        
        _stprintf(pControlsJoy[0], "%s", langEnumControlsJoyNone());
        _stprintf(pControlsJoy[1], "%s", langEnumControlsJoyNumpad());
        _stprintf(pControlsJoy[2], "%s", langEnumControlsJoyKeysetA());
        _stprintf(pControlsJoy[3], "%s", langEnumControlsJoyKeysetB());
        _stprintf(pControlsJoy[4], "%s", langEnumControlsJoyPCjoy1());
        _stprintf(pControlsJoy[5], "%s", langEnumControlsJoyPCjoy2());
        _stprintf(pControlsJoy[6], "%s", langEnumControlsJoyMouse());

        _stprintf(pControlsAutofire[0], "%s", langEnumControlsAfOff());
        _stprintf(pControlsAutofire[1], "%s", langEnumControlsAfSlow());
        _stprintf(pControlsAutofire[2], "%s", langEnumControlsAfMedium());
        _stprintf(pControlsAutofire[3], "%s", langEnumControlsAfFast());

        SendMessage(GetDlgItem(hDlg, IDC_JOYPORT1GROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropJoyPort1GB());
        SendMessage(GetDlgItem(hDlg, IDC_JOYPORT2GROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropJoyPort2GB());
        SendMessage(GetDlgItem(hDlg, IDC_JOYAUTOFIRE1TEXT), WM_SETTEXT, 0, (LPARAM)langPropJoyAutofireText());
        SendMessage(GetDlgItem(hDlg, IDC_JOYAUTOFIRE2TEXT), WM_SETTEXT, 0, (LPARAM)langPropJoyAutofireText());
        SendMessage(GetDlgItem(hDlg, IDC_JOYKEYSETGROUPBOX), WM_SETTEXT, 0, (LPARAM)langPropJoyKeysetGB());
        
        SetWindowText(GetDlgItem(hDlg, IDC_KEYSET1), langPropJoyKeyest1());
        SetWindowText(GetDlgItem(hDlg, IDC_KEYSET2), langPropJoyKeyest2());

        SetWindowText(GetDlgItem(GetParent(hDlg), IDOK), langDlgOK());
        SetWindowText(GetDlgItem(GetParent(hDlg), IDCANCEL), langDlgCancel());

        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        initDropList(hDlg, IDC_JOY1, pControlsJoy, pProperties->joy1.type);
        initDropList(hDlg, IDC_AUTOFIRE1, pControlsAutofire, pProperties->joy1.autofire);
        initDropList(hDlg, IDC_JOY2, pControlsJoy, pProperties->joy2.type);
        initDropList(hDlg, IDC_AUTOFIRE2, pControlsAutofire, pProperties->joy2.autofire);

        return FALSE;

    case WM_NOTIFY:
        if ((((NMHDR FAR *)lParam)->code) != PSN_APPLY) {
            return FALSE;
        }

        pProperties->joy1.type       = getDropListIndex(hDlg, IDC_JOY1, pControlsJoy);
        pProperties->joy1.autofire   = getDropListIndex(hDlg, IDC_AUTOFIRE1, pControlsAutofire);
        pProperties->joy2.type       = getDropListIndex(hDlg, IDC_JOY2, pControlsJoy);
        pProperties->joy2.autofire   = getDropListIndex(hDlg, IDC_AUTOFIRE2, pControlsAutofire);

        propModified = 1;
        
        return TRUE;
    }

    return FALSE;
}

int showProperties(Properties* pProperties, HWND hwndOwner, PropPage startPage, Mixer* mixer) {
	HINSTANCE       hInst = (HINSTANCE)GetModuleHandle(NULL);
    PROPSHEETPAGE   psp[5];
    PROPSHEETHEADER psh;
    Properties oldProp = *pProperties;

    theMixer = mixer;

    psp[0].dwSize = sizeof(PROPSHEETPAGE);
    psp[0].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[0].hInstance = hInst;
    psp[0].pszTemplate = MAKEINTRESOURCE(IDD_EMULATION);
    psp[0].pszIcon = NULL;
    psp[0].pfnDlgProc = emulationDlgProc;
    psp[0].pszTitle = langPropEmulation();
    psp[0].lParam = (LPARAM)pProperties;
    psp[0].pfnCallback = NULL;

    psp[1].dwSize = sizeof(PROPSHEETPAGE);
    psp[1].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[1].hInstance = hInst;
    psp[1].pszTemplate = MAKEINTRESOURCE(IDD_VIDEO);
    psp[1].pszIcon = NULL;
    psp[1].pfnDlgProc = videoDlgProc;
    psp[1].pszTitle = langPropVideo();
    psp[1].lParam = (LPARAM)pProperties;
    psp[1].pfnCallback = NULL;

    psp[2].dwSize = sizeof(PROPSHEETPAGE);
    psp[2].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[2].hInstance = hInst;
    psp[2].pszTemplate = MAKEINTRESOURCE(IDD_SOUND);
    psp[2].pszIcon = NULL;
    psp[2].pfnDlgProc = soundDlgProc;
    psp[2].pszTitle = langPropSound();
    psp[2].lParam = (LPARAM)pProperties;
    psp[2].pfnCallback = NULL;

    psp[3].dwSize = sizeof(PROPSHEETPAGE);
    psp[3].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[3].hInstance = hInst;
    psp[3].pszTemplate = MAKEINTRESOURCE(IDD_CONTROLS);
    psp[3].pszIcon = NULL;
    psp[3].pfnDlgProc = controlsDlgProc;
    psp[3].pszTitle = langPropControls();
    psp[3].lParam = (LPARAM)pProperties;
    psp[3].pfnCallback = NULL;

    psp[4].dwSize = sizeof(PROPSHEETPAGE);
    psp[4].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[4].hInstance = hInst;
    psp[4].pszTemplate = MAKEINTRESOURCE(IDD_PERFORMANCE);
    psp[4].pszIcon = NULL;
    psp[4].pfnDlgProc = performanceDlgProc;
    psp[4].pszTitle = langPropPerformance();
    psp[4].lParam = (LPARAM)pProperties;
    psp[4].pfnCallback = NULL;

    psh.dwSize = sizeof(PROPSHEETHEADER);
    psh.dwFlags = PSH_USEICONID | PSH_PROPSHEETPAGE | PSH_NOAPPLYNOW;
    psh.hwndParent = hwndOwner;
    psh.hInstance = hInst;
    psh.pszIcon = NULL;
    psh.pszCaption = langPropTitle();
    psh.nPages = sizeof(psp) / sizeof(PROPSHEETPAGE);
    psh.nStartPage = startPage;
    psh.ppsp = (LPCPROPSHEETPAGE) &psp;
    psh.pfnCallback = NULL;

    propModified = 0;

    PropertySheet(&psh);

    if (propModified) {
        propModified = memcmp(&oldProp, pProperties, sizeof(Properties));
    }

    return propModified;
}

/* Alternative property default settings */
#ifdef PROPERTIES_DEFAULTS_ALT_1

void propInitDefaults(Properties* pProperties) 
{
    int i;
    pProperties->language                 = EMU_LANG_ENGLISH;
    
    pProperties->emulation.statsDefDir[0] = 0;
    pProperties->emulation.family         = P_EMU_MSX2;
    pProperties->emulation.speed          = 50;
    pProperties->emulation.RAMsize        = P_EMU_RAM128;
    pProperties->emulation.VRAMsize       = P_EMU_VRAM128;
    pProperties->emulation.syncMethod     = P_EMU_SYNC1MS;
    
    pProperties->video.monType            = P_VIDEO_COLOR;
    pProperties->video.palEmu             = P_VIDEO_PALNONE;
    pProperties->video.videoType          = P_VIDEO_NTSC;
    pProperties->video.size               = P_VIDEO_SIZEX2;
    pProperties->video.driver             = P_VIDEO_DRVDIRECTX;
    pProperties->video.frameSkip          = P_VIDEO_FSKIP0;
    pProperties->video.fullRes            = P_VIDEO_FRES640X480_32;

    pProperties->sound.driver             = P_SOUND_DRVDIRECTX;
    pProperties->sound.frequency          = P_SOUND_FREQ44;
    pProperties->sound.bufSize            = P_SOUND_BUF150;
    pProperties->sound.syncMethod         = P_SOUND_SYNCQADJUST;

    pProperties->sound.stereo = 1;
    pProperties->sound.masterVolume = 100;
    pProperties->sound.chip.msxmusic = 1;
    pProperties->sound.chip.msxaudio = 1;

    pProperties->sound.mixerChannel[MCT_PSG].enable = 1;
    pProperties->sound.mixerChannel[MCT_PSG].pan = 50;
    pProperties->sound.mixerChannel[MCT_PSG].volume = 100;

    pProperties->sound.mixerChannel[MCT_SCC].enable = 1;
    pProperties->sound.mixerChannel[MCT_SCC].pan = 50;
    pProperties->sound.mixerChannel[MCT_SCC].volume = 100;

    pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable = 1;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan = 50;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume = 100;

    pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable = 1;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan = 50;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume = 100;

    pProperties->sound.mixerChannel[MCT_KEYBOARD].enable = 1;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].pan = 50;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].volume = 100;

    pProperties->sound.mixerChannel[MCT_CASSETTE].enable = 1;
    pProperties->sound.mixerChannel[MCT_CASSETTE].pan = 50;
    pProperties->sound.mixerChannel[MCT_CASSETTE].volume = 100;
    
    pProperties->joy1.type              = P_JOY_NUMPAD;
    pProperties->joy1.autofire          = P_JOY_AFOFF;
    pProperties->joy1.keyUp             = 0xff;
    pProperties->joy1.keyDown           = 0xff;
    pProperties->joy1.keyLeft           = 0xff;
    pProperties->joy1.keyRight          = 0xff;
    pProperties->joy1.button1           = 0xff;
    pProperties->joy1.button2           = 0xff;
    
    pProperties->joy2.type              = P_JOY_NONE;
    pProperties->joy2.autofire          = P_JOY_AFOFF;
    pProperties->joy2.keyUp             = 0xff;
    pProperties->joy2.keyDown           = 0xff;
    pProperties->joy2.keyLeft           = 0xff;
    pProperties->joy2.keyRight          = 0xff;
    pProperties->joy2.button1           = 0xff;
    pProperties->joy2.button2           = 0xff;
    
    pProperties->keyboard.keySet        = P_CHAR_EUROPEAN;
    
    pProperties->cartridge.defDir[0]    = 0;
    pProperties->cartridge.slotA[0]     = 0;
    pProperties->cartridge.slotB[0]     = 0;
    pProperties->cartridge.slotAZip[0]  = 0;
    pProperties->cartridge.slotBZip[0]  = 0;
    pProperties->cartridge.autoReset    = 1;

    pProperties->diskdrive.defDir[0]    = 0;
    pProperties->diskdrive.slotA[0]     = 0;
    pProperties->diskdrive.slotB[0]     = 0;
    pProperties->diskdrive.slotAZip[0]  = 0;
    pProperties->diskdrive.slotBZip[0]  = 0;
    pProperties->diskdrive.autostartA   = 0;
    
    pProperties->cassette.defDir[0]     = 0;
    pProperties->cassette.tape[0]       = 0;
    pProperties->cassette.tapeZip[0]    = 0;

    for (i = 0; i < MAX_HISTORY; i++) {
        pProperties->filehistory.cartridgeA[i][0] = 0;
        pProperties->filehistory.cartridgeB[i][0] = 0;
        pProperties->filehistory.diskdriveA[i][0] = 0;
        pProperties->filehistory.diskdriveB[i][0] = 0;
        pProperties->filehistory.cassette[i][0] = 0;
    }

}

#else

/* Default property settings */
void propInitDefaults(Properties* pProperties) 
{
    int i;
    pProperties->language                 = EMU_LANG_ENGLISH;

    pProperties->emulation.statsDefDir[0] = 0;
    pProperties->emulation.family         = P_EMU_MSX2;
    pProperties->emulation.speed          = 50;
    pProperties->emulation.RAMsize        = P_EMU_RAM2048;
    pProperties->emulation.VRAMsize       = P_EMU_VRAM128;
    pProperties->emulation.syncMethod     = P_EMU_SYNCAUTO;
    
    pProperties->video.monType          = P_VIDEO_COLOR;
    pProperties->video.palEmu           = P_VIDEO_PALNYC;
    pProperties->video.videoType        = P_VIDEO_PAL;
    pProperties->video.size             = P_VIDEO_SIZEX2;
    pProperties->video.driver           = P_VIDEO_DRVDIRECTX;
    pProperties->video.frameSkip        = P_VIDEO_FSKIP0;
    pProperties->video.fullRes          = P_VIDEO_FRES640X480_32;

    pProperties->sound.driver           = P_SOUND_DRVDIRECTX;
    pProperties->sound.frequency        = P_SOUND_FREQ44;
    pProperties->sound.bufSize          = P_SOUND_BUF150;
    pProperties->sound.syncMethod       = P_SOUND_SYNCQADJUST;

    pProperties->sound.stereo = 1;
    pProperties->sound.masterVolume = 92;
    pProperties->sound.chip.msxmusic = 1;
    pProperties->sound.chip.msxaudio = 1;


    pProperties->sound.mixerChannel[MCT_PSG].enable = 1;
    pProperties->sound.mixerChannel[MCT_PSG].pan = 42;
    pProperties->sound.mixerChannel[MCT_PSG].volume = 100;

    pProperties->sound.mixerChannel[MCT_SCC].enable = 1;
    pProperties->sound.mixerChannel[MCT_SCC].pan = 58;
    pProperties->sound.mixerChannel[MCT_SCC].volume = 86;

    pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable = 1;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan = 58;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume = 100;

    pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable = 1;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan = 50;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume = 100;

    pProperties->sound.mixerChannel[MCT_KEYBOARD].enable = 1;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].pan = 54;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].volume = 55;

    pProperties->sound.mixerChannel[MCT_CASSETTE].enable = 0;
    pProperties->sound.mixerChannel[MCT_CASSETTE].pan = 50;
    pProperties->sound.mixerChannel[MCT_CASSETTE].volume = 0;
    
    pProperties->joy1.type              = P_JOY_NONE;
    pProperties->joy1.autofire          = P_JOY_AFOFF;
    pProperties->joy1.keyUp             = 0xff;
    pProperties->joy1.keyDown           = 0xff;
    pProperties->joy1.keyLeft           = 0xff;
    pProperties->joy1.keyRight          = 0xff;
    pProperties->joy1.button1           = 0xff;
    pProperties->joy1.button2           = 0xff;
    
    pProperties->joy2.type              = P_JOY_NONE;
    pProperties->joy2.autofire          = P_JOY_AFOFF;
    pProperties->joy2.keyUp             = 0xff;
    pProperties->joy2.keyDown           = 0xff;
    pProperties->joy2.keyLeft           = 0xff;
    pProperties->joy2.keyRight          = 0xff;
    pProperties->joy2.button1           = 0xff;
    pProperties->joy2.button2           = 0xff;
    
    pProperties->keyboard.keySet        = P_CHAR_EUROPEAN;
    
    pProperties->cartridge.defDir[0]    = 0;
    pProperties->cartridge.slotA[0]     = 0;
    pProperties->cartridge.slotB[0]     = 0;
    pProperties->cartridge.slotAZip[0]  = 0;
    pProperties->cartridge.slotBZip[0]  = 0;
    pProperties->cartridge.autoReset    = 1;
    
    pProperties->diskdrive.defDir[0]    = 0;
    pProperties->diskdrive.slotA[0]     = 0;
    pProperties->diskdrive.slotB[0]     = 0;
    pProperties->diskdrive.slotAZip[0]  = 0;
    pProperties->diskdrive.slotBZip[0]  = 0;
    pProperties->diskdrive.autostartA   = 0;

    pProperties->cassette.defDir[0]     = 0;
    pProperties->cassette.tape[0]       = 0;
    pProperties->cassette.tapeZip[0]    = 0;

    for (i = 0; i < MAX_HISTORY; i++) {
        pProperties->filehistory.cartridgeA[i][0] = 0;
        pProperties->filehistory.cartridgeB[i][0] = 0;
        pProperties->filehistory.diskdriveA[i][0] = 0;
        pProperties->filehistory.diskdriveB[i][0] = 0;
        pProperties->filehistory.cassette[i][0] = 0;
    }

}

#endif

void propLoad(Properties* pProperties) 
{
    int i;
    getIntValue(registryKey, "EmuLanguage", (DWORD*)&pProperties->language);

    getStrValue(registryKey, "StateDefDir", (char*)pProperties->emulation.statsDefDir);
    getIntValue(registryKey, "EmuFamily", (DWORD*)&pProperties->emulation.family);
    getIntValue(registryKey, "EmulationSpeed", (DWORD*)&pProperties->emulation.speed);
    getIntValue(registryKey, "RamSize", (DWORD*)&pProperties->emulation.RAMsize);
    getIntValue(registryKey, "VramSize", (DWORD*)&pProperties->emulation.VRAMsize);
    getIntValue(registryKey, "Sync", (DWORD*)&pProperties->emulation.syncMethod);
    
    getIntValue(registryKey, "Monitor", (DWORD*)&pProperties->video.monType);
    getIntValue(registryKey, "VideoEmu", (DWORD*)&pProperties->video.palEmu);
    getIntValue(registryKey, "VideoType", (DWORD*)&pProperties->video.videoType);
    getIntValue(registryKey, "VideoSize", (DWORD*)&pProperties->video.size);
    getIntValue(registryKey, "VideoDriver", (DWORD*)&pProperties->video.driver);
    getIntValue(registryKey, "FrameSkip", (DWORD*)&pProperties->video.frameSkip);
    getIntValue(registryKey, "FullRes", (DWORD*)&pProperties->video.fullRes);
    
    getIntValue(registryKey, "Sound-Out", (DWORD*)&pProperties->sound.driver);
    getIntValue(registryKey, "Frequency", (DWORD*)&pProperties->sound.frequency);
    getIntValue(registryKey, "BufferSize", (DWORD*)&pProperties->sound.bufSize);
    getIntValue(registryKey, "SyncMethod", (DWORD*)&pProperties->sound.syncMethod);

    getIntValue(registryKey, "StereoMono", (DWORD*)&pProperties->sound.stereo);
    getIntValue(registryKey, "MasterVol", (DWORD*)&pProperties->sound.masterVolume);
    getIntValue(registryKey, "EnableMSXMusic", (DWORD*)&pProperties->sound.chip.msxmusic);
    getIntValue(registryKey, "EnableMSXAudio", (DWORD*)&pProperties->sound.chip.msxaudio);

    getIntValue(registryKey, "PSGEnable", (DWORD*)&pProperties->sound.mixerChannel[MCT_PSG].enable);
    getIntValue(registryKey, "PSGPAN", (DWORD*)&pProperties->sound.mixerChannel[MCT_PSG].pan);
    getIntValue(registryKey, "PSGVOL", (DWORD*)&pProperties->sound.mixerChannel[MCT_PSG].volume);
    getIntValue(registryKey, "SCCEnable", (DWORD*)&pProperties->sound.mixerChannel[MCT_SCC].enable);
    getIntValue(registryKey, "SCCPAN", (DWORD*)&pProperties->sound.mixerChannel[MCT_SCC].pan);
    getIntValue(registryKey, "SCCVOL", (DWORD*)&pProperties->sound.mixerChannel[MCT_SCC].volume);
    getIntValue(registryKey, "MSXMUSICEnable", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable);
    getIntValue(registryKey, "MSXMUSICPAN", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan);
    getIntValue(registryKey, "MSXMUSICVOL", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume);
    getIntValue(registryKey, "MSXAUDIOEnable", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable);
    getIntValue(registryKey, "MSXAUDIOPAN", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan);
    getIntValue(registryKey, "MSXAUDIOVOL", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume);
    getIntValue(registryKey, "KEYBOARDEnable", (DWORD*)&pProperties->sound.mixerChannel[MCT_KEYBOARD].enable);
    getIntValue(registryKey, "KEYBOARDPAN", (DWORD*)&pProperties->sound.mixerChannel[MCT_KEYBOARD].pan);
    getIntValue(registryKey, "KEYBOARDVOL", (DWORD*)&pProperties->sound.mixerChannel[MCT_KEYBOARD].volume);
    getIntValue(registryKey, "CASSETTEEnable", (DWORD*)&pProperties->sound.mixerChannel[MCT_CASSETTE].enable);
    getIntValue(registryKey, "CASSETTEPAN", (DWORD*)&pProperties->sound.mixerChannel[MCT_CASSETTE].pan);
    getIntValue(registryKey, "CASSETTEVOL", (DWORD*)&pProperties->sound.mixerChannel[MCT_CASSETTE].volume);
    
    getIntValue(registryKey, "JoyType", (DWORD*)&pProperties->joy1.type);
    getIntValue(registryKey, "JoyAutoFire", (DWORD*)&pProperties->joy1.autofire);
    getIntValue(registryKey, "JoyKeyUp", (DWORD*)&pProperties->joy1.keyUp);
    getIntValue(registryKey, "JoyKeyDown", (DWORD*)&pProperties->joy1.keyDown);
    getIntValue(registryKey, "JoyKeyLeft", (DWORD*)&pProperties->joy1.keyLeft);
    getIntValue(registryKey, "JoyKeyRight", (DWORD*)&pProperties->joy1.keyRight);
    getIntValue(registryKey, "JoyButton1", (DWORD*)&pProperties->joy1.button1);
    getIntValue(registryKey, "JoyButton2", (DWORD*)&pProperties->joy1.button2);
    
    getIntValue(registryKey, "Joy2Type", (DWORD*)&pProperties->joy2.type);
    getIntValue(registryKey, "Joy2AutoFire", (DWORD*)&pProperties->joy2.autofire);
    getIntValue(registryKey, "Joy2KeyUp", (DWORD*)&pProperties->joy2.keyUp);
    getIntValue(registryKey, "Joy2KeyDown", (DWORD*)&pProperties->joy2.keyDown);
    getIntValue(registryKey, "Joy2KeyLeft", (DWORD*)&pProperties->joy2.keyLeft);
    getIntValue(registryKey, "Joy2KeyRight", (DWORD*)&pProperties->joy2.keyRight);
    getIntValue(registryKey, "Joy2Button1", (DWORD*)&pProperties->joy2.button1);
    getIntValue(registryKey, "Joy2Button2", (DWORD*)&pProperties->joy2.button2);
    
    getIntValue(registryKey, "KeyBoardKeySet", (DWORD*)&pProperties->keyboard.keySet);
    
    getStrValue(registryKey, "CartDefDir", (char*)pProperties->cartridge.defDir);
    getStrValue(registryKey, "CartSlotA", (char*)pProperties->cartridge.slotA);
    getStrValue(registryKey, "CartSlotB", (char*)pProperties->cartridge.slotB);
    getStrValue(registryKey, "CartSlotA-ZIP", (char*)pProperties->cartridge.slotAZip);
    getStrValue(registryKey, "CartSlotB-ZIP", (char*)pProperties->cartridge.slotBZip);
    getIntValue(registryKey, "CartAutoReset", (DWORD*)&pProperties->cartridge.autoReset);

    getStrValue(registryKey, "DriveDefDir", (char*)pProperties->diskdrive.defDir);
    getStrValue(registryKey, "DriveSlotA", (char*)pProperties->diskdrive.slotA);
    getStrValue(registryKey, "DriveSlotB", (char*)pProperties->diskdrive.slotB);
    getStrValue(registryKey, "DriveSlotA-ZIP", (char*)pProperties->diskdrive.slotAZip);
    getStrValue(registryKey, "DriveSlotB-ZIP", (char*)pProperties->diskdrive.slotBZip);
    getIntValue(registryKey, "DriveAutoStart", (DWORD*)&pProperties->diskdrive.autostartA);

    getStrValue(registryKey, "CasseteDefDir", (char*)pProperties->cassette.defDir);
    getStrValue(registryKey, "Cassete", (char*)pProperties->cassette.tape);
    getStrValue(registryKey, "Cassete-ZIP", (char*)pProperties->cassette.tapeZip);

    for (i = 0; i < MAX_HISTORY; i++) {
        char str[32];
        sprintf(str, "CartA-History%x",(0x00+i));
        getStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeA[i]);
        sprintf(str, "CartB-History%x",(0x00+i));
        getStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeB[i]);
        sprintf(str, "DiskDriveA-History%x",(0x00+i));
        getStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveA[i]);
        sprintf(str, "DiskDriveB-History%x",(0x00+i));
        getStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveB[i]);
        sprintf(str, "Cassete-History%x",(0x00+i));
        getStrValue(registryKey, str, (char*)pProperties->filehistory.cassette[i]);
    }
}

void propSave(Properties* pProperties) 
{
    int i;
    setIntValue(registryKey, "EmuLanguage", pProperties->language);

    setStrValue(registryKey, "StateDefDir", (char*)pProperties->emulation.statsDefDir);
    setIntValue(registryKey, "EmuFamily", pProperties->emulation.family);
    setIntValue(registryKey, "EmulationSpeed", pProperties->emulation.speed);
    setIntValue(registryKey, "RamSize", pProperties->emulation.RAMsize);
    setIntValue(registryKey, "VramSize", pProperties->emulation.VRAMsize);
    setIntValue(registryKey, "Sync", pProperties->emulation.syncMethod);
    
    setIntValue(registryKey, "Monitor", pProperties->video.monType);
    setIntValue(registryKey, "VideoEmu", pProperties->video.palEmu);
    setIntValue(registryKey, "VideoType", pProperties->video.videoType);
    setIntValue(registryKey, "VideoSize", pProperties->video.size);
    setIntValue(registryKey, "VideoDriver", pProperties->video.driver);
    setIntValue(registryKey, "FrameSkip", pProperties->video.frameSkip);
    setIntValue(registryKey, "FullRes", pProperties->video.fullRes);
    
    setIntValue(registryKey, "Sound-Out", pProperties->sound.driver);
    setIntValue(registryKey, "Frequency", pProperties->sound.frequency);
    setIntValue(registryKey, "BufferSize", pProperties->sound.bufSize);
    setIntValue(registryKey, "SyncMethod", pProperties->sound.syncMethod);

    setIntValue(registryKey, "StereoMono", pProperties->sound.stereo);
    setIntValue(registryKey, "MasterVol", pProperties->sound.masterVolume);
    setIntValue(registryKey, "EnableMSXAudio", pProperties->sound.chip.msxaudio);
    setIntValue(registryKey, "EnableMSXMusic", pProperties->sound.chip.msxmusic);
    
    setIntValue(registryKey, "PSGEnable", pProperties->sound.mixerChannel[MCT_PSG].enable);
    setIntValue(registryKey, "PSGPAN", pProperties->sound.mixerChannel[MCT_PSG].pan);
    setIntValue(registryKey, "PSGVOL", pProperties->sound.mixerChannel[MCT_PSG].volume);
    setIntValue(registryKey, "SCCEnable", pProperties->sound.mixerChannel[MCT_SCC].enable);
    setIntValue(registryKey, "SCCPAN", pProperties->sound.mixerChannel[MCT_SCC].pan);
    setIntValue(registryKey, "SCCVOL", pProperties->sound.mixerChannel[MCT_SCC].volume);
    setIntValue(registryKey, "MSXMUSICEnable", pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable);
    setIntValue(registryKey, "MSXMUSICPAN", pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan);
    setIntValue(registryKey, "MSXMUSICVOL", pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume);
    setIntValue(registryKey, "MSXAUDIOEnable", pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable);
    setIntValue(registryKey, "MSXAUDIOPAN", pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan);
    setIntValue(registryKey, "MSXAUDIOVOL", pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume);
    setIntValue(registryKey, "KEYBOARDEnable", pProperties->sound.mixerChannel[MCT_KEYBOARD].enable);
    setIntValue(registryKey, "KEYBOARDPAN", pProperties->sound.mixerChannel[MCT_KEYBOARD].pan);
    setIntValue(registryKey, "KEYBOARDVOL", pProperties->sound.mixerChannel[MCT_KEYBOARD].volume);
    setIntValue(registryKey, "CASSETTEEnable", pProperties->sound.mixerChannel[MCT_CASSETTE].enable);
    setIntValue(registryKey, "CASSETTEPAN", pProperties->sound.mixerChannel[MCT_CASSETTE].pan);
    setIntValue(registryKey, "CASSETTEVOL", pProperties->sound.mixerChannel[MCT_CASSETTE].volume);

    setIntValue(registryKey, "JoyType", pProperties->joy1.type);
    setIntValue(registryKey, "JoyAutoFire", pProperties->joy1.autofire);
    setIntValue(registryKey, "JoyKeyUp", pProperties->joy1.keyUp);
    setIntValue(registryKey, "JoyKeyDown", pProperties->joy1.keyDown);
    setIntValue(registryKey, "JoyKeyLeft", pProperties->joy1.keyLeft);
    setIntValue(registryKey, "JoyKeyRight", pProperties->joy1.keyRight);
    setIntValue(registryKey, "JoyButton1", pProperties->joy1.button1);
    setIntValue(registryKey, "JoyButton2", pProperties->joy1.button2);

    setIntValue(registryKey, "Joy2Type", pProperties->joy2.type);
    setIntValue(registryKey, "Joy2AutoFire", pProperties->joy2.autofire);
    setIntValue(registryKey, "Joy2KeyUp", pProperties->joy2.keyUp);
    setIntValue(registryKey, "Joy2KeyDown", pProperties->joy2.keyDown);
    setIntValue(registryKey, "Joy2KeyLeft", pProperties->joy2.keyLeft);
    setIntValue(registryKey, "Joy2KeyRight", pProperties->joy2.keyRight);
    setIntValue(registryKey, "Joy2Button1", pProperties->joy2.button1);
    setIntValue(registryKey, "Joy2Button2", pProperties->joy2.button2);
    
    setIntValue(registryKey, "KeyBoardKeySet", pProperties->keyboard.keySet);
    
    setStrValue(registryKey, "CartDefDir", (char*)pProperties->cartridge.defDir);
    setStrValue(registryKey, "CartSlotA", (char*)pProperties->cartridge.slotA);
    setStrValue(registryKey, "CartSlotB", (char*)pProperties->cartridge.slotB);
    setStrValue(registryKey, "CartSlotA-ZIP", (char*)pProperties->cartridge.slotAZip);
    setStrValue(registryKey, "CartSlotB-ZIP", (char*)pProperties->cartridge.slotBZip);
    setIntValue(registryKey, "CartAutoReset", pProperties->cartridge.autoReset);

    setStrValue(registryKey, "DriveDefDir", (char*)pProperties->diskdrive.defDir);
    setStrValue(registryKey, "DriveSlotA", (char*)pProperties->diskdrive.slotA);
    setStrValue(registryKey, "DriveSlotB", (char*)pProperties->diskdrive.slotB);
    setStrValue(registryKey, "DriveSlotA-ZIP", (char*)pProperties->diskdrive.slotAZip);
    setStrValue(registryKey, "DriveSlotB-ZIP", (char*)pProperties->diskdrive.slotBZip);
    setIntValue(registryKey, "DriveAutoStart", pProperties->diskdrive.autostartA);

    setStrValue(registryKey, "CasseteDefDir", (char*)pProperties->cassette.defDir);
    setStrValue(registryKey, "Cassete", (char*)pProperties->cassette.tape);
    setStrValue(registryKey, "Cassete-ZIP", (char*)pProperties->cassette.tapeZip);

    for (i = 0; i < MAX_HISTORY; i++) {
        char str[32];
        sprintf(str, "CartA-History%x",(0x00+i));
        setStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeA[i]);
        sprintf(str, "CartB-History%x",(0x00+i));
        setStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeB[i]);
        sprintf(str, "DiskDriveA-History%x",(0x00+i));
        setStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveA[i]);
        sprintf(str, "DiskDriveB-History%x",(0x00+i));
        setStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveB[i]);
        sprintf(str, "Cassete-History%x",(0x00+i));
        setStrValue(registryKey, str, (char*)pProperties->filehistory.cassette[i]);
    }
}

char* getLocalDirectory() {
    static char dir[1024];
    char* ptr;
    DWORD size;

    size = GetModuleFileName(NULL, dir, 1024);

    if (size == 0) {
        return NULL;
    }

    ptr = dir + size;

    while (ptr > dir && *ptr != '\\') {
        ptr--;
    }
    *ptr = 0;

    if (ptr == dir) {
        return NULL;
    }

    return dir;
}

Properties* propCreate(int useDefault, int useIniFile) 
{
    Properties* pProperties;

    canUseRegistry = 0;

#ifndef PROPERTIES_NO_REGISTRY
    if (!useIniFile) {
        DWORD test = 0;

        setRegIntValue(registryKey, "Test", 42);
        getRegIntValue(registryKey, "Test", &test);

        canUseRegistry = test == 42;
    }
#endif

#ifdef PROPERTIES_LOCAL_INI_FILE
    keyPath = getLocalDirectory();
#endif

    pProperties = malloc(sizeof(Properties));
    pProperties->joy1.id = 1;
    pProperties->joy2.id = 2;
    propInitDefaults(pProperties);
    if (!useDefault) {
        propLoad(pProperties);
    }
    return pProperties;
}


void propDestroy(Properties* pProperties) {
    propSave(pProperties);

    free(pProperties);
}

BOOL propSetVideoSize(Properties* pProperties, PropVideoSize size) {
    BOOL changed = pProperties->video.size != size;
    pProperties->video.size = size;
    return changed;
}
