/*****************************************************************************
** File:        ym2413.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the YM2413 sound chip. 
**              Wraps the c++ implementation taken from openMSX
**
** History:     1.0 - 11/5 2003 Initial version
**
******************************************************************************
*/
#include "YM2413.h"
#include "openMsxYM2413.h"


#define FREQUENCY        GLOBAL_FREQ //3570259 //3568884 //3579545
#define SAMPLERATE       44100
#define BUFFER_SIZE      1024
 
extern "C" Int16* ym2413Sync(void* ref, UInt32 count);

#define OFFSETOF(s, a) ((char*)&ym2413->ym2413.a - (char*)&ym2413->ym2413)
OpenMixer dummyMixer;

OpenMixer* OpenMixer::instance() 
{
    return &dummyMixer;
}

struct YM_2413 {
    YM_2413() : ym2413("ym2413", 100, 0), address(0) {
        memset(defaultBuffer, 0, sizeof(defaultBuffer));
    }

    Mixer* mixer;
    OpenYM2413 ym2413;
    UInt8  address;
    UInt8  registers[256];
    Int16  buffer[BUFFER_SIZE];
    Int16  defaultBuffer[BUFFER_SIZE];
};


extern "C" int ym2413LoadState(YM_2413* ym2413, FILE* file, UInt32 cpuClock) {
    int offset = OFFSETOF(OpenYM2413, firstMember);
    ym2413->ym2413.reset(cpuClock);

    if (fread((char*)ym2413->registers, 1, 256, file) != 256) {
        return 0;
    }

    if (fread(((char*)&ym2413->ym2413) + offset, 1, sizeof(ym2413->ym2413) - offset, file) != sizeof(ym2413->ym2413) - offset) {
        return 0;
    }

    return 1;
}

extern "C" int ym2413SaveState(YM_2413* ym2413, FILE* file, UInt32 cpuClock) {
    int offset = OFFSETOF(OpenYM2413, firstMember);
    if (fwrite((char*)ym2413->registers, 1, 256, file) != 256) {
        return 0;
    }
    
    if (fwrite(((char*)&ym2413->ym2413) + offset, 1, sizeof(ym2413->ym2413) - offset, file) != sizeof(ym2413->ym2413) - offset) {
        return 0;
    }
    return 1;
}


extern "C" YM_2413* ym2413Create(Mixer* mixer, UInt32 cpuClock)
{
    YM_2413* ym2413 = new YM_2413;

    ym2413->mixer = mixer;
    mixerRegisterChannel(mixer, MIXER_CHANNEL_MSX_MUSIC, ym2413Sync, ym2413);

    ym2413->ym2413.setSampleRate(SAMPLERATE);
	ym2413->ym2413.setVolume(32767 / 11);
    return ym2413;
}


extern "C" void ym2413Destroy(YM_2413* ym2413) {
    if (ym2413 != NULL) {
        mixerUnregisterChannel(ym2413->mixer, MIXER_CHANNEL_MSX_MUSIC);
        delete ym2413;
    }
}


extern "C" void ym2413WriteAddress(YM_2413* ym2413, UInt8 address)
{
    ym2413->address = address;
}

extern "C" void ym2413WriteData(YM_2413* ym2413, UInt8 data, UInt32 cpuClock)
{
    mixerSync(ym2413->mixer, cpuClock);
    ym2413->registers[ym2413->address & 0xff] = data;
    ym2413->ym2413.writeReg(ym2413->address, data, cpuClock);
}

extern "C" Int16* ym2413Sync(void* ref, UInt32 count) 
{
    YM_2413* ym2413 = (YM_2413*)ref;
    int* genBuf = ym2413->ym2413.updateBuffer(count);
    UInt32 i;

    if (genBuf == NULL) {
        return ym2413->defaultBuffer;
    }

    for (i = 0; i < count; i++) {
        ym2413->buffer[i] = (Int16)genBuf[i];
    }

    return ym2413->buffer;
}
