/*****************************************************************************
** File:
**      Cassette.c
** 
** Author:
**      Daniel Vik
**
** Description:
**      Bios patch for cassette support
**
******************************************************************************
*/

#include "Casette.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>
#include "ziphelper.h"


static char* toLower(const char* str) {
    static char tempname[256];
    char *ptr = tempname;

    while (*str) {
        *ptr++ = tolower(*str++);
    }

    return tempname;
}


static FILE* tapeFile = NULL;
static char* ramImageBuffer = NULL;
static int   ramImageSize = 0;
static int   ramImagePos = 0;
extern int   casBusy;
static int   tapeDelayEnable = 0;

static void TapeDelay(UInt32 bytes)
{
    if (!tapeDelayEnable) {
        return;
    }
}

UInt8 TapeRead(UInt8* value) 
{
    int data;

    if (ramImageBuffer != NULL) {
        if (ramImagePos < ramImageSize) {
            TapeDelay(1);
            *value = ramImageBuffer[ramImagePos++];
            casBusy = 1;
            return 1;
        }
        return 0;
    }

    if(tapeFile) {
        *value = data = fgetc(tapeFile);
        if (data >= 0) {
            TapeDelay(1);
        }
        return data >= 0;
    }
    return 0;
}

UInt8 TapeWrite(UInt8 value) 
{
    if (ramImageBuffer != NULL) {
        if (ramImagePos >= ramImageSize) {
            char* newBuf = realloc(ramImageBuffer, ramImageSize + 1024);
            if (newBuf) {
                ramImageSize += 1024;
                ramImageBuffer = newBuf;
            }
        }

        if (ramImagePos < ramImageSize) {
            TapeDelay(1);
            ramImageBuffer[ramImagePos++] = value;
            casBusy = 1;
            return 1;
        }
        return 0;
    }

    if (tapeFile) {
        TapeDelay(1);
        fputc(value, tapeFile);
        return 1;
    }
    return 0;
}

UInt8 TapeReadHeader() 
{
    static const UInt8 TapeHeader[8] = { 0x1F,0xA6,0xDE,0xBA,0xCC,0x13,0x7D,0x74 };
    
    UInt8 buf[8];

    if (ramImageBuffer != NULL) {
        while (ramImagePos & 7) {
            if (!TapeRead(buf)) {
                return 0;
            }
        }

        do {
            int i;
            for (i = 0; i < 8; i++) {
                if (!TapeRead(buf + i)) {
                    return 0;
                }
            }
            TapeDelay(8);
        } while (memcmp(buf, TapeHeader, 8));

        return 1;
    }

    if(tapeFile) {
        long pos = ftell(tapeFile);
        if(pos & 7) {
            if (fseek(tapeFile, 8 - (pos&7), SEEK_CUR)) {
                return 0;
            }
        }

        while (8 == fread(buf, 1, 8, tapeFile)) {
            TapeDelay(8);
            if (!memcmp(buf, TapeHeader, 8)) {
                return 1;
            }
        }
    }
    return 0;
}

UInt8 TapeWriteHeader() {
    static const UInt8 TapeHeader[8] = { 0x1F,0xA6,0xDE,0xBA,0xCC,0x13,0x7D,0x74 };

    if (ramImageBuffer != NULL) {
        int i;

        while (ramImagePos & 7) {
            if (!TapeWrite(0)) {
                return 0;
            }
        }

        for (i = 0; i < 8; i++) {
            if (!TapeWrite(TapeHeader[i])) {
                return 0;
            }
        }
        TapeDelay(8);
        return 1;
    }

    if(tapeFile) {
        long pos = ftell(tapeFile);
        if(pos & 7) {
            if (fseek(tapeFile, 8 - (pos&7), SEEK_CUR)) {
                return 0;
            }
        }

        TapeDelay(8);
        return 8 == fwrite(TapeHeader,1,8,tapeFile);
    }
    return 0;
}

int tapeInsert(char *name, const char *fileInZipFile) 
{
    if(tapeFile) {
        fclose(tapeFile);
        tapeFile = NULL;
    }

    if (ramImageBuffer != NULL) {
        // Flush to file??
        free(ramImageBuffer);
        ramImageBuffer = NULL;
    }

    if(!name) {
        return 1;
    }

    if (fileInZipFile != NULL) {
        ramImageBuffer = zipLoadFile(name, fileInZipFile, &ramImageSize);
        ramImagePos = 0;
        return ramImageBuffer != NULL;
    }

    tapeFile = fopen(name,"a+b");
    if(tapeFile == NULL) {
        FILE* file = fopen(name,"rb");
        if (file != NULL) {
            // File is read only, load it into ram
            fseek(file, 0, SEEK_END);
            ramImageSize = ftell(file);
            fseek(file, 0, SEEK_SET);
            if (ramImageSize > 0) {
                ramImagePos = 0;
                ramImageBuffer = malloc(ramImageSize);
                if (ramImageBuffer != NULL) {
                    if (ramImageSize != fread(ramImageBuffer, 1, ramImageSize, file)) {
                        free(ramImageBuffer);
                        ramImageBuffer = NULL;
                    }
                }
            }
            fclose(file);
        }
    }
    if (tapeFile) {
        fseek(tapeFile, 0, SEEK_SET);
    }

    return tapeFile != NULL;
}

UInt32 tapeGetLength()
{
    UInt32 length = 0;

    if(tapeFile) {
        int curPos = ftell(tapeFile);
        fseek(tapeFile, 0, SEEK_END);
        length = ftell(tapeFile);
        fseek(tapeFile, curPos, SEEK_SET);
    }
    else {
        length = ramImageSize;
    }

    return length;
}

UInt32 tapeGetCurrentPos()
{
    UInt32 pos = 0;

    if(tapeFile) {
        pos = ftell(tapeFile);
    }
    else {
        pos = ramImagePos;
    }

    return pos;
}

void tapeSetCurrentPos(UInt32 pos)
{
    UInt32 length = tapeGetLength();

    if (pos > length) {
        pos = length;
    }

    if(tapeFile) {
        fseek(tapeFile, pos, SEEK_SET);
    }
    else {
        ramImagePos = pos;
    }
}

void tapeSetRealTiming(int enable) {
    tapeDelayEnable = enable;
}