/*
Copyright (C) 2003 Parallel Realities

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "init.h"


#include <xtl.h>
#include "iosupport.h"
CIoSupport m_io ;

void checkForLicense()
{
	if (!engine.loadData("data/LICENSE"))
		graphics.showLicenseErrorAndExit();
}

/*
Show the GNU Public License the first time the game is played. Waits 4 seconds
and then proceeds. THIS MUST NOT BE REMOVED!!!!!
*/
void showLicense()
{
	SDL_FillRect(graphics.screen, NULL, graphics.black);
	graphics.delay(1000);

	SDL_FillRect(graphics.screen, NULL, graphics.black);
	SDL_Surface *pic = graphics.loadImage("gfx/main/licensePic.png");
	graphics.blit(pic, 0, 0, graphics.screen, false);
	SDL_FreeSurface(pic);

	checkForLicense();

	char line[255];
	int y = 0;

	char *token = strtok((char*)engine.dataBuffer, "\n");

	while (true)
	{
		sscanf(token, "%d %[^\n]", &y, line);

		graphics.drawString(line, 320, y, true, graphics.screen);

		token = strtok(NULL, "\n");

		if (token == NULL)
			break;
	}

	//graphics.delay(4000);

	graphics.drawString("Press Space to Continue...", 320, 440, true, graphics.screen);

	engine.flushInput();
	engine.clearInput();

	while (true)
	{
		graphics.updateScreen();
		engine.getInput();
		if (engine.keyState[SDLK_SPACE])
			break;
	}
	
	SDL_FillRect(graphics.screen, NULL, graphics.black);
	graphics.delay(1000);
}

/*
This bit is just for Linux and Unix users. It attempts to get the user's
home directory, then creates the .parallelrealities and .parallelrealities/blobwars
directories so that saves and temporary data files can be written there. Good, eh? :)
*/
#if UNIX
void setupUserHomeDirectory()
{
	char *userHome;

	char *name = getlogin();

	passwd *pass;

	if (name != NULL)
		pass = getpwnam(name);
	else
		pass = getpwuid(geteuid());

	if (pass == NULL)
	{
		printf("Couldn't determine the user home directory. Exitting.\n");
		exit(1);
	}

	userHome = pass->pw_dir;

	char dir[PATH_MAX];
	strcpy(dir, "");

	sprintf(dir, "%s/.parallelrealities", userHome);
	if ((mkdir(dir, S_IRWXU|S_IRWXG|S_IROTH|S_IXOTH) != 0) && (errno != EEXIST))
	{
		printf("Couldn't create required directory '%s'", dir);
		exit(1);
	}

	sprintf(dir, "%s/.parallelrealities/blobwars", userHome);
	if ((mkdir(dir, S_IRWXU|S_IRWXG|S_IROTH|S_IXOTH) != 0) && (errno != EEXIST))
	{
		printf("Couldn't create required directory '%s'", dir);
		exit(1);
	}

	char gameSavePath[PATH_MAX];
	sprintf(gameSavePath, "%s/.parallelrealities/blobwars/", userHome);
	engine.setUserHome(gameSavePath);
}

#endif

bool loadConfig()
{
	float version = 0;
	int release = 0;
	bool rtn = false;

	char configPath[PATH_MAX];

	sprintf(configPath, "%sconfig", engine.userHomeDirectory);

	debug(("Loading Config from %s\n", configPath));

	FILE *fp = fopen(configPath, "rb");

	if (!fp)
		return true;

	fscanf(fp, "%f %d", &version, &release);

	debug(("Version = %.1f - Expected %.1f\n", version, VERSION));
	debug(("Release = %d - Expected %d\n", release, RELEASE));

	if ((version != VERSION) && (release != RELEASE))
		rtn = true;

	fscanf(fp, "%d %d %d %d %d", &engine.fullScreen, &game.musicVol, &game.soundVol, &game.output, &game.brightness);

	fclose(fp);

	debug(("Output Type = %d\n", game.output));

	// Override audio if there is no sound available
	if (engine.useAudio)
		engine.useAudio = game.output;
		
	engine.loadJoystickConfig();

	return rtn;
}

void saveConfig()
{
	char configPath[PATH_MAX];

	sprintf(configPath, "%sconfig", engine.userHomeDirectory);

	FILE *fp = fopen(configPath, "wb");

	if (!fp)
	{
		printf("Error Saving Config to %s\n", configPath);
		return;
	}

	fprintf(fp, "%f %d\n", VERSION, RELEASE);
	fprintf(fp, "%d %d %d %d %d\n", engine.fullScreen, game.musicVol, game.soundVol, game.output, game.brightness);

	fclose(fp);
	
	debug(("Output Type = %d\n", game.output));
}

/*
Chugg chugg chugg.... brrr... chugg chugg chugg...brrrrrr... chugg ch..
BRRRRRRRRRRRRRRRRRMMMMMMMMMMMMMMMMMMM!! Well, hopefully anyway! ;)
*/
void initSystem()
{
	m_io.Unmount("C:") ;
	m_io.Unmount("E:") ;
	m_io.Unmount("F:") ;
	m_io.Unmount("X:") ;
	m_io.Unmount("Y:") ;
	m_io.Unmount("Z:") ;
	m_io.Unmount("R:") ;
	m_io.Mount("C:", "Harddisk0\\Partition2");
	m_io.Mount("E:", "Harddisk0\\Partition1");
	m_io.Mount("F:", "Harddisk0\\Partition6");
	m_io.Mount("X:", "Harddisk0\\Partition3");
	m_io.Mount("Y:", "Harddisk0\\Partition4");
	m_io.Mount("Z:", "Harddisk0\\Partition5");
	m_io.Mount("R:","Cdrom0");

	CreateDirectory( "E:\\SAVES", NULL ) ;
	CreateDirectory( "E:\\SAVES\\BLOBWARSX", NULL ) ;

	strcpy(engine.userHomeDirectory, "E:\\SAVES\\BLOBWARSX\\");

	#if UNIX
		setupUserHomeDirectory();
	#endif

	bool displayLicense = loadConfig();

	/* Initialize the SDL library */
	if (SDL_Init(SDL_INIT_VIDEO|SDL_INIT_AUDIO|SDL_INIT_JOYSTICK) < 0)
	{
		printf("Couldn't initialize SDL: %s\n", SDL_GetError());
		exit(1);
	}

	if (!engine.fullScreen)
		graphics.screen = SDL_SetVideoMode(640, 480, 0, SDL_HWPALETTE);
	else
		graphics.screen = SDL_SetVideoMode(640, 480, 16, SDL_HWPALETTE | SDL_FULLSCREEN);

	if (graphics.screen == NULL)
	{
		printf("Couldn't set 640x480x16 video mode: %s\n", SDL_GetError());
		exit(1);
	}

	// This (attempts to) set the gamma correction. We attempt to catch an error here
	// in case someone has done something really stupid in the config file(!!)
	Math::limitInt(&game.brightness, 1, 20);
	float brightness = game.brightness;
	brightness /= 10;
	SDL_SetGamma(brightness, brightness, brightness);

	if (TTF_Init() < 0)
	{
		printf("Couldn't initialize SDL TTF: %s\n", SDL_GetError());
		exit(1);
	}

	if (engine.useAudio)
	{
		if (Mix_OpenAudio(22050, AUDIO_S16, game.output, 1024) < 0)
		{
			printf("Warning: Couldn't set 22050 Hz 16-bit audio - Reason: %s\n", Mix_GetError());
			printf("Sound and Music will be disabled\n");
			engine.useAudio = 0;
		}
	}

	debug(("Found %d Joysticks...\n", SDL_NumJoysticks()));

	if (SDL_NumJoysticks() > 0)
	{
		SDL_JoystickEventState(SDL_ENABLE);
		SDL_EventState(SDL_JOYHATMOTION, SDL_IGNORE);

		for (int i = 0 ; i < SDL_NumJoysticks() ; i++)
		{
			debug(("Opening Joystick #%d - %s...\n", i, SDL_JoystickName(i)));
			engine.joystick[i] = SDL_JoystickOpen(i);
			if (i == 1)
				break;
		}
	}

	SDL_ShowCursor(SDL_DISABLE);
	SDL_EventState(SDL_MOUSEMOTION, SDL_DISABLE);

	graphics.registerEngine(&engine);
	graphics.mapColors();

 	audio.registerEngine(&engine);
	audio.setSoundVolume(game.soundVol);
	audio.setMusicVolume(game.musicVol);

	debug(("Sound Volume = %d\n", game.soundVol));
	debug(("Music Volume = %d\n", game.musicVol));
	debug(("Output Type = %d\n", game.output));
	debug(("Brightness = %d\n", game.brightness));
	debug(("tmp dir = %s\n", TEMPDIR));
	debug(("Pack Dir = %s\n", PAKLOCATION));
	debug(("Loading Fonts...\n"));

	#if USEPAK
		remove(TEMPDIR"font.ttf");
		SDL_Delay(1000); // wait one second, just to be sure!
		if (!engine.unpack("data/vera.ttf", PAK_FONT))
			engine.reportFontFailure();
	#endif

		graphics.loadFont(0, "D:\\data\\vera.ttf", 12);
	graphics.loadFont(1, "D:\\data\\vera.ttf", 14);
	graphics.loadFont(2, "D:\\data\\vera.ttf", 22);
	graphics.loadFont(3, "D:\\data\\vera.ttf", 26);
	graphics.loadFont(4, "D:\\data\\vera.ttf", 32);

	audio.loadSound(SND_CHEAT, "d:\\sound\\Lock And Load!!!");
	audio.loadSound(SND_HIGHLIGHT, "d:\\sound\\menu.wav");
	audio.loadSound(SND_SELECT, "d:\\sound\\select.wav");
	
	//graphics.infoBar = graphics.createSurface(640, 25);
	//graphics.messageBar = graphics.createSurface(640, 25);

	SDL_Surface *device = graphics.loadImage("gfx/main/alienDevice.png");

	SDL_WM_SetIcon(device, NULL);
	SDL_WM_SetCaption("Blob Wars : Metal Blob Solid", "Blob Wars");

	SDL_FreeSurface(device);

	if (displayLicense)
		showLicense();
	else
		checkForLicense();
		
	srand(time(NULL));
	
	engine.saveConfig = true;

	debug(("Init Complete...\n"));
}

/*
Removes [hopefully] all the resources that has been
loaded and created during the game. This is called by
atexit();
*/
void cleanup()
{
	try
	{
		debug(("Cleaning Up...\n"));

		debug(("Freeing Audio...\n"));
		audio.destroy();

		debug(("Removing Music...\n"));
		remove(TEMPDIR"music.mod");

		debug(("Freeing Game Info...\n"));
		game.destroy();

		debug(("Freeing Map Data...\n"));
		map.destroy();

		debug(("Freeing Engine Data...\n"));
		engine.destroy();

		debug(("Freeing Graphics...\n"));
		graphics.destroy();

		debug(("Saving Config...\n"));
		if (engine.saveConfig)
			saveConfig();

		debug(("Removing Font File...\n"));
		remove(TEMPDIR"font.ttf");
		
		if (SDL_NumJoysticks() > 0)
		{
			SDL_JoystickEventState(SDL_DISABLE);
			for (int i = 0 ; i < SDL_NumJoysticks() ; i++)
			{
				debug(("Closing Joystick #%d - %s...\n", i, SDL_JoystickName(i)));
				SDL_JoystickClose(engine.joystick[i]);
				if (i == 1)
					break;
			}
		}

		debug(("Closing Audio...\n"));
		if (engine.useAudio)
			Mix_CloseAudio();

		debug(("Closing TTF...\n"));
		TTF_Quit();

		debug(("Closing SDL Sub System...\n"));
		SDL_Quit();

		debug(("All Done.\n"));
	}
	catch (...)
	{
		printf("WARNING: Something went wrong whilst trying to clean up...\n");
		printf("trying to quit SDL anyway...\n");
		SDL_Quit();
	}
}

