/*
Copyright (C) 2003 Parallel Realities

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "map.h"

void drawMap()
{
	SDL_Rect r;

	int offsetX = (int)(engine.playerPosX);
	int offsetY = (int)(engine.playerPosY);

	map.offsetX = offsetX;
	map.offsetY = offsetY;

	Math::limitInt(&offsetX, 0, ((MAPWIDTH - 40) * BRICKSIZE));
	Math::limitInt(&offsetY, 0, ((MAPHEIGHT - 30) * BRICKSIZE));

	int mapx = offsetX >> BRICKSHIFT;
	int mapy = offsetY >> BRICKSHIFT;
	int brick = 0;

	for (int x = 0 ; x < 21 ; x++)
	{
		for (int y = 0 ; y < 16 ; y++)
		{
			brick = map.data[mapx + x][mapy + y];

			r.x = ((x * BRICKSIZE) - (offsetX & BRICKSIZE - 1));
			r.y = ((y * BRICKSIZE) - (offsetY & BRICKSIZE - 1));
			r.w = r.h = BRICKSIZE;

   		if ((brick >= MAP_BREAKABLE) && (brick < MAP_WATERANIM))
				graphics.blit(graphics.tile[brick], r.x, r.y, graphics.screen, false);
		}
	}
}

void drawMapTopLayer()
{
	SDL_Rect r;

	int offsetX = map.offsetX;
	int offsetY = map.offsetY;

	Math::limitInt(&offsetX, 0, ((MAPWIDTH - 40) * BRICKSIZE));
	Math::limitInt(&offsetY, 0, ((MAPHEIGHT - 30) * BRICKSIZE));

	int mapx = offsetX >> BRICKSHIFT;
	int mapy = offsetY >> BRICKSHIFT;

	int brick;

	for (int x = 0 ; x < 21 ; x++)
	{
		for (int y = 0 ; y < 16 ; y++)
		{
			r.x = ((x * BRICKSIZE) - (offsetX & BRICKSIZE - 1));
			r.y = ((y * BRICKSIZE) - (offsetY & BRICKSIZE - 1));
			r.w = r.h = BRICKSIZE;

			brick = map.data[mapx + x][mapy + y];

			if (brick == 0)
				continue;

			if ((brick < MAP_BREAKABLE) || (brick >= MAP_WATERANIM))
			{
				if (brick == MAP_WATER)
				{
					if (map.data[mapx + x][mapy + y + 1] >= MAP_BREAKABLE)
						addBubble((mapx + x) * BRICKSIZE, (mapy + y) * BRICKSIZE);
				}

				if (brick == MAP_WATERANIM)
				{
					brick = graphics.getWaterAnim();
				}
				else if (brick == MAP_SLIME)
				{
					brick = graphics.getSlimeAnim();
				}
				else if ((brick >= MAP_LAVAANIM) && (brick < MAP_TOPLAYER))
				{
					map.data[mapx + x][mapy + y] = graphics.getLavaAnim(brick);
					brick = map.data[mapx + x][mapy + y];
				}

				graphics.blit(graphics.tile[brick], r.x, r.y, graphics.screen, false);
			}
		}
	}
}

RadarBlip *addBlips(RadarBlip *blipTail, int mapX, int mapY, int type)
{
	Sprite *blipType;
	Entity *ent;

	switch (type)
	{
		case 1:
			blipType = graphics.getSprite("MIAArrow", true);
			ent = (Entity*)map.miaList.getHead();
			break;
		case 2:
			blipType = graphics.getSprite("ItemArrow", true);
			ent = (Entity*)map.itemList.getHead();
			break;
		case 3:
			blipType = graphics.getSprite("EnemyArrow", true);
			ent = (Entity*)map.enemyList.getHead();
			break;
	}

	RadarBlip *blip;
	int x, y;

	while (ent->next != NULL)
	{
		ent = (Entity*)ent->next;

		if (ent->health <= 0)
			continue;

		if (type == 3)
			if (!requiredEnemy(ent->name))
				continue;

		// Items
		if ((type == 2) && (ent->id < ITEM_MISC))
			continue;

		x = (int)(ent->x + ent->width) >> BRICKSHIFT;
		y = (int)(ent->y + ent->height) >> BRICKSHIFT;

		x -= mapX;
		y -= mapY;

		x = (160) + (x * 5);
		y = (120) + (y * 5);

		if ((x >= 165) && (y >= 125) && (x <= 475) && (y <= 355))
		{
			blip = new RadarBlip();
			blip->set(x, y, type);
			blipTail->next = blip;
			blipTail = blip;
		}
		else
		{
			if (y < 125)
				graphics.blit(blipType->image[0], 220 + (type * 50), 100, graphics.screen, true);
			if (x > 475)
				graphics.blit(blipType->image[1], 510, 140 + (type * 50), graphics.screen, true);
			if (y > 355)
				graphics.blit(blipType->image[2], 220 + (type * 50), 380, graphics.screen, true);
			if (x < 165)
				graphics.blit(blipType->image[3], 125, 140 + (type * 50), graphics.screen, true);
		}
	}
	
	return blipTail;
}

void showMap(int centerX, int centerY)
{
	char string[100];
	int x1, y1, x2, y2;

	x1 = centerX - 32;
	x2 = centerX + 32;
	y1 = centerY - 24;
	y2 = centerY + 24;

	Math::limitInt(&x1, 0, MAPWIDTH);
	Math::limitInt(&x2, 0, MAPWIDTH);
	Math::limitInt(&y1, 0, MAPHEIGHT);
	Math::limitInt(&y2, 0, MAPHEIGHT);

	SDL_FillRect(graphics.screen, NULL, graphics.black);
	graphics.updateScreen();
	SDL_Delay(500);
	
	SDL_Surface *panel = graphics.createSurface(320, 240);

	SDL_Surface *background = graphics.loadImage("gfx/main/mapBackground.png");
	SDL_SetAlpha(background, SDL_SRCALPHA|SDL_RLEACCEL, 128);
	
	int color = graphics.black;

	for (int y = 0 ; y < 48 ; y++)
	{
		for (int x = 0 ; x < 64 ; x++)
		{
			color = -1;

			if (map.data[x1 + x][y1 + y] == MAP_AIR)
				color = -1;
			else if (map.data[x1 + x][y1 + y] == MAP_WATER)
				color = graphics.blue;
			else if (map.data[x1 + x][y1 + y] == MAP_SLIME)
				color = graphics.green;
			else if (map.data[x1 + x][y1 + y] == MAP_LAVA)
				color = graphics.red;
			else if (map.data[x1 + x][y1 + y] < MAP_WATERANIM)
				color = graphics.white;

			if (color > -1)
				graphics.drawRect(x * 5, y * 5, 5, 5, color, panel);
		}
	}

	graphics.blit(background, 0, 0, panel, false);
	graphics.blit(background, 0, 0, graphics.screen, false);
	SDL_FreeSurface(background);

	List blipList;

	RadarBlip *blip = new RadarBlip();
	blip->set(160 + ((centerX - x1) * 5), 120 + ((centerY - y1) * 5), 0);
	blipList.add(blip);

	blipList.setTail(addBlips((RadarBlip*)blipList.getTail(), x1, y1, 1));
	blipList.setTail(addBlips((RadarBlip*)blipList.getTail(), x1, y1, 2));
	blipList.setTail(addBlips((RadarBlip*)blipList.getTail(), x1, y1, 3));
	Sprite *enemySignal = graphics.getSprite("EnemySignal", true);
	Sprite *miaSignal = graphics.getSprite("MIASignal", true);
	Sprite *bobSignal = graphics.getSprite("BobSignal", true);
	Sprite *itemSignal = graphics.getSprite("ItemSignal", true);

	graphics.setFontColor(0xff, 0xff, 0xff, 0x00, 0x00, 0x00);
	graphics.setFontSize(3);
	graphics.drawString(game.stageName, 320, 30, TXT_CENTERED, graphics.screen);

	graphics.setFontSize(0);
	
	graphics.drawRect(160, 414, 7, 7, graphics.yellow, graphics.white, graphics.screen);
	graphics.drawString("MIAs", 175, 410, TXT_LEFT, graphics.screen);

	graphics.drawRect(290, 414, 7, 7, graphics.blue, graphics.white, graphics.screen);
	graphics.drawString("Items", 320, 417, TXT_CENTERED, graphics.screen);

	graphics.drawRect(415, 414, 7, 7, graphics.red, graphics.white, graphics.screen);
	graphics.drawString("Enemies", 480, 410, TXT_RIGHT, graphics.screen);

	graphics.setFontSize(1);
	sprintf(string, "%s - %.2d:%.2d:%.2d", engine.translate("Mission Time"), game.currentMissionHours, game.currentMissionMinutes, game.currentMissionSeconds);
	graphics.drawString(string, 320, 60, TXT_CENTERED, graphics.screen);
	graphics.drawString("Press Tab to Continue...", 320, 450, TXT_CENTERED, graphics.screen);

	engine.flushInput();
	engine.clearInput();

	while (true)
	{
		engine.getInput();
		graphics.updateScreen();
		graphics.animateSprites();

		graphics.drawRect(160, 120, 320, 240, graphics.black, graphics.white, graphics.screen);
		graphics.blit(panel, 160, 120, graphics.screen, false);

		if (engine.keyState[SDLK_TAB])
			break;

		blip = (RadarBlip*)blipList.getHead();

		while (blip->next != NULL)
		{
			blip = (RadarBlip*)blip->next;

			switch (blip->type)
			{
				case 0:
					graphics.blit(bobSignal->getCurrentFrame(), blip->x, blip->y, graphics.screen, true);
					break;
				case 1:
					graphics.blit(miaSignal->getCurrentFrame(), blip->x, blip->y, graphics.screen, true);
					break;
				case 2:
					graphics.blit(itemSignal->getCurrentFrame(), blip->x, blip->y, graphics.screen, true);
					break;
				case 3:
					graphics.blit(enemySignal->getCurrentFrame(), blip->x, blip->y, graphics.screen, true);
					break;
			}
		}
	}

	blipList.clear();

	SDL_FillRect(graphics.screen, NULL, graphics.black);
	graphics.updateScreen();
	SDL_Delay(500);

	engine.flushInput();
	engine.clearInput();
}

void evaluateMapAttribute(Entity *ent, int mapAttribute)
{
	switch (mapAttribute)
	{
		case MAP_AIR:
			if ((ent->environment != ENV_AIR) && (!(ent->flags & ENT_INANIMATE)))
			{
				if (!(ent->flags & ENT_SWIMS))
				{
					if (ent->dy < 0)
						ent->dy = PLAYER_JUMP_SPEED;

					if (ent == &player)
					{
						if ((ent == &player) && ((game.hasAquaLung) || (engine.cheatExtras)))
						{
							player.setSprites(graphics.getSprite("BobRight", true), graphics.getSprite("BobLeft", true), graphics.getSprite("BobSpin", true));
						}

						if (engine.cheatJump)
							ent->dy *= 3;
					}

					ent->environment = ENV_AIR;
					
					if (ent->dy < 0)
						audio.playSound(SND_WATEROUT, CH_TOUCH);

					ent->checkEnvironment();
				}
			}
			ent->falling = true;
		case MAP_SOLID:
			break;
		case MAP_WATER:
		case MAP_SLIME:
		case MAP_LAVA:

			ent->falling = false;

			if (ent->environment == ENV_AIR)
			{
				audio.playSound(SND_WATERIN, CH_TOUCH);
				if ((mapAttribute == MAP_SLIME) || (mapAttribute == MAP_LAVA))
					ent->thinktime = 1;
			}

			if (mapAttribute == MAP_WATER)
			{
				ent->environment = ENV_WATER;
				if ((ent == &player) && ((game.hasAquaLung) || (engine.cheatExtras)))
				{
					player.setSprites(graphics.getSprite("AquaBobRight", true), graphics.getSprite("AquaBobLeft", true), graphics.getSprite("AquaBobSpin", true));
				}
			}
			else if (mapAttribute == MAP_SLIME)
			{
				ent->environment = ENV_SLIME;
			}
			else if (mapAttribute == MAP_LAVA)
			{
				ent->environment = ENV_LAVA;
			}
			break;
	}
}

void raiseWaterLevel()
{
	if (map.waterLevel == map.requiredWaterLevel)
		return;

	int y = (int)map.waterLevel;

	if ((int)map.waterLevel != map.requiredWaterLevel)
	{
		for (int x = 0 ; x < MAPWIDTH ; x++)
		{
			if ((map.data[x][y] == MAP_AIR) || (map.isLiquid(x, y)) || (map.data[x][y] == 247))
				map.data[x][y] = MAP_WATER;


			if ((map.data[x][y - 1] == MAP_AIR) || (map.isLiquid(x, y - 1)) || (map.data[x][y] == 247))
				map.data[x][y - 1] = MAP_WATERANIM;
		}

		map.waterLevel -= 0.1;

		int x = (int)(player.x + player.dx) >> BRICKSHIFT;
		int y = (int)(player.y + player.height - 1) >> BRICKSHIFT;

		int mapAttribute = map.data[x][y];

		if ((mapAttribute == MAP_WATER) && (player.environment == MAP_AIR))
			evaluateMapAttribute(&player, mapAttribute);
	}
}

void parseMapDataLine(char *line, int y)
{
	int tileIndex = 0;
	int x = 0;

	while (true)
	{
		sscanf(line, "%d", &tileIndex);

		map.data[x][y] = tileIndex;

		while (true)
		{
			*line++;

			if (*line == ' ')
				break;
		}

		x++;

		if (x == MAPWIDTH)
			break;
	}
}

bool loadMapData(char *filename)
{
	map.clear();

	char skillLevel[10];

	char mapEntity[255];

	char string[5][1024];
	int param[10];

	if (!engine.loadData(filename))
		graphics.showErrorAndExit("The requested map '%s' was not found.", filename);

	char *token = strtok((char*)engine.dataBuffer, "\n");
	parseMapDataLine(token, 0);

	int y = 1;

	while (true)
	{
		token = strtok(NULL, "\n");

		parseMapDataLine(token, y);

		y++;
		if (y == MAPHEIGHT)
			break;
	}

	int errors = 0;

	bool allowAtSkillLevel = false;

	while (true)
	{
		token = strtok(NULL, "\n");

		#if USEPAK
		graphics.showLoading(1, 100);
		graphics.delay(1);
		#endif

		allowAtSkillLevel = false;

		sscanf(token, "%s", skillLevel);

		if ((strstr(skillLevel, "E")) && (game.skill == 0))
			allowAtSkillLevel = true;

		if ((strstr(skillLevel, "M")) && (game.skill == 1))
			allowAtSkillLevel = true;

		if ((strstr(skillLevel, "H")) && (game.skill >= 2))
			allowAtSkillLevel = true;
			
		// This is just for practice missions
		if (game.skill == -1)
			allowAtSkillLevel = true;

		// Ignore comments
		if (strstr(skillLevel, "//"))
			allowAtSkillLevel = false;

		if (strcmp("@EOF@", skillLevel) == 0)
		{
			break;
		}

		if (allowAtSkillLevel)
		{
			sscanf(token, "%*s %s", mapEntity);

			if (strcmp("STAGENAME", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %*c %[^\"] %*c", string[0]);

				map.setName(string[0]);
				game.setStageName(string[0]);
			}
			else if (strcmp("TRAIN", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s %d %d %d %d %d %s %s", string[0], &param[0], &param[1], &param[2], &param[3], &param[4], string[1], string[2]);
				map.addTrain(string[0], param[0], param[1], param[2], param[3], param[4], engine.getValueOfDefine(string[1]), engine.getValueOfDefine(string[2]));
			}
			else if (strcmp("DOOR", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s %s %d %d %d %d %s", string[0], string[1], &param[0], &param[1], &param[2], &param[3], string[2]);
				map.addDoor(string[0], engine.getValueOfDefine(string[1]), param[0], param[1], param[2], param[3], engine.getValueOfDefine(string[2]));
			}
			else if (strcmp("SWITCH", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %*c %[^\"] %*c %s %*c %[^\"] %*c %*c %[^\"] %*c %s %d %d %s", string[0], string[1], string[2], string[3], string[4], &param[0], &param[1], string[5]);
				map.addSwitch(string[0], string[1], string[2], string[3], engine.getValueOfDefine(string[4]), param[0], param[1], engine.getValueOfDefine(string[5]));
			}
			else if (strcmp("ITEM", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %d %*c %[^\"] %*c %d %d %s", &param[0], string[0], &param[1], &param[2], string[1]);

				addItem(param[0], string[0], param[1], param[2], string[1], 60, 1, 0, false);

				if (param[0] >= ITEM_MISC)
					map.totalItems++;
			}
			else if (strcmp("OBSTACLE", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %*c %[^\"] %*c %d %d %s", string[0], &param[0], &param[1], string[1]);

				addObstacle(string[0], param[0], param[1], string[1]);
			}
			else if (strcmp("OBJECTIVE", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %*c %[^\"] %*c %*c %[^\"] %*c %d %s", string[0], string[1], &param[0], string[2]);

				map.addObjective(string[0], string[1], param[0], engine.getValueOfDefine(string[2]));
			}
			else if (strcmp("START", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %d %d", &param[0], &param[1]);

				player.place(param[0], param[1]);

				game.setCheckPoint(param[0], param[1]);
				game.setObjectiveCheckPoint();

			}
			else if (strcmp("ENEMY", mapEntity) == 0)
			{
				if (!engine.devNoMonsters)
				{
					sscanf(token, "%*s %*s %*c %[^\"] %*c %d %d", string[0], &param[0], &param[1]);
					addEnemy(string[0], param[0], param[1], 0);
				}
			}
			else if (strcmp("MIA", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %*c %[^\"] %*c %d %d %s", string[0], &param[0], &param[1], string[1]);
				addMIA(string[0], param[0], param[1], engine.getValueOfDefine(string[1]));
				map.totalMIAs++;
			}
			else if (strcmp("REQUIREDMIAS", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %d", &param[0]);
				map.requiredMIAs = param[0];
			}
			else if (strcmp("LINEDEF", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %*c %[^\"] %*c %s %*c %[^\"] %*c %d %d %d %d", string[0], string[1], string[2], &param[0], &param[1], &param[2], &param[3]);

				addLineDef(string[0], string[1], string[2], param[0], param[1], param[2], param[3]);
			}
			else if (strcmp("SPAWNPOINT", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s %d %d %s %s %d %d %s", string[0], &param[0], &param[1], string[1], string[2], &param[2], &param[3], string[3]);
				map.addSpawnPoint(string[0], param[0], param[1], engine.getValueOfDefine(string[1]), engine.getValueOfDefine(string[2]), param[2], param[3], engine.getValueOfDefine(string[3]));
			}
			else if (strcmp("SPAWNABLE_ENEMY", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %*c %[^\"] %*c", string[0]);
				map.setAllowableEnemy(getDefinedEnemy(string[0]));
			}
			else if (strcmp("TELEPORTER", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s %d %d %d %d %s", string[0], &param[0], &param[1], &param[2], &param[3], string[1]);
				addTeleporter(string[0], param[0], param[1], param[2], param[3], engine.getValueOfDefine(string[1]));
			}
			else if (strcmp("TRAP", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s %s %d %d %d %d %d %d %d %d %s %s", string[0], string[1], &param[0], &param[1], &param[2], &param[3], &param[4], &param[5], &param[6], &param[7], string[2], string[3]);
				addTrap(string[0], engine.getValueOfDefine(string[1]), param[0], param[1], param[2], param[3], param[4], param[5], param[6], param[7], string[2], engine.getValueOfDefine(string[3]));
			}
			else if (strcmp("SPRITE", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %[^\n]", string[0]);
				loadSprite(string[0]);
			}
			else if (strcmp("DEFENEMY", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %[^\n]", string[0]);
				loadEnemy(string[0]);
			}
			else if (strcmp("TILESET", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s", string[0]);
				graphics.loadMapTiles(string[0]);
			}
			else if (strcmp("CLIPPING", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %d %d %d %d", &param[0], &param[1], &param[2], &param[3]);
				map.setClipping(param[0], param[1], param[2], param[3]);
			}
			else if (strcmp("AMBIENCE", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s", string[0]);
				if (!audio.loadSound(SND_AMBIANCE, string[0]))
					graphics.showErrorAndExit(ERR_FILE, string[0]);
			}
			else if (strcmp("WATERLEVEL", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %d", &param[0]);
				map.waterLevel = map.requiredWaterLevel = param[0];
			}
			else if (strcmp("ALPHATILES", mapEntity) == 0)
			{
				for (int i = 0 ; i < 15 ; i++)
					*token++;

				while (true)
				{
					sscanf(token, "%d", &param[0]);

					if (param[0] == -1)
						break;
						
					debug(("Setting Alpha for Tile %d\n", param[0]));

					SDL_SetAlpha(graphics.tile[param[0]], SDL_SRCALPHA|SDL_RLEACCEL, 128);

					while (true)
					{
						*token++;

						if (*token == ' ')
							break;
					}
				}
			}
			else if (strcmp("BACKGROUND", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s", string[0]);
				graphics.loadBackground(string[0]);
			}
			else if (strcmp("MUSIC", mapEntity) == 0)
			{
				sscanf(token, "%*s %*s %s", string[0]);
				audio.loadMusic(string[0]);
			}
			else if (allowAtSkillLevel)
			{
				graphics.showErrorAndExit("Symbol '%s' not recognised or not implemented in map data", mapEntity);
				errors++;
			}

			if (errors > 100)
				break;
		}
		else
		{
			//printf("Ignoring Line: %s\n", token);
			errors++;
		}
	}

	if (errors > 100)
	{
		printf("Too many errors in map data. Exitting.\n");
		return false;
	}

	adjustObjectives();
	initMIAPhrases();

	return true;
}
