
#include <stdio.h>
#include <memory.h>
#include "SDLAudioOutputDevice.h"

#define WORKING_BUFFER_SIZE 4096
#define HARDWARE_BUFFER_SIZE 2048

INT16     samples[WORKING_BUFFER_SIZE];
UINT32    playbackPosition    = 0;
int       queueStart    = 0;
int       queueFillSize = 0;


// Text for error messages
const CHAR* audioErrorMessages[2] = {
    "Unable to initialize audio",
    "Unable to open audio device",
};


void audioCallback(void *unused, Uint8 *stream, int len)
{
#ifdef _TRACE_AUDIO
    fprintf(stderr, "::audioCallBack()\n");
#endif

    // Don't stick around in this loop forever, because when the
    // program is ended cleanup will not begin until this callback
    // has returned - but it won't end without this check because
    // nothing new is being added to the queue.
    if (!queueFillSize)
        return;

    UINT16 copySize = (queueFillSize > HARDWARE_BUFFER_SIZE
            ? HARDWARE_BUFFER_SIZE : queueFillSize);
    if (queueStart+copySize > WORKING_BUFFER_SIZE) {
        int endChunk = WORKING_BUFFER_SIZE-queueStart;
        memcpy(stream, samples+queueStart, endChunk<<1);
        memcpy(stream+(endChunk<<1), samples, (copySize-endChunk)<<1);
    }
    else
        memcpy(stream, samples+queueStart, copySize<<1);

    queueStart = (queueStart+copySize)%WORKING_BUFFER_SIZE;
    queueFillSize -= copySize;

#ifdef _TRACE_AUDIO
    fprintf(stderr, "exiting ::audioCallBack()\n");
#endif

}


SDLAudioOutputDevice::SDLAudioOutputDevice()
{
    // Init all variables
    m_desired.freq        = 22050;
    m_desired.format      = AUDIO_S16MSB;
    m_desired.samples     = HARDWARE_BUFFER_SIZE;
    m_desired.channels    = 1;
    m_desired.userdata    = NULL;
    m_desired.callback    = audioCallback;

    // clear buffer
    memset(samples, 0, WORKING_BUFFER_SIZE << 1);
}

SDLAudioOutputDevice::~SDLAudioOutputDevice()
{
    // Cleanup before shutdown
    release();
}


const CHAR* SDLAudioOutputDevice::getErrorDescription(ERRCODE errorCode)
{
    return audioErrorMessages[errorCode-1];
}


// init: Activate audio
ERRCODE SDLAudioOutputDevice::init()
{
    if(SDL_Init(SDL_INIT_AUDIO))
        return ERR_INIT_AUDIO;

    SDL_AudioSpec* desired    = &m_desired;
    SDL_AudioSpec* obtained    = &m_obtained;
    if ( SDL_OpenAudio(desired, obtained) < 0 )
        return ERR_OPEN_AUDIO;

    // TODO: CHECK TO SEE IF OBTAINED = DESIRED, AND ACCOUNT FOR DIFFERENCES!!!!

    SDL_PauseAudio(0);
    return OK;
}

// release: Clean up
void SDLAudioOutputDevice::release()
{
    SDL_PauseAudio(true);
    SDL_CloseAudio();
    SDL_QuitSubSystem(SDL_INIT_AUDIO);
}

void SDLAudioOutputDevice::playSamples(INT16* sampleBuffer, UINT16 sampleCount)
{
#ifdef _TRACE_AUDIO
    fprintf(stderr, "SDLAudioOutputDevice::playSamples()\n");
#endif

    SDL_LockAudio();

    while (WORKING_BUFFER_SIZE-queueFillSize < sampleCount) {
        SDL_UnlockAudio();
        SDL_Delay(1);
        SDL_LockAudio();
    }

    int queuePosition = (queueStart + queueFillSize)%WORKING_BUFFER_SIZE;
    if (queuePosition+sampleCount > WORKING_BUFFER_SIZE) {
        int endChunk = WORKING_BUFFER_SIZE-queuePosition;
        memcpy(samples+queuePosition, sampleBuffer, endChunk<<1);
        memcpy(samples, sampleBuffer+endChunk, (sampleCount-endChunk)<<1);
    }
    else
        memcpy(samples+queuePosition, sampleBuffer, sampleCount<<1);

    queueFillSize += sampleCount;

    SDL_UnlockAudio();

#ifdef _TRACE_AUDIO
    fprintf(stderr, "exiting SDLAudioOutputDevice::playSamples()\n");
#endif
}
