
#include <iostream>
#include <string.h>
#include "AY38900.h"
#include "MOBRect.h"

using namespace std;

const INT32 AY38900::TICK_LENGTH_SCANLINE             = 228;
const INT32 AY38900::TICK_LENGTH_FRAME                = 59736;
const INT32 AY38900::TICK_LENGTH_VBLANK               = 15164;
const INT32 AY38900::TICK_LENGTH_START_ACTIVE_DISPLAY = 112;
const INT32 AY38900::TICK_LENGTH_IDLE_ACTIVE_DISPLAY  = 456;
const INT32 AY38900::TICK_LENGTH_FETCH_ROW            = 456;
const INT32 AY38900::TICK_LENGTH_RENDER_ROW           = 3192;

const INT32 AY38900::LOCATION_BACKTAB    = 0x0200;
const INT32 AY38900::LOCATION_GROM       = 0x3000;
const INT32 AY38900::LOCATION_GRAM       = 0x3800;
const INT32 AY38900::FOREGROUND_BIT = 0x0010;

const UINT32 AY38900::palette[32] = {
    0x000000, 0x002DFF, 0xFF3D10, 0xC9CFAB,
    0x386B3F, 0x00A756, 0xFAEA50, 0xFFFCFF,
    0xBDACC8, 0x24B8FF, 0xFFB41F, 0x546E00,
    0xFF4E57, 0xA496FF, 0x75CC80, 0xB51A58,
    0x000000, 0x002DFF, 0xFF3D10, 0xC9CFAB,
    0x386B3F, 0x00A756, 0xFAEA50, 0xFFFCFF,
    0xBDACC8, 0x24B8FF, 0xFFB41F, 0x546E00,
    0xFF4E57, 0xA496FF, 0x75CC80, 0xB51A58,
};

AY38900::AY38900()
{
	horizontalOffset	= 0;
	verticalOffset		= 0;
	blockTop			= 0;
	blockLeft			= 0;
	mode				= 0;

	SR1 = NULL;
	SR2 = NULL;
	cpu = NULL;

	reset();
}

void AY38900::init(CP1610* cpu, SignalLine* SR1,
        SignalLine* SR2, SignalLine* SST)
{
    this->cpu = cpu;
    this->SR1 = SR1;
    this->SR2 = SR2;
    this->SST = SST;

    //initialize the registers
    registers.init(this);

    memories[0] = &registers;
    memories[1] = &backtab;
    memories[2] = &gram;
    memories[3] = &grom;
}

INT32 AY38900::getMemoryCount()
{
    return 4;
}

Memory16Bit* AY38900::getMemory(INT32 i)
{
    return memories[i];
}

INT32 AY38900::getClockSpeed() {
    return 3579545;
}

void AY38900::reset()
{

//mob info
memset(mobBuffers, 0, sizeof(mobBuffers));


    inVBlank = FALSE;

    //reset the registers first
    registers.reset();
    backtab.reset();
    gram.reset();
    grom.reset();

    //switch to bus copy mode
    setGraphicsBusVisible(TRUE);

    //reset the mobs
    for (UINT16 i = 0; i < 8; i++)
        mobs[i].reset();

    //reset the state variables
    mode = -1;
	if(SR1)
		SR1->isHigh = TRUE;
	if(SR2)
		SR2->isHigh = TRUE;
	if(cpu)
		cpu->isIdle = FALSE;
    previousDisplayEnabled = TRUE;
    displayEnabled         = FALSE;
    colorStackMode         = FALSE;
    colorModeChanged       = TRUE;
    bordersChanged         = TRUE;
    colorStackChanged      = TRUE;
    offsetsChanged         = TRUE;
    imageBufferChanged     = TRUE;

    //local register data
    borderColor = 0;
    blockLeft = blockTop = FALSE;
    horizontalOffset = verticalOffset = 0;

    //blank the offscreen image
    for (INT32 x = 0; x < 15360; x++)
        combinedBuffer[x] = 0;
}

void AY38900::setGraphicsBusVisible(BOOL visible)
{
    registers.visible = visible;
    gram.visible = visible;
    grom.visible = visible;
}

INT32 AY38900::tick()
{
#ifdef _TRACE_TICKS
    fprintf(stderr, "AY38900::tick()\n");
#endif

    //move to the next mode
    mode++;

    switch (mode) {
        case 0:
            setGraphicsBusVisible(TRUE);
            if (previousDisplayEnabled)
                renderFrame();
            displayEnabled = FALSE;
            inVBlank = TRUE;

            //release SR2, allowing the CPU to run
            SR2->isHigh = TRUE;
            cpu->isIdle = FALSE;

            //kick the irq line
            SR1->isHigh = FALSE;

            return TICK_LENGTH_VBLANK;

        case 1:
            inVBlank = FALSE;

            //irq line comes back up
            SR1->isHigh = TRUE;

            //if the display is not enabled, skip the rest of the modes
            if (!displayEnabled) {
                previousDisplayEnabled = FALSE;
                if (previousDisplayEnabled) {
                    //render a blank screen
                    for (INT32 x = 0; x < 15360; x++)
                        combinedBuffer[x] = borderColor;
                }
                mode = -1;
                return (TICK_LENGTH_FRAME - TICK_LENGTH_VBLANK);
            }
            else {
                previousDisplayEnabled = TRUE;
                SR2->isHigh = FALSE;
                return TICK_LENGTH_START_ACTIVE_DISPLAY;
            }

        case 2:
            //switch to bus isolation mode, but only if the CPU has
            //acknowledged ~SR2 by asserting ~SST
            if (!SST->isHigh)
                setGraphicsBusVisible(FALSE);

            //release SR2
            SR2->isHigh = TRUE;
            cpu->isIdle = FALSE;

            return TICK_LENGTH_IDLE_ACTIVE_DISPLAY +
                    (2*verticalOffset*TICK_LENGTH_SCANLINE);

        case 3:
        case 5:
        case 7:
        case 9:
        case 11:
        case 13:
        case 15:
        case 17:
        case 19:
        case 21:
        case 23:
        case 25:
            SR2->isHigh = FALSE;
            return TICK_LENGTH_FETCH_ROW;

        case 4:
        case 6:
        case 8:
        case 10:
        case 12:
        case 14:
        case 16:
        case 18:
        case 20:
        case 22:
        case 24:
            SR2->isHigh = TRUE;
            cpu->isIdle = FALSE;
            return TICK_LENGTH_RENDER_ROW;

        case 26:
            SR2->isHigh = TRUE;
            cpu->isIdle = FALSE;

            //this mode could be cut off in tick length if the vertical
            //offset is greater than 1
            switch (verticalOffset) {
                case 0:
                    return TICK_LENGTH_RENDER_ROW;
                case 1:
                    mode = -1;
                    return TICK_LENGTH_RENDER_ROW - TICK_LENGTH_SCANLINE;
                default:
                    mode = -1;
                    return (TICK_LENGTH_RENDER_ROW - TICK_LENGTH_SCANLINE -
                            (2*(verticalOffset-1)*TICK_LENGTH_SCANLINE));
            }

        case 27:
        default:
            mode = -1;
            SR2->isHigh = FALSE;
            return TICK_LENGTH_SCANLINE;
    }
}

void AY38900::renderFrame()
{
    //render the next frame
    if (somethingChanged()) {
        renderBorders();
        renderBackground();
        renderMOBs();
        for (INT32 i = 0; i < 8; i++)
            mobs[i].collisionRegister = 0;
        copyMOBsToStagingArea();
        determineMOBCollisions();
        markClean();
    }
    for (INT32 i = 0; i < 8; i++)
        registers.memory[0x18+i] |= mobs[i].collisionRegister;
}

void AY38900::render(VideoOutputDevice* videoOutputDevice)
{
    videoOutputDevice->renderPalettedImage(combinedBuffer);
}

BOOL AY38900::somethingChanged() {
    return (offsetsChanged || bordersChanged || colorStackChanged ||
        colorModeChanged || backtab.isDirty() || gram.isDirty() ||
        mobs[0].changed || mobs[1].changed ||
        mobs[2].changed || mobs[3].changed ||
        mobs[4].changed || mobs[5].changed ||
        mobs[6].changed || mobs[7].changed);
}

void AY38900::markClean() {
    //everything has been rendered and is now clean
    offsetsChanged = FALSE;
    bordersChanged = FALSE;
    colorStackChanged = FALSE;
    colorModeChanged = FALSE;
    backtab.markClean();
    gram.markClean();
    for (INT32 i = 0; i < 8; i++)
        mobs[i].markClean();
}

void AY38900::renderBorders() {
    if (!bordersChanged && (!offsetsChanged || (blockLeft && blockTop)))
        return;

    //draw the borders, if necessary
    if (blockTop) {
        UINT8* rowOne = combinedBuffer;
        UINT8* rowTwo = combinedBuffer+29440;
        for (UINT16 x = 0; x < 1280; x++) {
            (*rowOne++) = borderColor;
            (*rowTwo++) = borderColor;
        }
    }
    else if (verticalOffset) {
        UINT8* rowOne = combinedBuffer;
        UINT16 numPixels = verticalOffset*320;
        for (UINT16 x = 0; x < numPixels; x++)
            (*rowOne++) = borderColor;
    }

    if (blockLeft) {
        UINT8* colOne = combinedBuffer;
        UINT8* colTwo = combinedBuffer+156;
        for (UINT16 y = 0; y < 192; y++) {
            for (UINT16 x = 0; x < 4; x++) {
                (*colOne++) = borderColor;
                (*colTwo++) = borderColor;
            }
            colOne += 156;
            colTwo += 156;
        }
    }
    else if (horizontalOffset) {
        UINT8* colOne = combinedBuffer;
        UINT8 offset = 160-horizontalOffset;
        for (UINT16 y = 0; y < 192; y++) {
            for (UINT16 x = 0; x < horizontalOffset; x++)
                (*colOne++) = borderColor;
            colOne += offset;
        }
    }
}

void AY38900::renderMOBs() {
    MOBRect* r;
    INT32 cardNumber;
    INT32 cardMemoryLocation;
    INT32 pixelSize;
    INT32 mobPixelHeight;
    BOOL doubleX;
    INT32 nextMemoryLocation;
    INT32 nextData;
    INT32 nextX;
    INT32 nextY;
    INT32 xInc;

    for (INT32 i = 0; i < 8; i++) {
        if (!mobs[i].changed && mobs[i].isGrom)
            continue;

        cardNumber = mobs[i].cardNumber;
        if (!mobs[i].isGrom)
            cardNumber = (cardNumber & 0x003F);
        cardMemoryLocation = (cardNumber << 3);

        r = mobs[i].getBounds();
        pixelSize = (mobs[i].quadHeight ? 4 : 1) *
                (mobs[i].doubleHeight ? 2 : 1);
        mobPixelHeight = 2 * r->height;
        doubleX = mobs[i].doubleWidth;

        for (INT32 j = 0; j < mobPixelHeight; j++) {
            nextMemoryLocation = (cardMemoryLocation + (j/pixelSize));
            //if (!mobs[i].changed && !gram.isDirty(nextMemoryLocation))
                //continue;

            nextData = (mobs[i].isGrom
                    ? (nextMemoryLocation >= (INT32)grom.getSize()
                        ? 0xFFFF : grom.peek(nextMemoryLocation))
                    : (nextMemoryLocation >= (INT32)gram.getSize()
                        ? 0xFFFF: gram.peek(nextMemoryLocation)));
            nextX = (mobs[i].horizontalMirror ? (doubleX ? 15 : 7) : 0);
            nextY = (mobs[i].verticalMirror
                    ? (mobPixelHeight - j - 1) : j);
            xInc = (mobs[i].horizontalMirror ? -1: 1);
            mobBuffers[i][nextX][nextY] = ((nextData & 0x0080) != 0);
            mobBuffers[i][nextX + xInc][nextY] = (doubleX
                    ? ((nextData & 0x0080) != 0)
                    : ((nextData & 0x0040) != 0));
            mobBuffers[i][nextX + (2*xInc)][nextY] = (doubleX
                    ? ((nextData & 0x0040) != 0)
                    : ((nextData & 0x0020) != 0));
            mobBuffers[i][nextX + (3*xInc)][nextY] = (doubleX
                    ? ((nextData & 0x0040) != 0)
                    : ((nextData & 0x0010) != 0));
            mobBuffers[i][nextX + (4*xInc)][nextY] = (doubleX
                    ? ((nextData & 0x0020) != 0)
                    : ((nextData & 0x0008) != 0));
            mobBuffers[i][nextX + (5*xInc)][nextY] = (doubleX
                    ? ((nextData & 0x0020) != 0)
                    : ((nextData & 0x0004) != 0));
            mobBuffers[i][nextX + (6*xInc)][nextY] = (doubleX
                    ? ((nextData & 0x0010) != 0)
                    : ((nextData & 0x0002) != 0));
            mobBuffers[i][nextX + (7*xInc)][nextY] = (doubleX
                    ? ((nextData & 0x0010) != 0)
                    : ((nextData & 0x0001) != 0));
            if (!doubleX)
                continue;

            mobBuffers[i][nextX + (8*xInc)][nextY] =
                    ((nextData & 0x0008) != 0);
            mobBuffers[i][nextX + (9*xInc)][nextY] =
                    ((nextData & 0x0008) != 0);
            mobBuffers[i][nextX + (10*xInc)][nextY] =
                    ((nextData & 0x0004) != 0);
            mobBuffers[i][nextX + (11*xInc)][nextY] =
                    ((nextData & 0x0004) != 0);
            mobBuffers[i][nextX + (12*xInc)][nextY] =
                    ((nextData & 0x0002) != 0);
            mobBuffers[i][nextX + (13*xInc)][nextY] =
                    ((nextData & 0x0002) != 0);
            mobBuffers[i][nextX + (14*xInc)][nextY] =
                    ((nextData & 0x0001) != 0);
            mobBuffers[i][nextX + (15*xInc)][nextY] =
                    ((nextData & 0x0001) != 0);
        }

    }
}

void AY38900::renderBackground() {
    if (backtab.isDirty() || gram.isDirty() || colorStackChanged ||
            colorModeChanged)
    {
        if (colorStackMode)
            renderColorStackMode();
        else
            renderForegroundBackgroundMode();
    }
    copyBackgroundBufferToStagingArea();
}

void AY38900::renderForegroundBackgroundMode()
{
    UINT8 nextx = 0;
    UINT8 nexty = 0;
    for (UINT8 i = 0; i < 240; i++) {
        UINT16 nextCard = backtab.peek(0x200+i);
        BOOL isGrom = !(nextCard & 0x0800);
        BOOL renderAll = backtab.isDirty(0x200+i) || colorModeChanged;
        UINT16 memoryLocation = (nextCard & 0x01F8);

        if (renderAll || (!isGrom && gram.isCardDirty(memoryLocation))) {
            UINT8 fgcolor = (nextCard & 0x0007) | FOREGROUND_BIT;
            UINT8 bgcolor = ((nextCard & 0x2000) >> 11) |
                    ((nextCard & 0x1600) >> 9);

            UINT16* memory = (isGrom ? grom.image : gram.image) +
                    memoryLocation;
            for (INT32 j = 0; j < 8; j++)
                renderLine((*memory++), nextx, nexty+j, fgcolor, bgcolor);
        }
        nextx += 8;
        if (nextx == 160) {
            nextx = 0;
            nexty += 8;
        }
    }
}

void AY38900::renderColorStackMode()
{
    UINT8 csPtr = 0;
    //if there are any dirty color advance bits in the backtab, or if
    //the color stack or the color mode has changed, the whole scene
    //must be rendered
    BOOL renderAll = backtab.areColorAdvanceBitsDirty() ||
            colorStackChanged || colorModeChanged;

    UINT8 nextx = 0;
    UINT8 nexty = 0;
    for (UINT8 h = 0; h < 240; h++) {
        UINT16 nextCard = backtab.peek(0x200+h);

        //colored squares mode
        if ((nextCard & 0x1800) == 0x1000) {
            if (renderAll || backtab.isDirty(0x200+h)) {
                UINT8 csColor = registers.memory[0x28 + csPtr];
                UINT8 color0 = nextCard & 0x0007;
                UINT8 color1 = (nextCard & 0x0038) >> 3;
                UINT8 color2 = (nextCard & 0x01C0) >> 6;
                UINT8 color3 = ((nextCard & 0x2000) >> 11) |
                        ((nextCard & 0x0600) >> 9);
                renderColoredSquares(nextx, nexty,
                        (color0 == 7 ? csColor : (color0 | FOREGROUND_BIT)),
                        (color1 == 7 ? csColor : (color1 | FOREGROUND_BIT)),
                        (color2 == 7 ? csColor : (color2 | FOREGROUND_BIT)),
                        (color3 == 7 ? csColor : (color3 | FOREGROUND_BIT)));
            }
        }
        //color stack mode
        else {
            //advance the color pointer, if necessary
            if (nextCard & 0x2000)
                csPtr = (csPtr+1) & 0x03;

            BOOL isGrom = !(nextCard & 0x0800);
            UINT16 memoryLocation = (isGrom ? (nextCard & 0x07F8)
                        : (nextCard & 0x01F8));

            if (renderAll || backtab.isDirty(0x200+h) ||
                    (!isGrom && gram.isCardDirty(memoryLocation)))
            {
                UINT8 fgcolor = ((nextCard & 0x1000) >> 9) |
                        (nextCard & 0x0007) | FOREGROUND_BIT;
                UINT8 bgcolor = registers.memory[0x28 + csPtr];
                UINT16* memory = (isGrom ? grom.image : gram.image) +
                        memoryLocation;
                for (INT32 j = 0; j < 8; j++)
                    renderLine((*memory++), nextx, nexty+j, fgcolor, bgcolor);
            }
        }
        nextx += 8;
        if (nextx == 160) {
            nextx = 0;
            nexty += 8;
        }
    }
}

void AY38900::copyBackgroundBufferToStagingArea()
{
    INT32 sourceWidthX = blockLeft ? 152 : (160 - horizontalOffset);
    INT32 sourceHeightY = blockTop ? 88 : (96 - verticalOffset);

    UINT8* nextSourceRow = backgroundBuffer +
            (blockLeft ? (8 - horizontalOffset) : 0) +
            ((blockTop ? (8 - verticalOffset) : 0) * 160);
    UINT8* nextTargetRow = combinedBuffer +
            (blockLeft ? 4 : 0) +
            ((blockTop ? 4 : 0) * 320);
    for (INT32 y = 0; y < sourceHeightY; y++) {
        memcpy(nextTargetRow, nextSourceRow, sourceWidthX);
        nextTargetRow += 160;
        memcpy(nextTargetRow, nextSourceRow, sourceWidthX);
        nextSourceRow += 160;
        nextTargetRow += 160;
    }
}

//copy the offscreen mob buffers to the staging area
void AY38900::copyMOBsToStagingArea() {
    for (INT8 i = 7; i >= 0; i--) {
        if (mobs[i].xLocation == 0 ||
                (!mobs[i].flagCollisions && !mobs[i].isVisible))
            continue;

        BOOL borderCollision = FALSE;
        BOOL foregroundCollision = FALSE;

        MOBRect* r = mobs[i].getBounds();
        UINT8 mobPixelHeight = r->height << 1;
        UINT8 fgcolor = (UINT8)mobs[i].foregroundColor;

        INT16 leftX = r->x + horizontalOffset;
        INT16 nextY = (r->y + verticalOffset) << 1;
        UINT16 rowPixelIndex = (leftX - (blockLeft ? 4 : 0)) +
                (160 * (nextY - (blockTop ? 8 : 0)));

        for (UINT8 y = 0; y < mobPixelHeight; y++) {
            for (UINT8 x = 0; x < r->width; x++) {
                //if this mob pixel is not on, then don't paint it
                if (!mobBuffers[i][x][y])
                    continue;

                INT32 nextX = leftX + x;
                //if the next pixel location is on the border, then we
                //have a border collision and we can ignore painting it
                if (nextX < (blockLeft ? 8 : 0) || nextX > 158 ||
                        nextY < (blockTop ? 16 : 0) || nextY > 191)
                {
                    borderCollision = TRUE;
                    continue;
                }

                UINT8 currentPixel = combinedBuffer[rowPixelIndex+x];
       
                //check for foreground collision
                if (currentPixel & FOREGROUND_BIT) {
                    foregroundCollision = TRUE;
                    if (mobs[i].behindForeground)
                        continue;
                }
      
                if (mobs[i].isVisible)
                    combinedBuffer[rowPixelIndex+x] = fgcolor |
                            (currentPixel & FOREGROUND_BIT);
            }
            rowPixelIndex += 160;
            nextY++;
        }

        //update the collision bits
        if (mobs[i].flagCollisions) {
            if (foregroundCollision)
                mobs[i].collisionRegister |= 0x0100;
            if (borderCollision)
                mobs[i].collisionRegister |= 0x0200;
        }
    }
}

void AY38900::renderLine(UINT8 nextByte, INT32 x, INT32 y,
        UINT8 fgcolor, UINT8 bgcolor)
{
    UINT8* nextTargetPixel = backgroundBuffer + x + (y*160);
    (*nextTargetPixel++) = (nextByte & 0x80) ? fgcolor : bgcolor;
    (*nextTargetPixel++) = (nextByte & 0x40) ? fgcolor : bgcolor;
    (*nextTargetPixel++) = (nextByte & 0x20) ? fgcolor : bgcolor;
    (*nextTargetPixel++) = (nextByte & 0x10) ? fgcolor : bgcolor;
    (*nextTargetPixel++) = (nextByte & 0x08) ? fgcolor : bgcolor;
    (*nextTargetPixel++) = (nextByte & 0x04) ? fgcolor : bgcolor;
    (*nextTargetPixel++) = (nextByte & 0x02) ? fgcolor : bgcolor;
    (*nextTargetPixel) = (nextByte & 0x01) ? fgcolor : bgcolor;
}

void AY38900::renderColoredSquares(INT32 x, INT32 y, UINT8 color0, UINT8 color1,
        UINT8 color2, UINT8 color3)
{
    
    UINT8* topLeftPixel = backgroundBuffer + x + (y*160);
    UINT8* topRightPixel = topLeftPixel+4;
    UINT8* bottomLeftPixel = topLeftPixel+640;
    UINT8* bottomRightPixel = bottomLeftPixel+4;

    for (UINT8 w = 0; w < 4; w++) {
        for (UINT8 x = 0; x < 4; x++) {
            (*topLeftPixel++) = color0;
            (*topRightPixel++) = color1;
            (*bottomLeftPixel++) = color2;
            (*bottomRightPixel++) = color3;
        }
        topLeftPixel += 156;
        topRightPixel += 156;
        bottomLeftPixel += 156;
        bottomRightPixel += 156;
    }
}

void AY38900::determineMOBCollisions()
{
    //check mob to mob collisions
    for (int i = 0; i < 7; i++) {
        if (mobs[i].xLocation == 0 || !mobs[i].flagCollisions)
            continue;

        for (int j = i+1; j < 8; j++) {
            if (mobs[j].xLocation == 0 || !mobs[j].flagCollisions)
                continue;

            if (mobsCollide(i, j)) {
                mobs[i].collisionRegister |= (1 << j);
                mobs[j].collisionRegister |= (1 << i);
            }
        }
    }
}

#define MIN(v1, v2) (v1 < v2 ? v1 : v2)
#define MAX(v1, v2) (v1 > v2 ? v1 : v2)

BOOL AY38900::mobsCollide(INT32 mobNum0, INT32 mobNum1) {
    MOBRect* r0;
    r0 = mobs[mobNum0].getBounds();
    MOBRect* r1;
    r1 = mobs[mobNum1].getBounds();
    if (!r0->intersects(r1))
        return FALSE;

    //iterate over the intersecting bits to see if any touch
    INT32 x0 = MAX(r0->x, r1->x);
    INT32 y0 = MAX(r0->y, r1->y);
    INT32 r0y = 2*(y0-r0->y);
    INT32 r1y = 2*(y0-r1->y);
    INT32 width = MIN(r0->x+r0->width, r1->x+r1->width) - x0;
    INT32 height = (MIN(r0->y+r0->height, r1->y+r1->height) - y0) * 2;
    for (INT32 x = 0; x < width; x++) {
        for (INT32 y = 0; y < height; y++) {
            if (mobBuffers[mobNum0][x0-r0->x+x][r0y+y] &&
                    mobBuffers[mobNum1][x0-r1->x+x][r1y+y])
                return TRUE;
        }
    }

    return FALSE;
}

void AY38900::setGROMImage(UINT16* gromImage)
{
    memcpy(grom.image, gromImage, sizeof(grom.image));
}

void AY38900::getOutputImageSize(UINT16* width, UINT16* height)
{
    (*width) = 160;
    (*height) = 192;
}

void AY38900::getPalette(const UINT32** p, UINT16* numEntries)
{
    (*p) = (UINT32*)&palette[0];
    (*numEntries) = 32;
}
