
#ifndef ROM_H
#define ROM_H

#include <string.h>
#include <stdio.h>
#include "PopMessage.h"
#include "Memory.h"
#include "FileInputStream.h"
#include "FileOutputStream.h"
#include "osd/types.h"
#include "unzip.h"

#ifndef _MAX_PATH
#define _MAX_PATH 255
#endif

#ifdef _WIN32
#pragma warning(disable:4786)    // Suppress STL debug info > 255 chars messages
#endif

template <typename A, typename W>
class ROM : public Memory<A, W>
{

    public:
        ROM(A size, A location);
        ROM(W* image, A size, A location);
        ~ROM();
        void loadImage(W* image);
        void reset();
        A getSize();
        A getNumLocations();
        A getLocation(A i);
        BOOL isLocationReadable(A i);
        BOOL isLocationWriteable(A i);
        W peek(A location);
        void poke(A location, W value);
        void save(FileOutputStream*);
        void load(FileInputStream*);

        static void loadROMImage(W** romImage, A* size,
                const CHAR* resourceName, BOOL is16bit);

    private:
        static void formInternalFilename(CHAR*, const CHAR*, const CHAR*);

        A   size;
        A   location;
        W*  image;

};

template <typename A, typename W>
ROM<A, W>::ROM(A size, A location)
{
    this->size = size;
    this->location = location;
    this->image = new W[size];
}

template <typename A, typename W>
ROM<A, W>::ROM(W* image, A size, A location)
{
    this->size = size;
    this->location = location;
    this->image = new W[size];
    loadImage(image);
}

template <typename A, typename W>
ROM<A, W>::~ROM()
{
    if (image != NULL)
        delete[] image;
}

template <typename A, typename W>
void ROM<A, W>::loadImage(W* image)
{
    memcpy(this->image, image, size*sizeof(W));
}

template <typename A, typename W>
void ROM<A, W>::reset()
{
    //ROM can't be changed, so there's nothing to reset
}

template <typename A, typename W>
A ROM<A, W>::getNumLocations() {
    return 1;
}

template <typename A, typename W>
A ROM<A, W>::getLocation(A i) {
    return location;
}

template <typename A, typename W>
BOOL ROM<A, W>::isLocationReadable(A i) {
    return TRUE;
}

template <typename A, typename W>
BOOL ROM<A, W>::isLocationWriteable(A i) {
    return FALSE;
}

template <typename A, typename W>
W ROM<A, W>::peek(A location)
{
    return image[location-this->location];
}

template <typename A, typename W>
void ROM<A, W>::poke(A location, W value)
{
    //can't change ROM
}

template <typename A, typename W>
A ROM<A, W>::getSize()
{
    return size;
}

template <typename A, typename W>
void ROM<A, W>::loadROMImage(W** romImage, A* size, const CHAR* fileName,
        BOOL is16bit)
{
    UINT64 currentSize = 4096;
    W* image = new W[currentSize];
    UINT64 totalSize = 0;
    INT32 nextRead;

    if(strstr(fileName, ".zip") != NULL)
    {
        unzFile file;
        file = unzOpen(fileName);
        if(file != NULL)
        {
            char name1[_MAX_PATH];
            char name2[_MAX_PATH];
            formInternalFilename(name1, fileName, ".bin");
            formInternalFilename(name2, fileName, ".int");

            BOOL opened = TRUE;
            if(unzLocateFile(file, name1, 2) != UNZ_OK) {
                if(unzLocateFile(file, name2, 2) != UNZ_OK) {
                    opened = FALSE;
                }
            }

            if (opened)
            {
                if(unzOpenCurrentFile(file) == UNZ_OK)
                {
                    while(unzReadCurrentFile(file, &nextRead, 1) > 0)
                    {
                        if (totalSize >= currentSize) {
                            W* newImage = new W[currentSize<<1];
                            memcpy(newImage, image, currentSize*sizeof(W));
                            currentSize <<= 1;
                            delete[] image;
                            image = newImage;
                        }
                        if (is16bit)
                        {
                            INT32 tmp_read = 0;
                            unzReadCurrentFile(file, &tmp_read, 1);
                            image[totalSize] = (nextRead << 8) | tmp_read;
                        }
                        else
                            image[totalSize] = nextRead;
    
                        totalSize++;
                    }
                    unzCloseCurrentFile(file);
                }
                else
                {
                    PopMessage("Error in attemting to read data from zip file, verify that zip file is not corrupt.\n", MSG_ERROR);
                }
            }
            else
            {
                PopMessage("Unable to locate rom in zip file.  Ensure that the rom has the same name as the zip file except with an extension of .bin and no embeded paths in the zip file.\n", MSG_ERROR);
            }
            unzClose(file);
        }
        else
        {
            PopMessage("There was an error opening the zip file, ensure that the zip file exists and is not corrupt.\n", MSG_ERROR);
        }

    }
    else
    {

        FILE* file = fopen(fileName, "rb");

        nextRead = fgetc(file);
        while (nextRead != EOF) {
            if (totalSize >= currentSize) {
                W* newImage = new W[currentSize<<1];
                memcpy(newImage, image, currentSize*sizeof(W));
                currentSize <<= 1;
                delete[] image;
                image = newImage;
            }
    
            if (is16bit)
                image[totalSize] = (nextRead << 8) | fgetc(file);
            else
                image[totalSize] = nextRead;
            
            totalSize++;
            nextRead = fgetc(file);
        }
    }

    W* finalImage = new W[totalSize];
    memcpy(finalImage, image, totalSize*sizeof(W));
    delete[] image;
    (*romImage) = finalImage;
    (*size) = totalSize;
}

template <typename A, typename W>
void ROM<A, W>::save(FileOutputStream*)
{
}

template <typename A, typename W>
void ROM<A, W>::load(FileInputStream*)
{
}

template <typename A, typename W>
void ROM<A, W>::formInternalFilename(CHAR* out, const CHAR* in, const CHAR* ext)
{
    strcpy(out, in);
    char *p = strstr(out, ".zip");
    *p = '\0';
    strcat(out, ext);
    p = &out[strlen((char *)out)];
    while(--p > out)
    {
        if(*p == '\\' || *p == '/')
            break;
    }
    strcpy(out, p+1);
}

typedef ROM<UINT16, UINT16> ROM16Bit;
typedef ROM<UINT16, UINT8> ROM8Bit;

#endif
