// ****************************************************************************
// * Bliss:  Video Game System Emulator
// *           Created by Kyle Davis
// *
// *           Kyle Davis    - Lead Programmer, Design, etc.
// *           Jesse Litton  - Additional programming/debugging
// *
// ****************************************************************************

#include "Bliss.h"
#include "cartridge/CRC32.h"
#include "ROM.h"
#include "std_file.h"

using namespace std;
#ifdef _WIN32
#pragma warning(disable:4786)    // Suppress STL debug info > 255 chars messages
#endif

VideoOutputDevice* f_vod;


// Menu callbacks
Container*  currentMenu;
Container*  previousMenu;

void changeMenu(Container* newMenu)
{
	if(newMenu) {
		previousMenu = currentMenu;
		currentMenu = newMenu;
		currentMenu->m_parent = previousMenu;
	}
    else {
        currentMenu = NULL;
    }
}

void changeMenu(void* lItem)
{
	changeMenu((Container*)((PListItem)lItem)->data);
}


void changeRez(void *data)
{
    char* cLine = (char*)((PListItem)data)->data;

    unsigned int x, y;
    int nParms = sscanf(cLine, "%ux%u", &x, &y);
    if(nParms == 2)
        f_vod->changeOutputSize(x, y);
}

void closeMenu(void *data)
{
    currentMenu = NULL;
}

bool emuClosed = false;
void closeEmu(void *data)
{
    emuClosed = true;
}


Bliss::Bliss()
{
    currentMenu = previousMenu = NULL;
	pauseOnMenu = false;

    currentEmu = NULL;
    frameSkip = 1;
    frameCounter = 0;
    paused = FALSE;
    f_vod = NULL;
    aod = NULL;
    id = NULL;
    cd = NULL;
}

void Bliss::init(VideoOutputDevice* invod, AudioOutputDevice* aod,
        InputDevice* id, ClockDevice* cd)
{
//    this->vod = vod;
    f_vod = invod;

    this->aod = aod;
    this->id = id;
    this->cd = cd;
}

void Bliss::setEmulator(Emulator* emu)
{
    this->currentEmu = emu;
}

void Bliss::setDefaultMenu(Container* menu)
{
    this->defaultMenu = menu;
}

void Bliss::setFrameSkip(INT32 frameSkip)
{
    this->frameSkip = frameSkip;
}

INT32 Bliss::getFrameSkip()
{
    return frameSkip;
}

INT32 Bliss::run()
{
    INT64 displayTickDelta = cd->getTickFrequency()/60;
    INT64 nextDisplayTick = 0; 

    const UINT32* palette;
    UINT16 numEntries;
    currentEmu->getPalette(&palette, &numEntries);
    f_vod->setPalette(palette, numEntries);

    UINT16 width;
    UINT16 height;
    currentEmu->getOutputImageSize(&width, &height);
    f_vod->changeInputSize(width, height);

    currentEmu->setOutputSampleRate(aod->getSampleRate());

    currentEmu->reset();

	INT64 now = cd->getTick();
	INT64 inputSpeed = cd->getTickFrequency() / 5;	// 1/5 sec

    while (true) {
        id->poll();

        if (id->getControlValue(EMULATOR_PAUSE)) {
            paused = !paused;
        }
        else if (id->getControlValue(EMULATOR_RESET)) {
            currentEmu->reset();
        }

        if (id->toggleRequested()) {
						f_vod->toggleScreen();
            now = cd->getTick() + 2 * cd->getTickFrequency();
        }
        else {
        
            // menu input
#ifdef _WIN32
            if(cd->getTick() > (now + inputSpeed)) {
#endif
	  			      if (id->getControlValue(MENU_DISPLAY)) {
  	  		 		     if(currentMenu) {
           			 	    currentMenu = currentMenu->m_parent;    // Back up one
						          if(!currentMenu)
											{
													if(pauseOnMenu) paused = FALSE;
			 									  id->setRepeatState(TRUE);
											}
				           }
        				   else {
										  id->setRepeatState(FALSE);
						          currentMenu = defaultMenu;              // Main menu
						          if(pauseOnMenu)
            					    paused = TRUE;
				           }

				           now = cd->getTick();
		     	      }
         		    else if(currentMenu) {    // Not escape, but menu up
 									  id->setRepeatState(FALSE);
                		if(id->getControlValue(MENU_SELECT) ) { //&& !id->altPressed()) {
                    		    currentMenu->processEvent(PRESS_COMPONENT);
							          if(!currentMenu)
												{
				 									  id->setRepeatState(TRUE);
												}
          	         		now = cd->getTick();
                    }
 		                else if(id->getControlValue(MENU_DOWN)) {
        		            currentMenu->processEvent(NEXT_COMPONENT);
            		        now = cd->getTick();
                		}
                 		else if(id->getControlValue(MENU_UP)) {
                        currentMenu->processEvent(PREV_COMPONENT);
  	                    now = cd->getTick();
   		              }
       		          else if(id->getControlValue(MENU_PGDOWN)) {
           		          currentMenu->processEvent(NEXT_PAGE);
               		      now = cd->getTick();
                    }
 		                else if(id->getControlValue(MENU_PGUP)) {
     		                currentMenu->processEvent(PREV_PAGE);
         		            now = cd->getTick();
             		    }
                }		
#ifdef _WIN32
            }
#endif

            if (f_vod->stopRequested() || emuClosed)
                break;
        }

        if (!paused) {
            if(!currentMenu)
                currentEmu->updateControllers(id);

            currentEmu->run();
            currentEmu->flushAudio(aod);
        }

        frameCounter--;
        if (frameCounter < 0) {
            frameCounter = frameSkip;

            currentEmu->render(f_vod);
            if(currentMenu)
                currentMenu->render(f_vod);
#ifdef _WIN32
            //smooth out the display of successive frames
            UINT64 nowTick = cd->getTick();
            if (nowTick < nextDisplayTick) {
                nextDisplayTick += displayTickDelta;
                while (cd->getTick() < nextDisplayTick);
            }
            else
                nextDisplayTick = nowTick + displayTickDelta;
#endif
            f_vod->present();
        }
#ifdef _WIN32
	else
            nextDisplayTick += displayTickDelta;
#endif

    }

    return 0;
}
