
#define _DEBUG

#ifdef _WIN32
#pragma warning(disable:4786)	// Suppress STL debug info > 255 chars messages
#endif

// ****************************************************************************
// * Bliss32:  Video Game System Emulator
// *           Created by Kyle Davis
// *
// *           Kyle Davis    - Lead Programmer, Design, etc.
// *           Jesse Litton  - Additional programming/debugging
// *
// ****************************************************************************

#include <time.h>

#include "Bliss32.h"
#include "SDL.h"
#include "config.h"
#include "../core/colors.h"
#include "../core/Container.h"
#include "../core/font.h"
#include "../core/image.h"
#include "../core/std_file.h"
#include "../core/PopMessage.h"
#ifdef _WIN32
#include "../core/Win32ReadDir.h"
#endif
#include "../core/cartridge/CRC32.h"
#include "../intellivision/cartridge/IntellivisionCartridge.h"
#include "../intellivision/cartridge/IntellivisionCartridgeConfiguration.h"
#include "../intellivision/cartridge/IntellivisionRecognitionDatabase.h"

const UINT32 Bliss32::CRC_EXECUTIVE_ROM = 0xCBCE86F7;
const UINT32 Bliss32::CRC_GROM          = 0x683A4158;
const UINT32 Bliss32::CRC_ECS_ROM       = 0xEA790A06;
const UINT32 Bliss32::CRC_IVOICE_ROM    = 0x0DE7579D;
const CHAR* Bliss32::FILENAME_EXECUTIVE_ROM = "exec.bin";
const CHAR* Bliss32::FILENAME_GROM          = "grom.bin";
const CHAR* Bliss32::FILENAME_ECS_ROM       = "ecs.bin";
const CHAR* Bliss32::FILENAME_IVOICE_ROM    = "ivoice.bin";

Bliss Bliss32::bliss;
Intellivision Bliss32::inty;

SDLVideoOutputDevice Bliss32::vod;
SDLAudioOutputDevice Bliss32::aod;
SDLInputDevice Bliss32::id;
#ifdef _WIN32
Win32ClockDevice Bliss32::cd;
Win32ReadDir rd;
#else
RDTSCClockDevice Bliss32::cd;
UNIXReadDir rd;
#endif

vector<IntellivisionCartridge*> cartridgeFileList;
IntellivisionRecognitionDatabase* pIrdb = &IntellivisionRecognitionDatabase::FACTORY;
char intvCartPath[MAX_GAMES][MAX_PATH];

IntellivisionCartridge* cart;

char lastcart[MAX_PATH];


int main(int argc, char* argv[])
{
    Bliss32 bliss32;

    char sCmd[MAX_PATH];
    sCmd[0] = '\0';
    if(argc > 1) {
        for(int x = 1; x < argc; x++)
        {
            if(x > 1)
                strcat(sCmd, " ");

            strcat(sCmd, argv[x]);
        }
    }

    bliss32.setProgramName(argv[0]);

    return (bliss32.run(sCmd));
}


void changeGame(void *lItem)
{
    Bliss32::inty.removeCartridge();
    delete cart;

    cart = IntellivisionCartridge::loadCartridge((char*)((PListItem)lItem)->data);
    Bliss32::inty.insertCartridge(cart);
    Bliss32::inty.reset();

    strcpy(lastcart, (char*)((PListItem)lItem)->data);

    changeMenu((Container*)NULL);
}


Bliss32::Bliss32()
{
    execImage = NULL;
    gromImage = NULL;
    ecsImage = NULL;
    ivoiceImage = NULL;
    
    memset(lastcart, '\0', sizeof(lastcart));
}

Bliss32::~Bliss32()
{
    if (execImage)
        delete[] execImage;
    if (gromImage)
        delete[] gromImage;
    if (ecsImage)
        delete[] ecsImage;
    if (ivoiceImage)
        delete[] ivoiceImage;
}


void Bliss32::updateSplash(image* Image, font* StatFont, char* sStatus, font* VerFont)
{
	Image->draw(0, 0);
	StatFont->write(2, 2, sStatus);
	VerFont->write(550, 445, BLISS_VERSION);
	vod.present();
}


INT32 Bliss32::run(char* sCart)
{
    Bliss	bliss;
	UINT8	splash=6;
	INT64	start = cd.getTick();
    char	status[MAX_PATH + 64];
		char cfg_File[MAX_PATH + 64];

   	UINT16	xf = 320, yf = 200;  // Default fullscreen rez
    UINT16	xw = 640, yw = 384;  // Default windowed rez

	// --------------------- Load title screen/mode preferences
	getProgramDir((char *)&cfg_File[0], MAX_PATH);
#ifdef _WIN32
  strcat((char *)cfg_File, "\\");
#else
  strcat((char *)cfg_File, "/");
#endif
  strcat((char *)cfg_File, "bliss.ini");

	Config ini(cfg_File);

	char cLine[32];
	unsigned int mode=0;
	if(ini.get("global", "mode", &cLine[0], 9)) {
		int nParms = sscanf(cLine, "%u", &mode);
		if(nParms == 1)
			vod.toggleScreen(mode);
	}
    else
#ifdef _DEBUG
    	vod.toggleScreen(MODE_WINDOWED);
#else
    	vod.toggleScreen(MODE_FULLSCREEN);
#endif

    // Set size for title screen, this has to be done before
    // initializing the video, because SDL won't re-center yet.
    vod.setFullScreenSize(640, 480);
    vod.setWindowSize(640, 480);


    //init the video device
	char sWindowTitle[32];

#ifdef _WIN32
	strcpy(sWindowTitle, "Bliss32 - \0");
#else
	strcpy(sWindowTitle, "BlissX - \0");
#endif
	strcat(sWindowTitle, BLISS_VERSION);
    INT32 result = vod.init(sWindowTitle, "Bliss", "b32.bmp");
    if (result) {

        sprintf(status, "%s.", vod.getErrorDescription(result));
        PopMessage(status, MSG_ERROR);

        return result;
	}

    //init the audio device
    result = aod.init();
    if (result) {
        vod.toggleScreen(MODE_WINDOWED);

        sprintf(status, "%s.", aod.getErrorDescription(result));
        PopMessage(status, MSG_ERROR);

        vod.release();
        return result;
    }

    //init the input device
    result = id.init();
    if (result) {
        vod.toggleScreen(MODE_WINDOWED);

        sprintf(status, "%s.", id.getErrorDescription(result));
        PopMessage(status, MSG_ERROR);

        aod.release();
        vod.release();
        return result;
    }


	// Start splashscreen
	font*  pfontNeon  = new font(&vod, "NeonFont.png");
	font*  pfontIntel = new font(&vod, "12p Intellifont Black.png");
	image* plogoImage = new image(&vod, "bliss.jpg", false);

	updateSplash(plogoImage, pfontIntel, "Loading Preferences...", pfontNeon);



	// --------------------- Now, load/set the preferred size settings
	if(ini.get("global", "fullscreen", &cLine[0], 9)) {
		sscanf(cLine, "%ux%u", &xf, &yf);
	}
	vod.setFullScreenSize(xf, yf);

	if(ini.get("global", "windowed", &cLine[0], 9)) {
		sscanf(cLine, "%ux%u", &xw, &yw);
	}
	vod.setWindowSize(xw, yw);


	updateSplash(plogoImage, pfontIntel, "Loading System ROMs...", pfontNeon);
    if(!loadSystemROMs()) {
		return 1;
	}
    inty.setExecImage(execImage);
    inty.setGROMImage(gromImage);
    if (ecsImage) {
        inty.enableECSSupport(ecsImage);
    }
    if (ivoiceImage) {
        inty.enableIntellivoiceSupport(ivoiceImage);
    }


	updateSplash(plogoImage, pfontIntel, "Setting ROM Paths...", pfontNeon);
	// --------------------- Load the ROM path
	if(!ini.get("intellivision", "rom_dir", romdir_intv, _MAX_DIR)) {
        strcpy(romdir_intv, "roms");
    }
	if(!ini.get("5200", "rom_dir", romdir_5200, _MAX_DIR)) {
        strcpy(romdir_5200, "roms");
    }


	updateSplash(plogoImage, pfontIntel, "Building Menus...", pfontNeon);
    // Menu Test......................................................
    font*   pfontGray = NULL;
	font*   pfontSmallNeon = NULL;
	font*   pfontSmall = NULL;
	font*   pfontBlue14 = NULL;
	font*   pfontYellow14 = NULL;

			pfontGray		= new font(&vod, "SmallGray.png");
			pfontSmallNeon	= new font(&vod, "SmallNeon.png");
			pfontSmall		= new font(&vod, "SmallStone.png");
			pfontBlue14		= new font(&vod, "14P_Copperplate_LBlue.png");
			pfontYellow14	= new font(&vod, "14P_Copperplate_Yellow.png");
//			pimageBack		= new image(&vod, "background2.png", true);

    image*	pimageBack = new image(&vod, "Background2.png", true);//NULL;

    ImageCtl    imgBack(pimageBack, 200, 0, 0);

    Container   menuMain(220, 180), menuLoad(220, 180), menuIntv(220, 180),
                menu5200(220, 180), menuVideo(220, 180), menuCredits(220, 180);

    // -------------- Main menu construction
    menuMain.addComponent(&imgBack);
    TextCtl  txtMainTitle(pfontGray, "Main Menu", 35, 4);
    menuMain.addComponent(&txtMainTitle);

    ListCtl  lstMain(pfontBlue14, pfontYellow14, 15, 44, 8);
//    ListCtl  lstMain(pfontIntel, pfontYellow14, 15, 44, 8);
    ListItem loadGame, videoSettings, credits, exitBliss;

    loadGame.callback = &changeMenu;
//    loadGame.data = &menuLoad;
    loadGame.data = &menuIntv;
    loadGame.description = "New Game";
    lstMain.addItem(&loadGame, false);

    videoSettings.callback = changeMenu;
    videoSettings.data = &menuVideo;
    videoSettings.description = "Video Settings";
    lstMain.addItem(&videoSettings, false);

    credits.callback = changeMenu;
    credits.data = &menuCredits;
    credits.description = "Credits";
    lstMain.addItem(&credits, false);
    
    exitBliss.callback = &closeEmu;
    exitBliss.description = "Exit Bliss32";
    lstMain.addItem(&exitBliss, false);

    menuMain.addComponent(&lstMain);

    // -------------- System menu construction
    menuLoad.addComponent(&imgBack);
    TextCtl  txtLoadTitle(pfontGray, "Load Game", 35, 4);
    menuLoad.addComponent(&txtLoadTitle);

    ListCtl  lstSystems(pfontBlue14, pfontYellow14, 15, 44, 6);
    ListItem itm5200, itmIntv;

    itm5200.callback    = &changeMenu;
    itm5200.data        = &menu5200;
    itm5200.description = "Atari 5200";
    lstSystems.addItem(&itm5200, false);

    itmIntv.callback = &changeMenu;
    itmIntv.data = &menuIntv;
    itmIntv.description = "Intellivision";
    lstSystems.addItem(&itmIntv, false);

    menuLoad.addComponent(&lstSystems);

    // -------------- Atari 5200 menu construction
    menu5200.addComponent(&imgBack);
    TextCtl  txt5200Title(pfontGray, "5200 Games", 29, 4);
    menu5200.addComponent(&txt5200Title);
    

    // -------------- Intellivision menu construction
    menuIntv.addComponent(&imgBack);
    TextCtl  txtIntvTitle(pfontGray, "Intellivision", 27, 4);
    menuIntv.addComponent(&txtIntvTitle);

    ListCtl  lstGames(pfontSmall, pfontSmallNeon, 15, 44, 8);
    lstGames.setMaxChars(21, 18);

		updateSplash(plogoImage, pfontIntel, "Scanning Intellivision ROMs...", pfontNeon);

    clearCartList_Intv();

    if (rd.open(romdir_intv))
		{
        int i = 0;
        char nextPath[MAX_PATH];
				while (rd.hasNext())
				{
            rd.getNext(nextPath);
            UINT32 crc = CRC32::getCrc(nextPath);
            IntellivisionCartridgeConfiguration* nextCartType =
                (IntellivisionCartridgeConfiguration*)
                pIrdb->getConfiguration(crc);
            if (nextCartType != NULL) {
                strcpy(intvCartPath[i], nextPath);
                IntellivisionCartridge* nextCartFile = new IntellivisionCartridge(
                    nextCartType->getName(),
                    NULL,
                    nextCartType->getMemoryCount(),
                    nextCartType->requiresECS(),
                    nextCartType->usesIntellivoice());

                if((!(nextCartType->requiresECS())) || ecsImage)
                {
                    cartridgeFileList.push_back(nextCartFile);

                    itmIntvGame[i].data = intvCartPath[i];
                    itmIntvGame[i].description = (*cartridgeFileList[i]).getName();
                    itmIntvGame[i].callback = &changeGame;
                    lstGames.addItem(&itmIntvGame[i], true);
                    i++;
                }
            }
        }
        rd.close();
	}

    menuIntv.addComponent(&lstGames);


    // -------------- Video menu construction
    menuVideo.addComponent(&imgBack);
    TextCtl   txtVideoTitle(pfontGray, "Video", 65, 4);
    menuVideo.addComponent(&txtVideoTitle);
 
    ListCtl  lstVideo(pfontBlue14, pfontYellow14, 15, 44, 6);
    ListItem itmVidRez[64];

		updateSplash(plogoImage, pfontIntel, "Enumerating Video Modes...", pfontNeon);

    char cmode[64][16];
    unsigned int modeCount = 0, ex = 0, ey = 0;
    while(vod.enumModes(cmode[modeCount]) && modeCount < 64)
    {
		INT16 nParms = sscanf(cmode[modeCount], "%ux%u", &ex, &ey);
        if(nParms == 2 && ( ex <= 1024 && ey <= 768))
        {
            itmVidRez[modeCount].description = cmode[modeCount];
            itmVidRez[modeCount].callback = &changeRez;
            itmVidRez[modeCount].data = &cmode[modeCount][0];
            lstVideo.addItem(&itmVidRez[modeCount], true);
            modeCount++;
        }
    }
    menuVideo.addComponent(&lstVideo);

    // -------------- Credits menu construction
    menuCredits.addComponent(&imgBack);
    TextCtl   txtCreditsTitle(pfontGray, "Credits", 60, 4);
    menuCredits.addComponent(&txtCreditsTitle);
    TextCtl   txtCredits1(pfontBlue14, "Kyle Davis", 5, 34);
    TextCtl   txtCredits1a(pfontSmall, "(core program & design)", 15, 54);
    TextCtl   txtCredits2(pfontBlue14, "Jesse Litton", 5, 64);
    TextCtl   txtCredits2a(pfontSmall, "(programming/debugging)", 15, 84);
    TextCtl   txtCredits3(pfontBlue14, "Joseph Zbiciak", 5, 94);
    TextCtl   txtCredits3a(pfontSmall, "(arcane knowledge)", 15, 114);
    menuCredits.addComponent(&txtCredits1);
    menuCredits.addComponent(&txtCredits1a);
    menuCredits.addComponent(&txtCredits2);
    menuCredits.addComponent(&txtCredits2a);
    menuCredits.addComponent(&txtCredits3);
    menuCredits.addComponent(&txtCredits3a);
    
    bliss.setDefaultMenu(&menuMain);

	unsigned int menuatstartup = 0;
	if(ini.get("global", "menu_at_startup", &cLine[0], 2)) {
		int nParms = sscanf(cLine, "%u", &menuatstartup);
		if(nParms == 1 && menuatstartup)
			changeMenu(&menuMain);
	}

	unsigned int pauseonmenu = 0;
	if(ini.get("global", "pause_on_menu", &cLine[0], 2)) {
		int nParms = sscanf(cLine, "%u", &pauseonmenu);
		if(nParms == 1 && pauseonmenu)
			bliss.menuPauses();
	}
 


    // Load Cartridge
	unsigned int randomcart = 0;
    if(strlen(sCart) == 0) {
	    if(ini.get("global", "random_cart", &cLine[0], 2)) {
		    int nParms = sscanf(cLine, "%u", &randomcart);
	    }

	    if(((!ini.get("global", "cartridge", lastcart, _MAX_FNAME)) || randomcart)
                && cartridgeFileList.size()) {
            srand( cd.getTick() );
            int nCart = (int)((cartridgeFileList.size()-1)*((float)rand()/RAND_MAX));
            strcpy(lastcart, intvCartPath[nCart]);
        }
    }
    else {
        strcpy(lastcart, sCart);
    }

    char fskip[2] = "1";
    if(strlen(lastcart))
    {
		if(fileExists(lastcart)) {

			char statusMsg[MAX_PATH + 64];
			sprintf(statusMsg, "Loading %s", lastcart);
			updateSplash(plogoImage, pfontIntel, statusMsg, pfontNeon);


			cart = IntellivisionCartridge::loadCartridge(lastcart);
            inty.insertCartridge(cart);

            if(ini.get("global", "frameskip", fskip, 2)) {
                bliss.setFrameSkip(atoi(fskip));
            }

			if(ini.get("global", "splashtime", &cLine[0], 9)) {
				int nParms = sscanf(cLine, "%u", &splash);
			}
			INT64 finish  = start + (cd.getTickFrequency()*splash);

			while(INT64 time=cd.getTick() < finish)
			{
				updateSplash(plogoImage, pfontIntel, "Ready:  Press Space!", pfontNeon);

				id.poll();
				if(id.getControlValue(KEYBOARD_SPACE) ||
					id.getControlValue(KEYBOARD_ENTER) ||
					id.getControlValue(KEYBOARD_ESCAPE))
					break;
			}  


			// Reclaim video memory
			delete pfontIntel;
			delete pfontNeon;
			delete plogoImage;


			if(mode)
				vod.changeOutputSize(xw, yw);
			else
				vod.changeOutputSize(xf, yf);

			bliss.init(&vod, &aod, &id, &cd);
            bliss.setEmulator(&inty);
            bliss.run();

            delete cart;
        }
        else
				{
            char line[MAX_PATH + 64];
            sprintf(line, "Cartridge '%s' not found.\n\n Delete the 'lastcartridge' \
line from the 'bliss.INI' file, or replace the ROM to correct this error.", lastcart);
                vod.toggleScreen(MODE_WINDOWED);
                PopMessage(line, MSG_ERROR);
        }
    }
    else
		{
        // ZERO INTELLIVISION ROMS - CANNOT RUN ANYTHING
        vod.toggleScreen(MODE_WINDOWED);
		char line[MAX_PATH + 128];
		sprintf(line, "No ROMs found.  Does the ROM directory (%s) exist?\n\n\
This can be caused by failure to decompress the zip distribution \
with the option to rebuild paths.", romdir_intv);

        PopMessage(line, MSG_ERROR);
    }

    //release input, audio and video devices
    id.release();
    aod.release();
    vod.release();

	SAFE_DELETE(pfontGray);
	SAFE_DELETE(pfontSmallNeon);
	SAFE_DELETE(pfontSmall);
	SAFE_DELETE(pfontBlue14);
	SAFE_DELETE(pfontYellow14);
	SAFE_DELETE(pimageBack);

    // ---------------------------- Update INI file
	vod.getOutputSize(xf, yf, xw, yw);
	char line[32];
	sprintf(line, "%ux%u\0", xf, yf);
	ini.put("global", "fullscreen", line);
	sprintf(line, "%ux%u\0", xw, yw);
	ini.put("global", "windowed", line);
	sprintf(line, "%u", vod.windowed());
	ini.put("global", "mode", line);

	sprintf(line, "%u", menuatstartup);
	ini.put("global", "menu_at_startup", line);
	sprintf(line, "%u", pauseonmenu);
	ini.put("global", "pause_on_menu", line);

	sprintf(line, "%u", splash);
	ini.put("global", "splashtime", line);

    ini.put("global", "cartridge", lastcart);
	sprintf(line, "%u", randomcart);
	ini.put("global", "random_cart", line);

    ini.put("global", "frameskip", fskip);

    ini.put("intellivision", "rom_dir", romdir_intv);
    ini.put("5200", "rom_dir", romdir_5200);

    return 0;
}

INT32 Bliss32::loadSystemROMs()
{
    CHAR currentDir[MAX_PATH], nextFile[MAX_PATH];
    
    // Get program directory
    if(!getProgramDir(currentDir, MAX_PATH)) {
        vod.toggleScreen(MODE_WINDOWED);
        PopMessage("Could not identify the program directory.", MSG_ERROR);
        return 0;
    }

    // Verify that the executive rom file exists
    strcpy(nextFile, currentDir);
#ifdef _WIN32
    strcat(nextFile, "\\");
#else
    strcat(nextFile, "/");
#endif
    strcat(nextFile, FILENAME_EXECUTIVE_ROM);

    if(fileExists(nextFile)) {
        //verify the CRC of the executive rom file
        UINT32 crc = CRC32::getCrc(nextFile);
        if (crc != CRC_EXECUTIVE_ROM) {
            vod.toggleScreen(MODE_WINDOWED);
            PopMessage("File 'exec.bin' does not contain the correct Executive ROM.", MSG_ERROR);
            return 0;
        }
    }
    else {
        vod.toggleScreen(MODE_WINDOWED);
        PopMessage("File 'exec.bin' does not exist in the program directory. Read docs/readme.doc for information about this file.", MSG_ERROR);
        return 0;
    }

    // Load the executive rom file
    UINT16 size;
    ROM16Bit::loadROMImage(&execImage, &size, nextFile, TRUE);


    // Verify the grom file exists.
    //
    strcpy(nextFile, currentDir);
#ifdef _WIN32
    strcat(nextFile, "\\");
#else
    strcat(nextFile, "/");
#endif
    strcat(nextFile, FILENAME_GROM);
    if(fileExists(nextFile)) {
        // Verify the CRC of the grom rom file
        UINT32 crc = CRC32::getCrc(nextFile);
        if (crc != CRC_GROM) {
            vod.toggleScreen(MODE_WINDOWED);
            PopMessage("File 'grom.bin' does not contain the correct GROM.", MSG_ERROR);
            delete[] execImage;
            return 0;
        }
    }
    else {
        vod.toggleScreen(MODE_WINDOWED);
        PopMessage("File 'grom.bin' does not exist in the program directory. Read docs/readme.doc for information about this file.", MSG_ERROR);
        return 0;
    }

    //load the grom rom file
    ROM16Bit::loadROMImage(&gromImage, &size, nextFile, FALSE);

    strcpy(nextFile, currentDir);
#ifdef _WIN32
    strcat(nextFile, "\\");
#else
    strcat(nextFile, "/");
#endif
    strcat(nextFile, FILENAME_ECS_ROM);
    if(fileExists(nextFile)) {
        // verify the CRC of the ECS ROM
        UINT32 crc = CRC32::getCrc(nextFile);
        if (crc == CRC_ECS_ROM) {
            //load the ECS rom file
            ROM16Bit::loadROMImage(&ecsImage, &size, nextFile, TRUE);
        }
    }
    else {
        //No error, can still run non-ECS carts
    }

    strcpy(nextFile, currentDir);
#ifdef _WIN32
    strcat(nextFile, "\\");
#else
    strcat(nextFile, "/");
#endif
    strcat(nextFile, FILENAME_IVOICE_ROM);
    if(fileExists(nextFile)) {
        //verify the CRC of the Intellivoice rom file
        UINT32 crc = CRC32::getCrc(nextFile);
        if (crc == CRC_IVOICE_ROM) {
            //load the intellivoice ROM file
            ROM16Bit::loadROMImage(&ivoiceImage, &size, nextFile, FALSE);
        }
    }
    else {
        //No error, can still run Intellivoice carts without voice
    }

    return 1;
}

// getProgramDir:  Retrieve current program directory.
//                 The return value is the length.
//
INT32 Bliss32::getProgramDir(CHAR* sDir, UINT32 maxLen)
{
#ifdef _WIN32
    if(GetModuleFileName(NULL, sDir, maxLen)) {
        INT32 len = strlen(sDir);
        while (sDir[len] != '\\')
            len--;
        sDir[len+1] = 0;
        return len;
    }
#else
    INT32 src_len = strlen(programName);
    if(src_len > maxLen)
    {
        strncpy((char *)sDir, (char *)programName, maxLen);
    }
    else
    {
        strncpy((char *)sDir, (char *)programName, src_len);
    }
    INT32 len = strlen(sDir);
    while (sDir[len] != '/')
        len--;
    sDir[len+1] = 0;
    return len;
#endif

    return 0;
}


void Bliss32::clearCartList_Intv()
{
    //delete the current list
    INT32 size = cartridgeFileList.size();
    for (INT32 i = 0; i < size; i++) {
        intvCartPath[i][0] = '\0';
        delete cartridgeFileList[i];
    }
    cartridgeFileList.erase(cartridgeFileList.begin(),
		cartridgeFileList.end());
}
